

# Copyright 2014 2017 Intel Corporation.
#
# The source code,  information  and material  ("Material") contained herein is
# owned  by  Intel Corporation or its suppliers or licensors, and title to such
# Material remains  with Intel  Corporation or its  suppliers or licensors. The
# Material contains proprietary information of Intel or its suppliers and
# licensors.
# The  Material is  protected by  worldwide copyright laws and treaty provisions.
# No part  of the Material may  be used, copied, reproduced, modified, published,
# uploaded, posted,  transmitted,  distributed  or  disclosed in any way without
# Intel's prior express written permission.
# No license under any patent, copyright  or  other intellectual property rights
# in  the  Material is  granted to  or conferred  upon you, either expressly, by
# implication, inducement, estoppel or otherwise.
# Any  license under such  intellectual  property  rights  must  be  express and
# approved by Intel in writing.



"""
This is an example script for creating components in code and then
writing them out as code or XML
"""
from __future__ import absolute_import
from __future__ import division
from __future__ import print_function

import os

from namednodes.comp import CreateComponentDefinition
from namednodes.nodes import (
                    NamedNodeDefinition,
                    NamedNodeArrayDefinition,
                    NamedNodeArrayItemDefinition
                    )
from namednodes.registers import RegisterDefinition, FieldDefinition, RegisterComponent
from namednodes import GetComponentFromFile

# plugins needed for the static_xml writer to work
from namednodes.plugins import lmdb_dict
# function for writing the built up component out to xml
from namednodes.discoveries import static

_static_out_dir = "static_example"
_static_out_file = "static_socket.xml"
_LMDB_FILE = "socket.lmdb"

try:
    _THIS_DIR = os.path.dirname(__file__)
except NameError:  # happens during debug
    _THIS_DIR = os.getcwd()


def build_component_definition():
    """Build up an example component that we will save to file"""

    ###############################################################
    # Create ComponentDefinitions that will hold certain registers
    ###############################################################
    socket  = CreateComponentDefinition( "socket" )
    core = CreateComponentDefinition( "core" , parent = socket)
    thread = CreateComponentDefinition( "thread" , parent = socket.core)
    uncore = CreateComponentDefinition( "uncore" , parent = socket, value_class=RegisterComponent)

    #############################################
    # uarch is used to hold internal structions
    #############################################
    uarch = CreateComponentDefinition( "uarch", parent = socket )


    ########################## Registers and Fields ##########################
    # now we have components, lets add some nodes to them
    # ...for backwards compatibility, we typically use RegisterDefinition
    # and FieldDefinition objects
    ########################## Registers and Fields ##########################
    mcg_cap = RegisterDefinition("ia32_mcg_cap",
                                 dict(default=0,
                                      description="machine check capabilities register")
                                 )

    # normally we would add "MSR" as the access type, but this is just for demo purposes
    # so we will use a 'constant' value...
    # AccessConstant - must be imported before it is actually used, but here we are
    # not going to use it, just define that the register _will_ use it, when accessed
    mcg_cap.add_access('constant', "AccessRegisterStoredValues")
    # reading this register will default to being 0x15, again - this is not going to use an
    # actual MSR read...but if it where, then we would set the address using something like:
    # mcg_cap.info['msr_offset'] = 0x179
    mcg_cap.info['default'] = 0xa

    # now...lets add some fields to this register...
    info = dict(upperbit=7,
                lowerbit=0,
                description="number of machine-check banks")
    mc_count = FieldDefinition("count",info=info)
    info=dict(upperbit=8,
              lowerbit=8,
              description="the processor implements mcg_ctl")
    mcg_ctl_p = FieldDefinition("mcg_ctl_p",info=info)
    # add fields to the register
    mcg_cap.add_node( mc_count )
    mcg_cap.add_node( mcg_ctl_p )
    # Finally, lets add our register to the thread, so that all thread objects
    # will have this register
    thread.add_node( mcg_cap )

    ####################### Add an uncore register ######################


    ######################################################################

    ########################## General Nodes #############################
    # if we are adding non-register state, then we can use the more general
    # nodes to build arrays or nodes that may have non-numeric values
    # general nodes
    ######################################################################
    voltage  = NamedNodeDefinition("voltage",info=dict(
                                                    constant_dict={},
                                                    default=1.0,
                                                    value_type="float")
                                   )
    voltage.add_access('constant',"AccessStoredValues")
    uncore.add_node( voltage )

    ############################################################
    # To demonstrate how array's work, we are going to need
    # a custom access, our access will just return: set*16 + way
    ############################################################
    # default for state node reads should use StateNodeAccess or StateNodeAccessNoLock
    # so that they read through children until a read actually occurrs, we use the nolock
    # so that we don't require actual target interaction
    # you end up with database that looks sort of like:
    # cache.set.set_item.way.way_item
    cache     = NamedNodeDefinition("cache", accesses={'state':'StateNodeAccessNoLock'})
    cset_item = NamedNodeArrayItemDefinition("set", accesses={'state':'StateNodeAccessNoLock'})
    cset       = NamedNodeArrayDefinition("set",
                                          item_keys=list(range(10)),
                                          item_definition=cset_item,
                                          accesses={'state':'StateNodeAccessNoLock'})
    # before we can use this cache.set, we will need to eventually write a
    # StateSetWayAccess class to perform the actual access, but that can be done
    # on the usage side
    cway_item = NamedNodeArrayItemDefinition("way", accesses={'state':'AccessSetWay'})
    cway       = NamedNodeArrayDefinition("way",
                                          item_keys=list(range(3)),
                                          item_definition=cway_item,
                                          accesses={'state':'StateNodeAccessNoLock'})

    cdata = FieldDefinition("data",info=dict(upperbit=16,lowerbit=0))
    cway_item.add_node( cdata )
    cset_item.add_node( cway )
    cache.add_node( cset )
    uarch.add_node( cache )


    ####################### For Enum Example ##############################
    ### Here we put some placeholder registers that we will later add enums to using
    ### the enum plugin
    pcie_header = RegisterDefinition(
            "pcie_header",
            info=dict(constant_dict={}, default=0xf, numbits=32)
        )
    pcie_header.add_access('constant', "AccessRegisterStoredValues")
    uncore.add_node(pcie_header)
    mycount = FieldDefinition("count",
        dict(upperbit=8, lowerbit=0,description="header counts"), parent=pcie_header)
    lookup = FieldDefinition("lookup",
        dict(upperbit=16, lowerbit=8, description="header lookup"), parent=pcie_header)
    # we could have added enums using functions here, but instead we will use the xml for this
    # particular example
    socket.enums.load_xml(os.path.join(_THIS_DIR, "example_enum.xml"),format='nanoscope')


    ####################### Register Grouping ##############################
    # reading and writing to various registers at the same time is possible
    # via register grouping, it is possible to group registers inside a
    # component or to create
    ########################################################################
    # in order to show how the grouping of registers in a component works
    # we are going to add various registers to create a group
    num_registers = 3
    for rn in range(num_registers):
        vmd_data = RegisterDefinition(
            "vmd%d_pmsixdata"%rn,
            info=dict(constant_dict={}, default=0xf, numbits=16)
        )
        vmd_data.add_access('constant',"AccessStoredValues")
        uncore.add_node(vmd_data)

    ##########################################################################
    # Finished. Return the socket definition that we created
    ##########################################################################
    return socket

def build_component( component_definition ):

    # simple rename to shorten things up for later in the code
    socketdef = component_definition

    ##########################################################################
    # For folks that need to build parts of the component based on some
    # ipc.devicelist info, you would use something like this flow
    ##########################################################################
    example_socket = component_definition.create("socket0")
    # add some cores/threads
    numcores = 4
    numthreads = 2
    for c in range(numcores):
        newcore = socketdef.core.create("core%d"%c)
        example_socket.add_component( newcore )
        for t in range(numthreads):
            newcore.add_component( socketdef.core.thread.create("thread%d"%t) )

    ##########################################################################
    # for folks whose 'in system' matches the database exactly, you can
    # use the create_tree function that is a 1:1 between the definition
    # and the actual target
    ##########################################################################
    example_socket.add_component( socketdef.uncore.create_tree(), "uncore" )
    example_socket.add_component( socketdef.uarch.create_tree(), "uncore_uarch" )

    ##########################################################################
    # This socket is built and pretty much ready for use. It is just missing
    # things like the "did" for the actual access mechanisms. We will fill that
    # in later during the actual discovery.
    ##########################################################################
    return example_socket

def get_out_path(filename):
    """
    simply return the full path to where-ever we are putting our output files
    """
    outdir = os.path.join( _THIS_DIR, _static_out_dir)
    if not os.path.exists(outdir):
        os.mkdir(outdir)
    return os.path.join(outdir,filename)

def build_all():
    ##########################################################################
    # FIRST, we build up our component definition, typically, this is
    # done by some collateral conversion utility like svtools.c3po
    ##########################################################################
    example_socket_definition = build_component_definition()
    #########################################################################
    # Write our database out to the file, and load back from the file
    # so that we can save references to it
    #########################################################################
    outlmdb = get_out_path(_LMDB_FILE)
    example_socket_definition.tolmdb.write(outlmdb)
    example_socket_definition = GetComponentFromFile(outlmdb)
    ##########################################################################
    # Now we take the definition and we create actual components
    # this is where you may have the same components in your data file
    # or you may need to tweak based on some parameters
    ##########################################################################
    example_socket = build_component( example_socket_definition )
    #########################################################################
    # Write our hard coded system representation out to a file
    #########################################################################
    outxml = get_out_path( _static_out_file )
    static.write( [ example_socket ], outxml)
    return example_socket

# this gets run in regression tests to make sure that this can be run
# successfully
def run_examples():
    build_all()

if __name__ == "__main__":
    # if this script is run, then generate the static xml
    run_examples()




