
# INTEL CONFIDENTIAL
# Copyright 2014 2018 Intel Corporation
#
# The source code  contained or  described herein and  all documents related to
# the source code  ("Material") are owned by Intel Corporation or its suppliers
# or licensors.  Title to the  Material  remains with  Intel Corporation or its
# suppliers  and licensors. The Material contains trade secrets and proprietary
# and  confidential  information  of  Intel  or  its  suppliers  and  licensors.
# The Material  is protected  by worldwide  copyright and trade secret laws and
# treaty provisions.  No part of the Material  may  be used, copied, reproduced,
# modified, published, uploaded, posted, transmitted, distributed, or disclosed
# in any way without Intel's prior express written permission. No license under
# any  patent,  copyright, trade secret or other intellectual property right is
# granted  to  or conferred upon you by disclosure or delivery of the Materials,
# either expressly, by implication, inducement, estoppel or otherwise.
# Any license under such intellectual property rights must be express and
# approved by Intel in writing.




"""
Module implementing accesses events functionality.
"""

from __future__ import absolute_import
from __future__ import division
from __future__ import print_function

from ._manager import EventManager, Event
from ..utils._py2to3 import *

# initialize AccessEventManager
AccessEventManager = EventManager()

# Wrappers for hiding event manager from users.
subscribe = AccessEventManager.subscribe
unsubscribe = AccessEventManager.unsubscribe
send_event = AccessEventManager.send_event


class AccessEventTypes(object):
    """
    Storage for the different access event types.
    """
    _pre_bitmask = 0x100
    _post_bitmask = 0x200
    #: called before read occurrs
    pre_read = _pre_bitmask | 0x1
    #: called after read occurrs
    post_read = _post_bitmask | 0x1
    #: called before write occurrs
    pre_write = _pre_bitmask | 0x2
    #: called after write occurrs
    post_write = _post_bitmask | 0x2
    #: called before a read-modify-write,
    pre_read_write = _pre_bitmask | 0x3
    #: called after a read-modify-write,
    post_read_write = _post_bitmask | 0x3
    #: called before a store
    pre_store = _pre_bitmask | 0x4
    #: called after a store
    post_store = _post_bitmask | 0x4
    #: called before a flush
    pre_flush = _pre_bitmask | 0x5
    #: called after a flush
    post_flush = _post_bitmask | 0x5
    #: called before an update_value (not always the same as read)
    pre_update_value = _pre_bitmask | 0x6
    #: called after an update_value (not always the same as read)
    post_update_value = _post_bitmask | 0x6

    # used to store the pre event types
    pre_event_types = [
        pre_read,
        pre_write,
        pre_read_write,
        pre_store,
        pre_flush,
        pre_update_value,
        ]

    # used to store the post event types
    post_event_types = [
        post_read,
        post_write,
        post_read_write,
        post_store,
        post_flush,
        post_update_value,
        ]

    # used for caching reverse lookup
    _reverse = None

    @classmethod
    def get_str_value(cls, value):
        """
        Given the access event type, find out its name.
        """
        if cls._reverse is None:
            cls._reverse = {}
            for k, v in cls.__dict__.items():
                if isinstance(v, (int, long, basestring)):
                    cls._reverse[v] = k
        return cls._reverse.get(value, "Unknown")


class AccessEvent(Event):
    """
    Implementation of an access event. AccessEvents are passed
    in to any handler registered with the subscribe function
    """

    def __init__(self, event_type, node, method_args, method_kwargs):
        """
        Constructor.

        Args:
            event_type (AccessEventTypes.*): The event type.
            node (NamedNodeValue): The node that triggered the event.
        """
        Event.__init__(self, event_type, None)

        # used to store the node that triggered this event
        self._node = node

        # used to store the *args and **kwargs passed on to the
        # read, write, read_write, store and flush access methods.
        # this will be useful when implementing the log functionality
        # for backwards compatibility with PythonSv.
        self._event_data = {
            'args': method_args,
            'kwargs': method_kwargs
            }

    @property
    def event_type_str(self):
        """
        Get the string representation of the event type.
        """
        if not self._event_type_str:
            self._event_type_str = (
                AccessEventTypes.get_str_value(self._event_type)
            )
        return self._event_type_str

    @property
    def node(self):
        """
        Get the node that the access related to
        """
        return self._node

    @property
    def event_data(self):
        """
        Get parameters passed on to the access method that raised the event.

        The returned object is a dictionary with the following structure:

        event_data = {
            'args': [],
            'kwargs': {}
            }
        """
        return self._event_data
