
# INTEL CONFIDENTIAL
# Copyright 2014 2018 Intel Corporation
#
# The source code  contained or  described herein and  all documents related to
# the source code  ("Material") are owned by Intel Corporation or its suppliers
# or licensors.  Title to the  Material  remains with  Intel Corporation or its
# suppliers  and licensors. The Material contains trade secrets and proprietary
# and  confidential  information  of  Intel  or  its  suppliers  and  licensors.
# The Material  is protected  by worldwide  copyright and trade secret laws and
# treaty provisions.  No part of the Material  may  be used, copied, reproduced,
# modified, published, uploaded, posted, transmitted, distributed, or disclosed
# in any way without Intel's prior express written permission. No license under
# any  patent,  copyright, trade secret or other intellectual property right is
# granted  to  or conferred upon you by disclosure or delivery of the Materials,
# either expressly, by implication, inducement, estoppel or otherwise.
# Any license under such intellectual property rights must be express and
# approved by Intel in writing.




"""
Event manager for namednodes.
"""
from __future__ import absolute_import
from __future__ import division
from __future__ import print_function

from ..logging import getLogger
from namednodes.utils.ordereddict import odict

_LOG = getLogger()


class EventManager(object):
    """
    Event manager.
    """
    _event_handlers = odict()

    def subscribe(self, handler, user_info=None):
        """
        Subscribe a handler to be notified when events are triggered.

        Args:
            handler (function): The function to call when an event is
                triggered.
            user_info (object): Might be needed for when we have the C API.
        """
        self._event_handlers[handler] = user_info

    def unsubscribe(self, handler):
        """
        Remove handler.

        Args:
            handler (function): The function to remove from handlers list.
        """
        if handler in self._event_handlers:
            del self._event_handlers[handler]

    def send_event(self, event):
        """
        Notify registered handlers that an event has been triggered.
        """
        for handler, user_info in self._event_handlers.items():
            try:
                if user_info is None:
                    handler(event)
                else:
                    handler(event, user_info)
            except Exception:
                _LOG.exception("Unable to notify event '{0}' to handler '{1}'".format(event.event_type, handler))


class Event(object):
    """
    Base class for event objects which are passed to the EventManager.send_event() method
    """

    def __init__(self, event_type, event_type_str):
        """
        Constructor.

        Args:
            event_type: The event type.
        """
        # used to store the event type
        self._event_type = event_type

        # used to store the string representation of the event type
        self._event_type_str = event_type_str

    @property
    def event_type(self):
        """
        Get the event type.
        """
        return self._event_type

    @property
    def event_type_str(self):
        """
        Get the string representation of the event type.
        """
        return self._event_type_str
