
# INTEL CONFIDENTIAL
# Copyright 2014 2018 Intel Corporation
#
# The source code  contained or  described herein and  all documents related to
# the source code  ("Material") are owned by Intel Corporation or its suppliers
# or licensors.  Title to the  Material  remains with  Intel Corporation or its
# suppliers  and licensors. The Material contains trade secrets and proprietary
# and  confidential  information  of  Intel  or  its  suppliers  and  licensors.
# The Material  is protected  by worldwide  copyright and trade secret laws and
# treaty provisions.  No part of the Material  may  be used, copied, reproduced,
# modified, published, uploaded, posted, transmitted, distributed, or disclosed
# in any way without Intel's prior express written permission. No license under
# any  patent,  copyright, trade secret or other intellectual property right is
# granted  to  or conferred upon you by disclosure or delivery of the Materials,
# either expressly, by implication, inducement, estoppel or otherwise.
# Any license under such intellectual property rights must be express and
# approved by Intel in writing.




from __future__ import absolute_import
from __future__ import division
from __future__ import print_function

from .logging import getLogger
from .events.discovery import subscribe, send_event, DiscoveryEventTypes as Events
from .utils._py2to3 import *
from .utils import PluginManager
from .utils.ordereddict import odict
from .comp import ComponentGroup
from .telemetry import record_discovery_event

from . import settings

from collections import OrderedDict
from copy import copy
import importlib
import warnings
import os
import sys
import types
import traceback
import six
import time
from xml.etree import cElementTree as ET

_LOG = getLogger()


class DiscoveryPlugins(PluginManager):
    basetype = "Discovery"
    _plugins = {} # important to have plugins local to this class

@with_metaclass(DiscoveryPlugins)
class Discovery(object):
    """
    A Discovery object knows to *find* objects and return the NamedComponents
    that represent them
    """
    #: This is for documenting and reporting what the discovery returns
    _name = None
    #: it is perfectly legal for another discovery to change these two variables
    #: that way the getall/findall will affect multiple discovery objects
    discovered = None
    discovered_dict = None

    def __init__(self, name=None):
        self._name = name
        grouppath = "" if self.name is None else self.name+"s"
        self.discovered = ComponentGroup(name = grouppath, grouppath=grouppath)
        self.discovered_dict = {}

    @property
    def name(self):
        return self._name

    @name.setter
    def name(self, value):
        # remove old name if we have that attribute
        if self._name in self.__dict__:
            del self.__dict__[self._name]
        self._name = value
        grouppath = value + "s"
        self.discovered.set_grouppath( grouppath )
        self.discovered.set_groupname( grouppath )
        # add new attribute
        setattr(self, grouppath, self.discovered)

    @classmethod
    def create(cls,obj=None):
        # not used for discovery, but needed to match other plugins behavior
        return cls()

    def getbypath(self, *args, **kwargs):
        """for compatibility with previous pythonsv generations, see get_by_path for help"""
        return self.get_by_path(*args, **kwargs)

    def get_by_path(self, path, **kwargs):
        """
        return the node or component based on the path provided

        Args:
            path (str,list) : Path is a "." delimitted string or a list, that
                              is used to traverse the nodes and components to
                              return whatever object the path points to
            default (optional, bool) : if path is not found return this instead
            missing (optional, bool) : if missing return list of [foundpart, path_left]
        Returns:
            returns either a node or component depending on what path is

        Raises:
            ValueError if the path can not be found
        """
        # doc string copy/pasted from component.get_by_path
        default = kwargs.pop("default", KeyError)
        path_parts = path.split(".")
        comp_name = path_parts[0]
        if comp_name not in self.discovered_dict:
            if default is KeyError:
                raise ValueError("Unknown path: %s" % path)
            else:
                return default
        return self.discovered_dict[comp_name].get_by_path(path)

    def get_all(self, refresh=False, **kwargs):
        """
        Return any components that have already been created through
        discovery, or run the discovery functions to find and create the
        components

        Args:
            refresh (bool) : can be used to force rediscovery of the components
            stop_on_error (false) : can be used to force exceptions during find_all to be raised

        kwargs will be passed along to the discovery code after any arguments
        for this functions have been removed.
        """
        stop_on_error = kwargs.pop("stop_on_error", settings.GET_ALL_STOP_ON_ERROR)
        # if we already have list and refresh not reqeusted, return it..
        if not refresh and len(self.discovered) != 0:
            return self.discovered
        elif refresh:
            # refresh requested, and we have a list...empty it
            self.clear_known_items()

        # OK, now we have an empty list due to refresh, or nothing to begin
        # with, go do some discovery

        # Trigger a pre_find_all event for the discovery event manager
        if settings.EVENTS_ON_DISCOVERY:
            event_data = {'item': self.name, 'refresh': refresh}
            send_event(Events.pre_find_all(event_data))

        try:
            discovered_tmp = self.find_all(**kwargs)
            if discovered_tmp is None:
                raise RuntimeError("find_all should have returned a list, but it did not:\n\t%s"%self.find_all)
        except Exception as e:
            if stop_on_error:
                _LOG.info("Error during discovery", exc_info=True)
                raise
            else:
                # need to run first to create the filename
                _LOG.info('%s find_all error' % self.name, exc_info=True)
                msg = "ERROR! Skipping '%s', error hit during discovery, specify: stop_on_error=True to see traceback\n" % self.name
                # this *should* always be set, but seems there are some cases where we can end up with logging completely turned off
                if(_LOG._root_filehandler):
                    msg += "ERROR! When emailing the project enabler about this error, be sure to email them:\n"
                    msg += "  "+str(_LOG._root_filehandler.baseFilename)
                _LOG.error(msg)
                discovered_tmp = []

        # add to our list and dictionary
        for d in discovered_tmp:
            self.add_discovered(d)

        # Trigger a post_find_all event for the discovery event manager
        if settings.EVENTS_ON_DISCOVERY:
            event_data = {'item': self.name, 'items_found': copy(self.discovered_dict), 'refresh': refresh}
            send_event(Events.post_find_all(event_data))

        return self.discovered

    def add_discovered(self, discovered):
        # move from the findall to the discovered list/group
        self.discovered.append(discovered)
        if getattr(discovered,"name",None) != None:
            self.discovered_dict[discovered.name] = discovered
        # force saving a reference to this discovery object to help with various debug
        discovered.__dict__['_discovery'] = self
        # indeed sometimes the discovery can return something other than namednodes components
        tinfo = getattr(discovered,"target_info", None)
        if tinfo:
            tinfo['_discovery_name'] = self.name

    def clear_known_items(self):
        while len(self.discovered)>0:
            self.discovered.pop()
        # simply wipe out dictionary vs. saving it for now
        self.discovered_dict.clear()

    def refresh(self, **kwargs):
        return self.get_all(refresh=True, **kwargs)

    def find_all(self, *args, **kwargs):
        """
        Must be implemented to return the list of components for this discovery
        """
        raise Exception("Must be implemented")

    def __getattr__(self, attr):
        if attr in self.discovered_dict:
            return self.discovered_dict[attr]
        else:
            return object.__getattribute__(self, attr)

    def __dir__(self):
        """
        default dir of a discovery is to show the names of the things discovered
        """

        dirlist = list(self.__dict__.keys())
        dirlist.extend( dir(self.__class__) )
        for i in self.discovered:
            dirlist.append( i.name )
        return dirlist

class _InitializeStages:
    not_started = 0
    in_progress = 1
    complete = 2

class DiscoveryManager(object):
    """
    A singleton for managing the various discovery items and project plugin information
    """
    #: For returning the *global* discovery manager
    _manager = None
    #: Internal project variable so that we can track if project gets changed
    _project = None
    #: Internal list of known projects and their initialization file, share
    #: amongst all discovery manager classes
    _projects = None
    #: also shared across all discovery managers so that any change to one
    #: will be reflected in all
    _all_discoveries = None
    # to track whether we have initialized at least once
    # this has three stages (for now)
    _initialized = _InitializeStages.not_started

    def __init__(self):
        """this should not be called directly"""
        # if manager not present, clear out all the class variable
        self._projects = odict()
        self._all_discoveries = odict()
        self._project = None
        # load discovery xml at this point and be reading to initialize
        self.load_discovery_xml()

    @classmethod
    def get_manager(cls, discovery_names=None):
        """
        Return the primary DiscoveryManager object or a subset if that is requested
        """
        # dont let the global get_manager function limit the items...
        if cls._manager is None:
            cls._manager = cls()
        if discovery_names is None:
            return cls._manager
        else:
            return DiscoveryManagerSubset( discovery_names )

    def initialize(self, refresh=False):
        """
        Used to initialize the discovery manager and load plugins

        Args:
            refresh (bool) : force initialization even if we have loaded a project
                             already
        """
        # if inside sphinx dont run initialization
        if 'DOCUTILSCONFIG' in os.environ:
            self._initialized = _InitializeStages.complete
            return
        # this is to prevent us from coming back through 
        # here and getting in to a recusive loop due to __getattr__ overrides
        if sys.version_info[:2] == [2, 6]:
            print("!!!!! Python2.6 not supported !!!!")
        if not (sys.maxsize>2**32):
            print("!!!!! You must use 64bit Python !!!!")

        if not self._initialized:
            subscribe(record_discovery_event)

        if not self._initialized or refresh is True:
            self._initialized = _InitializeStages.in_progress
            # Trigger a pre_initialize event for the discovery event manager
            if settings.EVENTS_ON_DISCOVERY:
                event_data = {'refresh': refresh}
                send_event(Events.pre_initialize(event_data))

            self._project = None
            # wipe out existing discoveries
            self._all_discoveries.clear()
            DiscoveryPlugins._plugins.clear()
            self._load_project_discoveries()
            self._initialized = _InitializeStages.complete

            # Finished initialize; trigger a post_initialize event for the discovery event manager
            if settings.EVENTS_ON_DISCOVERY:
                event_data = {'items': copy(self._all_discoveries), 'project': self._project,
                              'refresh': refresh}
                send_event(Events.post_initialize(event_data))

    @property
    def discoveries(self):
        try:        
            # make sure our project is up-to-date
            if self._project is None:
                self.initialize()
            # return shallow copy so they can't change on us
            return copy(self._all_discoveries)
        except AttributeError as e:
            # turn it in to a runtime error so that it will propagate up
            raise RuntimeError(e)

    @property
    def project(self):
        """return the current project, or determine the project if it is unknown"""
        # only load the project if we haven't already
        if self._project is None:
            #####################
            # Check settings for how to determine project
            #####################
            #
            # no project to load
            #
            if settings.PROJECT == None:
                return None
            #
            # we need to use pysv config to load project
            #
            if settings.PROJECT == "__PYSVCONFIG__":
                # try to get project from the PYSVCONFIG
                try:
                    from svtools.common.pysv_config import CFG as pysvCFG
                except ImportError:
                    _LOG.debug("__PYSVCONFIG__ requires 'svtools.common.pysv_config' to be present")
                    return
                # baseaccess will ALWAYS be in CFG
                if 'project' not in pysvCFG.baseaccess:
                    _LOG.debug("pysv_config has not been run and settings.PROJECT is not set")
                    return
                self._project = pysvCFG.baseaccess.project
                if isinstance(self._project, basestring) and self._project.strip().lower() == "none":
                    self._project = None
            #
            # Very simple, just use the project settings
            #
            else:
                self._project = settings.PROJECT
        # project has been initialized at this point, return it
        return self._project

    def load_discovery_xml(self):
        """this is loading of discovery xml to create our discovery what objects
        """
        # open our discovery XML
        thisdir = os.path.dirname(__file__)
        config_xmlpath = os.path.join( thisdir, "config", "Discovery.xml")
        if not os.path.exists(config_xmlpath):
            return  # discovery XML not present
        tree = ET.parse( config_xmlpath )
        root = tree.getroot()
        for pnode in root.findall("Project"):
            pname = pnode.get("name")
            assert isinstance(pname, basestring),"project name missing in XML"
            pname = pname.strip()
            import_path = pnode.get("plugins")
            assert isinstance(pname,basestring),(
                    "Project {0} missing from plugin node in XML".format(pname))
            # add the specified path to the list of known projects
            self.add_project(pname, import_path)

    def add_project(self, projectname, plugin_path):
        """
        Add a supported project and specify path to what to import
        if this project is set in the settings
        """
        self._projects[projectname] = plugin_path

    def refresh(self, **kwargs):
        """Refresh all known items that this manager can discover"""
        return self.get_all(True, **kwargs)

    def get_all(self, refresh=False, **kwargs):
        """
            Args:
                refresh (bool) : can be used to force rediscovery of the components
        """
        # reset list of known items for refresh
        for dname,discovery in self.discoveries.items():
            discovery.get_all(refresh, **kwargs)

    def _load_project_discoveries(self):
        """
        load any plugins that we need to based on the project name
        """
        # if project is still None, nothing we can do...
        if self.project is None:
            return
        # reset our known discoveries when we load a project
        self._all_discoveries = odict()
        # if first time, then load plugins...
        # if first time, then load plugins...
        default_import_path = "{0}.toolext.namednodes:init".format(self.project)
        import_path = self._projects.get(self.project, KeyError)
        # no import path in the file, we will need to try the defaults
        if import_path is KeyError:
            _LOG.debug("Project %s does not have a path for loading plugins"%self.project)
            # make sure to prioritize package over the one in SVN....
            check_for = [
                "pysvext.{0}.namednodes:init".format(self.project),
                 "{0}.toolext.namednodes:init".format(self.project),
            ]
        elif import_path is None:
            # project is a known project and specifically has None for its known plugins
            return
        else:
            # project not in the list of known projects
            check_for = [import_path]

        loaded = False
        # save this before our imports since each may add something...or in a case of
        # migration from toolext to pysvext, it may be that both add the same thing
        errs = []
        for check_path in check_for:
            if check_path.count(":") != 1:
                raise ValueError("ERROR: Invalid import path (%),\nERROR: must specify module:function"%check_path)
            module_path, import_func_name = check_path.split(":")
            try:
                plugins = importlib.import_module( module_path )
                loaded = True
            except ImportError as ierr:
                errs.append(traceback.format_exc())
                # log the exception for debug purposes
                _LOG.debug("-- ImportError on %s, this may be ok if other import paths succeed"%module_path)
                _LOG.debug("%s"%traceback.format_exc())
                _LOG.debug("-- ")
                # import error had nothing to do with tool ext or pysvext or namednodes
                if (str(ierr).count("toolext") == 0 and
                            str(ierr).count("pysvext") == 0 and
                            str(ierr).count(self.project)==0 and
                            str(ierr).count("namednodes")==0):
                    # module was there, but it did not import properly due to some other error
                    _LOG.error("ERROR trying to load class plugins '{0}' for project {1}".
                               format(module_path, self._project))
                else:
                    # expected import error when the project doesn't define this...
                    continue
            except:
                errs.append(traceback.format_exc())
                # module was there, but it did not import properly due to some other error
                _LOG.debug("-- Exception importing %s"%module_path)
                _LOG.debug("%s"%traceback.format_exc())
                _LOG.debug("-- ")
                _LOG.error( "ERROR trying to load class plugins '{0}' for project {1}".
                    format(module_path, self._project))
            else:
                import_func = getattr(plugins, import_func_name, None)
                if import_func is None:
                    _LOG.error("'{}' function is missing from '{}' and must be present even if it does nothing".format(
                        import_func_name,
                        module_path,
                    ))
                    time.sleep(4)
                else:
                    import_func()

        if loaded == False:
            # must have been in the XML
            if len(check_for)==1:
                _LOG.error( "ERROR trying to load class plugins '{0}' for project {1}".
                    format(module_path,self._project))
                raise ImportError(module_path)
            else:
                # show error for each one that failed...
                _LOG.error( "ERROR trying to load class plugins for project {1}")
                for c, check_path in enumerate(check_for):
                    _LOG.error(errs[c])
                    _LOG.error( "ERROR could not find: %s"%check_path)
                raise ImportError(str(check_for))

        # this section of code will go away once we have killed the DiscoveryPlugin manager
        for name in DiscoveryPlugins.plugins():
            # don't override anything that was explicitly registered
            if name is not None and name not in self._all_discoveries:
                disc_object = DiscoveryPlugins.get(name, None)
                if disc_object is not None:
                    self._register(name, disc_object)

    def _register(self, name, discovery):
        """
        Register the specified discovery.

        Args:
            name (str) : name of discovery to register
            discovery (obj) : discovery object to register

        for now this is not public since we only do this while/after project is known
        """
        # check first, only update if we need to
        # make sure the two match
        discovery.name = name
        self._all_discoveries[name] = discovery

    def _unregister(self, name):
        """remove the specified discovery from our known discoveries dictionary"""
        del DiscoveryManager.get_manager()._all_discoveries[name]

    def __getattr__(self,attr):
        # if we have not run initialization, run it...we have to put special
        # try/except to make sure traceback properly propogates
        if self._initialized == _InitializeStages.not_started:
            try:
                self.initialize()
            except Exception as e:
                # we have to switch to runtime error to make sure exceptions
                # loading discoveries get bubbled up
                _LOG.exception("exception during getattr, so this this will likely get hidden...")
                raise RuntimeError(e)
        # ok, proceed with checking for discoveries
        if attr in self._all_discoveries:
            return self._all_discoveries[attr]
        # check if any of our discoveries have found this
        for disc in self._all_discoveries.values():
            if attr in disc.discovered_dict:
                return disc.discovered_dict[attr]
            elif attr == disc.discovered.grouppath:
                return disc.discovered
        # must be class attribute or something else:
        return object.__getattribute__(self,attr)

    def __dir__(self, supported=None):
        """supported is a way of checking only certain discovery classes"""
        if supported is None:
            supported = self._all_discoveries.keys()
        mydir = []
        # first show the list of discoveries here
        mydir.extend( supported )
        for disc in supported:
            disc_item = self._all_discoveries.get(disc, None)
            if disc_item is None:
                print("WARNING: %s requested but is not a known discovery type"%disc)
                continue
            # now show the group names
            if disc_item.discovered.grouppath is not None:
                mydir.append( disc_item.discovered.grouppath )
            # now show each item
            names = [d.name for d in disc_item.discovered]
            mydir.extend(names)
        if settings.HIDE_METHODS is False:
            # even here...lets only show a few things
            mydir.extend(['project', 'get_all', 'refresh'])
        return mydir

    def get_by_path(self, path, **kwargs):
        """
                return the node or component based on the path provided

        Args:
            path (str,list) : Path is a "." delimitted string or a list, that
                              is used to traverse the nodes and components to
                              return whatever object the path points to
            default (optional, bool) : if path is not found return this instead
        Returns:
            returns either a node or component depending on what path is

        Raises:
            ValueError if the path can not be found
        """
        assert isinstance(path, six.string_types), "Must be a string"
        default = kwargs.pop("default", KeyError)
        # if we have no known items, call get_all to attempt to get items
        # what check here to know when to call get_all ?
        path_parts = path.split(".")
        if len(path_parts)==0:
            raise ValueError("Invalid path given: %s"%path)
        # if here, then it is supported, see if we have it...
        comp = getattr(self, path_parts[0], None)
        if comp is None:
            # nothing found return default or raise error
            if default is not KeyError:
                return default
            raise ValueError("Unknown path: '%s'" % path)
        # found it, call next level of get by path
        if len(path_parts)==1:
            return comp
        else:
            # use known item and call its get_by_path
            next_path = ".".join(path_parts[1:])
            return comp.get_by_path(next_path, default=default)

class DiscoveryManagerSubset(object):
    """for representing a subset of discovery items"""
    #: for tracking subset of discoveries vs. the global manager
    _approved_discoveries = None

    def __init__(self, discovery_names):
        # default is an empty project
        if not isinstance(discovery_names,list):
            raise Exception("Must be a list or of items")
        self._approved_discoveries = set(discovery_names)

    def refresh(self, **kwargs):
        """Refresh all known items that this manager can discover"""
        return self.get_all(True, **kwargs)

    def get_all(self, refresh=False, **kwargs):
        """get all the items that this subset supports"""
        # make sure top level is initialized
        disc_manager = DiscoveryManager.get_manager()
        if disc_manager._initialized == _InitializeStages.not_started:
            try:
                disc_manager.initialize()
            except Exception as e:
                # we have to switch to runtime error to make sure exceptions
                # loading discoveries get bubbled up
                _LOG.exception("exception during getattr, so this this might get hidden...")
                raise RuntimeError(e)

        # tell main disc_manager to do get our approved subset
        for dname in self._approved_discoveries:
            disc_obj = disc_manager.discoveries.get(dname, None)
            if disc_obj is not None:
                disc_obj.get_all(refresh, **kwargs)
            # else: dont break anything here due to a typo on what should be discoverable

    @property
    def project(self):
        """return project that is being used by discovery manager"""
        return DiscoveryManager.get_manager().project

    @property
    def discoveries(self):
        """return project that is being used by discovery manager"""
        our_discoveries = {}
        all_discoveries = DiscoveryManager.get_manager().discoveries
        for name in self._approved_discoveries:
            if name in all_discoveries:
                our_discoveries[name] = all_discoveries[name]
        return our_discoveries

    def initialize(self,refresh=False):
        """
        Args:
            refresh (bool) : used to force initialization even if project
                             has not changed
        """
        DiscoveryManager.get_manager().initialize(refresh)

    def __getattr__(self,attr):
        # if we have not run initialization, run it...we have to put special
        # try/except to make sure traceback properly propogates
        disc_manager = DiscoveryManager.get_manager()
        # THE ONE place we use a protected variable for a friend class here
        # only because we are doing initialization in a getattr call
        if disc_manager._initialized == _InitializeStages.not_started:
            try:
                disc_manager.initialize()
            except Exception as e:
                # we have to switch to runtime error to make sure exceptions
                # loading discoveries get bubbled up
                _LOG.exception("exception during getattr, so this this might get hidden...")
                raise RuntimeError(e)

        # if it is in the discovery manager, start there...
        if attr in disc_manager.__dir__(self._approved_discoveries):
            return getattr(disc_manager,attr)
        # otherwise must be a class attribute or something else
        return object.__getattribute__(self,attr)

    def __dir__(self):
        return DiscoveryManager.get_manager().__dir__(self._approved_discoveries)

    def get_by_path(self, path, **kwargs):
        """
                return the node or component based on the path provided

        Args:
            path (str,list) : Path is a "." delimitted string or a list, that
                              is used to traverse the nodes and components to
                              return whatever object the path points to
            default (optional, bool) : if path is not found return this instead
        Returns:
            returns either a node or component depending on what path is

        Raises:
            ValueError if the path can not be found
        """
        # I could not figure out how to do this without duplicating the work in the primary component manager
        # due to the check for "does this sub manager suport it"
        assert isinstance(path, six.string_types), "Must be a string"
        default = kwargs.pop("default", KeyError)
        # if we have no known items, call get_all to attempt to get items
        # what check here to know when to call get_all ?
        path_parts = path.split(".")
        if len(path_parts)==0:
            raise ValueError("Invalid path given: %s"%path)
        # if here, then it is supported, see if we have it...
        comp = getattr(self, path_parts[0], None)
        if comp is None:
            # nothing found return default or raise error
            if default is not KeyError:
                return default
            raise ValueError("Unknown path: '%s'" % path)
        # found it, call next level of get by path
        if len(path_parts) == 1:
            return comp
        else:
            # use known item and call its
            next_path = ".".join(path_parts[1:])
            return comp.get_by_path(next_path, default=default)

def register(name, discovery):
    """
    use this to register your discovery object
    """
    DiscoveryManager.get_manager()._register(name, discovery)

def unregister(name):
    """
    use this to register your discovery object
    """
    DiscoveryManager.get_manager()._unregister(name)

def get_project():
    """

    returns a what the current known project is.

    Side Effects:
       This will load the projects plugins if they have
       not already been loaded.

    """
    return DiscoveryManager.get_manager().get_project()
