
# INTEL CONFIDENTIAL
# Copyright 2014 2018 Intel Corporation
#
# The source code  contained or  described herein and  all documents related to
# the source code  ("Material") are owned by Intel Corporation or its suppliers
# or licensors.  Title to the  Material  remains with  Intel Corporation or its
# suppliers  and licensors. The Material contains trade secrets and proprietary
# and  confidential  information  of  Intel  or  its  suppliers  and  licensors.
# The Material  is protected  by worldwide  copyright and trade secret laws and
# treaty provisions.  No part of the Material  may  be used, copied, reproduced,
# modified, published, uploaded, posted, transmitted, distributed, or disclosed
# in any way without Intel's prior express written permission. No license under
# any  patent,  copyright, trade secret or other intellectual property right is
# granted  to  or conferred upon you by disclosure or delivery of the Materials,
# either expressly, by implication, inducement, estoppel or otherwise.
# Any license under such intellectual property rights must be express and
# approved by Intel in writing.




from __future__ import absolute_import
from __future__ import division
from __future__ import print_function

import sys
import os
import hashlib
import uuid
import filelock
from typing import Dict, List
from distutils.version import StrictVersion

import ipccli


try:
    import svtools.common as common
    import svtools.common.baseaccess as common_baseaccess
except ImportError:
    common = None
    common_baseaccess = None

from ..discovery import Discovery
from ..utils.unzip import UnzipCheck, _DummyLock
from ..plugins import tdef, lmdb_dict
from ..utils.ordereddict import odict, _dict_update
from ..comp import GetComponentFromFile
from ..components.uarch_comp import UarchComponent
from ..errors import VersionError, DeviceNotFound
from ..import settings
from ..logging import getLogger

_LOG = getLogger("uarch")

## Used for base class so that downstream tools
## can easily tell state discovery
class BaseUarchDiscovery(Discovery):

    #: these baseaccess modes support uarch discovery
    baseaacceses_with_uarch = ['itpii', 'ipc']

    # save bundle code the same for both Uarch and Tdef discoveries
    def save_bundles(self, overrides=None,
                     nodenames=None,
                     which="include",
                     per_item=False,
                     compress=False):
        """
        Execute save_bundle on the specified nodes so that we save a jtag
        bundle operation for quickly retrieving the data on subsequent

        Args:
            overrides : dictionary where: key = string of function to override
                        value = function / object to override
            nodenames : list of node names to save/not save the bundle for
                        (if None=all nodes)
            which : "include" - save all the specified nodenames
                    "exclude" - save all except for the specified nodenames
            per_item : True/(False) build a bundle per item for the specified
                       nodes (so each slice of the array gets its own bundle)
            compress : True/(False) whether to create bz2 file after creating bundles

        Examples:

            >> from project.toolext.namednodes import tdef_overrides
            >> sv.uarch.save_bundles({"RegBus":tdef_overrides.RegBusBundle})
            >> # save only the l2cache
            >> sv.uarch.save_bundles({"RegBus":tdef_overrides.RegBusBundle}, ["l2cache"], "include")

        Notes:

            - This only saves the bundles, to use the bundle, setaccess('bundle') must
              be called for each uearch
            - This saves the bundle information to the disk
            - overrides can override functions or objects (like RegBus)

        Warning:
            there is no way for the code here to know that the override is
            successfully calling the bundling API
        """
        print("WARNING: this is still POC code and requires special builds of OpenIPC to use")
        assert which in ['include','exclude'],\
            "which must be 'include' or 'exclude', %s is unknown" % which
        # if no overrides we still need a dictionary to override the locker
        if overrides is None:
            overrides = {}
        # at this level we know that all the uarchs are the same type
        uarchs = self.get_all()
        assert len(uarchs) != 0, "No uarchs found, cannot capture bundles"
        uarch = uarchs[0]
        if nodenames is None:
            nodenames = uarch.nodenames
        if which == "exclude":
            # build list of nodenames except for the onese we are excluding
            exclude_what = set(nodenames)
            nodenames = [n for n in uarch.nodenames if n not in exclude_what]
            # make sure excludes don't have paths
            for n in exclude_what:
                if n.count("."):
                    raise ValueError("excluded nodes cannot contain paths: %s" % n)

        # we do not need/want a lock while building bundles, so we use
        # this to prevent locking from occurring
        class DummyLocker(object):
            def __enter__(dself):
                yield dself
            def __exit__(dself,*args,**kwargs):
                return
        def dummy_locker(*args,**kwargs):
            return DummyLocker()

        overrides['device_locker'] = dummy_locker

        #############
        # save off previous access method before we override for bundling
        #############
        access_module_name = uarch.definition.info['access_modname']
        access_module = sys.modules[access_module_name]
        previous_ops = {}
        for opname, override in overrides.items():
            if opname.count("."):
                raise ValueError("complex operation overrides with paths not supported yet: %s"%opname)
            previous_op = getattr( access_module, opname, None)
            if previous_op is None:
                print("Warning %s not found to override"%opname)
                continue
            previous_ops[opname] = previous_op
            setattr(access_module, opname, override)
        # operations save off, now get our scans
        try:
            for nodename in nodenames:
                node = uarch.get_by_path(nodename)
                if per_item:
                    node.bundle.save_bundle_per_item()
                else:
                    node.bundle.save_bundle()

            # save our changes back to the disk

        finally:
            for opname, original in previous_ops.items():
                setattr(access_module, opname, original)

        if compress:
            uarch.definition.tolmdb.compress()

        return

class UarchDiscovery(BaseUarchDiscovery):
    # not meant for directy use
    name = None
    # IPC device to component file we should attach it to
    components = {}
    #: controls where we are using IPC or DAL
    env = "ipc"

    def baseaccess(self):
        """returns IPC or ITP baseaccess"""
        if self.env in ["itpii", "dal"]:
            import itpii
            if itpii==ipccli:
                _LOG.warning("WARNING! mode set to itpii, but itpii override is in place")
            return itpii.baseaccess()
        elif self.env == "ipc":
            return ipccli.baseaccess()
        else:
            raise Exception("Unsupported env: %s"%self.env)

    def find_all(self):
        #import pdb;pdb.set_trace()
        found = []
        if common_baseaccess and common_baseaccess.getaccess() not in self.baseaacceses_with_uarch:
            return found
        ipc = ipccli.baseaccess()
        cache = {}
        for devtype,comppath in self.components.items():
            devices = ipc.devicelist.search(devicetype="^{0}$".format(devtype))
            if not type(comppath) is list:
                component_path = [comppath]
            else:
                component_path = comppath

            for dev in devices:
                first_comp = None
                instance_num = 1
                for comp_path in component_path:         
                    if comp_path not in cache:
                        cache[comp_path] = GetComponentFromFile(comp_path)
                    uarch_comp = cache[comp_path]                    
                    nn_version_builtwith = StrictVersion(uarch_comp.info['namednodes_version_builtwith'])
                    kwargs = dict(value_class=UarchComponent)
                    if nn_version_builtwith >= StrictVersion("1.34"):
                        uarch = uarch_comp.create(name="uarch_"+dev.alias, device=dev.alias, **kwargs) # maybe object ?
                    else:                         
                        uarch = uarch_comp.create(name="uarch_" + dev.alias, device=dev, **kwargs)  # maybe object ?
                    if first_comp is None:
                        first_comp = uarch
                        dev.node.state.uarch = first_comp
                        found.append( first_comp )
                      
                    else:
                        if uarch.name in first_comp.sub_components:                             
                            previous = first_comp.sub_components.pop(uarch.name)
                            previous.set_name(uarch.name+"0")
                            first_comp.access_classes.update(uarch.access_classes)
                            first_comp.add_component(previous)
                            first_comp.add_component(uarch, uarch.name +str(instance_num))
                            instance_num = instance_num + 1
                        else:
                            first_comp.access_classes.update(uarch.access_classes)
                            first_comp.add_component(uarch, uarch.name)
                        

                    
                    
        return found


class TdefDiscovery(BaseUarchDiscovery):

    #: type of device to attached our uarchs to in the devicelist
    devicetype = None

    def __init__(self, devicetype, lmdb_path, tdef_files, access_code="", similar_steppings=None, compress=False, **kwargs):
        """
        Args:
            devicetype:   type of device to attached our uarchs to in the devicelist
            lmdb_path:    path to store tdef converterd data
            tdef_files:   list of tdef files to parse in to namednodes
            access_code:  additional code to add to generated access file
            similar_steppings: mapping where key is a tuple of steppings that are similar and
                               value is the stepping to use in collateral
            compress: whether to compress after lmdb after generation
            validate_function (func) : (optional kwarg) function used to validate all the nodes encountered are 'valid'
            minimum_namednodes (str) : (optional kwarg) if an lmdb exists, make sure it was created with a version the
                                        same or newer than the one specified

        """
        super(TdefDiscovery,self).__init__()
        self.devicetype = devicetype
        self.lmdb_path = lmdb_path
        self.tdef_files = tdef_files
        # it may be that we don't have any tdef files
        #if self.tdef_files is not None:
        #   assert len(self.tdef_files) > 0, "At least one tdef file must be specified"

        # dal support was just experimental, do not use on DAL
        self._env = "ipc"
        self.access_code = access_code
        self._known_tdef_commands = {}
        self.similar_steppings = similar_steppings
        self.compress = compress
        # variable to know when we generated new LMDB file
        self.generated = False
        self._validate_nodes = kwargs.pop("validate_function", None)
        self.minimum_namednodes = kwargs.pop("minimum_namednodes", None)
        if self.minimum_namednodes is not None:
            # convert to version for comparison later
            self.minimum_namednodes = StrictVersion(self.minimum_namednodes)
        assert len(kwargs) == 0, "Unknown args: %s"%str(list(kwargs.keys()))

    # dal not actually supported, but was experimented with as part of a POC at one point
    def baseaccess(self):
        """returns IPC or ITP baseaccess"""
        import ipccli
        return ipccli.baseaccess()

    def find_all(self):
        # generated will be set to True if it turns out we did not read from the lmdb and had to read from the tdef
        if common_baseaccess and common_baseaccess.getaccess() not in self.baseaacceses_with_uarch:
            return []
        self.generated = False
        uarchs = []
        # update our paths with stepping information...
        tdef_files = self.tdef_files if self.tdef_files is not None else []
        paths = [self.lmdb_path] + tdef_files
        paths = update_stepping_in_paths(self.devicetype, paths, self.similar_steppings)
        lmdb_path = paths[0]
        if self.tdef_files is not None:
            tdef_files = paths[1:]
        # make sure no zips have been updated
        tdef_files = self._unzip_check(tdef_files)
        # see if lmdb file is up-to-date compared to the tdef file
        filelock_path = lmdb_path + ".lock"
        if settings.NO_FILELOCKS is False:
            # if dir does not exist, i guess we should make it
            lockdir = os.path.dirname(os.path.abspath(filelock_path))
            if not os.path.exists(lockdir):
                os.makedirs(lockdir)
            lock = filelock.FileLock(filelock_path)
        else:
            lock = _DummyLock(filelock_path)

        # not sure of default time since this can take quite a while?
        try:
            try:
                lock.acquire(1)
                lock.release()
            except filelock.Timeout:
                print("Some other process is currently writing the components, we are waiting...")

            with lock.acquire(settings.UNZIP_LOCK_TIMEOUT*2):
                compdef = self._lmdb_load_and_check(lmdb_path, tdef_files)
                if compdef == False:  # didn't get from lmdb, get from tdef
                    # path for access file will be in using lmdb path
                    if not os.path.exists( lmdb_path ):
                        os.makedirs( lmdb_path )
                    access_filename = lmdb_path + os.sep + "accesses.py"
                    compdef = self._load_tdef_component(tdef_files, access_filename)
                    # because we print '.' when loading tdefs...
                    sys.stdout.write("\n")
                    sys.stdout.flush()
                    # _save_components, but if we have lmdb files...
                    self._save_component(compdef, lmdb_path)
                    # make sure we are consistently using the comp from the file
                    compdef = self._lmdb_load_and_check(lmdb_path, tdef_files)
                    # for other parts of the stack that need to know we regenerated the files
                    self.generated = True
                if compdef == False:
                    # still false...
                    raise Exception("unknown error with lmdb generation for %s" % self.devicetype)
                #try:
                #    os.remove(filelock_path)
                #except:
                #    pass
        except filelock.Timeout:
            msg = ("TDEF conversion hit timeout acquiring lock. You may need to delete this file "
                    "and re-run:\n\t %s"%filelock_path)
            _LOG.error("\n"+msg)
            raise RuntimeError(msg)

        ####
        # now add the comp to appropriate device
        #
        ipc = self.baseaccess()
        devices = []
        for dev in ipc.devicelist:
            # sometimes these attributes are present but are none...
            dtype = getattr(dev, "devicetype", None)
            dtype = "" if dtype is None else dtype
            subtype = getattr(dev, "devicesubtype", None)
            subtype = "" if subtype is None else "_" + subtype
            devlist_devtype = dtype + subtype
            if devlist_devtype == self.devicetype:
                devices.append(dev)
        if len(devices) == 0:
            _LOG.info("no devices found that match {0}".format(self.devicetype))
        nn_version_builtwith = StrictVersion(compdef.info['namednodes_version_builtwith'])
        for dev in devices:
            # newer namednodes use alias instead of device
            kwargs = dict(value_class=UarchComponent)
            if nn_version_builtwith >= StrictVersion("1.34a1"):
                uarch = compdef.create("uarch_{0}".format(dev.alias), device=dev.alias, **kwargs)
            else:
                uarch = compdef.create("uarch_{0}".format(dev.alias), device=dev, **kwargs)
            # so we know what we attached it to
            uarch.target_info["devtype"] = self.devicetype
            # append to list of all found and created
            uarchs.append( uarch )
            # save off on the node for backwards compatibility with DAL
            dev.node.state.uarch = uarch
        return uarchs

    def support_command(self, xml_tag, function_name=None, first_param="device"):
        """
        add new tag that should be supported when we parse the tdef
        """
        function_name = function_name or xml_tag
        self._known_tdef_commands[xml_tag] = tdef.TdefCommand(
                                                    xml_tag,
                                                    function_name,
                                                    first_param)

    def _unzip_check(self, files):
        correct_filenames = []
        for fpath in files:
            if fpath.endswith(".zip"):
                opath = fpath
                fpath = UnzipCheck(fpath).decompress(silent=True)
                # in case it was not named .xml.zip as expected
                if not fpath.endswith(".xml"):
                    fpath += ".xml"
            # updated file list
            correct_filenames.append(fpath)
        return correct_filenames

    def _from_pysv_path(self, relative_path):
        """given a filepath, try add pysv_path if needed"""
        # nothing we can do
        if common_baseaccess is None:
            return os.path.normcase(relative_path)
        if "{pysv_root}" in relative_path:
            if common_baseaccess is None:
                raise Exception("common module missing but file needs pysvpath")
            fullpath = relative_path.format(pysv_root=common.path.getpysvpath())
        else:
            # older files dont have pysv_root or we were created without pysv_root
            fullpath = relative_path
        # make sure path is absolute and simplified to match others
        return os.path.normcase(os.path.abspath(fullpath))

    @staticmethod
    def _to_pysv_path(filepath):
        """given a filepath, try and find the path relative to pysvroot"""
        # nothing we can do
        if common_baseaccess is None:
            return filepath
        pysv_base = common.path.getpysvpath()
        if pysv_base and pysv_base in filepath:
            relative_path = "{pysv_root}"+os.sep+filepath.replace(pysv_base,"")
        else:
            # nothing we can do
            relative_path = filepath
        return relative_path


    def _lmdb_load_and_check(self, lmdb_path, tdef_files):
        """
        Returns:
            False means we need to load tdefs, else returns loaded components
        """
        # all those checks...does lmdb_path exists yet? if not, it never will...
        found = False
        if os.path.exists(lmdb_path+"/data.mdb"):
            found = True
        else:
            # make sure compressed file exists or lmdb path exists
            for ext in UnzipCheck.extensions.keys():
                if os.path.exists(lmdb_path + ext):
                    found = True
                    break
        if found == False:
            # could not find file or a compressed version
            return False
        # if we get a version error we need to regenerate, or on really old
        # comps, if we are missing tdef info, assume it is out of date
        try:
            comp = GetComponentFromFile(lmdb_path)
            if self.minimum_namednodes is not None:
                rev_used = comp.info.get("namednodes_version_builtwith", None)
                # minimum version requested, but somehow that data is missing, lets regen
                if rev_used is None: # must use 'is' here as versions cant compare against None
                    return False
                if rev_used < self.minimum_namednodes:
                    return False
                # must have been ok...nothing needed
        except VersionError:
            return False

        # if no tdef files specified then we will just use the component
        if tdef_files is None:
            return comp

        # tdefs given make sure component has that info too
        if 'tdefs' not in comp.info:
            return False

        # convert to a dictionary of checksums with full path
        full_db_paths = { self._from_pysv_path(f):comp.info['tdefs'][f] for f in comp.info['tdefs']}

        missing = []
        for fpath in tdef_files:
            # may need to revisit this abspath if the BKM becomes to NOT submit the tdef's to SVN
            fpath = os.path.normcase(os.path.abspath(fpath))
            if not os.path.exists(fpath):
                missing.append(fpath)
            # a new tdef was given, we will have to recompile
            if fpath not in full_db_paths:
                _LOG.debug("database missing checksuum for file for file {0}, rebuilding database".format(fpath))
                return False
            with open(fpath,"rb") as f:
                md5 = hashlib.md5(f.read())
            checksum = md5.digest()
            hex_checksum = md5.hexdigest()
            # check againsted
            if checksum != full_db_paths[fpath] and hex_checksum != full_db_paths[fpath]:
                _LOG.debug("checkshum didn't match for file {0}, rebuilding database".format(fpath))
                return False
        # if ALL tdefs are missing, its ok, just use the lmdb, but if even
        # just one is missing, we should return false
        needs_compiling = set(tdef_files) - set(missing)
        len_needs = len(needs_compiling)
        if len_needs != 0 and len_needs != len(tdef_files):
            raise RuntimeError("some tdefs present, some missing, for example: %s"%list(needs_compiling)[0])
        # all comp loaded and match what is in their database file
        return comp

    def _load_tdef_component(self, tdef_files, access_filename="accesses.py"):
        """using the tdef_files dictionary with a list of files, built the neccessary components

        Returns:
            version of the component dictionary with a ComponentDefinition for each key

        """
        final_comp = None
        files_seen = set()

        for fpath in tdef_files:
            # may need to revisit this abspath if the BKM becomes to NOT submit the tdef's to SVN
            fpath = os.path.abspath(fpath)
            sys.stdout.write(".")
            sys.stdout.flush()
            # create loader directly because we need to specify additional information
            loader = tdef.TdefLoader.create(fpath, additional_access_code=self.access_code)
            if loader is None:
                if not os.path.exists(fpath):
                    raise RuntimeError("TDEF loader couldn't find parse: %s"%fpath)
                else:
                    raise ValueError("TDEF loader doesn't know how to parse: %s"%fpath)
            # add in the commands that we have been told to support/add
            for tdef_cmd in self._known_tdef_commands.values():
                loader.support_command( tdef_cmd.xml_tag,
                                        tdef_cmd.function_name,
                                        tdef_cmd.first_param)
            modname = "namednodes.tdef.access."+self.devicetype+"_"+"uarch"
            ##### if first time we have seen this file...
            overwrite = not( access_filename in files_seen )
            if overwrite: files_seen.add( access_filename )
            ##### PARSE  ####
            this_comp = loader.parse(access_filename, modname, overwrite = overwrite)
            #### add nodes to existing components
            # first comp, just uses its nodes
            rel_fpath = self._to_pysv_path(fpath)
            if final_comp is None:
                final_comp = this_comp
                # add path to each node so we can find what tdef it came from later
                for node in final_comp.nodes:
                    node.info['tdef_path'] = rel_fpath
            else:
                for node in this_comp.nodes:
                    # add path to each node so we can find what tdef it came from later
                    node.info['tdef_path'] = rel_fpath
                    # move nodes over to the final component
                    final_comp.add_node( node )

            if self._validate_nodes is not None:
                for node in this_comp.walk_nodes():
                    self._validate_nodes(node)

            _dict_update(final_comp.info, this_comp.info)
            # add tdef info to the component
            tdef_info = final_comp.info.setdefault('tdefs', {})
            # compute and save checksum...so we know if xml changed
            # save hex string so it is friendly to pickle
            with open(fpath,"rb") as f:
                checksum = hashlib.md5(f.read()).hexdigest()
            tdef_info[rel_fpath] = checksum

            if "deviceType" in final_comp.info:
                if final_comp.info['deviceType'] != self.devicetype:
                    _LOG.debug("FYI: Uarch discovery specified XML for device type '{0}', but XML had '{1}' in it".\
                                format(self.devicetype,final_comp.info['deviceType']))
        final_comp.name = "uarch_"+self.devicetype
        return final_comp

    def _save_component(self,component, lmdb_path):
        # write components to lmdb
        # make sure we are starting from scratch (?)
        _LOG.result("Writing lmdb file: {0} ...".format(lmdb_path))
        # First, we need to update some path info...make sure access file is relative
        # to the the lmdb file
        lmdb_file = os.path.normpath( lmdb_path )
        access_filename = component.info.get("access_filename",None)
        if access_filename != None:
            # change the access file to be relative tot the lmdb file
            access_filename = os.path.relpath(access_filename, lmdb_file)
            _LOG.debug("Updated accessfilename to be: %s"%access_filename)
            component.info['access_filename'] = access_filename
        # this used to be access='w', but that caused some regression
        # error due to database still being open. not 100% sure how to close here
        component.tolmdb.write(lmdb_file, compress=self.compress)

class TdefUarchDiscovery(BaseUarchDiscovery):
    """
    """
    #: should be set by ineriting classes to report what is being discovered
    name = None
    #: this should have a devicetype:path mapping to where we should put
    #: the generated lmdb output after we have parsed TDEF
    lmdb_paths = {}
    # IPC device to component file we should attach it to
    tdef_files = {}
    #: dictionary to hold the corresponding discovery classes
    #: the key = device_type (same as tdef_files and lmdb_paths)
    tdef_discoveries = None
    #: any additional custom access code to include in our tdef access files
    access_code = None
    #: can be filled in if we want to map actual steppings to single stepping id
    #: example: { ("A0","A1"): "A0"}
    #: will fill in {stepping} in the paths with A0 even if the devicelist has A1
    similar_steppings = None
    #: whether to compress lmdb's after they are generated from tdef
    compress = False
    # dal not actually supported, but was experimented with as part of a POC at one point
    env = "ipc"
    # if there is some minimum version of namednodes that the collateral/lmdb must have
    # been generated with
    minimum_namednodes = None
    #: when set, we will re-run "save_bundles" if any of the tdef files
    #: were modified and therefore wiped out bundles
    auto_save_bundles = False
    #: internal variable to track supported commands that we need to pass on to various tdef discoveries
    _supported_commands = None
    #: to track whether init_tdef  has been called

    def __init__(self):
        super(TdefUarchDiscovery,self).__init__()
        self._supported_commands = {}

    def init_tdef_discoveries(self):
        if set(self.lmdb_paths.keys()) != set(self.tdef_files.keys()):
            _LOG.error("LMDB path keys: %s"%list(self.lmdb_paths.keys()))
            _LOG.error("Component Types keys: %s"%list(self.tdef_files.keys()))
            raise Exception("keys in 'lmdb_paths' must match 'tdef_files")

        # build ourselves up as a superset of our children
        self.tdef_discoveries = {}
        for devtype in self.lmdb_paths:
            if self.similar_steppings and devtype in self.similar_steppings:
                steppings = self.similar_steppings[devtype]
            else:
                steppings = self.similar_steppings
            self.tdef_discoveries[devtype] = TdefDiscovery(devtype,
                                                  self.lmdb_paths[devtype],
                                                  self.tdef_files[devtype],
                                                  self.access_code,
                                                  steppings,
                                                  self.compress,
                                                  validate_function=getattr(self, "validate_nodes", None),
                                                  minimum_namednodes=self.minimum_namednodes,
                                                )
            for xml_tag, args in self._supported_commands.items():
                self.tdef_discoveries[devtype].support_command(*args)

    def baseaccess(self):
        """returns IPC or ITP baseaccess"""
        if self.env in ["itpii", "dal"]:
            import itpii
            if itpii==ipccli:
                _LOG.warning("WARNING! mode set to itpii, but itpii override is in place")
            return itpii.baseaccess()
        elif self.env == "ipc":
            return ipccli.baseaccess()
        else:
            raise Exception("Unsupported env: %s"%self.env)

    def find_all(self):
        # these modes dont support AFD currently
        if common_baseaccess and common_baseaccess.getaccess() not in self.baseaacceses_with_uarch:
            return []

        self.init_tdef_discoveries()
        uarchs = []
        # we initialize both here and in the get_all in case someone overwrote one method but
        # not the other
        self._run_save_bundles = False
        for disc in self.tdef_discoveries.values():
            # must flow through the get_all otherwise the discovery objects
            # don't get properly updated
            try:
                # always set stop on error and let OUR getall handle the exception handling or not...
                items = disc.get_all(refresh=True, stop_on_error=True)
                # keep up with whether any discovery caused us to generate lmdbs
                self._run_save_bundles |= disc.generated
            except DeviceNotFound as e:
                _LOG.warning("Device not found: {}".format(str(e)))
                items = []
            uarchs.extend(items)
        return uarchs

    def get_all(self, refresh=False, stop_on_error=False):
        self._run_save_bundles = False
        # some projects apparenlty override find_all and dont call super
        # so we need to not break them (but the auto_save wont work for them either)
        the_list = super(TdefUarchDiscovery, self).get_all(refresh, stop_on_error=stop_on_error)
        # how do we do this??? we need get all to be "done" otherwise if save_bundles calls get all, we hit
        # some recursive nightmare...
        if self._run_save_bundles and self.auto_save_bundles:
            # save bundles may call get all, so we need to clear the flag here so that
            # this function can be re-entrant...
            self._run_save_bundles = False
            self.save_bundles()
            # reset flag so we dont run again
        return the_list

    def support_command(self, xml_tag, function_name=None, first_param="device"):
        """
        add the supported command to all our tdef loaders

        Args:

            xml_tag : tag in the xml to support
            function_name : function to call when tag is encountered (default
                            is that the xml_tag and function_name are the same)
            first_param : whether first parameter should be device, node, or no
                          special parameter needed
        """
        # add to the list of supported commands so that during initialization we will have the parameters
        self._supported_commands[xml_tag] = (xml_tag, function_name, first_param)
        if self.tdef_discoveries is not None:
            # initialize already called, so pass along the support command request
            for discovery in self.tdef_discoveries.values():
                discovery.support_command(xml_tag, function_name, first_param)

    def save_bundles(self, *args, **kwargs):
        """
        see TdefDiscovery.save_bundles for help
        """
        for disc in self.tdef_discoveries.values():
            disc.save_bundles(*args, **kwargs)

    def _test_read(self, stop_on_fail=False):
        """simple flow to test read all nodes

        WARNING: this could take quite a *long* time to run

        """
        uarchs = self.get_all()
        # just test one of each device type
        tested = set()
        failures = []
        for uarch in uarchs:
            devtype = uarch.target_info['device'].devicetype
            if devtype in tested:
                continue
            tested.add(devtype)
            print("Testing %s"%devtype)
            for node in uarch.nodes:
                if node.name.count("l2_cache"):
                    # commonly this is very big and takes a while, so we skip
                    # for our smoke testing
                    print('skipping l2')
                    continue
                print("."*4+"%s"%node.name)
                try:
                    a=node.read()
                except:
                    print("."*8+"failed")
                    failures.append("%s - %s"%(devtype,node.name))
        if len(failures)>0:
            print('Failure Summary')
            print('-'*20)
            for f in failures:
                print(f)



def update_stepping_in_paths(devicetype, paths, similar_steppings=None):
    """
    Helper function to update paths with the current steppings

    Args:
        devicetype : device type to check for stepping
        paths: list of paths to udpate
        similar_steppings: mapping used so that similar steppings can
            be sent to the same path. Example: { ("A0","A1"): "A0" }

    Returns:
        list of updated paths
    """
    import ipccli
    ipc = ipccli.baseaccess()
    for dev in ipc.devicelist:
        # sometimes these attributes are present but are none...
        dtype = getattr(dev, "devicetype", None)
        dtype = "" if dtype is None else dtype
        subtype = getattr(dev, "devicesubtype", None)
        subtype = "" if subtype is None else "_"+subtype
        devlist_devtype = dtype + subtype
        if devlist_devtype == devicetype:
            break
    else: # else on "for" means we didn't hit a break...
        raise DeviceNotFound("no devices found that match {0}".format(devicetype))
        #_LOG.warning("no devices found that match {0}".format(devicetype))
        #return paths
    stepping = dev.stepping
    # see if we should convert the found stepping in to some other stepping key that
    # will be used in our collateral lookup
    if similar_steppings is not None:
        for stepping_tuple in similar_steppings:
            if stepping in stepping_tuple:
                stepping = similar_steppings[stepping_tuple]
                break

    newpaths = []
    for f, fname in enumerate(paths):
        newpaths.append(fname.format(stepping=stepping))
    return newpaths



def convert_tdefs_to_uarch(
        tdef_paths: List[str],
        lmdb_path: str,
        *,
        supported_commands: List[tdef.TdefCommand] = None,
        additional_access_code: str = "",
        compress: bool = True,
        name: str = None,
) -> UarchComponent:
    """Given the specified tdef paths create a single lmdb component
    Args:
        tdef_paths : list of paths to tdef files
        lmdb_path : path to lmdb file to write out to
        supported_commands: this needs to be a list of TdefCommand objects (namednodes.plugins.tdef.TdefCommand)
        additional_access_code : string of additional code that should be added to the generated access.py file
    """
    final_comp = None
    files_seen = set()
    access_filename = os.path.join(lmdb_path, "access.py")
    if not os.path.exists(access_filename):
        os.makedirs(lmdb_path)
    for fpath in tdef_paths:
        # may need to revisit this abspath if the BKM becomes to NOT submit the tdef's to SVN
        fpath = os.path.abspath(fpath)
        sys.stdout.write(".")
        sys.stdout.flush()
        # create loader directly because we need to specify additional information
        loader = tdef.TdefLoader.create(fpath, additional_access_code=additional_access_code)
        if loader is None:
            if not os.path.exists(fpath):
                raise RuntimeError("TDEF loader couldn't find parse: %s" % fpath)
            else:
                raise ValueError("TDEF loader doesn't know how to parse: %s" % fpath)
        # add in the commands that we have been told to support/add
        if supported_commands:
            for tdef_cmd in supported_commands:
                loader.support_command(tdef_cmd.xml_tag,
                                       tdef_cmd.function_name,
                                       tdef_cmd.first_param)
        modname = "tdef_" + str(uuid.uuid4()).replace("-", "_")
        ##### if first time we have seen this file...
        overwrite = not (access_filename in files_seen)
        if overwrite:
            files_seen.add(access_filename)
        ##### PARSE  ####
        this_comp = loader.parse(access_filename, modname, overwrite=overwrite)
        #### add nodes to existing components
        # first comp, just uses its nodes
        rel_fpath = TdefDiscovery._to_pysv_path(fpath)
        if final_comp is None:
            final_comp = this_comp
            # add path to each node so we can find what tdef it came from later
            for node in final_comp.nodes:
                node.info['tdef_path'] = rel_fpath
        else:
            for node in this_comp.nodes:
                # add path to each node so we can find what tdef it came from later
                node.info['tdef_path'] = rel_fpath
                # move nodes over to the final component
                final_comp.add_node(node)

        _dict_update(final_comp.info, this_comp.info)
        # add tdef info to the component
        tdef_info = final_comp.info.setdefault('tdefs', {})
        # compute and save checksum...so we know if xml changed
        # save hex string so it is friendly to pickle
        with open(fpath, "rb") as f:
            checksum = hashlib.md5(f.read()).hexdigest()
        tdef_info[rel_fpath] = checksum

    if name:
        final_comp.name = name

    if name is None:
        name = os.path.split(lmdb_path)[-1].replace(".lmdb", "")
        final_comp.name = name
    final_comp.tolmdb.write(lmdb_path, compress=compress)
    # open it back up using the lmdb path for easing the save flow
    final_comp = GetComponentFromFile(lmdb_path)

    return final_comp