
# INTEL CONFIDENTIAL
# Copyright 2014 2018 Intel Corporation
#
# The source code  contained or  described herein and  all documents related to
# the source code  ("Material") are owned by Intel Corporation or its suppliers
# or licensors.  Title to the  Material  remains with  Intel Corporation or its
# suppliers  and licensors. The Material contains trade secrets and proprietary
# and  confidential  information  of  Intel  or  its  suppliers  and  licensors.
# The Material  is protected  by worldwide  copyright and trade secret laws and
# treaty provisions.  No part of the Material  may  be used, copied, reproduced,
# modified, published, uploaded, posted, transmitted, distributed, or disclosed
# in any way without Intel's prior express written permission. No license under
# any  patent,  copyright, trade secret or other intellectual property right is
# granted  to  or conferred upon you by disclosure or delivery of the Materials,
# either expressly, by implication, inducement, estoppel or otherwise.
# Any license under such intellectual property rights must be express and
# approved by Intel in writing.



"""
=================================
Static XML Discovery and Loading
=================================

The static XML writer is used to help write a component (and its subcomponents) to
the specified storage format along with an XML file that will describe how
to recreate the objects.

This assumes that you don't really need to do any (or much) discovery and that the
component should always have the same sub components and always created using the
same values for the constructor

For writing the component to disk, the :py:func:`write` function is what should be
to store the components and create an XML that describes how to re-create your
component hiearchy.

For loading your XML back in, you will need to subclass :py:class:`StaticDiscovery` and
define what the :py:data:`StaticDiscovery.static_path` is for where you wrote the file
in write_component.


.. autofunction:: write
.. autofunction:: load
.. autofunction:: load_offline

.. autoclass:: StaticDiscovery
    :members:


"""
from __future__ import absolute_import
from __future__ import division
from __future__ import print_function


# Internal modules
import json
from ..utils._ascii_menu import menu_prompt
from ..utils._py2to3 import *
from ..utils.ordereddict import odict
from ..utils.unzip import UnzipCheck
from ..utils.xml import indent as indent_xml
from ..utils.xml import CDATA
from ..discovery import Discovery
from ..plugins.nn_logregisters import _read_if_available
from ..registers import RegisterComponent
from ..access import NodeAccess
from ..logging import getLogger
from .. import settings
import ipccli

from ..errors import StaticDiscoveryError

from ..comp import (
    ComponentGroup,
    GetComponentFromFile,
    NamedComponent,
    NamedComponentDefinition,
    )
from ..accesses import (
    AccessSnapshot,
    AccessStoredValues,
    AccessRegisterStoredValues,
    AccessRegister,
    AccessField,
    StateNodeAccess,
    StateNodeOfflineAccess,
    StateNodeAccessNoLock,
    ScanFieldAccess,
    )
from ..accesses.general import (
    _StoredValuesFormat
)

# other plugins required
from copy import copy
import site

if PY2:
    from cStringIO import StringIO
    from backports import lzma
    import cPickle as pickle
else:
    from io import BytesIO
    from io import StringIO
    import lzma
    import pickle

try:
    import svtools.common as common
    import svtools.common.path
except ImportError:
    common = None


# extention for pickle with comp creators
_EXT_XZ_PICKLE = ".spkx"
_EXT_PICKLE = ".spk"
_EXT_XML = ".xml"
_EXT_JSON = ".sjson"
_EXT_XZ_JSON = ".sjson.xz"
# supported extensions
_STATIC_EXT = (_EXT_XZ_PICKLE,
               _EXT_PICKLE,
               _EXT_XML,
               _EXT_JSON,
                _EXT_XZ_JSON
               )

_LOG = getLogger("static")

# acceses where we do not need to save the values in offline mode
_ACCESSES_WITHOUT_VALUES = {
        AccessField,
        StateNodeAccessNoLock,
        ScanFieldAccess,
        AccessStoredValues,
        AccessRegisterStoredValues, # because the dictionary will already have its saved value
}

# mapping of known accesses that should be converted to other accesses
# when we are in offline mode
_ACCESSES_OFFLINE_MAP = {
    # FROM , TO
    AccessRegister: AccessRegisterStoredValues,
    # remove lock, but make reads call reads/reprs on our children
    StateNodeAccess: StateNodeOfflineAccess,
    StateNodeAccessNoLock: StateNodeOfflineAccess,
    # to make sure we don't remap safe ones
    AccessField: AccessField,
    ScanFieldAccess: ScanFieldAccess,
    AccessRegisterStoredValues: AccessRegisterStoredValues,
}

# Python builtins
import os,sys

# we cannot use the cET for writing CDATA tags, but we want it for parsing
from xml.etree import cElementTree as cET
from xml.etree import ElementTree as ET

# Python 2.7 and 3
if hasattr(ET, '_serialize_xml'):
    # only override if have to
    if not hasattr(ET,"_original_serialize_xml"):
        ET._original_serialize_xml = ET._serialize_xml
        def _serialize_xml(write, elem, *args):
            if elem.tag == '![CDATA[':
                write("<%s%s]]>" % (
                    elem.tag, elem.text))
                return
            return ET._original_serialize_xml(write, elem, *args)
        ET._serialize_xml = ET._serialize['xml'] = _serialize_xml


class _CompCreator(object):
    """
    Holds the information that is needed to create a component value object
    """

    @classmethod
    def from_component(cls, comp_value, output_path=None, definition_path=None):
        """
        from a component value, create a CompCreator

        Args:
            definition_path (str) : override for where to pull the definition from vs.
                              querying the definition objects
            output_path (str) : path to where this data will be stored, so that
                                we can determine relative path to the definition file
        """
        subs = [_CompCreator.from_component(sub, output_path, definition_path)
                for sub in comp_value.sub_components.values()
                if not sub.definition.info.get("never_save_offline", False)]

        if definition_path:
            def_file_path = definition_path
        else:
            def_file_path = comp_value.definition.origin.info.get("definition_filepath", None)
            # if def_file_path is None:
            #     msg = ("Saving info for a component '%s' that did not come from a file, will require a "
            #           "definition object at load time"%comp_value.definition.path)
            #     _LOG.debug(msg)
            #     _LOG.debug("\t%s"%comp_value.definition.path)
            #     import warnings
            #     warnings.warn(msg, UserWarning)
        # theoretically, file_path only is None, if we built using a 'live' component definition
        # and at that point it is 'on the user' to specify the definition when they load
        # the offline file back in
        if def_file_path is None or def_file_path is None:
            file_path = None
        else:
            # if output_path given get its relative position to the pickle we are about to write
            file_path = _calc_definition_path(output_path, def_file_path)
        # save register or "base" component
        if isinstance(comp_value, RegisterComponent):
            value_class = RegisterComponent
        else:
            value_class = NamedComponent

        return cls(comp_value.name,
                   comp_value.target_info,
                   comp_value.definition.path,
                   subs,
                   file_path,
                   value_class=value_class,
                   )

    @classmethod
    def from_xml(cls, elem, output_path=None, definition_path=None):
        """from XML create our comp creator"""
        name = elem.find("name").text
        def_path = elem.find('comppath').text
        # definition file path has to be passed in OR it needs
        # to be in the xml
        if definition_path is None:
            def_file_path = elem.find('def_file_path')
            if def_file_path is not None:
                def_file_path = def_file_path.text
        else:
            def_file_path = definition_path
        # info dictionary also used in constructor
        target_info = odict()
        for item in elem.find("info").findall("item"):
            if PY2:
                target_info[item.get("name") ] = eval(item.text)
            else:
                target_info[item.get("name") ] = eval(item.text.replace("\\","\\\\"))
        # now grab subs
        subs = [_CompCreator.from_xml(c, output_path, definition_path)
                for c in elem.findall("Component")]
        # now create our objects
        return cls(name,
                    target_info,
                    def_path,
                    subs,
                    def_file_path,
                   )
    # must have defaults for these to support older pickle files
    name = None
    target_info = None
    def_path = None
    subs = None
    def_file_path = None
    value_class = None

    def __init__(self, name, target_info, def_path, subs, def_file_path=None, value_class=None):
        """
        Args:
            name: name of component to create
            target_info: target info of component we are saving
            def_path: path to definition of the component
            subs: list of Comp Creators representing sub components
            def_file_path: path to definition file
            value_class: used as a backup so that if a component is missing we will use value_class
                         to create a temporary run
        """
        self.name = name
        self.target_info = target_info
        self.def_path = def_path
        # list of comp creators
        self.subs = subs
        self.def_file_path = def_file_path
        self.value_class = value_class

    def to_xml(self, xmlnode=None):
        """
        return xml that can be used to build a component

        Args:
            xmlnode : parent xml node to attach new nodes to

        """
        if xmlnode is None:
            node_comp = ET.Element("Component")
        else:
            node_comp = ET.SubElement(xmlnode, "Component")

        name = ET.SubElement( node_comp, "name")
        name.text = self.name
        comppath = ET.SubElement(node_comp, "comppath")
        comppath.text = self.def_path
        if self.def_file_path is not None:
            def_file = ET.SubElement(node_comp, "def_file_path")
            def_file.text = self.def_file_path
        node_info = ET.SubElement(node_comp, "info")
        for key,value in self.target_info.items():
            node_item = ET.SubElement(node_info, "item")
            node_item.set("name", key)
            # write out value as a string assuming we can eval it later
            # this will allow us to support dicts and lists
            #node_item.text =  "<![CDATA[" + str(value) + "]]"
            if isinstance(value,basestring):
                node_item.append(CDATA("'%s'" % str(value)))
            else:
                node_item.append(CDATA(str(value)))
        # now write subcomponents
        for sub in self.subs:
            sub.to_xml(node_comp)
        # finished, return our parent
        return xmlnode

    def create(self, comp_origin_def,
                    parent=None,
                    offline_filepath=None,
                ):
        """take a component definition and create values
        Args:
            parent : used to our newly created component to
            logfile : used to send creation error messages to
            offline_filepath : needed for knowing what path is to definition file

        Returns:
            None : if component could not be created

        Raises:
            ValueError if logfile is not given AND path is missing from
            the definition component
        """
        # know the definition file path AND the offline file path, use that to get the
        # definition object
        my_def = None
        database_path = None
        if offline_filepath is not None:
            if self.def_file_path:
                database_path = self.def_file_path
                comp_origin_def = _load_definition_from_path(self.def_file_path, offline_filepath)
            if comp_origin_def is None and self.value_class and not settings.STATIC_REQUIRES_DATABASE:
                my_def = NamedComponentDefinition(self.name, value_class=self.value_class)
            if comp_origin_def is None and parent is None and my_def is None:
                raise RuntimeError("We could not find lmdb for top level component, cannot proceed, missing: %s"%self.def_file_path)
            elif comp_origin_def is None and my_def is None:
                # this may should be a runtime error not sure howe we get here
                # not top level component but couldn't get definition
                # assume layer below us has warning/exception handling
                return None
            # else: we must have gotten back our definition object
        elif comp_origin_def is None:
            raise RuntimeError("Not enough information to create a component from this offline file")

        try:
            if my_def is None:
                my_def = comp_origin_def
                paths_s = self.def_path.split(".")
                if len(paths_s) == 1 and paths_s[0] != my_def.name:
                    my_def = NamedComponentDefinition(self.name, value_class=self.value_class)   
                else:
                    for path in paths_s[1:]:
                        my_def = my_def.sub_components.get(path, None)
                        if my_def is None:
                            raise ValueError("Missing {}".format(path))                                           
        except KeyboardInterrupt:
            raise
        except ValueError:
            # couldn't find the component we needed in definition path we had...we'll need to just make something to survive
            _LOG.info("Definition path: %s was not in in file '%s'\n" % (self.def_path, self.def_file_path))
            if self.value_class and not settings.STATIC_REQUIRES_DATABASE:
                my_def = NamedComponentDefinition(self.name, value_class=self.value_class)
            else:
                return None
        except Exception as e:
                _LOG.info("Unexpected error %s searching for path: %s in file '%s'\n",(
                    str(e),
                    self.def_path,
                    self.def_file_path))
                return None

        my_comp = my_def.create(name=self.name,
                                info=self.target_info,
                                offline_filepath=offline_filepath,
                                lmdb_filepath=database_path,
                                )
        if parent is not None:
            parent.add_component(my_comp, skip_sort=True)

        for sub in self.subs:
            newcomp = sub.create(comp_origin_def, parent=my_comp, offline_filepath=offline_filepath)
            # sort component groups after done adding all of them
            if newcomp is not None:
                newcomp.sort_component_groups()


        return my_comp


_path_cache = {}
def _calc_definition_path(output_path, definition_path):
    """calculatate path from original to definition"""
    definition_path = os.path.normpath(definition_path)
    existing = _path_cache.get((output_path, definition_path), None)
    if existing:
        return existing

    # we will attempt to calculate relative path vs. pythonsv root, and if that
    # doesn't work, we will do relative to the definition file
    relative_path = None
    # try to do relative to pysv root

    if common is not None and not settings.USE_ONLY_RELATIVE_PATHS:
        base_path = svtools.common.path.getpysvpath()
        # if definition path is definitely in pysv tree:
        if base_path is not None and base_path in definition_path:
            relative_path = "{path}" + os.sep + definition_path.replace(base_path, "")

    # try to find relative to the site packages directory
    # -- this is missing in some py2 virutal environments
    if relative_path is None:
        getsitepackages = getattr(site, "getsitepackages", None)
        if getsitepackages is not None:
            check_dirs = copy(getsitepackages())
            if sys.prefix in check_dirs:
                check_dirs.remove(sys.prefix)
        else:
            from distutils.sysconfig import get_python_lib
            check_dirs = [get_python_lib()]
        # see if the lmdb path has any of these paths
        for p in check_dirs:
            if p in definition_path:
                relative_path = "{path}" + os.sep + definition_path.replace(p, "")
                break


    if relative_path is None:
        # we didn't use pysv_root so use relative path
        base_path = definition_path
        # calculate relative path
        relative_path = os.path.relpath(
            os.path.abspath(base_path),
            os.path.dirname(
                os.path.abspath(output_path)
            ),
        )
    # make sure we use ONLY / because the lmdb may later be used in linux
    relative_path = relative_path.replace("\\","/")
    _path_cache[(output_path, definition_path)] = relative_path
    return relative_path


def _pickle_debug(creator_obj):
    try:
        # we assume it is the target info that is the problem
        pickle.dumps(creator_obj.target_info)
    except:
        _LOG.info("pickle exception", exc_info=True)         # not sure  if we should log again...
        _LOG.result("Failed while pickling: %s, attempting to recover"%(creator_obj.name))
        _LOG.result("\tdef_path: %s"%(creator_obj.def_path))
        _LOG.result("\tdef_file_path: %s"%(creator_obj.def_file_path))
        target_info = {}
        for k, v in creator_obj.target_info.items():
            next_k = k
            try:
                pickle.dumps(k)
            except Exception as e:
                try:
                    next_k = 'REMOVED BAD KEY: \'%s\''%repr(k)
                    _LOG.result("\tremoved bad key: %s" %repr(k))
                except Exception as e:
                    next_k = 'REMOVED BAD KEY'
                    _LOG.result("\tremoved bad key")
            try:
                pickle.dumps(v)
            except Exception as e:
                try:
                    v = 'REMOVED BAD VALUE: \'%s\''%repr(v)
                except Exception as e:
                    v = 'REMOVED BAD VALUE'
                _LOG.result("\tremoved bad value for key: %s"%next_k)
            target_info[next_k] = v
        creator_obj.target_info = target_info
    # On to the sub components
    for sub in creator_obj.subs:
        _pickle_debug(sub)


def write(components, output_path, definition_path=None):
    """
    Write an XML that statically describes how to build the components.

    Args:
        components : list of component objects that we should save the targetinfo for
        output_path : path to write the static xml or dictionary to
        definition_path : write the definition to disk also

    - output_path will choose pickle if path ends in ".spk" or XML if path
      ends in '.xml'

    Note:
        when writing the value file the saved path is relative to the pythonsv
        root directory if it is installed and is within the same directory tree

    """
    global _PATH_CHECKS
    _PATH_CHECKS = {}
    assert isinstance(components, (list, ComponentGroup)), "Components must be a list"
    assert isinstance(output_path, basestring), "output path must be a string"
    #######################################################
    # if definition path given, write out definition first
    #######################################################
    if definition_path is not None:
        # consider dropping this support as it is unlikely used
        # if writing definitions, we assume they are all from the same definition object
        components[0].definition.tolmdb.write(definition_path, access="w")

    # needed only for XML...delete when we are done updating it to support definitions
    # from other files
    # try to build path for future reference between our static file and definition file
    if definition_path is not None:
        # consider dropping this support as it is unlikely used
        relative_path = _calc_definition_path(output_path, definition_path)

    # create our creator objects
    # pickle it out to a file
    if output_path.endswith((_EXT_PICKLE, _EXT_XZ_PICKLE)):
        comps = [_CompCreator.from_component(c, output_path, definition_path) for c in components]
        system = {
            "version": 2,
        }
        if output_path.endswith(_EXT_XZ_PICKLE):
            f = lzma.open(output_path, 'w', preset=settings.STATIC_DISCOVERY_COMPRESSION)
        else:
            f = open(output_path, 'wb')
        with f:
            pickle.dump(system, f, protocol=2)
            try:
                pickle.dump(comps, f, protocol=2)
            except Exception as e:
                _LOG.error("Error pickling something during save...attempting to find out what")
                _LOG.info("picking error info", exc_info=True)
                for c in comps:
                    _pickle_debug(c)

    elif output_path.endswith(_EXT_XML):
        comps = [_CompCreator.from_component(c, output_path, definition_path) for c in components]
        system = {
            "version": 2,
        }
        # Create our top level XML nodes and add components to them
        root = ET.Element('StaticNamedNodes')
        if definition_path is not None:
            # if a definition path was given, then save that in to the file as well
            xmlnode = ET.SubElement(root, "ComponentFile")
            # if definition path was given, then relative path was calculated
            xmlnode.text = relative_path
        # ok, now...get the components
        for c in comps:
            c.to_xml(root)
        # no clue why but i couldn't figure out how to get etree to just
        # return the string...had to still call ET.tostring here...
        indent_xml(root)
        with open(output_path, "wb") as f:
            f.write(ET.tostring(root))
    elif output_path.endswith((_EXT_JSON, _EXT_XZ_JSON)):
        if not settings.ALPHA_FEATURE_STATIC_JSON:
            raise ValueError("extension not officially supported yet")
        if definition_path:
            raise RuntimeError("definition database path override not supported for JSON formats")
        encoder = NamedNodesValueEncoder(output_path)
        output = {
            "static_discovery": {
                'version': 1,
                'components': list(components),
            }
        }
        comp_json = encoder.encode(output)
        if output_path.endswith(_EXT_XZ_JSON):
            with lzma.open(output_path, 'wb', preset=settings.STATIC_DISCOVERY_COMPRESSION) as f:
                f.write(comp_json.encode())
        else:
            with open(output_path, "w") as f:
                f.write(comp_json)
    else:
        raise ValueError("extension not supported for: %s"%output_path)

class NamedNodesValueEncoder(json.JSONEncoder):
    def __init__(self, outputpath, *args, **kwargs):
        self.output_path = outputpath
        super().__init__(*args, **kwargs)

    def default(self, obj):
        if isinstance(obj, NamedComponent):
            def_file_path = obj.definition.origin.info.get("definition_filepath", None)
            # try:
            #     json.dumps(dict(obj.target_info))
            # except:
            #     import traceback
            #     traceback.print_exc()
            #     import pdb;pdb.set_trace()
            data = dict(
                name=obj.name,
                target_info=obj.target_info
            )
            if def_file_path:
                def_file_path = _calc_definition_path(self.output_path, def_file_path)
                data['definition_file_path'] = def_file_path
                data['definition_comp_path'] = obj.definition.path
            if obj.sub_components:
                data['subs'] = list(obj.sub_components.values())
            if obj.__class__ != NamedComponent:
                data['value_class'] = "{}:{}".format(obj.__class__.__module__, obj.__class__.__name__)
            return data
        elif isinstance(obj, odict):
            return obj._data
        elif isinstance(obj, ipccli.BitData):
            return int(obj)
        elif obj is LookupError:
            # return "<LookupError>"  # ?
            return "NaN"
        # not one of our objects
        return json.JSONEncoder.default(self, obj)

def _create_from_json(parent, comp_dict, input_path, error_on_missing):
    # TODO: important for performance to add to parent as we are created instead of the whole tree
    definition_file_path = comp_dict.get('definition_file_path', None)
    definition_comp_path = comp_dict.get('definition_comp_path', None)
    value_class = comp_dict.get("value_class", "namednodes.comp:NamedComponent")
    if definition_file_path is None:
        comp_def = NamedComponentDefinition(comp_dict['name'], value_class=value_class)
    else:
        if definition_comp_path is None:
            raise RuntimeError("definition file path present, but missing component path")
        comp_def = _load_definition_from_path(definition_file_path, input_path)
        if comp_def is None:
            if error_on_missing:
                raise RuntimeError("definition file path present, but missing component path")
            else:
                comp_def = NamedComponentDefinition(comp_dict['name'], value_class=value_class)
        else:
            comp_def = comp_def.get_by_path(definition_comp_path, default=None)
            if comp_def is None:
                if error_on_missing:
                    raise RuntimeError("definition file path present, but missing component path")
                else:
                    comp_def = NamedComponentDefinition(comp_dict['name'], value_class=value_class)

    comp = comp_def.create(comp_dict['name'], info=comp_dict['target_info'])
    if parent:
        parent.add_component(comp, skip_sort=True)
    for sub in comp_dict.get('subs', []):
        child_comp = _create_from_json(comp, sub, input_path, error_on_missing)
    return comp

# used to short cut finding compdef everytime through this during a load
_LOAD_PATH_CHECKS = {}
def _load_definition_from_path(compdef_path, static_input_path):
    """
    helper function used to calculate full path to definition file
    Args:
        compdef_path : relative path to definition file that is from our static file
        input_path : path to our static info file

    Returns:
        component definition object
    """
    if compdef_path is None:
        raise ValueError(
            "component definition not passed in and is not in static file")
    if compdef_path in _LOAD_PATH_CHECKS:
        return _LOAD_PATH_CHECKS[compdef_path]
    # default
    full_def_path = None
    # make some attempts to fill in full definition path with the common's getpysvpath
    # try to fill in pysv_root if it exists
    if compdef_path.find("{pysv_root}")>=0:
        if common is not None:
            full_def_path = compdef_path.format(pysv_root=svtools.common.path.getpysvpath())
        else:
            raise Exception("common module missing but definitin file specifies pysv_root\n"
                            "make sure you setup the pythonsv path")
    elif compdef_path.find("{path}") >= 0:
        # make sure we find any compressed files as well...
        def all_paths():
            for p in sys.path:
                test_path = compdef_path.format(path=p)
                yield test_path
                for ext in UnzipCheck.extensions:
                    yield test_path + ext
        for p in all_paths():
            if os.path.exists(p):
                full_def_path = p
                break

    # we didn't use pysv_root apparently
    if full_def_path is None:
        # now build up full path
        full_def_path = os.path.join(
            os.path.dirname(static_input_path),
            compdef_path)
    try:
        component = GetComponentFromFile(full_def_path, cache=True)
        _LOAD_PATH_CHECKS[compdef_path] = component
    except ValueError as e:
        #if str(e).endswith("not found") and (settings.CLASSIFICATION not in [600,None] or (not settings.STATIC_REQUIRES_DATABASE)):
        if str(e).endswith("not found") and (not settings.STATIC_REQUIRES_DATABASE):
            _LOAD_PATH_CHECKS[compdef_path] = None
            # print warning?
        else:
            # unexpected error, or classifications is higher, if this happens internally
            # we want to debug these
            raise
    return _LOAD_PATH_CHECKS[compdef_path]


def write_values(components, output_path, **kwargs):
    """
    Write the given components to an offline file

    components : list of components to write
    output_path : file to write to
    comp_paths : optional list of paths to use for saving only some values instead of all values
    raise_exceptions : to stop on first error when saving

    Note:
        for now comp_paths is assumed to be a starting point, and all components and nodes
        under those paths will be logged

    """
    global _LOAD_PATH_CHECKS
    _LOAD_PATH_CHECKS = {}
    assert len(components) > 0, "expecting at least one component"
    # used to specify which paths to save the values for
    comp_paths = kwargs.pop("comp_paths", None)
    raise_exceptions = kwargs.pop("raise_exceptions", False)
    if isinstance(comp_paths, basestring):
        comp_paths = [comp_paths]
    # not implemented, but may be needed one day
    #skip_paths = kwargs.pop("skip_paths", None)
    #recursive = kwargs.pop("recursive", True)
    assert len(kwargs)==0,"Unexpected kwargs: %s"%(list(kwargs.keys()))

    # 1. create an offline version of the components
    # 1a. create things that know how to create components
    comp_creators = [_CompCreator.from_component(c, output_path=output_path) for c in components]
    definition = components[0].definition
    # 2b. load it load them back
    offline_components = []
    for creator in comp_creators:
        comp = creator.create(definition, offline_filepath=output_path)
        # manually sort all the component groups after they have all been added
        comp.sort_component_groups()
        if comp is not None:
            offline_components.append(comp)
    # turn offline
    for offlinec in offline_components:
        default_access = AccessRegisterStoredValues if isinstance(offlinec, RegisterComponent) else AccessStoredValues
        _switch_offline(offlinec, default_access=default_access)

    from ..plugins.lmdb_dict import CacheContext

    # 2. copy values for all nodes that are not 'field like' nodes
    access_cache = {}
    # turn definition caching off so memory doesn't balloon:
    last_message_len = 0
    # paths not given write ALL
    with CacheContext(False):
        for c, orig_comp in enumerate(components):
            # no comp path provided just use the original comp as the starting point
            if comp_paths is None:
                start_comps = [ orig_comp ]
            else:
                # we have comp_paths, turn them in to a list of 'start_comps' that
                # we should walk through
                start_comps = []
                for comp_path in comp_paths:
                    try:
                        sub_comp = orig_comp.get_by_path(comp_path)
                        if isinstance(sub_comp, ComponentGroup):
                            start_comps.extend( list(sub_comp) )
                        else:
                            start_comps.append(sub_comp)
                    except ValueError:
                        _LOG.warning("%s does not have path %s"%(orig_comp.name, comp_path))
            # ok, now we have our list of starting points, go for it:
            for top_comp in start_comps:
                for comp in top_comp.walk_components():
                    message = "Saving: %s"%comp.path
                    print("\r" + " "*last_message_len + "\r"+ message[:79], end="")
                    last_message_len = len(message)%79 # never more than 79 chars
                    # start component walks with a clean is available cache
                    comp.target_info['is_available_cache'] = {}
                    for node in comp.walk_nodes():
                        # if we are not going to change access in offline, then it is
                        # because it reads from parent or in general we must not need
                        # to save its value (like fields)
                        try:
                            access_class = node.access_class
                        except RuntimeError: # this means we could not find the right access class
                            access_class = None

                        if access_class in _ACCESSES_WITHOUT_VALUES:
                            continue
                        elif _read_if_available(node, raise_exceptions=raise_exceptions):
                            value = node.value
                        else:
                            # -1 is used for stubs and means to just put the default
                            # when we are taking a snapshot we want to show we are missing
                            # the info somehow due to couldnt read the register
                            value = -2
                        offline_components[c].get_by_path(node.path).write(value)
    print()
    # 3. write to file
    write(offline_components, output_path)


def load(input_path, definition=None, **kwargs):
    """
    load and return a component group from the specified path
    Args:
        input_path : path to xml or pickle to load from
        definition : component definition to use while creating components

    if definition not provided, it is assumed that the input_path
    was saved along with a definition filepath.
    """
    global _LOAD_PATH_CHECKS
    # use to cut down on path checks done during load
    _LOAD_PATH_CHECKS = {}

    # used mostly for unittests, that is why it is not documented
    error_on_missing = kwargs.pop("error_on_missing", False)
    if len(kwargs) > 0:
        raise ValueError("unexpected arguments: %s"%str(kwargs))
    if isinstance(definition, basestring):
        # don't check for file existance, it may be a path to a file that needs to be uncompressed
        definition = GetComponentFromFile(definition)

    # see if it is a compressed file
    if input_path.endswith((".zip", ".gz")):
        # should be a single file
        input_path = UnzipCheck(input_path).decompress()[0]

    # ########### read Pickled static file
    if input_path.endswith((_EXT_PICKLE, _EXT_XZ_PICKLE)):
        # read in our system dictionary
        if input_path.endswith(_EXT_PICKLE):
            f = open(input_path, "rb")
        else:
            # maybe some check that if the file size is huge
            # we would use lzma.open instead
            # must be xz pickle
            if PY2:
                f = StringIO(
                    lzma.decompress(open(input_path, 'rb').read())
                )
            else:
                f = BytesIO(
                    lzma.decompress(open(input_path, 'rb').read())
                )
        try:
            system = pickle.load(f)
            if system['version'] < 2:
                # version 1 only existed for short bit in the very early days of namednodes
                # and did not support the fact that a tree can be built from multiple lmdbs
                raise DeprecationError("Version 1 of SPK is no longer supported")
            comp_creators = pickle.load(f)
        finally:
            f.close()

        # ######## get definition if not passed in
        if definition is None and system["version"] == 1:
            compdef_path = system.get("definition_filepath", None)
            definition = _load_definition_from_path(compdef_path, input_path)

    # ########### read XML static file
    elif input_path.endswith(_EXT_XML):
        root = ET.parse(input_path).getroot()
        # #####
        if definition is None:
            # older version of the xml had a ComponentFile at the top level
            compdef_path = root.find("ComponentFile")
            if compdef_path is not None:
                compdef_path = compdef_path.text
        else:
            compdef_path = None
        # #####
        # turn them in to component creator objects
        comp_creators = [_CompCreator.from_xml(c, input_path, definition_path=compdef_path)
                         for c in root.findall("Component")]

    elif input_path.endswith((_EXT_JSON, _EXT_XZ_JSON)):
        if definition is not None:
            raise RuntimeError("JSON formats do not support definition database overrides")
        if input_path.endswith(_EXT_XZ_JSON):
            with lzma.open(input_path) as f:
                jsondata = json.loads(f.read().decode(), parse_constant=(lambda x: KeyError))
        else:
            import time
            s = time.time()
            with open(input_path, "r") as f:
                jsondata = json.loads(f.read(), parse_constant=(lambda x: KeyError))
        comp_list = [_create_from_json(None, c, input_path, error_on_missing) for c in jsondata['static_discovery']['components']]
        return comp_list
    else:
        ext = os.path.splitext(input_path)
        raise ValueError("Unsupported format extension: %s" % ext)

    # should have a list of creators at this point
    # do creation and return
    clist = []
    # specifying no log will cause error to get immediately raised
    error_file = None if error_on_missing else StringIO()
    for creator in comp_creators:
        comp = creator.create(definition, offline_filepath=input_path)
        if comp is not None:
            clist.append(comp)

    if error_file is not None:
        messages = error_file.getvalue()
        if len(messages):
            warning_file = "static_load_warnings.txt"
            _LOG.warning("Some paths were in your offline file, but not in your definition")
            _LOG.warning("    mismatches can be found in %s"%warning_file)
            with open(warning_file, "w") as f:
                f.write(messages)

    # if this file contains an old version of the offline data...we should conver it
    # if the file was in the previous format, then we should convert it
    for c in clist:
        # if no offline data, dont worry about it
        if AccessSnapshot._cache_name in c.target_info:
            stored_cache_format = c.target_info.get(AccessSnapshot._cache_format, _StoredValuesFormat.path_tuple)
            if stored_cache_format == _StoredValuesFormat.path_tuple:
                AccessStoredValues._convert_tuple_to_dict(c)

    return clist

def load_offline(input_path, definition=None, value_file=None,**kwargs):
    """
    create components from the specified files and turn all
    the accesses into "AccessStoredValue"
    """
    comps = load(input_path, definition, **kwargs)
    # update accesses

    # add values if needed
    if value_file is not None:
        raise ValueError("not supported yet")

    for comp in comps:
        default_access = AccessRegisterStoredValues if isinstance(comp, RegisterComponent) else AccessStoredValues
        _switch_offline(comp, default_access=default_access)

    return comps

# this code is shared with snapshot
def _switch_offline( comp, **kwargs):
    """turn a list of components in to offline accesss"""
    accesses_map = kwargs.pop("accesses_map", _ACCESSES_OFFLINE_MAP)
    default_access = kwargs.pop("default_access", AccessStoredValues)
    if len(kwargs):
        raise ValueError("Unknown kwargs: %s"%str(kwargs))

    comp.target_info['offline'] = True
    # only save mapping if a previous save is not already there
    if '_previous_access_mapping' not in comp.__dict__:
        comp.__dict__['_previous_access_mapping'] = []

    # add a new previous mapping to the list...
    previous_mapping = {}
    previous_mapping.update(comp.access_classes)
    comp.__dict__['_previous_access_mapping'].append(previous_mapping)
    # save off any previous defaults
    previous_mapping['_previous_default'] = comp.access_classes.get('default', LookupError)
    comp.access_classes['default'] = default_access

    # for classname and class object
    for cname, aclass in comp.access_classes.items():
        next_class = accesses_map.get(aclass, None)
        if next_class is not None:
            comp.access_classes[cname] = next_class
            continue
        else:
            # need to look for 'issubclass' due to things like AFD where
            # we generate a ton of sub access classes
            for orig_class, next_class in accesses_map.items():
                if issubclass(aclass, orig_class):
                    comp.access_classes[cname] = next_class
                    break
            # for loop didn't break
            else:
                # no class mapping found, must be default stored values one
                comp.access_classes[cname] = default_access


# put a previously "offline" component back to being online...
def _switch_online(comp, **kwargs):
    if '_previous_access_mapping' not in dir(comp):
        raise RuntimeError("cannot restore this component since is missing original class mapping")
    # pop off the last previous mapping and restore it
    previous_mapping =  comp._previous_access_mapping.pop()
    # because access class dictionaries are kep as references, we can't simply copy to fix the original one..
    comp.access_classes.update(previous_mapping)
    if previous_mapping.get('_previous_default', LookupError) is not LookupError:
        comp.access_classes['default'] = previous_mapping['_previous_default']
    # no longer in offline (or snapshot) mode
    if len(comp._previous_access_mapping) == 0:
        comp.target_info['offline'] = False
        del comp.__dict__['_previous_access_mapping']

def add_offline_access(original, replacement):
    """
    Add an access class mapping to determine what class should be substitued when
    we turn a component in to offline mode

    original (type) : class that we should replace in offline moode
    replacement (type) : class that should be used instead of original in offline mode
    """
    global _ACCESSES_OFFLINE_MAP
    assert issubclass(original, NodeAccess), "original must be a NodeAccess class"
    assert issubclass(replacement, NodeAccess), "replacement must be a NodeAccess class"
    _ACCESSES_OFFLINE_MAP[original] = replacement


def add_access_without_value(access_class):
    """
    Add to our list of known accesses that we should not save the value for when someone
    creates a new offline file with the current values in the component
    """
    assert issubclass(access_class, NodeAccess), "access_class must be a NodeAccess class"
    global _ACCESSES_WITHOUT_VALUES
    _ACCESSES_WITHOUT_VALUES.add(access_class)

class StaticDiscovery(Discovery):
    """
    .. code-block:: python

        from namednodes.discoveries.static_xml import StaticDiscovery
        example_discovery = StaticDiscovery( static_xml_path, lmdb_filepath)
        namednodes.discovery.register("example", example_discovery)
    """
    #: filename of component XML to load
    static_filename = ""
    #: path to definition of component
    definition_path = None
    #: Used to make the discovery load an offline version of the components
    offline = False
    _offline_dir = None

    def __init__(self, static_path=None, definition_path=None, offline_dir=None, offline=False):
        """
        Args:
            static_path (str) : path or filename to offline file to use for discovery
                                combined with offline_dir if it is given
            definition_path (str) : used to specify where definition components
                                    should be pulled from
            offline_dir (str) : directory of where to save/load static files to/from
                                 (if not specified, current-working-dir is used)
            offline (bool) : whether the default behaviour is offline mode or not
        """
        super(StaticDiscovery, self).__init__()
        self.definition_path = definition_path
        self.static_filename = static_path
        self._offline_dir = offline_dir
        self.offline = offline

    def share_items(self, discovery):
        """
        use this so that static functions operate on the same discovery
        items as the provided discovery
        """
        self.discovered = discovery.discovered
        self.discovered_dict = discovery.discovered_dict

    @property
    def offline_dir(self):
        """"""
        return self._offline_dir

    @offline_dir.setter
    def offline_dir(self,value):
        if not os.path.exists(value):
            raise SystemError("Path not found: %s"%value)
        if not os.path.isdir(value):
            raise SystemError("Path is not a directory: %s"%value)
        self._offline_dir = value

    @property
    def static_path(self):
        if self.offline_dir is not None:
            assert self.static_filename is not None, "offline dir specified, but no file name has been setup or chosen"
            static_path = os.path.join(self._offline_dir, self.static_filename)
        else:
            static_path = self.static_filename
        return static_path

    def find_all(self):
        """
        Parse the file specified in static_path and create the components
        specified within it
        """
        # our component files are assumed to be relative to this output directory
        if self.offline:
            return load_offline( self.static_path,
                                self.definition_path)
        else:
            return load(self.static_path, self.definition_path)

    def save(self, filename=None):
        """
        save known components to filename

        Args:
            filename (str) : file *name* (full path not needed) of file to write to.
                supported extensions are "spk" or "xml". "spk" is a compressed pickle
                file. xml is readable but quite large.

        Note:
            - this does not save all the values, just the discovery and/or
            - this function is most useful and fastest if you are already using
              an offline component and want to save any updated values to a new file
        """
        if filename is None:
            filename = self.static_path
        elif self.offline_dir is not None:
            filename = os.path.join(self.offline_dir, filename )
        # make sure we save with a known extension
        if not filename.endswith(_STATIC_EXT):
            filename += _EXT_PICKLE

        comps = self.get_all()
        # for discovery we assume the component definitions are already on disc
        print('Saving to %s'%filename)
        write(comps, filename)

    def get_choices(self):
        """return list of filenames from the offline path"""
        assert self.offline_dir is not None,"Only valid if offline path was set"
        found = []
        for f in os.listdir(self.offline_dir):
            if f.endswith(_STATIC_EXT):
                found.append(f)
        found.sort()
        return found

    def choose(self, **kwargs):
        """choose the next offline file to load from"""
        # this is for debug/testing
        if len(kwargs)>1 or (len(kwargs)==1 and'choice' not in kwargs):
            raise ValueError("kwargs not supported: %s"%str(kwargs))
        # actual relavent code for getting values
        choices = self.get_choices()
        if len(choices)==0:
            raise ValueError("no other offline files found in: %s"%self.offline_dir)
        self.static_filename = menu_prompt(choices,**kwargs)











