
# INTEL CONFIDENTIAL
# Copyright 2014 2018 Intel Corporation
#
# The source code  contained or  described herein and  all documents related to
# the source code  ("Material") are owned by Intel Corporation or its suppliers
# or licensors.  Title to the  Material  remains with  Intel Corporation or its
# suppliers  and licensors. The Material contains trade secrets and proprietary
# and  confidential  information  of  Intel  or  its  suppliers  and  licensors.
# The Material  is protected  by worldwide  copyright and trade secret laws and
# treaty provisions.  No part of the Material  may  be used, copied, reproduced,
# modified, published, uploaded, posted, transmitted, distributed, or disclosed
# in any way without Intel's prior express written permission. No license under
# any  patent,  copyright, trade secret or other intellectual property right is
# granted  to  or conferred upon you by disclosure or delivery of the Materials,
# either expressly, by implication, inducement, estoppel or otherwise.
# Any license under such intellectual property rights must be express and
# approved by Intel in writing.



"""
===================
Socket Discovery
===================

.. autoclass:: SocketDiscovery
    :members:

"""
from __future__ import absolute_import
from __future__ import division
from __future__ import print_function

from ..discovery import Discovery
from ..comp import (
    NamedComponent,
    NamedComponentDefinition,
    ComponentGroup,
    GetComponentFromFile,
)
from ..utils._py2to3 import *
from ..utils import SimilarSteppings
from ..logging import getLogger
import imp
import sys

_LOG = getLogger("socket")

class SocketDiscovery(Discovery):
    # must be inherited from
    _installed = None
    # will provide a mapping of components to device alias
    _component_to_device = None
    # use to control whether we load socket override
    # derivied classes can set this to false
    override_components_socket = True


    def __init__(self,*args,**kwargs):
        super(SocketDiscovery,self).__init__(*args,**kwargs)
        # created only on initial socket discovery creation, after
        # this we will use "clear" to allow folks to keep a reference
        self._component_to_device = {}
        # install our override
        if not self._installed and self.override_components_socket:
            sys.meta_path.insert(0, ModuleOverride("components.socket", self))
            SocketDiscovery._installed=True
            # print warning if pysv is already installed ??
            previous_mod = sys.modules.get('components.socket', None)
            if previous_mod is not None:
                if isinstance(previous_mod, SocketDiscovery):
                    # we have already intercepted socket discovery, it is ok...
                    # TODO: should we replace with 'newer' one here?
                    sys.modules['components.socket'] = self
                    #return
                else:
                    print("WARNING! someone already imported or overrode components.socket")
                    del sys.modules['components.socket']
                    # only redo import here because someone else had done an import
                    # typically do not attempt components.socket imports, let user ask for them
                    import components.socket

    @property
    def component_to_device (self):
        # do thought of doing this as a copy, but we changed that so folks get a
        return self._component_to_device


    def getAll(self, forceSearch=False):
        """Returns a list of process socket objects found in the system

        Args:
            forceSearch (bool) : whether to force discovery (default=False)


        For forceSearch: set this to True to force the code to redo the
        discovery process and return the list of socket obejcts found.
        It is not recommended forceSearch=True within scripts
        """
        return self.get_all(refresh=forceSearch)

    def getBySocket(self, socketNumber=None, forceSearch=False):
        """
        Returns the processor socket specified.

        Args:
            socketNumber (int) : number of socket object to search for
            forceSearch (bool) : whether to force discovery

        If a socket is passed in (instead of a number) then it is simply
        returned instead of searching. This is so that everyone's script
        can be flexible on taking a socket number or a socket object
        and just call this function to make sure the end result
        is just an object

        If socketNumber is None, then the first socket found will be returned
        """
        if forceSearch is False and isinstance(socketNumber, NamedComponent):
            return socketNumber
        socket_group = self.get_all(refresh=forceSearch)
        if socketNumber is None and len(socket_group)>0:
            return socket_group[0]

        # turn the socket object in to a number
        if isinstance(socketNumber, NamedComponent):
            socketNumber = socketNumber.target_info.get('socket_num', None)
            if socketNumber is None:
                raise ValueError("Unexpected object passed to "
                                 "getBySocket and missing socket_num")

        for s in socket_group:
            if s.target_info.get('socket_num', None) == socketNumber or \
               s.target_info.get('socketNum', None) == socketNumber:
                return s

        raise ValueError("Socket %s not present"%socketNumber)

    def getSocketGroup(self, socketNumber=None, forceSearch=False):
        """
        Returns a group of the processer socket objects specified

        Args:
            socketNumber (int) : (optional) number of socket object to
                search for
            forceSearch (bool) : whether to force discovery

        A combo of getBySocket and getAll, this function will return
        all socket objects by default. However, if socketNumber is
        specified, then only that socket will be returned
        """
        slist = self.get_all(refresh=forceSearch)
        # if no socket number just return the whole list
        if socketNumber is None:
            return slist
        # pysv supported a list for socketNumber, so make
        # sure we have a list before we filter out the sockets
        # we want
        if isinstance(socketNumber,(list, ComponentGroup)):
            # make a copy because we modify and fill in with numbers
            num_list = list(socketNumber)
        else:
            num_list = [ socketNumber ]
        # we could still have a list of numbers OR a list of socket objs
        for n,sobj in enumerate(num_list):
            # if is a component, make sure we have a socket num?
            if isinstance(sobj, NamedComponent):
                # replace object with number
                snum = sobj.target_info.get('socket_num', None)
                snum = sobj.target_info.get('socketNum', snum)
                if snum is None:
                    raise ValueError("Unexpected object passed to "
                                 "getBySocket and missing socket_num")
                # save off number
                num_list[n] = snum
            # else: must have been a number
        # ---- now add to new group based on socket number(s)
        # list of components that we will add to our new component_group
        comp_list = []
        for sobj in slist:
            # not sure which way socket num was specified....
            snum = sobj.target_info.get('socket_num', None)
            snum = sobj.target_info.get('socketNum', snum)
            if snum in num_list:
                comp_list.append( sobj )
        # return our newly created group
        return ComponentGroup(comp_list, "sockets")

    def getByName(self, name, forceSearch=False):
        """
        returns the process socket oby specified by name
        """
        for s in self.get_all(refresh=forceSearch):
            if s.name == name:
                return s
        raise ValueError("Socket %s not present"%name)

    def _add_to_mapping(self, comp):
        """simple function for adding to our mapping"""
        alias = comp.target_info.get('alias', None)
        if alias is not None:
            self._component_to_device[comp.path] = comp.target_info['alias']

    def _create_cpu(self, device, last_socket, cpu_path, cpu_similar_steppings=None):
        """create new cpu object from socket or from lmdb path"""
        cpu_def = last_socket.definition.sub_components.get("cpu", None)
        if cpu_def is None and cpu_path is not None:
            cpu_def = self._get_cpu_def(device, cpu_path, cpu_similar_steppings)
        elif cpu_def is not None and cpu_path is not None:
            _LOG.warning("socket_discovery warning: cpu path was provided, but socket database also has a cpu")
            cpu_def = self._get_cpu_def(device, cpu_path, cpu_similar_steppings)

        last_cpu = cpu_def.create_tree("cpu", skip=['module','core'])
        last_socket.add_component(last_cpu)
        return last_cpu

    def _create_module(self, device, last_cpu):
        """create module object if it exists in the cpu database, else, return the cpu definition back"""
        cpu_def = last_cpu.definition
        instance = getattr(device,"instanceid", None)
        if instance is None: # then hope it was a cpu
            instance = getattr(device, "coreid")
        if instance is None:
            raise RuntimeError("Could not find instanceid or coreid while building module object")
        if 'module' in cpu_def.sub_components:
            last_module = last_cpu.definition.module.create_tree(
                "module%d" % instance,
                skip=['core'])
            last_module.target_info['did'] = device.did
            last_module.target_info['alias'] = device.alias
            last_cpu.add_component(last_module)
        else:
            # some teams insist on NO module, if that is the case
            # we redirect back to the cpu
            last_module = last_cpu
        return last_module

    def _create_core(self, device, last_module):
        """create new core from the cpu/module definition"""
        core_def = last_module.definition.sub_components.get("core", None)
        assert core_def is not None, "module or cpu is missing core definition object"
        last_core = core_def.create_tree("core%d" % device.coreid, skip=['thread'])
        last_core.target_info['did'] = device.did
        last_core.target_info['alias'] = device.alias
        coreid = getattr(device,"coreid", None)
        if coreid is None:
            coreid = getattr(device,"instance", 0) # default?
        last_core.target_info['instance'] = coreid
        last_module.add_component(last_core)
        # ?? not sure why this was here...
        # last_sock.target_info['instance'] = socketid
        return last_core

    def _create_thread(self, device, last_core):
        thread_def= last_core.definition.sub_components.get("thread",None)
        assert thread_def is not None, "database missing thread component"
        last_thread = thread_def.create_tree("thread%d"%device.threadid)
        last_thread.target_info['did'] = device.did
        last_thread.target_info['alias'] = device.alias
        last_thread.target_info['msr_did'] = device.did
        last_thread.target_info['isenabled'] = device.isenabled
        last_core.add_component(last_thread)
        return last_thread

    def find_all_from_baseaccess(self, definition_path, similar_steppings=None, **kwargs):
        """
        This find all uses the baseaccess (globalbase) to get the device-list,
        and restarts the baseaccess before calling find_all_from_devicelist.
        All arguments and options are as described in find_all_from_devicelist.
        """
        from svtools.common import baseaccess
        base = baseaccess.getglobalbase()
        restart = getattr(base, "restart", None)
        if restart:
            restart()
        return self.find_all_from_devicelist(base.devicelist, definition_path, similar_steppings, **kwargs)

    def find_all_from_devicelist(self, devicelist,
                                 definition_path,
                                 similar_steppings=None,
                                 **kwargs):
        """
        this find all uses the given devicelist and assumes that we have a
        network that nests the modules/cores/threads under the cltapc/uncore. If a module
        is not found, then it is assumed to have one module per core so that all projects
        have the same hierarchy.

        Args:
            definition_path : definition object to build our components from OR path to a definition file
            similar_steppings: tuple used to fill in socketdef (if it is a path) with the correct
                stepping to add to the path

            optional:

            cpu_path (str) : path to core definition object to use instead of definition_path
            cpu_similar_steppings (dict) : core specific similar_steppings to use
            alias_cpu_to_socket_tap (bool) : create socket.tap.name for socket.path.name.tap components (default False)
            show_disabled (bool) : whether to force even disabled to be in the devicelist
            core_devicetype (str) : create core components only for devices of this devicetype
            thread_devicetype (str) : create thread components only for devices of this devicetype

        The socketdef must have the following components with this specific path:

            socketdef.cpu.module.core.thread

        """
        # each time we create a component we need to fill in the device alias
        # do not create a new one, simply clear it
        self._component_to_device.clear()
        socket_list = []
        # we have a whole bunch of smaller "create" calls since we sometimes have all the devicelist hiearchy
        # but we have the component hiearchy

        # if you add something here, be sure to update it in the section that finds the uncore
        last_sock = None
        last_cpu = None
        last_core = None
        last_thread = None
        last_module = None
        # used to flag that we didn't find a module
        no_modules = True
        # versions 0.50.0 and before passed in a definition object
        # wait till we get to seeing the first uncore (and can get the stepping) before
        # we load the file
        if isinstance(definition_path, NamedComponentDefinition):
            socketdef = definition_path
            cpu_def = getattr(socketdef, "cpu", None)
            # set the path to be None so we know we got a definition passed in
            definition_path = None
            assert 'cpu_path' not in kwargs,"if cpu path passed in, definition_path must be a path as well"
        else:
            socketdef = None
            cpu_def = None

        similar_steppings = similar_steppings or {}
        # in case cpu is in different file than socket
        cpu_path = kwargs.pop("cpu_path", None)
        # default to use sockets similar steppings
        cpu_similar_steppings = kwargs.pop("cpu_similar_steppings", similar_steppings)
        alias_cpu_to_socket_tap = kwargs.pop("alias_cpu_to_socket_tap", False)
        show_disabled = kwargs.pop("show_disabled", False)
        # used to find non-GPC or fewer GPC devicetypes:
        core_devicetype = kwargs.pop("core_devicetype", None)
        thread_devicetype = kwargs.pop("thread_devicetype", None)

        assert len(kwargs) == 0, "Unexpected keyword arguments: {}".format(str(list(kwargs.keys())))

        # turn the tuple mapping in to a lookup if we didn't get our similar steppings class
        if isinstance(similar_steppings,dict):
            similar_steppings = SimilarSteppings(similar_steppings)
        if isinstance(cpu_similar_steppings, dict):
            cpu_similar_steppings = SimilarSteppings(cpu_similar_steppings)

        for device in devicelist:
            if device.nodetype == "uncore":
                # if we don't have a socket definition, then we must need to grab it using path and
                if socketdef is None:
                    # try to get stepping from our similar map, but if it is not there, use default
                    # of what we found in the devicelist
                    socket_stepping = similar_steppings.get_primary(device.stepping)
                    socketdef = GetComponentFromFile(definition_path.format(stepping=socket_stepping))
                socketid = getattr(device,"socketid", None)
                if socketid is None:
                    socketid = getattr(device, "instanceid", None)
                if socketid is None:
                    socketid = getattr(device, "packageid", None)
                if isinstance(socketid, basestring):
                    socketid = int(socketid)
                if socketid is None:
                    raise RuntimeError("could not find a valid socketid, instanceid, or packageid off the device")
                last_sock = socketdef.create('socket%d'%socketid)
                socket_list.append(last_sock)
                last_sock.target_info['socketNum'] = last_sock.target_info['socket_num'] = socketid
                last_sock.target_info['deviceName'] = last_sock.target_info['device_name'] = device.devicetype
                last_sock.target_info['alias'] = device.alias
                last_sock.target_info['instance'] = socketid
                self._add_to_mapping(last_sock)
                last_sock.target_info['stepping'] = device.stepping
                ## create uncore and related info
                last_sock.add_component(socketdef.uncore.create_tree(), "uncore")
                last_sock.uncore.target_info['deviceName'] = last_sock.uncore.target_info['device_name'] = device.devicetype
                last_sock.uncore.target_info['stepping'] = device.stepping
                last_sock.uncore.target_info['did'] = device.did
                last_sock.uncore.target_info['alias'] = device.alias
                self._add_to_mapping(last_sock.uncore)
                # reset a few of our "last" objects....
                last_core = None
                last_thread = None
                last_module = None
                last_cpu = None
                # make sure a tap exists under the socket
                if alias_cpu_to_socket_tap:
                    # due to my bug in using tap instead of taps, we have to check for both
                    if 'tap' in socketdef.sub_components:
                        socket_tap = socketdef.tap.create("tap")
                        socket_tap.target_info['did'] = device.did
                        last_sock.add_component(socket_tap)
                    elif 'taps' in socketdef.sub_components:
                        # the same but it is set as taps
                        socket_tap = socketdef.taps.create("taps")
                        socket_tap.target_info['did'] = device.did
                        last_sock.add_component(socket_tap)
                    else:
                        # warning or just debug ???
                        _LOG.warning("tap component not found in socket definition")
                        socket_tap = None
                    if socket_tap is not None:
                        self._add_to_mapping(socket_tap)

            elif (device.nodetype == "box" and device.devicetype.count("MODULE")) or (
                        device.nodetype == "module"):
                assert last_sock is not None, "socket not found yet"
                if last_cpu is None:
                    last_cpu = self._create_cpu(device, last_sock, cpu_path, cpu_similar_steppings)
                # so that we know when we get to cores if the module is 'fake' or not
                no_modules = False
                # add module
                last_module = self._create_module(device, last_cpu)
                self._add_to_mapping(last_module)
                # now check for tap, and if it exists, create an alias under the socket
                if alias_cpu_to_socket_tap and socket_tap is not None:
                    if 'tap' in cpu_def.module.sub_components:
                        module_tap = cpu_def.module.tap.create_tree()
                        module_tap.target_info['did'] = device.did
                        module_tap.target_info['alias'] = device.alias
                        socket_tap.add_component(module_tap)

            elif device.nodetype == "core" and (
                    (getattr(device, "coregroup", "GPC") == "GPC" and core_devicetype is None) or
                    (core_devicetype and core_devicetype == device.devicetype)
            ):
                last_core_name = "core%d"%device.coreid
                # only add enabled threads
                isenabled = getattr(device,"isenabled", True)
                if isenabled or show_disabled:
                    # make sure we follow everyone's hiearchy and have a module
                    if last_cpu is None:
                        last_cpu = self._create_cpu(device, last_sock, cpu_path, cpu_similar_steppings)
                    if last_module is None or no_modules:
                        last_module = self._create_module(device, last_cpu)
                    # keep in mind last module MIGHT actually be the CPU object IF
                    # the team insisted on not have modules
                    last_core = self._create_core(device, last_module)
                    # this is not in create core since we do not do these flows
                    # in the thread creation....
                    stateports = getattr(
                            getattr(
                                getattr(device, "node", None),
                                "stateport", None),
                                "stateport_names",[]
                    )
                    self._add_to_mapping(last_core)  # must be after add component
                    # whether core was enabled or not, we have to check for adding
                    # in taps
                    # NOW check for taps and alis if needed
                    if alias_cpu_to_socket_tap and socket_tap is not None:
                        if 'tap' in last_cpu.definition.sub_components:
                            core_tap = last_cpu.definition.tap.create_tree()
                            core_tap.target_info['did'] = device.did
                            core_tap.name = last_core_name
                            socket_tap.add_component(core_tap, last_core_name)
                            self._add_to_mapping(core_tap)

            elif device.nodetype == "thread" and (
                    (getattr(device, "coregroup", "GPC") == "GPC" and thread_devicetype is None) or
                    (thread_devicetype and device.devicetype == thread_devicetype)
            ):
                # only add enabled threads
                isenabled = getattr(device,"isenabled", True)
                # !! - IT IS NOT SUPPORTED HERE FOR CORE TO BE DISABLED BUT THREAD BE ENABLED!!!
                if isenabled or show_disabled:
                    # go make any part of the hiearchy we are missing....
                    if last_cpu is None:
                        last_cpu = self._create_cpu(device, last_sock, cpu_path, cpu_similar_steppings)
                    if last_module is None:
                        last_module = self._create_module(device, last_cpu)
                    if last_core is None:
                        last_core = self._create_core(device, last_module)

                    # see if we have hit a new core...
                    # try to use coreid if it exists, otherwise, use instance
                    coreid = getattr(device, "coreid", getattr(device, "instance", 0))
                    if last_core.target_info['instance'] != coreid:
                        last_core = self._create_core(device, last_module)

                    last_thread = self._create_thread(device, last_core)
                    self._add_to_mapping(last_thread) # make sure to call after add component
                    # the first thread should be used for MSR commands for the core and the uncore
                    last_core.target_info.setdefault('msr_did', device.did)
                    last_sock.uncore.target_info.setdefault('msr_did', device.did)

                    # always check for adding tap
                    # NOW check for taps and alis if needed
                    if alias_cpu_to_socket_tap and socket_tap is not None:
                        if 'tap' in core_def.thread.sub_components:
                            thread_tap = core_def.thread.tap.create_tree()
                            thread_tap.target_info['did'] = device.did
                            self._add_to_mapping(thread_tap)
                            socket_tap.add_component(thread_tap, last_thread.name)

        return socket_list

    def _get_cpu_def(self, device, definition_path, similar_steppings):
        """Helper function for creating cpu object"""
        ## create placeholder cpu module
        cpu_stepping = similar_steppings.get_primary(device.stepping)
        full_path = definition_path.format(stepping=cpu_stepping)
        cpu_def = GetComponentFromFile(full_path)
        if hasattr(cpu_def,"cpu"):
            return cpu_def.cpu
        elif cpu_def.name == "cpu":
            return cpu_def
        else:
            raise ValueError("Could not find 'cpu' in:\n\t%s\n\t"%(definition_path,full_path))


class ModuleOverride(object):
    def __init__(self,module_path, module_object):
        """
        Args:
            module_path : path to look for for override
            module_obj : object to return instead of typical module

        Note:
            Currently only single "." is supported for module path

        """
        self.module_path = module_path
        self.module_obj = module_object

    def find_module(self, module_name, package_path):
        """this function and its parameters are determined by PEP302"""
        # need to also check for when "components" not present at all
        components = module_name.split(".")
        parent = components[0]
        # our path is no where near that one...not for us
        if not self.module_path.startswith(parent):
            return None
        try:
            file,filename,stuff = imp.find_module(module_name)
        except ImportError:
            # need to make sure parent is in the list...
            if module_name==parent:
                new_module = imp.new_module(parent)
                new_module.__file__ = "dummy"
                # needed so that it is identified as a package
                # that may have sub-modules/packages
                new_module.__path__ = []
                loader = ModuleOverride(parent, new_module)
                new_module.__loader__ = loader
                return loader
        else:
            # if parent is actually out there...then use it as normal..
            return None

        # done if not *really* this module? we chated on import above...
        if module_name != self.module_path:
            return None

        return self

    def load_module(self,fullname):
        """this function and its parameters are determined by PEP302"""
        _LOG.debugall("ENTER: load_module")
        if fullname in sys.modules:
            return sys.modules[fullname]
        sys.modules[self.module_path] = self.module_obj
        return self.module_obj
