
# INTEL CONFIDENTIAL
# Copyright 2014 2018 Intel Corporation
#
# The source code  contained or  described herein and  all documents related to
# the source code  ("Material") are owned by Intel Corporation or its suppliers
# or licensors.  Title to the  Material  remains with  Intel Corporation or its
# suppliers  and licensors. The Material contains trade secrets and proprietary
# and  confidential  information  of  Intel  or  its  suppliers  and  licensors.
# The Material  is protected  by worldwide  copyright and trade secret laws and
# treaty provisions.  No part of the Material  may  be used, copied, reproduced,
# modified, published, uploaded, posted, transmitted, distributed, or disclosed
# in any way without Intel's prior express written permission. No license under
# any  patent,  copyright, trade secret or other intellectual property right is
# granted  to  or conferred upon you by disclosure or delivery of the Materials,
# either expressly, by implication, inducement, estoppel or otherwise.
# Any license under such intellectual property rights must be express and
# approved by Intel in writing.



"""
===================
PCI Scan Discovery
===================

.. autoclass:: PciScanDiscovery
    :members:

"""
from __future__ import absolute_import
from __future__ import division
from __future__ import print_function

import os
from itertools import product
from svtools.common import baseaccess
from ..logging import getLogger
from ..discovery import Discovery
from ..comp import GetComponentFromFile
from ..plugins.lmdb_dict import LmdbComponent

_LOG = getLogger("pciscan")


class PciScanDiscovery(Discovery):
    """
    Search for known PCI devices in the system

    Attributes:
        max_segment (int): Max PCI segment value to scan. Default is 0x6.
        max_bus (int): Max PCI bus value to scan. Default is 0xff.
        max_device (int): Max PCI device value to scan. Defauls is 0x0.
        max_function (int): Max PCI function value to scan. Defauls is 0x0.
        saved_info (list). Information that is saved in the component (under
            target_info). Default is ["segment", "bus", "device", "function"].


    Example:
        >>> from namednodes.discoveries.pciscan import PciScanDiscovery
        >>> discovery = PciScanDiscovery()
        >>> definition = module_path + r'toolext/namednodes/tool.lmdb.tar.bz2'
        >>> discovery.add_known_device(0x1234, 0x8086, definition)
        >>> discovery.get_all()
    """
    max_segment = 0x6
    max_bus = 0xff
    max_device = 0
    max_function = 0
    saved_info = None
    # _known_devices (dict): Known PCI devices to look for.
    _known_devices = None
    # ToDo: share _found_devices across instances to optimize scan
    # _found_devices (list): Caches devices found in previous scan.
    _found_devices = None

    def __init__(self, *args, **kwargs):
        super(PciScanDiscovery, self).__init__(*args, **kwargs)
        self._known_devices = {}
        self._found_devices = []
        self.saved_info = ["segment", "bus", "device", "function"]

    def add_known_device(self, did, vid, definition, rid=-1):
        """
        Add a device to search for during scan.

        Args:
            did (int): Device Id.
            vid (int): Vendor Id.
            definition (string): Path to lmdb definition file or LmdbComponent.
            rid (Optional[int]): Revision Id, use < 0 to match any Revision
                Id. Default is -1.
        """
        # Make sure parameters are the right size
        assert vid <= 0xffff, "vid size must be 2 bytes"
        assert did <= 0xffff, "did size must be 2 bytes"
        assert rid <= 0xff, "rid size must be 1 byte"

        # Build device key
        if rid < 0:
            key = (did << 16) + vid
        else:
            key = (did << 24) + (vid << 8) + rid

        # Add device to _known_devices
        self._known_devices[key] = definition

    def find_all(self,**kwargs):
        """
        Returns the list of components found in the system.

        Args:

            dev_increment (bool) : optional kwarg used to have an incrementing number for
                                   the suffix instead of the bus/device/function
        """
        components = []
        # Enumerate devices
        self._scan()
        dev_increment = kwargs.pop('dev_increment', None)
        if dev_increment:
            dev_counter = 0
        else:
            dev_counter = None
        # Build component for known devices:
        for did_vid, rid, device_info in self._found_devices:
            key = (did_vid << 8) + rid
            if key not in self._known_devices.keys():
                key = did_vid
                if key not in self._known_devices.keys():
                    continue
            definition = self._known_devices[key]
            components.append(self._get_new(device_info, definition, dev_counter))
            if dev_increment:
                dev_counter = dev_counter + 1

        # Return components list
        return components

    def _scan(self):
        """
        Enumerate PCI devices in the system.
        """
        self._found_devices = []
        base = baseaccess.getglobalbase()
        # Halt system if baseaccess allows it
        resume_system = False
        if hasattr(base, "isrunning") and base.isrunning():
            resume_system = True
            base.halt()

        # Calculate pci space to scan
        segments = range(self.max_segment + 1)
        buses = range(self.max_bus + 1)
        devices = range(self.max_device + 1)
        functions = range(self.max_function + 1)
        pci_space = product(segments, buses, devices, functions)

        # Enumerate present devices
        for seg, bus, dev, func in pci_space:
            did_vid = base.pcicfg(bus, dev, func, 0, 4, segment=seg)
            if did_vid == 0xffff:
                # Master abort
                continue
            rid = base.pcicfg(bus, dev, func, 8, 1, segment=seg)
            self._found_devices.append((did_vid, rid, (seg, bus, dev, func)))

        # Resume system if baseaccess allows it
        if resume_system:
            base.go()

    def _get_new(self, device_info, definition, dev_counter = None):
        """
        Create new device component.

        Args:
            device_info: Segment, Bus, Device, Function
            definition: Path to device's definition lmdb file or LmdbComponent.
            dev_counter: Device instance.
        """
        if type(definition) == LmdbComponent:
            def_object = definition
        else:
            # Get definition from lmdb file
            def_object = GetComponentFromFile(definition, refresh=True)

        # Get device PCI info
        seg, bus, dev, func = device_info
        # Create component from definition
        cmp_object = def_object.create_tree()
        # Name component
        suffix_args  = {'seg':seg, 'bus':bus, 'dev':dev, 'func':func,'counter':dev_counter}
        suffix = self._get_suffix(suffix_args)
        cmp_object.set_name(def_object.name + suffix)
        # Update target_info
        if "segment" in self.saved_info:
            cmp_object.target_info["pci_segment"] = seg
        if "bus" in self.saved_info:
            cmp_object.target_info["pci_bus"] = bus
        if "device" in self.saved_info:
            cmp_object.target_info["pci_device"] = dev
        if "function" in self.saved_info:
            cmp_object.target_info["pci_function"] = func
        return cmp_object

    def _get_suffix(self, suffix_components):
        # override this if you want a simpler
        if suffix_components['counter'] is None:
            seg = suffix_components.get('seg')
            bus = suffix_components.get('bus')
            dev = suffix_components.get('dev')
            func = suffix_components.get('func')
            return "_s%d_b%d_d%d_f%d" % (seg, bus, dev, func)
        else:
            return "%d"%suffix_components['counter']
