
# INTEL CONFIDENTIAL
# Copyright 2014 2018 Intel Corporation
#
# The source code  contained or  described herein and  all documents related to
# the source code  ("Material") are owned by Intel Corporation or its suppliers
# or licensors.  Title to the  Material  remains with  Intel Corporation or its
# suppliers  and licensors. The Material contains trade secrets and proprietary
# and  confidential  information  of  Intel  or  its  suppliers  and  licensors.
# The Material  is protected  by worldwide  copyright and trade secret laws and
# treaty provisions.  No part of the Material  may  be used, copied, reproduced,
# modified, published, uploaded, posted, transmitted, distributed, or disclosed
# in any way without Intel's prior express written permission. No license under
# any  patent,  copyright, trade secret or other intellectual property right is
# granted  to  or conferred upon you by disclosure or delivery of the Materials,
# either expressly, by implication, inducement, estoppel or otherwise.
# Any license under such intellectual property rights must be express and
# approved by Intel in writing.


"""
"""
from __future__ import absolute_import
from __future__ import division
from __future__ import print_function

# standard modules
import os, copy
import operator
import re
import weakref
import importlib
import itertools
from functools import cmp_to_key
from contextlib import contextmanager

# namednodes modules
from .logging import getLogger
from .utils.ordereddict import odict
from .utils import PluginManager
from .utils.str2py import str2py
from .nodes import (
    NamedNodeValue,
    NodeTypes,
    NamedNodeDefinition,
    NamedNodeDefinitionAlias
)
from .utils._py2to3 import *
from .utils import UnzipCheck
from .utils.stringcmp import alpha_numeric_cmp
from .errors import PluginError
from . import access as _access
from . import settings
import collections
from six import PY2,PY3

if PY2:
    from collections import MutableMapping, MutableSequence
if PY3:
    from collections.abc import MutableMapping, MutableSequence


_LOG = getLogger()



class NamedComponentDefinition(object):
    #: parent component to this definition, None, if no parent
    parent = None
    #: highest level component in the database, default is self, and is
    #: updated when we get added to anther component
    origin = None
    #: information that may be needed across all value's of this component
    info = None
    #: can be filled in so that required args needed to create the component
    #:  are understood
    requires = None
    #: to hold list of defintions for sub components in case value constructor
    #: knows how to build out all the needed components
    _sub_components = None
    #: to hold the dictionary with all the available nodes for this component
    _nodes = None

    _sub_groups = None

    def __init__(self, name, info=None, nodes=None, value_class=None):
        """NamedComponentDefinition(name,info=None)

        Args:
            name (str) : name that will be used when displaying this component
            info (obj) : odict object for meta data related to this object
            nodes (dict) : optional dictionary with all the nodes in this
                component definition
            value_class (type) : Component class to use when we create one of
                these components
        """
        self.name = name
        self.info = info or odict()
        self._nodes = nodes or odict()
        self._sub_components = odict()
        # store value class as part of the info
        if value_class is not None:
            self.info['value_class'] = value_class
        self.parent = None  # this is changed by add_component
        self.origin = weakref.proxy(self, _proxy_debug)
        self._sub_groups = odict()

    @property
    def path(self):
        """
        Path to this NamedComponentDefinition object if you started from the origin
        """
        # path is either our name or our parents path + our name
        # p = self.name if self.parent is None else self.parent.path + "." + self.name
        if self.parent is None:
            return self.name
        else:
            if self.parent is self:
                raise RuntimeError("parent was self somehow...%s" % self.name)
            else:
                return self.parent.path + "." + self.name

    def get_by_path(self, path, **kwargs):
        """
        return the NamedNodeDefinition or NamedComponentDefinition based on the path provided

        Args:
            path (str,list) : Path is a "." delimitted string or a list, that
                              is used to traverse the nodes and components to
                              return whatever object the path points to
            default (optional) : if path is not found return this instead
        Returns:
            returns either a node or component depending on what path is

        Raises:
            ValueError if the path can not be found
        """
        """
        return the node or component based on the path provided

        Args:
            path (str,list) : Path is a "." delimitted string or a list, that
                              is used to traverse the nodes and components to
                              return whatever object the path points to
            default (optional, bool) : if path is not found return this instead
            missing (optional, bool) : if missing return list of [foundpart, path_left]
        Returns:
            returns either a node or component depending on what path is

        Raises:
            ValueError if the path can not be found
        """
        default = kwargs.pop("default", KeyError)
        missing = kwargs.pop("missing", False)
        # internally we re-use this single get_by_path function for the definition AND for the value object
        start_comp = kwargs.pop('start_comp', self)
        known_classes = (
            NamedComponent,
            NamedComponentDefinition,
            NamedNodeDefinition,
            NamedNodeValue,
            ComponentGroup,
        )
        if len(kwargs):
            raise ValueError("Unknown kwargs: %s" % str(list(kwargs.keys())))
        if isinstance(path, basestring):
            parts = path.split('.')
        elif isinstance(path, list):
            parts = path
        else:
            raise TypeError('Error: wrong argument type for get_by_path')

        obj = start_comp

        for p, part in enumerate(parts):
            # to know if we obj has been advanced to the next object yet...
            start_obj = obj
            # this gets used if we are in component definitions for arrays
            aitem = part.find("[#]")  # if it is an array
            if aitem >= 0:
                if not isinstance(start_comp, NamedComponentDefinition):
                    raise ValueError("cannot use [#] when searching value objects")
                # array item's have the same name as their array
                obj = getattr(obj, part[:aitem], AttributeError)
                if getattr(obj, "type", None) != NodeTypes.Array:
                    raise ValueError("expected array due to path name %s " % part[p])
                obj = obj.item_definition
            # this gets used if we are in component *value* for arrays
            elif part.count("["):  # if it is an array
                nextattr, item = re.match(r"(\w+)\[(.*)\]", part).groups()
                nextobj = getattr(obj, nextattr, AttributeError)
                if nextobj is AttributeError:
                    if missing:
                        return [obj, ".".join(parts[p:])]
                    # dont error, let the default getattr code give it a shot first
                else:
                    # assume that if it is a string it will have quotes '
                    item_key = eval(item)
                    if item_key in nextobj or not missing:
                        obj = nextobj[eval(item)]
                    else:  # must be missing
                        return [nextobj, item_key]

            # yep, compare exact instance to make sure we didn't get a hit
            # on the array code above
            if start_obj is obj:
                # check for sub component
                next = getattr(obj, part, None)
                # if we didn't get a known thing, clear it back to be None
                if not isinstance(next, known_classes):
                    next = None
                    # we got function, or something like that...later we wil consider that "ok" but..
                    # first check to see if our parent happens to have a node with the same name, and
                    # make sure that gets prioritized over this function
                    if isinstance(obj, known_classes) and part in obj:
                        return obj.get_node(part)
                    # must not have found it...back to other checking...

                # it will be a somewhat common problem where folks do a get_by_path
                # that already includes this component
                if next is None and part == start_comp.name:
                    continue

                # for legacy cases were get_by_path may be getting abused to get a function at the end of that path
                if next is None and p == len(parts)-1:
                    next = getattr(obj, part, None)

                if next is None:  # p != self.name
                    # before giving up...try backing up a bit
                    parent = getattr(obj, "parent", None)
                    if parent is None:
                        parent = getattr(obj, "component", None)

                    # this falls back to typical get_node recursive call, only works if you make it
                    # to at least a component using the getattr methods
                    # FYI we still walk this bit even if the parent bit above was None
                    nextp = p
                    while nextp >= 0:
                        attempt = ".".join(parts[nextp:])
                        if isinstance(parent, known_classes) and attempt in parent:
                            return parent.get_node(attempt)
                        if isinstance(obj, known_classes) and attempt in obj:
                            return obj.get_node(attempt)
                        else:
                            # nothing found, back up further
                            nextp = nextp - 1
                    else:
                        # else on while loop == no break happened, we didn't find it on the
                        # parent either
                        if missing:
                            if obj is self:
                                return [None, ".".join(parts[p:])]
                            else:
                                return [obj, ".".join(parts[p:])]
                        elif default is KeyError:
                            raise ValueError("Unknown path: %s" % path)
                        else:
                            return default
                else:
                    obj = next
        return obj

    def getbypath(self, *args, **kwargs):
        """for compatibility with previous pythonsv generations, see get_by_path for help"""
        return self.get_by_path(*args, **kwargs)

    def __getattr__(self, attr):
        # odd python3 workaround to make sure we report False to hasattr when
        # folks try to copy this object
        if attr == "__setstate__":
            raise AttributeError("setstate not implemented")

        # use the property in case it is overridden...
        # check our dict first
        if attr.endswith("s"):
            self._build_group(attr, self)

        if attr in ComponentDefinitionPlugins.plugins():
            # else we need to create it
            plugin = ComponentDefinitionPlugins.get(attr, self)
            if plugin is not None:
                # save off for future use
                self.__dict__[attr] = plugin
                return plugin
            else:
                raise PluginError("The plugin for %s does not support this component type" % attr)
        # put these last...
        elif attr in self.sub_components:
            return self.sub_components[attr]
        elif attr in self._sub_groups:
            return self._sub_groups[attr]
        elif attr in self:
            # the 'in' == contains for a component checks only for node names
            return self.get_node(attr)
        else:
            return object.__getattribute__(self, attr)

    def _load_all_plugins(self):
        """Loads all the supported plugins for this node"""
        for plugin_name in ComponentDefinitionPlugins.plugins():
            if plugin_name not in self.__dict__:
                plugin = ComponentDefinitionPlugins.get(plugin_name, self)
                if plugin is not None:
                    self.__dict__[plugin_name] = plugin

    def __dir__(self):
        mydir = []
        if settings.HIDE_METHODS is False:
            self._load_all_plugins()
            mydir.extend(dir(type(self)))
            mydir.extend(self.__dict__.keys())
        mydir.extend(self.nodenames)
        mydir.extend(self.sub_components.keys())
        mydir.extend(self._sub_groups.keys())

        return mydir

    def _build_group(self, part, obj):
        """This method is used to generate component groups for 'similar'
        nodes. Example: you have several cbdmas nodes: cbdma_0...cbdma_7.
        This method will create a new group cbdmas containing cbdma0_7
        Args:
            part (str): The name of the proposed group name
            obj: the current component
        """
        if part in obj._sub_groups:
            return
        attributes_in_obj = dir(obj)
        for attribute in attributes_in_obj:
            match = self._groups_regexp_match(attribute, part)
            if match:
                groupobj = obj.__getattr__(attribute)
                if part not in obj._sub_groups:
                    newgroup = ComponentGroup([groupobj],
                                              part,
                                              grouppath=(self.path + "." + part))
                    obj._sub_groups[part] = newgroup
                else:
                    obj._sub_groups[part].append(groupobj)

    def _groups_regexp_match(self, attributes_string, given_groupname):
        """This group takes the attributes_string which is basically a nodename
        and applies certain rules to obtain a group name. If it matches the
        given_groupname then return True, so build group can proceed to build
        a new group as there are valid nodenames in the component
        Args:
            attributes_string (str): a node name, like cbdma_0
            given_groupname (str): a proposed group name, like cbdmas
        """
        retval = False
        default_re = r"([a-zA-Z_0-9]+?)(\d+)$"
        re_str = default_re
        group_re = re.compile(re_str)
        match = group_re.match(attributes_string)
        if match is None:  # no match, we're done now
            return False
        groupname = match.group(1).strip("_") + "s"
        if groupname == given_groupname:
            retval = True
        return retval

        # # request for noodes and components first, and to be sorted
        # self._load_all_plugins()
        # mydir = []
        # nodenames = list(self.nodenames)
        # mydir.extend(sorted(nodenames))
        # subcomps = list(self.sub_components.keys())
        # mydir.extend(sorted(subcomps))
        # methods = dir(type(self))
        # methods.extend(self.__dict__.keys())
        # mydir.extend(sorted(methods))
        # mydir.extend("blah")
        # return mydir

    def create(self, name, info=None, **kwargs):
        """
        Create new component with the specified name, and the info specific
        to this new component

        Args:
            name (str) : name to assign to the new component
            info (dict) : dictionary of information related to this new
                          component
            kwargs : instead of passing "info" the kwargs can be used to
                     initialize the info for the new component, but keep in
                     mind that order is not preserved using this method (since
                     python does not use an ordered dict by default)

            value_class : override definition's value class and use this one instead
        """
        vclass = kwargs.get('value_class', None)
        if info is not None:
            vclass = info.get('value_class', vclass)
        if vclass is not None:
            if isinstance(vclass, basestring):
                value_class = str2py(vclass)
            else:
                # if not a string assume it is a class, but we want to always save
                # as a string
                kwargs['value_class'] = vclass.__module__ + ":" + vclass.__name__
                value_class = vclass
        else:
            value_class = self.value_class
        return value_class(self, name, info, **kwargs)

    @property
    def value_class(self):
        """
        Return the value class used to create a Component object for
        this definition
        """
        # figure out the default
        value_class = self.info.get("value_class", None)
        if value_class is None:
            return NamedComponent
        # we got a value class, just return it
        if type(value_class) == type:
            return value_class
        elif isinstance(value_class, basestring):
            return str2py(value_class)
        else:
            raise ValueError("Unexpected value for value_class")

    def _set_origin(self, origin):
        """helper function so that we can recursively set origin"""
        # we should not rely on walk here since that is a plugin and this here
        # is one of our base classes
        self.origin = origin
        for sub in self.sub_components.values():
            sub._set_origin(origin)

    def add_component(self, component, name=None):
        """
        Add this component definition to this component's definition hierarchy.

        Args:
            component (NamedComponentDefinition): Component definition object
                to be added.
            name (str): Name of component definition to be used for getting to
                this component definition, component.name will be used if no
                name is provided.

        Returns:
            a component object

        Note::  the component returned may not be the exact same type
                as the one passed in (depending on how data storage is being
                used). Users should use the component returned here for future
                modifications instead of the component that is passed in.
        """
        # if this component already belong to another component, then we need to create
        # a copy of it first
        if component.parent is not None:
            _LOG.warning("You are adding to a component something that already has a parent, you should either\n"
                         "copy it first, or remove it from the old component first, old location: %s" % component.path)
        name = name or component.name
        component.parent = self  # weakref.proxy(self, _proxy_debug)
        component._set_origin(self.origin)
        self._sub_components[name] = component
        return component

    @property
    def __self__(self):
        """
        returns self object - Needed in order to get self from weakref objects
        """
        return self

    def __deepcopy__(self, memo):
        # the_copy = super(NamedComponentDefinition,self).__deepcopy__(self,memo)
        # pdb;pdb.set_trace()
        # return the_copy
        cls = self.__class__
        result = cls.__new__(cls)
        memo[id(self)] = result
        for k, v in self.__dict__.items():
            # keep up with our parent weakref
            if type(v) == weakref.ProxyType:
                copied = memo[id(v.__self__)]
                setattr(result, k, weakref.proxy(copied))
            elif id(v) in memo:
                setattr(result, k, memo[id(v)])
            else:
                setattr(result, k, copy.deepcopy(v, memo))
        return result

    def remove_component(self, component):
        """
        Remove the sub component definition with the specified name
        from the sub components definition list.

        Args:
            component (NamedComponentDefinition): Component definition object
                to be removed.
            name (str): The name to get to the component definition to remove,
                component.name will be used if no name is provided.
        """
        if isinstance(component, basestring):
            name = component
        else:
            name = component.name
        assert name in self._sub_components, \
            "%s subcomponent definition not in %s" % (name, self.name)
        oldcomp = self._sub_components.pop(name)
        oldcomp.parent = None
        return

    def add_node(self, node):
        """
        add a NamedNodeDefinition to this ComponentDefinition

        Args:
                node (obj) : a NamedNodeDefinition object
        """
        if not isinstance(node, NamedNodeDefinition):
            raise TypeError("you can only add NamedNodeDefinition objects here")
        self._nodes[node.name] = node
        node.component = weakref.proxy(self)
        return node

    def get_node(self, nodename, default=KeyError):
        """Get the """
        node = self._nodes.get(nodename, default)
        if node is not KeyError:
            return node
        else:
            raise KeyError("Unknown node %s on component %s"%(nodename, self.path))

    def __contains__(self, nodename):
        return nodename in self._nodes

    @property
    def nodes(self):
        """A list of children NodeDefinitions that are on this object"""
        return list(self.iter_nodes())

    @property
    def nodenames(self):
        """A list of names of the nodes that are children of this object"""
        return list(self.iter_nodenames())

    def iter_nodes(self):
        """For iterating through the children nodes of this object"""
        for name in self.iter_nodenames():
            yield self.get_node(name)

    def iter_nodenames(self):
        """For iterating through the names of the children nodes of this object"""
        for k in self._nodes.keys():
            yield k

    @property
    def sub_components(self):
        """
        A dictionary with a mapping of {name:NamedComponentDefinition} for ComponentDefinition
        objects that are children of this object
        """
        return self._sub_components

    @property
    def access_groups(self):
        """access groups should be dictionary in the information object"""
        return self.info.setdefault("access_groups", {"default": []})

    @access_groups.setter
    def access_groups(self, groups):
        # validation ?
        self.info['access_groups'] = groups

    def remove_node(self, name):
        """remove node from our known definitions"""
        try:
            del self._nodes[name]
        except KeyError:
            raise ValueError("Unknown node: %s" % name)


class NamedComponent(object):
    """NamedComponent(definition, name,target_info=None)

    Should only be created via a NamedComponentDefintion's create function.
    **Never instantiate this directly**

    Args:
        definition (obj) : ComponentDefinition to use for referencing nodes
                           or shared info
        name (str) : name that will be used when displaying this component
        target_info (obj) : odict object for meta data related to this
                            particular component

    Examples of name would be: "socket0", "uncore0", etc...
    Examples of target_info would be: "device=TLA_UC0"
    """
    #######################################################################################
    # Data that is should live at the class level, and not overridden by the object ##
    #######################################################################################
    #: Specify the pieces of information that this component class requires
    #: to be passed in to the target_info when creating objects
    requires = []
    ######
    # Data that is Object specific
    ######
    #: Name that will be used for the NamedComponent objects when referring to
    #: it in log files or in the CLI
    name = None
    #: odict object that is particular to this particular component instance.
    #: Passed in when the object is created if the created component is a added
    #: to another component, then this variable is set to that parent
    parent = None
    ##########
    # The next two could potentially be implemented as properties and would be
    # less prone to error
    # but performance might be an issue if they are used often...
    ########
    #: if the created component is part of a component hiearchy, this variable
    #: is set to the top level parent
    origin = None
    #: Used to specify the path to this component object
    path = None
    #: Contains all the known access names to access class mappings
    #: that are used by this component
    access_classes = None
    #: this will be an ordered dictionary to contain sub component references
    _sub_components = None
    #: Internal only - used to make sure users get a warning if they mispel a
    #: node when trying to do a write
    _frozen = False
    #: Used to maintain a list of our component groups
    _sub_groups = None
    #: Internal only - used to track nodes that are not in this component but
    #: may be in the definition
    _removed_nodes = None

    def __init__(self, definition, name, target_info=None, **kwargs):
        if str(self.__class__).endswith(".NamedComponent"):
            raise Exception(
                "NamedComponents should not be created directly. "
                "ComponentTypes should be built from 'CreateComponentType'")
        # Definition that will be used to get our nodes
        self.definition = definition
        # Name that will be used for this component when referring to it in
        # files or in the CLI
        self.name = name
        # default for path is just the name of the component
        self.path = name
        # save off target_info passed in...for performance reasons...dont deepcopy, leave that on the user
        if target_info is not None:
            if isinstance(target_info, odict):
                self.target_info = target_info
            else:
                self.target_info = odict(target_info.items())
        else:
            self.target_info = odict()

        for k, v in kwargs.items():
            self.target_info[k] = v

        # make sure our info has the specified requirements
        for r in self.requires:
            if r not in self.target_info:
                raise ValueError(
                    "This component class requires: '{0}' to be passed in via "
                    "the target_info dictionary".format(r))

        # track access settings as part of target_info
        self.target_info.setdefault('access_settings',
                                    self.definition.info.get("default_access_settings", {"default": "default"}))

        # for tracking what registers exist in the definition, but NOT in this
        # instance of the component
        self._removed_nodes = set()

        # initialize sub components to be empty dictionary
        self._sub_components = odict()

        # initialize sub groups to be an empty dictionary
        self._sub_groups = odict()

        # origin starts off defaulting to ourself
        self.origin = weakref.proxy(self, _proxy_debug)

        # no parent until we are added to subcomponent, but we need
        # a local reference for some odd setattr reasons, dont delete this default
        self.parent = None

        # create a copy of all the currently known accesses
        self.access_classes = copy.copy(_access.NodeAccessManager)

        # no more attribute adds, throw exceptions in case someone
        # is trying to write to a node
        self._frozen = True

    def __getattr__(self, attr):
        # odd python3 workaround to make sure we report False to hasattr when
        # folks try to copy this object
        if attr == "__setstate__":
            raise AttributeError("setstate not implemented")
        # look if it is in subcompnents first
        if attr in self._sub_components:
            return self._sub_components[attr]
        elif attr in self._sub_groups:
            return self._sub_groups[attr]
        # need to test performance vs. a check of "in nodes
        elif attr in self.definition and (attr not in self._removed_nodes):
            # get node definition, get value class, and create it....
            nodedef = self.definition.get_node(attr)
            return nodedef.value_class(attr, self, nodedef, None)
        elif attr in ComponentPlugins.plugins():
            # else we need to create it
            plugin = ComponentPlugins.get(attr, self)
            if plugin is not None:
                self.__dict__[attr] = plugin
                return plugin
            else:
                raise PluginError("The plugin for %s does not support this component type" % attr)
        else:
            raise AttributeError("Unknown Attribute {0}".format(attr))
            # double check that it is OUR key error and node something
            # lower in the stack failing
            # if attr not in self.definition._nodes:
            #    raise AttributeError("Unknown Attribute {0}".format(attr))
            # else:
            #    # attr was in our nodes dictionary, something else
            #    # in the stack failed, raise that up, since we don't know what
            #    #  failed
            #    raise

    def __setattr__(self, attr, value):
        # i setting known attributes (?)
        # we are partially doing this for performance when adding sub components
        # if (attr in self.__dict__):
        #    # fast track frozen
        #    object.__setattr__(self,attr,value)
        # dont check for nodes until we are frozen
        if self._frozen and self._sub_components and (
                attr in self._sub_components):
            raise AttributeError(
                "Component with that name exists, "
                "cannot overwrite it in this manner")
        elif (self._frozen and (attr in self.definition) and
              (attr not in self._removed_nodes)):
            # get value object and write it
            nodedef = self.definition.get_node(attr)
            nodeval = nodedef.value_class(attr, self, nodedef, None)
            nodeval.write(value)
        elif not self._frozen:
            object.__setattr__(self, attr, value)
        elif attr in dir(type(self)) or attr in self.__dict__:
            # not in our own dictionary, that check was first
            # but this should cover everything else
            object.__setattr__(self, attr, value)
        else:
            # we used to allow setting of clas
            raise Exception("cannot add new attributes at this point")

    def _load_all_plugins(self):
        """Loads all the supported plugins for this node"""
        for plugin_name in ComponentPlugins.plugins():
            if plugin_name not in self.__dict__:
                plugin = ComponentPlugins.get(plugin_name, self)
                if plugin is not None:
                    self.__dict__[plugin_name] = plugin

    def __dir__(self):
        mydir = []
        if settings.HIDE_METHODS is False:
            self._load_all_plugins()
            mydir = dir(type(self))
            mydir.extend(self.__dict__.keys())
        # make sure to not show removed nodenames
        # dont use definition due to removed nodes
        mydir.extend(self.nodenames)
        mydir.extend(self._sub_components.keys())
        mydir.extend(self._sub_groups.keys())
        return sorted(set(mydir))

    @property
    def info(self):
        """Shortcut to definition.info similar to what nodes has"""
        return self.definition.info

    @property
    def nodenames(self):
        """
        A list of node names that are available for access
        """
        # return list of nodes as long as they are not in removed nodes
        return list(self.iter_nodenames())

    @property
    def nodes(self):
        """
        A list of node that are available for access
        """
        # this cannot just use component definition because eventually we will
        # have ability to filter out certain nodes based on skew
        return list(self.iter_nodes())

    def iter_nodes(self):
        """
        An iterator for nodes that are available for accessing
        """
        for n in self.iter_nodenames():
            yield self.get_node(n)

    def iter_nodenames(self):
        """iterator over nodes for using in loops"""
        # I think this is temporary...eventually the value can have a list
        # of nodes that are different than the definition
        for name in self.definition.iter_nodenames():
            if name not in self._removed_nodes:
                yield name

    def __contains__(self, nodename):
        if nodename in self._removed_nodes:
            return False
        else:
            return nodename in self.definition

    def get_by_path(self, path, **kwargs):
        """
        return the node or component based on the path provided

        Args:
            path (str,list) : Path is a "." delimitted string or a list, that
                              is used to traverse the nodes and components to
                              return whatever object the path points to
            default (optional, bool) : if path is not found return this instead
            missing (optional, bool) : if missing return list of [foundpart, path_left]
        Returns:
            returns either a node or component depending on what path is

        Raises:
            ValueError if the path can not be found
        """
        kwargs['start_comp'] = self
        return self.definition.get_by_path(path, **kwargs)

    def getbypath(self, *args, **kwargs):
        """for compatibility with previous pythonsv generations, see get_by_path for help"""
        return self.get_by_path(*args, **kwargs)

    def get_node(self, name, default=KeyError):
        """Return a Named Node Value without reading"""
        # just make sure it hasn't been removed, then let the definition figure
        # out if it is a legit node...
        if name not in self._removed_nodes:
            nodedef = self.definition.get_node(name, default=RuntimeError)
            if nodedef is not RuntimeError:
                # must have gotten a node back
                return nodedef.value_class(name, self, nodedef, None)
            elif default is not KeyError:
                return default
            else:
                raise KeyError("unknown node: %s on component %s" % (name, self.path))
        else:
            raise KeyError("unknown node: %s on component %s" % (name, self.path))

    def remove_node(self, name):
        """remove the node with the specified name from this component"""
        if name not in self.definition:
            raise ValueError("Unknown node")
        self._removed_nodes.add(name)

    def get_node_definition(self, name):
        """Return a Named Node Value without reading"""
        return self.definition.get_node(name)

    def get_access(self, group="default"):
        """get_access(group)

        This function will return the current access being used by the
        specified access_group

        Args:
            group (str) : Access Group to find the types of access we are
                          currently using for that group

        Example:
            >>> socket0.get_access('pci')
            'mmcfg'
        """
        access_groups = self.definition.access_groups
        if group not in access_groups:
            _LOG.debug("The access_group (%s) is unknown for this component, "
                       "returning 'default'" % group)
            return "default"
        # this is a known group, return its value or "default" if it has not been set yet
        access = self.target_info.get('access_settings').get(group, "default")
        _LOG.debug("get_access(group='%s') == %s", group, access)
        return access

    def get_access_choices(self, **kwargs):
        """
        Poll all the registers and return a dictionary of the known groups
        and the known accesses in each group

        Args:
            recursive : (True)/False - optional argument so that only the
                        current component can be searched if that is desired
        """
        recursive = kwargs.pop("recursive", True)
        choices = {}
        if self.definition.access_groups:
            for k, v in self.definition.access_groups.items():
                choices.setdefault(k, []).extend(v)

        if recursive:
            for sub in self._sub_components.values():
                for k, v in sub.get_access_choices().items():
                    choices.setdefault(k, []).extend(v)
        for k in choices.keys():
            choices[k] = list(set(choices[k]))
        return choices

    def set_access(self, access, group=None, **kwargs):
        """
        set access for all the nodes in the group specified

        Args:
            access (str) : name of access that tells Nodes which Access Class to use
            group (str) : string that nodes should use when requesting the current access
            recursive (bool) : (True)/False optional keyword to specify whether the change
                               applies to subcomponents

        If group is left as default of None, then all groups are searched and any of them that
        have a matching access are updated to use the specified access

        Example:
            >>> socket0.set_access('mmcfg')
            >>> socket0.set_access('mmcfg','pci')
            >>> socket0.set_access('mmcfg','pci', recursive=False)

        """
        access_settings = self.target_info['access_settings']
        # it shouldn't be possible that this is missing..but just in case
        recursive = kwargs.pop('recursive', True)
        # by default silent is dependent on recursive
        silent = kwargs.pop('silent', recursive)
        # this function is recursive, we use this to know that we are the first one called...
        # we keep up with "found" count across recursive call, and this helps with that.
        firstcall = kwargs.pop('firstcall', True)
        if firstcall:
            _LOG.debug("set_access(access=%s, group='%s', recursive=%s)", access, group, recursive)
        assert len(kwargs) == 0, "Unknown arguments: '%s'" % ",".join(kwargs.keys())
        access_groups = self.definition.access_groups
        found = 0
        # unknown group
        if group is not None and group != "default":
            if group not in access_groups:  # not a known group, skip...
                pass
            # group must be there
            elif access not in access_groups[group]:
                # assume it is for some child component....could perhaps throw an exception
                # if the recursive == False....but hopefully 'found' and warning will be enough
                pass
            else:  # must have found it
                access_settings[group] = access
                found += 1
        else:  # group is none search everything
            for group2check, accesses in access_groups.items():
                if access in accesses:
                    access_settings[group2check] = access
                    found += 1
                elif group2check == "default" and accesses == []:
                    # do this for all defaults? or just if not found?
                    # dont set found ?
                    access_settings['default'] = access

        # not recursive
        if not recursive:
            if not found and not silent:
                _LOG.warning("WARNING: no access groups found for: group=%s, access=%s",
                             group, access)
                return
            else:
                return None  # None is what user should get, not found count
        # recursive...
        else:
            for sub in self._sub_components.values():
                found += sub.set_access(access, group, recursive=recursive, firstcall=False)

        # done with recrusive, the next return goes to the user, so warn if we didn't find anything
        if firstcall:
            if found == 0:
                _LOG.warning("WARNING: no access groups found for: group=%s, access=%s", group, access)
                return
        else:  # not first call, we are in recursion here, so just report back what we have found so far
            return found


    @contextmanager
    def temporary_access(self, access, group=None, *, recursive=True):
        """temporarily set access for the specified Group, used as a with statement

        Args:

            access (str) : name of access that tells Nodes which Access Class to use
            group (str) : string that nodes should use when requesting the current access
            recursive (bool) : (True)/False optional keyword to specify whether the change
                               applies to subcomponents

        Example::

            with comp.temporary_access("pcicfg"):
                # registers that have pcicfg will use it here
                ...
            # accesses will go back to previous setting
            ...
        """
        # we need to store previous accesses for ALL sub-components
        previous_accesses = {}
        if recursive:
            max_depth = -1
        else:
            max_depth = 0

        for comp in self.walk_components(max_depth=max_depth):
            previous_accesses[comp] = copy.copy(comp.target_info.get('access_settings', {}))

        for comp in self.walk_components(max_depth=max_depth):
            comp.set_access(access, group, recursive=False, silent=True)

        yield

        for prev_comp, prev_access in previous_accesses.items():
            prev_comp.target_info['access_settings'] = prev_access

    def _fix_path(self):
        """
        Used to update the path variable to be accurate
        """
        if self.parent is not None:
            self.path = self.parent.path + "." + self.name
        else:
            self.path = self.name
        # now, make sure all subcomponents are fixed as well
        for subcomp in self._sub_components.values():
            subcomp._fix_path()

    def set_name(self, compname):
        """
        used when we change the name of the component
        to update the name + path variable

        Args:
            compname : new name for the component

        """
        self.name = compname
        # TODO: pysv had some code here around argument saving...do we
        # need this here?
        self._fix_path()

    def _set_origin(self, origin):
        """
        helper function so that we can recursively set *origin* and
        *access_classes* variables appropriately
        """
        # we should not rely on walk here since that is a plugin and this here
        # is one of our base classes
        self.origin = origin
        self.access_classes = origin.access_classes
        for sub in self.sub_components.values():
            sub._set_origin(origin)

    def add_component(self, newcomp, compname=None, **kwargs):
        """
        Add this component to be a sub component of this one.

        Args:
            newcomp : component object to be added
            compname : name of component to be used for getting to this
                       component, by default component.name will be used
            skip_sort : (optional) True/False (default=False) to control whether
                        we will sort the component group after the new component is added

        Returns:
            component object, typically it is the same one passed in, but

        Note:
            The exact type of newcomp.definition object may be tweaked after
            being passed in, if the component we are adding to is some file
            based storage, but the newcomp.definition is not in that storage
            already

        Note:
            By default, the added component will also be added to a component
            group under this component.
        """
        # if this component is also somewhere else in the component tree
        # make a new copy of it first...
        if newcomp.parent is not None:
            _LOG.warning("You are adding to a component something that already has a parent, you should either\n"
                         "copy it first, or remove it from the old component first, old location: %s" % newcomp.path)
        skip_sort = kwargs.pop("skip_sort", False)
        if len(kwargs) > 0:
            raise TypeError("Unknown arguments passed in: %s"%list(kwargs.keys()))
        # From PythonSv:
        # Add reference to this component as the parent, this needs to be a
        # weak reference so that that the memory management can still delete
        # this object if it needs to.
        newcomp.__dict__['_frozen'] = False
        try:
            newcomp.parent = self  # weakref.proxy(self, _proxy_debug)
            # origin is already created as a weakref.proxy, so that is why it is
            # not needed again
            newcomp._set_origin(self.origin)

            compname = compname or newcomp.name
            # we need to know later whether we just clobbered an existing component
            # still not 100% sure if this is safe, but it helps with debug
            if settings.DEBUG and compname in self._sub_components:
                raise ValueError("A component with that name already exists, remove it first")
            self._sub_components[compname] = newcomp
            # update name and path
            newcomp.set_name(compname)
            # also...make sure our definitions also have definition hiearchy if
            # it was not already there
            # ONLY tie in to hierarchy if the definition was not already in a hiearchy
            # if newcomp.definition.parent is None:
            #    self.definition.add_component(newcomp.definition)

            # attempt to get RE from definition object if it has one
            default_re = r"([a-zA-Z_0-9]+?)(\d+)$"
            re_str = self.definition.info.get("component_group_re", default_re)
            group_re = re.compile(re_str)
            # create / add to component group
            # now see if it is worth matching to a group...
            match = group_re.match(compname)
            if match is None:  # no match, we're done now
                return newcomp
            # all group names are just plurals
            groupname = match.group(1).strip("_") + "s"
            # attempt to save 'instance' number from the grouping
            try:
                newcomp.target_info['instance'] = int(match.group(2))
            except ValueError:
                pass
            # easy...matches an existing group
            if groupname in self._sub_groups:
                # check other members for "_"
                prev_ = group_re.match(self._sub_groups[groupname][0].name).group(1).endswith("_")
                this_ = match.group(1).endswith("_")
                # make sure we dont mess up _, only add if other members of the group
                # also ended with _
                if not (prev_ ^ this_):  # only add if neither has an underscore
                    # we need to make sure we don't already have a component with this name
                    # since the rest of the stack also forces us to have uniquely named components
                    for c in self._sub_groups[groupname]:
                        if c.name == newcomp.name:
                            self._sub_groups[groupname].remove(c)
                            break
                    self._sub_groups[groupname].append(newcomp)
                    if not skip_sort:
                        self._sub_groups[groupname].sort(key=cmp_to_key(lambda x, y: alpha_numeric_cmp(x.name, y.name)))
            else:
                # need new group for this thing, may have groups with one entry,
                # but that way we consistently have these lists even if there's
                # only one
                newgroup = ComponentGroup([newcomp],
                                          groupname,
                                          grouppath=(self.path + "." + groupname))
                self._sub_groups[groupname] = newgroup
            # all done, either new group added...or not...
        finally:
            newcomp.__dict__['_frozen'] = True
        return newcomp

    def remove_component(self, compname, **kwargs):
        """
        Remove the sub component with the specified name from the sub components
        list.

        Args:
            compname (str): The name to get to the sub component to remove.

        Note:
            By default, the component will also be removed from its component
            group.
        """
        assert compname in self._sub_components, \
            "%s not a subcomponent of %s" % (compname, self.name)

        assert len(kwargs) == 0, "Unknown kwargs: %s" % (list(kwargs.keys()))

        # Remove sub component from any component groups.
        # Notice we don't check for empty groups, after removing
        # the component, and this is ok since we are not interested
        # in removing those.
        found_component = False
        for groupname, component_group in self._sub_groups.items():
            for component in component_group:
                if component.name == compname:
                    found_component = True
                    component_group.remove(component)
                    break

            if found_component:
                break

        if found_component:
            # if we there is no longer any components in the component group
            # then go ahead and remove it
            if len(self._sub_groups[groupname]) == 0:
                del self._sub_groups[groupname]

        # Leave the definition hierarchy in our definition intact. We don't
        # remove component definitions unless specifically requested.

        # Finally, remove the component from our sub components.
        old_comp = self._sub_components.pop(compname)
        old_comp.parent = None

    def sort_component_groups(self):
        """do a one-time sort of component groups by name"""
        for group in self._sub_groups.values():
            group.sort(key=cmp_to_key(lambda x, y: alpha_numeric_cmp(x.name, y.name)))

    @property
    def sub_components(self):
        return self._sub_components

    def __copy__(self):
        # similar to deep copy, but we dont copy the target info and instead reference the original
        # this allows for something similar to an alias...
        newcomp = self.definition.create(self.name, info=self.target_info)
        # create makes a copy, so we need to set it back here...
        newcomp.target_info = self.target_info
        for subcomp in self.sub_components.values():
            newcomp.add_component(copy.deepcopy(subcomp))
        return newcomp

    def __deepcopy__(self, memodict={}):
        newcomp = self.definition.create(self.name, info=copy.deepcopy(self.target_info))
        for subcomp in self.sub_components.values():
            newcomp.add_component(copy.deepcopy(subcomp))
            newcomp._removed_nodes = copy.deepcopy(self._removed_nodes)
        return newcomp

    def __lt__(self, other):
        # for sorting...
        if not isinstance(other, NamedComponent):
            raise ValueError("Cannot sort against anything other than NamedComponent classes")
        if 'instance' in self.target_info and 'instance' in other.target_info:
            return self.target_info['instance'] < (other.target_info['instance'])
        else:
            return self.name.__lt__(other.name)

class ComponentPlugins(PluginManager):
    """
    Developers should not inherit from this class, it is a meta-class and
    its purpose is to track all the plugins that have been created
    """
    basetype = "ComponentPlugin"
    _plugins = {}  # important to have plugins local to this class


@with_metaclass(ComponentPlugins)
class ComponentPlugin(object):
    """
    """
    # Attributes required by subclasses:
    #: name of object that will be used to access plugin functions from the
    #: component
    name = ""
    #: If this component has storage files, then it should report the file
    #: extension it is using. This is not used by the infastructure, but it is
    #: encouraged to put so others can determine.
    #:
    #: .. note:: file_ext should not include the starting "."
    #:
    file_ext = ""
    #: can be used to disable a plugin
    enabled = True

    def __init__(self, component):
        """
        Default plugin constructor simply keeps a reference to the
        component that this plugin is attached to
        """
        self.component = component

    @classmethod
    def create(cls, component):
        """
        Function for retuning a new (or existing) instance of the plugin

        Args:
            component (obj) : Named component instance that this pluging
                              will be added to

        Returns:
            - Plugin instance, if this plugin is supported for this component

            - **None** if this plugin does not support this type of component

        The create function is encouraged to use the component.filepath
        to determine where to pull any pulgin specific data from.
        """
        # return cls() # if the plugin supports this NamedComponent
        # return None  # if the plugin does not support this NamedComponent
        return cls(component)


class ComponentDefinitionPlugins(PluginManager):
    """
    Developers should not inherit from this class, it is a meta-class and its
    purpose is to track all the plugins that have been created
    """
    basetype = "ComponentDefinitionPlugin"
    _plugins = {}  # important to have plugins local to this class


@with_metaclass(ComponentDefinitionPlugins)
class ComponentDefinitionPlugin(object):
    """
    """
    # Attributes required by subclasses:
    #: name of object that will be used to access plugin functions from the
    #: component
    name = None
    #: If this component has storage files, then it should report the file
    #: extension it is using. This is not used by the infastructure, but it is
    #: encouraged to put so others can determine.
    #:
    #: .. note:: file_ext should not include the starting "."
    #:
    file_ext = ""
    #: can be used to disable a plugin
    enabled = True

    def __init__(self, definition):
        """
        Default plugin constructor simply keeps a reference to the component
        that this plugin is attached to
        """
        self.definition = definition

    @classmethod
    def create(cls, definition):
        """Function for retuning a new (or existing) instance of the plugin
        Args:
            component (obj) : Named component instance that this pluging will
                              be added to

        Returns:
            - Plugin instance, if this plugin is supported for this component

            - **None** if this plugin does not support this type of component

        The create function is encouraged to use the component.filepath to
        determine where to pull any pulgin specific data from.
        """
        # return cls() # if the plugin supports this NamedComponent
        # return None  # if the plugin does not support this NamedComponent
        return cls(definition)


class ComponentLoaderPluginManager(type):
    """
    Developers should not inherit from this class, it is a meta-class and its
    purpose is to track all the plugins that have been created
    """
    #: dictionary of 'name':class for all plugins that have been created
    plugins = odict()
    files_to_comp_def = {}

    def __new__(meta, name, bases, atts):
        newcls = super(ComponentLoaderPluginManager, meta). \
            __new__(meta, name, bases, atts)
        # no need to save a reference to this one since it is the base
        # class for our plugins
        if name != "ComponentLoaderPlugin":
            meta.plugins[name] = newcls
        return newcls


@with_metaclass(ComponentLoaderPluginManager)
class ComponentLoaderPlugin(object):
    """
    A Loader Plugin knows how to take a path to a file and return a
    NamedComponentType. Note, that the NamedComponentType is a class that must
    still be instantiated once this function returns it.

    When the primary load function is run, then it will examine the file_ext
    to determine which plugins know how to handle the specified file being
    loaded. The create function is expected to parse the specified path
    and return the requested component type.

    Note:
        the filepath could still be a directory, and *file_ext* can simply
        mean that the directory ends with the text specified in the file_ext

    Note:
        *file_ext* should not include the starting '.'
    """
    #: when the loader checks for file parser, it will see if the path ends
    #: with this extension and use this parser if it does
    file_ext = ""

    def __init__(self, filepath, *args, **kwargs):
        self.filepath = filepath

    @classmethod
    def create(cls, filepath):
        """
        May do additional checking the filepath and confirms that it supports
        the file by returning a instnace of the loader
        """
        return cls(filepath)

    def parse(self):
        """
        parses the filepath provided during creation and creates the
        various ComponentTypes
        """
        raise Exception("must be implemented")


def CreateComponentDefinition(name, info=None, parent=None, value_class=None):
    """
    Create a new component object and return it.

    Args:
        name : name of the component definition (default name if
               NamedComponent is not given a name)
        info : dictionary with information that will be common across all
               components created from this definition
        parent : (optional) parent component to add this to OR
        value_class : (optional) class that must be derived from NamedComponent

    Create a new Component definition object, If parent is provided, then
    it will be added as a sub_component defintion to the that class.

    """
    new_comp = NamedComponentDefinition(name, info, value_class=value_class)
    if parent is not None:
        parent.add_component(new_comp)
    return new_comp


def _get_classification_filepaths(filepath):
    """
    adds classification to path for non-600
    returns a list of possible alternative locations for the file
    """
    possible_paths = []
    if settings.CLASSIFICATION not in [None, 600]:
        import namednodes
        parts = os.path.split(filepath)
        dirname = "/".join(parts[:-1])
        filename = parts[-1]
        filename, ext = os.path.splitext(filename)
        # filepath already had classification, return it without mods
        if filename.endswith(str(settings.CLASSIFICATION)):
            return [filepath]
        # add this to possible paths
        if ext.startswith("."):
            ext = ext[1:]
        ## append for file.<classification>.ext
        possible_paths.append("{dirname}/{filename}.{classification}.{ext}".format(
            dirname=dirname, filename=filename, classification=str(settings.CLASSIFICATION), ext=ext))
        # also add in option of having the the directory with the classification:
        # originaldir/<classification>/filename.ext
        # this is "old dir" + classification as a dir + original filename
        new_parts = "{dirname}/{classification}/{filename}.{ext}".format(
            dirname=dirname,
            classification=settings.CLASSIFICATION,
            filename=filename,
            ext=ext)
        ## append for <classification>/file.ext
        possible_paths.append(new_parts)
        # and yet another option, classification in the directory AND in the filename
        # this is "old dir" + classification as a dir + filename with classification
        new_parts = "{dirname}/{classification}/{filename}.{classification}.{ext}".format(
            dirname=dirname,
            classification=settings.CLASSIFICATION,
            filename=filename,
            ext=ext)
        possible_paths.append(new_parts)
        return possible_paths
    else:
        return [filepath]


_files_to_comp_def = {}


def GetComponentFromFile(filepath, **kwargs):
    """
    Given a path to the specified file, find a loader that can process
    this file and return us a new component definition -- or an actual
    component object depeneding on what the file type supports.

    Args:
        cache : (optional) Used to pull from cache if we have already loaded the defining
                True/False (default=False)
        refresh : (optional) The opposite to cache argument, added to not break existing users.


    """
    global _files_to_comp_def
    use_cache = kwargs.pop("cache", False)
    # to not break any existing folks
    refresh = kwargs.pop("refresh", True)
    use_cache = use_cache or (not refresh)
    warn = kwargs.pop("warn", False)
    # make sure we don't have any DOS paths passed in so that the
    # rest of the code will work in both Windows and Linux
    orig_filepath = filepath
    filepath = filepath.replace("\\", "/")

    # use extension to find the most likely parser
    # THIS happens again late
    orig_filepath = filepath  # save a reference for easy debug
    # if filepath already has a compression extension remove it
    for ext in UnzipCheck.extensions:
        if filepath.endswith(ext):
            filepath = filepath[:-len(ext)]

    # now check against cache....
    if use_cache and (orig_filepath in _files_to_comp_def):
        return _files_to_comp_def[filepath]

    # to confirm we have file..
    found = False

    filepaths_modded = _get_classification_filepaths(filepath)
    for filepath_modded in filepaths_modded:
        for ext in UnzipCheck.extensions:
            if os.path.exists(filepath_modded + ext):
                UnzipCheck(filepath_modded + ext, single=True).decompress()
                found = True
                break
        # compression not found...does file exists ?
        if not found and os.path.exists(filepath_modded):
            found = True
            break
        elif found:
            break

    # if still not found? -- it seems that there was a reason we can't do a file not found check here...

    # use extension to find the most likely parser
    filename, ext = os.path.splitext(filepath_modded)
    if ext.startswith("."):
        ext = ext[1:]
    parser = None
    for name, plugin in ComponentLoaderPluginManager.plugins.items():
        if plugin.file_ext == ext:
            # confirm with the plugin that it supports this file
            parser = plugin.create(filepath_modded)
            if parser is not None:
                _LOG.debugall("{0} loader reports matching extension and returned"
                           " parser".format(name))
                break
            else:
                _LOG.debugall("{0} loader reports matching extension but parser "
                           "not returned".format(name))
    if parser is None:
        _LOG.error("No loader reports supporting this file: {0}"
                   .format(filepath_modded))
        return
    # this very well could be "None" since loading the files may have called
    # CreateComponentType directly and/or may have created multiple
    # components...but we still need to save off that we have loaded from this
    # file before
    comp_definition = parser.parse()
    if comp_definition is None:
        # GetComponentFromFile did: !! Warning, count of known named component
        # types didn't change"
        msg = "Loader did not return a component object"
        _LOG.warning(msg)
    # make sure we save path using / so that anything using this
    # will work on both windows and linux
    comp_definition.info['definition_filepath'] = filepath_modded
    ####!!!! save off original filepath and not original (assumes classification does not change)
    if use_cache:
        _files_to_comp_def[orig_filepath] = comp_definition
    return comp_definition


def ClearComponentCache():
    """
    for clearing the existing cache of components. so that
    the cache is rebuilt (if it is being used)
    """
    global _files_to_comp_def
    _files_to_comp_def = {}


class ComponentGroup(MutableSequence):
    """
    Defines a component group to help us group different named components.

    When a component group exists we are able to make calls to the grouped
    components methods and access their attributes in a unified way. As an
    example, changing an attribute on the component group will change that
    attribute for all the components it is grouping.

    By default, a component is added to a component group when it is added
    as a subcomponent of another component. Also, when a component is removed
    from a component, it is removed from the component group it was added to.

    The criteria for grouping components into a component group is based on
    the components' name.

    The component definition info can be used to specify the particular regular expression
    used to make new groups. Fill in for example::

        # this is the default regular expresssion used by namednodes
        component_definition.info['component_group_re'] = r"([a-zA-Z_0-9]+?)(\d+)$"


    To specify the regular expression used to create new component groups.

    Note:
        This class is not meant to be instantiated directly nor to be inherited
        from by users.
    """

    _nodes = None
    _groupname = None
    _grouppath = None
    _frozen = False

    def __init__(self, subs=None, name="", grouppath=""):
        """
        Initialize this component group.

        Args:
            subs (list): The list of nodes to initialize the component
                group.
            name (str): The name of the component group.
            grouppath (str): The path of the component group.
        """
        assert name is not None, "name cannot be None, must be a string"
        assert grouppath is not None, "grouppath cannot be None, must be a string"
        self._nodes = list()
        if subs is None:
            subs = []
        self._nodes.extend(subs)

        if grouppath == "":
            grouppath = name

        self._groupname = name
        self._grouppath = grouppath
        self._frozen = True

    def set_groupname(self, value):
        """set groupname after group was created"""
        # we have to do this nasty hack because frozen is set to true
        self.__dict__['_groupname'] = value

    def set_grouppath(self, value):
        """set groupname after group was created"""
        self.__dict__['_grouppath'] = value

    @property
    def groupname(self):
        """
        Get the name of this component group.
        """
        return self._groupname

    @property
    def grouppath(self):
        """
        Get the path of this component group.
        """
        return self._grouppath

    @property
    def matches(self):
        """
        Get a substring  of the group component names
        """
        parent = getattr(self, 'parent', None)
        if not parent:
            return
        groups = getattr(parent, 'groups', None)
        if not groups:
            return
        regexps_keys = groups._searches.keys()
        regexps_keys = list(itertools.chain.from_iterable(regexps_keys))
        # Remove duplicates but preserve keys order
        seen = set()
        regexps = [k for k in regexps_keys if k not in seen and not seen.add(k)]
        names = self.name
        list_to_ret = []
        for name in names:
            # iterate in reverse order to prioritize the regex added last
            for regexp in reversed(regexps):
                cre = re.compile(regexp)
                match = cre.match(name)
                if match:
                    list_to_ret.append(name[match.start(1):match.end(1)])
                    break
        return list_to_ret

    def __call__(self, *args, **kwargs):
        return_list = []
        for node in self._nodes:
            im_self = getattr(node, "im_self", None)
            if getattr(im_self, "__display__", None) == True:
                path = getattr(getattr(im_self, "nodevalue", None), "path", None)
                if path is not None:
                    print(path + ":")
            return_node = node(*args, **kwargs)
            return_list.append(return_node)

        # if everything is None, then we're done...
        count = 0
        for node in return_list:
            if node is None:
                count += 1
        if count == len(return_list):
            return

        return ComponentGroup(return_list)

    def __dir__(self):
        if len(self._nodes) > 0:
            return dir(self._nodes[0])
        else:
            return []

    def __setattr__(self, attr, value):
        # assume that the attribute is ours if not frozen
        if not self._frozen:
            self.__dict__[attr] = value
        else:
            # otherwise, try to set on our list
            for sub in self._nodes:
                setattr(sub, attr, value)

    def __getattr__(self, attr):
        next_node_list = []
        for node in self._nodes:
            attr_value = getattr(node, attr)
            if type(attr_value) == ComponentGroup:
                next_node_list.extend(attr_value)
            else:
                next_node_list.append(attr_value)

        # update our node list
        next_group = ComponentGroup(subs=next_node_list, name=attr,
                                    grouppath=self.grouppath + "." + attr)
        return next_group

    def __len__(self):
        return len(self._nodes)

    def __getitem__(self, i):
        return self._nodes[i]

    def __delitem__(self, i):
        del self._nodes[i]

    def __setitem__(self, i, v):
        self._nodes[i] = v

    def insert(self, i, v):
        self._nodes.insert(i, v)

    def sort(self, *args, **kwargs):
        return self._nodes.sort(*args, **kwargs)

    def __getslice__(self, i, j):
        assert i <= j, "Must specify slice with lower:upper"
        return ComponentGroup(self._nodes[i:j])

    def __iter__(self):
        # reset this ?
        return self._nodes.__iter__()

    def _checklist(self, other, operation):
        """
        Perform a check on our list against an object
        with the supplied operator.

        Args:
            other (object): The object to check against.
            operation (operator): The operator to perform
                the check (lt, le, eq, ne, gt, ge).
        """
        if not (isinstance(other, list) or isinstance(other, ComponentGroup)):
            for node in self._nodes:
                if operation(node, other) == False:
                    return False
            return True
        else:  # just do default behavior
            return operation(self._nodes, other)

    def __lt__(self, other):
        return self._checklist(other, operator.lt)

    def __le__(self, other):
        return self._checklist(other, operator.le)

    def __eq__(self, other):
        return self._checklist(other, operator.eq)

    def __ne__(self, other):
        # Don't be fooled into thinking this is just the "not of __eq__"
        return self._checklist(other, operator.ne)

    def __gt__(self, other):
        return self._checklist(other, operator.gt)

    def __ge__(self, other):
        return self._checklist(other, operator.ge)

    def __str__(self):
        if len(self._nodes) == 0:
            return "<ComponentGroupEmpty>"

        # take a sample from the list so we know what
        # we are dealing with.
        sample_node = self._nodes[0]

        if sample_node is None:
            return ""

        if isinstance(sample_node, NamedNodeValue):
            # This one handles both RegisterValue and FieldValue cases
            return_str = []
            for node in self._nodes:
                # get decode
                decode_str = getattr(node, "decode", lambda: ())()
                value = node.get_value()
                if isinstance(value, (long, int)):
                    value_str = "0x{:08x}".format(value)
                else:
                    value_str = str(value)
                if decode_str is None:
                    return_str.append("{} - {}".format(node.path, value_str))
                else:
                    return_str.append("{} - {} - {}".format(node.path, value_str, decode_str))
            return "\n".join(return_str)

        elif issubclass(type(sample_node), NamedComponent):
            # and this one handles the RegisterComponent case
            return_string = ["%s - %s\n" % (node.path, node)
                             for node in self._nodes]
            return "".join(return_string)
        elif isinstance(sample_node, (int, long)) and (
                not isinstance(sample_node, bool)):
            # sometimes we get mixed types....
            return_string = [hex(node) if isinstance(node, (int, long)) else repr(node) for node in self._nodes]
            return "[" + ", ".join(return_string) + "]"

        return str(self._nodes)

    def __repr__(self):
        return str(self)

    def __copy__(self):
        # create new object using the same nodes name and path
        newself = ComponentGroup(self._nodes, self._groupname, self._grouppath)
        return newself


class NamedComponentDefinitionAlias(NamedComponentDefinition):
    """
    Defines a NamedComponentDefinition alias to help us group and rename
    nodes from different named components.

    When nodes are added to this, we keep a reference to the original
    node definition and component value.
    """

    def __init__(self, name, value_class=None):
        super(NamedComponentDefinitionAlias, self).__init__(
            name, info=None, nodes=None, value_class=value_class
        )

    def add_node(self, node, newname=None):
        """
        Add a node definition to this definition alias.

        What we store per node is a NamedNodeDefinitionAlias with
        the node's definition and named component.

        Args:
            node (NamedNodeValue): The node value from which to get
                the definition we want to add.
            newname (str, optional): The name of the node definition.
                Defaults to the node's name.
        """
        newname = newname or node.name
        node_definition_alias = NamedNodeDefinitionAlias(
            newname,
            node.definition,
            node.component,
        )
        self._nodes[newname] = node_definition_alias
        return node


def CreateComponentDefinitionAlias(name, value_class=None):
    """
    Create a new component definition alias object and return it.

    Args:
        name (str): Name of the component definition alias (default name
            if NamedComponent is not given a name)
        value_class (cls, optional): Class that must be derived from
            NamedComponent. Defaults to None.
    """
    new_component_alias = NamedComponentDefinitionAlias(
        name,
        value_class=value_class
    )
    return new_component_alias


def _proxy_debug(proxy):
    return
    # import pdb;pdb.set_trace()