
# INTEL CONFIDENTIAL
# Copyright 2014 2018 Intel Corporation
#
# The source code  contained or  described herein and  all documents related to
# the source code  ("Material") are owned by Intel Corporation or its suppliers
# or licensors.  Title to the  Material  remains with  Intel Corporation or its
# suppliers  and licensors. The Material contains trade secrets and proprietary
# and  confidential  information  of  Intel  or  its  suppliers  and  licensors.
# The Material  is protected  by worldwide  copyright and trade secret laws and
# treaty provisions.  No part of the Material  may  be used, copied, reproduced,
# modified, published, uploaded, posted, transmitted, distributed, or disclosed
# in any way without Intel's prior express written permission. No license under
# any  patent,  copyright, trade secret or other intellectual property right is
# granted  to  or conferred upon you by disclosure or delivery of the Materials,
# either expressly, by implication, inducement, estoppel or otherwise.
# Any license under such intellectual property rights must be express and
# approved by Intel in writing.





from __future__ import absolute_import
from __future__ import division
from __future__ import print_function

from .register import AccessRegister
import ipccli
from ..utils.str2py import str2py
try:
    from svtools.common import baseaccess
except ImportError:
    baseaccess = None

__all__ = [
         "AccessTap",
         "AccessTapBaseaccess",
        ]

class AccessTap(AccessRegister):
    """

    Performs a tap read/write.
        For reads: uses ipc.irdrscanreplace
        For writes: use ipc.irdrscan
        For read-writes: uses ipc.irdrscanrmw

    """
    #:
    requires = [
                'tap_ir',
                'numbits'
                ]
    requires_from_target = [('did', 'tap_name')]

    @classmethod
    def read(cls, node):
        """
        performs an irdrscanreplace to read a tap node

        Args:
            node : NodeValue object that we are trying to return the data for

        """
        ipc = ipccli.baseaccess()
        did = cls.lookup_info(node, ('did', 'tap_name'))
        tap_ir = node.info['tap_ir']  # ir to read from
        numbits = node.numbits
        return ipc.irdrscanreplace(did, tap_ir, numbits, None)

    @classmethod
    def write(cls, node, value):
        """
        performs an irdrscan to write a tap register

        Args:
            node : NodeValue object that we are trying to return the data for
            value : value to write to the node
        """
        ipc = ipccli.baseaccess()
        did = cls.lookup_info(node, ('did', 'tap_name'))
        tap_ir = node.info['tap_ir']  # ir to read from
        numbits = node.numbits
        return ipc.irdrscan(did, tap_ir, numbits, value)

    @classmethod
    def read_write(cls, node, mask, value):
        """
        performs an irdrscanrmw to write a tap register
        Args:
            node : NodeValue object that we are trying to return the data for
            mask : what bits to modify
            value : value to write to the node
        """
        ipc = ipccli.baseaccess()
        did = cls.lookup_info(node, ('did', 'tap_name'))
        tap_ir = node.info['tap_ir']  # ir to read from
        numbits = node.numbits
        # note, we don't do anything special for w1c since we should be only modifying
        # the bits specified...this may be incorrect though, I don't know if rmw actually
        # does anything 'special' to make sure we are only touching the masked bits
        return ipc.irdrscanrmw(did, tap_ir, value, mask, numbits)

class AccessTapNoWrData(AccessTap):
    """

    Performs a tap read/write, but writes will not return data
        For reads: uses ipc.irdrscanreplace
        For writes: use ipc.irdrscan
        For read-writes: uses ipc.irdrscanrmw

    """
    #:
    requires = [
                'tap_ir',
                'numbits'
                ]
    requires_from_target = [('did', 'tap_name')]

    @classmethod
    def write(cls, node, value):
        """
        performs an irdrscan to write a tap register

        Args:
            node : NodeValue object that we are trying to return the data for
            value : value to write to the node
        """
        ipc = ipccli.baseaccess()
        did = cls.lookup_info(node, ('did', 'tap_name'))
        tap_ir = node.info['tap_ir']  # ir to read from
        numbits = node.numbits
        return ipc.irdrscan(did, tap_ir, numbits, value, returnData=False)

    @classmethod
    def read_write(cls, node, mask, value):
        """
        performs an irdrscanrmw to write a tap register
        Args:
            node : NodeValue object that we are trying to return the data for
            mask : what bits to modify
            value : value to write to the node
        """
        ipc = ipccli.baseaccess()
        did = cls.lookup_info(node, ('did', 'tap_name'))
        tap_ir = node.info['tap_ir']  # ir to read from
        numbits = node.numbits
        # note, we don't do anything special for w1c since we should be only modifying
        # the bits specified...this may be incorrect though, I don't know if rmw actually
        # does anything 'special' to make sure we are only touching the masked bits
        return ipc.irdrscanrmw(did, tap_ir, value, mask, numbits, returnData=False)

class AccessTapNoReplaceNoWrData(AccessTap):
    """
    Performs a tap read/write, but writes will not return data
        For reads: uses ipc.irdrscan
        For writes: use ipc.irdrscan
        For read-writes: uses ipc.irdrscanrmw

    """
    #:
    requires = [
                'tap_ir',
                'numbits'
                ]
    requires_from_target = [('did', 'tap_name')]

    @classmethod
    def read(cls, node):
        """
        performs an irdrscanreplace to read a tap node

        Args:
            node : NodeValue object that we are trying to return the data for

        """
        ipc = ipccli.baseaccess()
        did = cls.lookup_info(node, ('did', 'tap_name'))
        tap_ir = node.info['tap_ir']  # ir to read from
        numbits = node.numbits
        return ipc.irdrscan(did, tap_ir, numbits, None)

    @classmethod
    def write(cls, node, value):
        """
        performs an irdrscan to write a tap register

        Args:
            node : NodeValue object that we are trying to return the data for
            value : value to write to the node
        """
        ipc = ipccli.baseaccess()
        did = cls.lookup_info(node, ('did', 'tap_name'))
        tap_ir = node.info['tap_ir']  # ir to read from
        numbits = node.numbits
        return ipc.irdrscan(did, tap_ir, numbits, value, returnData=False)

    @classmethod
    def read_write(cls, node, mask, value):
        """
        performs an irdrscanrmw to write a tap register
        Args:
            node : NodeValue object that we are trying to return the data for
            mask : what bits to modify
            value : value to write to the node
        """
        ipc = ipccli.baseaccess()
        did = cls.lookup_info(node, ('did', 'tap_name'))
        tap_ir = node.info['tap_ir']  # ir to read from
        numbits = node.numbits
        # note, we don't do anything special for w1c since we should be only modifying
        # the bits specified...this may be incorrect though, I don't know if rmw actually
        # does anything 'special' to make sure we are only touching the masked bits
        return ipc.irdrscanrmw(did, tap_ir, value, mask, numbits, returnData=False)

class AccessTapBaseaccess(AccessRegister):
    """
    This acces works on DAL or OpenIPC, but it will not
    have the perfomance benefits possible on the OpenIPC
    with using the device locker

    Performs a tap read/write. This i
        For reads: uses base.irdrscanreplace
        For writes: use base.irdrscan
        For read-writes: uses base.irdrscanrmw

    """
    #:
    requires = [
                'tap_ir',
                'numbits'
                ]
    requires_from_target = [('did', 'tap_name')]

    @classmethod
    def read(cls, node):
        """
        performs an irdrscanreplace to read a tap node

        Args:
            node : NodeValue object that we are trying to return the data for

        """
        assert baseaccess is not None, "pythonsv-common must be installed for this access to work"
        base = baseaccess.getglobalbase()
        did = cls.lookup_info(node, ('did', 'tap_name'))
        tap_ir = node.info['tap_ir']  # ir to read from
        numbits = node.numbits
        return base.irdrscanreplace(did, tap_ir, numbits, None)

    @classmethod
    def write(cls, node, value):
        """
        performs an irdrscan to write a tap register

        Args:
            node : NodeValue object that we are trying to return the data for
            value : value to write to the node
        """
        assert baseaccess is not None, "pythonsv-common must be installed for this access to work"
        base = baseaccess.getglobalbase()
        did = cls.lookup_info(node, ('did', 'tap_name'))
        tap_ir = node.info['tap_ir']  # ir to read from
        numbits = node.numbits
        return base.irdrscan(did, tap_ir, numbits, value)

    @classmethod
    def read_write(cls, node, mask, value):
        """
        performs an irdrscanrmw to write a tap register
        Args:
            node : NodeValue object that we are trying to return the data for
            mask : what bits to modify
            value : value to write to the node
        """
        assert baseaccess is not None, "pythonsv-common must be installed for this access to work"
        base = baseaccess.getglobalbase()
        did = cls.lookup_info(node, ('did', 'tap_name'))
        tap_ir = node.info['tap_ir']  # ir to read from
        numbits = node.numbits
        return base.irdrscanrmw(did, tap_ir, value, mask, numbits)


# class AccessTapWtihPrescan(AccessTap):
#     requires = AccessTap.requires + ['tap_prescan']
#     _str2py_cache = {}
#
#     @classmethod
#     def prescan(cls, node):
#         prescan = cls.lookup_info(node, 'tap_prescan')
#         # convert the string to a python function and save a reference
#         if prescan not in cls._str2py_cache:
#             cls._str2py_cache[prescan] = str2py.str2py(prescan)
#         # call the pre-scan, assume that it needs the node as a parameter (?)
#         cls._str2py_cache[prescan](node)
#
#     @classmethod
#     def read(cls, node):
#         cls.prescan(node)
#         return super(AccessTapWtihPrescan, cls).read(node)
#
#     @classmethod
#     def write(cls, node, value):
#         cls.prescan(node)
#         return super(AccessTapWtihPrescan, cls).write(node, value)
#
#     @classmethod
#     def read_write(cls, node, value):
#         cls.prescan(node)
#         # assumes read_write does NOT call read then write, otherwise, the prescan
#         # will get run twice
#         return super(AccessTapWtihPrescan, cls).read_write(node, value)
