
# INTEL CONFIDENTIAL
# Copyright 2014 2018 Intel Corporation
#
# The source code  contained or  described herein and  all documents related to
# the source code  ("Material") are owned by Intel Corporation or its suppliers
# or licensors.  Title to the  Material  remains with  Intel Corporation or its
# suppliers  and licensors. The Material contains trade secrets and proprietary
# and  confidential  information  of  Intel  or  its  suppliers  and  licensors.
# The Material  is protected  by worldwide  copyright and trade secret laws and
# treaty provisions.  No part of the Material  may  be used, copied, reproduced,
# modified, published, uploaded, posted, transmitted, distributed, or disclosed
# in any way without Intel's prior express written permission. No license under
# any  patent,  copyright, trade secret or other intellectual property right is
# granted  to  or conferred upon you by disclosure or delivery of the Materials,
# either expressly, by implication, inducement, estoppel or otherwise.
# Any license under such intellectual property rights must be express and
# approved by Intel in writing.



from __future__ import absolute_import
from __future__ import division
from __future__ import print_function

from ..access import NodeAccess
from ..nodes import NodeTypes
from ..logging import getLogger
from ..utils._py2to3 import *
from ..utils.bitmap import BitMapConstantEntry, BitMapEntry
from .. import nodes # for getting to debug variable
from .. import settings
from .general import AccessStoredValues
import copy
import ipccli
from ..utils.bitmanip import num_to_buffer
try:
    import svtools.common.baseaccess as baseaccess
except ImportError:
    baseaccess = None

_LOG = getLogger()

__all__ = [
            "StateNodeAccess",
            "StateNodeOfflineAccess",
            "StateNodeAccessNoLock",
            "AccessBitMapFromParent",
            "ScanFieldAccess",
           ]


class StateNodeBase(NodeAccess):
    """
    Base Class for providing defautl 'repr' style that prints state children

    This class does not actually provide read, this still must be done.
    """
    methods = ['repr'] + NodeAccess.methods

    # this is used to prevent logging of state info in telemetry
    _logged_read = True
    _logged_write = True
    _logged_read_write = True
    _logged_store = True
    _logged_flush = True

    @classmethod
    def repr(cls, node, **kwargs):
        indent_level = kwargs.pop("indent", 0)
        space = "  "
        indent = indent_level * space
        indent2 = (indent_level + 1) * space
        mystr = []

        # if len(children)>0: # this check needed?
        mystr.append(indent + "{0} {{".format(node.name))

        # now the same for items
        if node.type == NodeTypes.Array:
            for n, child_node in enumerate(node):
                my_repr = getattr(child_node, "repr", None)
                if my_repr is not None:
                    mystr.append(my_repr(indent=indent_level + 1))
                else:
                    try:
                        mystr.append(indent2 + child_node.name + " = " + repr(child_node))
                    except:
                        print("Error calling repr on '{0}'".format(child_node.name))
                        raise
        # now do typical children
        for n, child_node in enumerate(node.nodes):
            my_repr = getattr(child_node, "repr", None)
            if my_repr is not None:
                mystr.append(my_repr(indent=indent_level + 1))
            else:
                try:
                    mystr.append(indent2 + child_node.name + " = " + repr(child_node))
                except:
                    print("Error calling repr on '{0}'".format(child_node.name))
                    raise
        mystr.append(indent + "}")
        return "\n".join(mystr)


class StateNodeAccess(StateNodeBase):
    """
    This should be the default access for any Node related to AFD. Reads request a lock and then
    try to get any children nodes to do a read

    Note:
        Returns 0 as the value since this does not do any accesses itself
    """
    #:
    #: does not require anything from node or component
    requires = []
    requires_from_target = []

    @classmethod
    def read(cls, node):
        if settings.DEBUG:
            _LOG.debugall("StateNodeAccess.read called for '{0}'".format(node.name))
        with _queue_and_execute():
            if node.type == NodeTypes.Array:
                # if we are an array
                for child in node:
                    # don't do get_value on the children that are the
                    # end of the tree...or they will ask their parents
                    # for the data and therefore kick off a read
                    child.read()
            # next read any fields we may have
            for child in node.nodes:
                # don't do get_value on the children that are the
                # end of the tree...or they will ask their parents
                # for the data and therefore kick off a read (which gets
                # us in to recursion)
                if len(child.nodes) > 0 or child.type == NodeTypes.Array:
                    child.read()
        # return a value to prevent a future implicit read
        return 0


    @classmethod
    def clear_child_values(cls, node):
        """for make sure we have no cached values in our child nodes"""
        for n in node.walk_nodes():
            n._value = None

class StateRegisterArrayAccess(StateNodeAccess):
    """
    Default access for register arrays
    """
    @classmethod
    def repr(cls, node, **kwargs):
        mystr = []

        if node.type == NodeTypes.Array:
            return node.valuestr(skip_fields=True)


class StateNodeAccessNoLock(StateNodeBase, AccessStoredValues):
    """
    This should be the default access for any Node related to AFD where we
    don't have CLI lock control

    Note:
        Returns 0 as the value since this does not do any accesses itself
    """
    #: does not require anything from node or component
    requires = []
    requires_from_target = []
    methods = ['repr'] + NodeAccess.methods

    @classmethod
    def read(cls, node):
        if settings.DEBUG:
            _LOG.debugall("StateNodeAccess.read called for '{0}'".format(node.name))
        if node.type == NodeTypes.Array:
            # if we are an array
            for child in node:
                # don't do get_value on the children that are the
                # end of the tree...or they will ask their parents
                # for the data and therefore kick off a read
                child.read()
        else:
            # next read any fields we may have
            for child in node.nodes:
                # don't do get_value on the children that are the
                # end of the tree...or they will ask their parents
                # for the data and therefore kick off a read
                if len(child.nodes)>0 or child.type==NodeTypes.Array:
                    child.read()
            # or if we are a
        # return a value to prevent a future implicit read
        return 0



# This access class is the base class of all generated accesses from the tdef parsing flow
class StateNodeImplementedAccess(StateNodeAccess):
    @classmethod
    def read(cls, node):
        raise NotImplemented("Expected child class to be implemented that fills in this read")


class StateNodeOfflineAccess(StateNodeBase, AccessStoredValues):
    pass



class AccessBitMapFromParent(NodeAccess):
    """
    BETA Warning! this access is subject to change.

    Currently this is used in AFD arrays
    """
    #: add repr so that we can show enum
    methods = ['repr'] + NodeAccess.methods
    #: parent must have definition.info['bitmaps'][node.name]
    requires = []
    requires_from_target = ['bitmaps', 'data']

    @classmethod
    def read(cls, node):
        if baseaccess is None:
            raise RuntimeError("svtools.common required for AFD flows")
        ipc = baseaccess.getglobalbase().getapi()
        BitData = ipc.BitData
        # make sure parent has a value that we can pull from
        if node.parent.value is None:
            node.parent.read()
        # we are going to assume our parent has filled in our info (?) and been read
        # assumes NO overlap
        # mapping of data to bits
        if 'bitmaps' not in node.parent.definition.info:
            raise Exception("FlowError for node: {0}, missing bitmaps".format(node.path))
        if node.name not in node.parent.definition.info['bitmaps']:
            # this happens if TDEF forgets to assign value
            _LOG.warning("ERROR: FlowError for node: {0}, ".format(node.path))
            _LOG.warning("ERROR:    parent is missing bitmap data for this field -- assuming 0")
            _LOG.warning("ERROR:    if used with TDEF this means the TDEF did not assign a value")
            return BitData(0, 0)
        if 'data' not in node.parent.target_info:
            raise Exception("Flow Error for node {0}, parent is missing data".format(node.name))
        # the node here is typicall a "field" in an array, and our parent
        # is the highest thing that has the dimensions - example
        # here we are in the access for the DATA of an l1, and our
        # parent is the particular L1 set/way that was read (but the entire
        # set#/way# was read, and we are grabbing our piece of the data from it)
        bitmap = node.parent.definition.info['bitmaps'][node.name]
        mapdata = node.parent.target_info['data']
        # using regular python numbers here for speed (instead of a bitdata object)
        myvalue = BitData(bitmap.totalbits, 0)
        # bitmap components: data,from_lower,dest_lower,numbits,invert)
        # i do not remember why this is re-implemented versus using the
        # get_value from the bitmap....
        for mapinfo in bitmap:
            # currently supported bitdata info
            if isinstance(mapinfo, BitMapEntry):
                d = mapdata.get(mapinfo.datasource, None)
                if d is None:
                    raise BaseException("Missing data source {0} for node '{1}'".format(mapinfo.datasource,
                                                                             node.name))
                # needed to keep legacy behavior of not throwing errors when the bitdata from the previous operation
                # was not big enough
                d.truncate = False
                # we could assume bit data and call extract, but every bit of performance counts:
                d = d[mapinfo.from_lower:mapinfo.from_lower+mapinfo.numbits-1]
                if mapinfo.invert:
                    # for invert, redo the mask so that we don't have issues with the shift below
                    d = ~d
                myvalue[mapinfo.dest_lower:mapinfo.dest_lower+mapinfo.numbits-1] = d

            elif isinstance(mapinfo, BitMapConstantEntry):
                if mapinfo.invert:
                    myvalue[mapinfo.dest_lower:mapinfo.dest_lower+mapinfo.numbits-1] = ~mapinfo.constant
                else:
                    myvalue[mapinfo.dest_lower:mapinfo.dest_lower+mapinfo.numbits-1] = mapinfo.constant

            else:
                raise RuntimeError("unexpected type found...")
        return myvalue

    @classmethod
    def write(cls, node, value):
        """default write for this class, is to set a stored value and ask parent to do write"""
        # make sure parent has been read at least once
        if node.parent.value is None:
            node.parent.read()
        cls.store(node, value)
        node.parent.flush()

    @classmethod
    def repr(cls, node, **kwargs):
        indent_level = kwargs.pop("indent", 0)
        space = "  "
        indent = indent_level * space
        value = node.get_value()
        if isinstance(value, (int,long)):
            valuestr = "0x{:x}".format(value)
        else:
            valuestr = repr(value)
        text = "{indent}{name} = {valuestr}".format(
            indent=indent,
            name=node.name,
            valuestr=valuestr,  # force to int?
        )
        the_decode = node.decode()
        if the_decode is not None:
            text += " (%s) " % str(the_decode)
        return text

class ScanFieldAccess(NodeAccess):
    """
    This class is used when the parent has a large chunk of data from the read.
    We use the maps to pull the correct bits from our parent in to our bits value
    """
    requires = ['bits', 'maps']
    requires_parent = []
    methods = ['repr'] + NodeAccess.methods


    @classmethod
    def read(cls, node):
        # would need to check map and call parent read if not set?
        # if node.parent._value == None:
        #    parent_value = node.parent.get_value()
        # else:
        #    parent_value = node.parent._value
        # this is a bit of a hack due to the fact that BitData does not have support for underlying value being a buffer
        ipc = baseaccess.getglobalbase().getapi()
        access = baseaccess.getaccess()
        # we know we are not bundle safe, but we rarely need to be, so just return 0
        if access == "ipc" and ipc.BitData != ipccli.BitData:
            return 0

        parent_buffer = node.parent.target_info.get('_scan_buffer_value', None)
        if parent_buffer is None:
            parent_bdata = node.parent.get_value()
            parent_buffer = num_to_buffer(long(parent_bdata), parent_bdata.BitSize)
            # save to target info so that it disappears with the scan
            node.parent.target_info['_scan_buffer_value'] = parent_buffer

        # assumes that length of bits and maps was already checked and
        # that the node definition was created correctly
        value = 0
        for bnum in range(len(node.definition.info.bits)):
            parentbit_num = node.definition.info.maps[bnum]
            bytepos,bitpos = parentbit_num // 8, parentbit_num % 8
            value |= ((int(parent_buffer[bytepos]) >> bitpos)&1) << node.definition.info.bits[bnum]

        # now...create bit data with max bit
        return ipc.BitData(node.definition.info.bits[-1]+1, value)

    @classmethod
    def repr(cls, node, **kwargs):
        indent_level = kwargs.pop("indent", 0)
        space = "  "
        indent = indent_level * space
        # BKM here?
        if node.value is None:
            node.read()
        max_bit = max(node.definition.info.bits)
        min_bit = min(node.definition.info.bits)
        # everything is contiguous
        if len(node.definition.info.bits) == (max_bit-min_bit+1):
            text = "{indent}{name}[{upper}:{lower}] = 0x{value:x}".format(
                        indent=indent,
                        name=node.name,
                        upper=max_bit,
                        lower=min_bit,
                        value=node.value,  # force to int?
                        )
            the_decode = node.decode()
            if the_decode is not None:
                text += " (%s) " % str(the_decode)
            return text
        else:
            # dang, not contiguous, this is tougher, make a copy so we don't mess
            # anything up
            all_bits = copy.copy(node.definition.info.bits)
            all_bits.sort()
            all_bits.reverse()
            # keep a list of what is contiguous
            prev_bit = all_bits[0]
            regions = [ [prev_bit, prev_bit] ]
            for this_bit in all_bits[1:]:
                # nothing special, simple increment
                if abs(this_bit-prev_bit) == 1:
                    prev_bit = this_bit
                else:  # assuming always >, because <= 0 would be a bug
                    # save off prev_bit as end of the last region
                    regions[-1][1] = prev_bit
                    prev_bit = this_bit
                    regions.append([prev_bit, prev_bit])
            # end of loop, update last region
            regions[-1][1] = this_bit
            # build crazy string
            region_str = []
            region_str.append("[")
            for r, region in enumerate(regions):
                start, stop = region
                if r > 0:
                    region_str.append(",")
                # region did not have a range
                if start-stop == 0:
                    region_str.append("%d" % start)
                # region had a range
                else:
                    region_str.append("%d:%d" % (start, stop))
            region_str.append("]")
            # finally finished....
            region_str = "".join(region_str)
            text = "{indent}{name}{region_str} = 0x{value:x}".format(
                indent=indent,
                name=node.name,
                region_str=region_str,
                value=node.value,  # force to int?
                )
            the_decode = node.decode()
            if the_decode is not None:
                text += " (%s) " % str(the_decode)
            return text


class _queue_and_execute(object):
    def __enter__(self):
        self.locker = None
        try:
            from svtools.common import baseaccess
        except ImportError:
            pass # Do nothing, assuming ipccli is not installed
        else:
            access = baseaccess.getaccess()
            base = baseaccess.getglobalbase()
            environment = getattr(base, "environment", None)
            if environment == "ipc":
                ipc = base.getapi()
                self.locker = ipc.device_locker()
                self.locker.__enter__()

    def __exit__(self, exception_type, exception_value=None, traceback=None):
        if self.locker is not None:
            self.locker.__exit__(exception_type, exception_value, traceback)