# INTEL CONFIDENTIAL
# Copyright 2019Intel Corporation
#
# The source code  contained or  described herein and  all documents related to
# the source code  ("Material") are owned by Intel Corporation or its suppliers
# or licensors.  Title to the  Material  remains with  Intel Corporation or its
# suppliers  and licensors. The Material contains trade secrets and proprietary
# and  confidential  information  of  Intel  or  its  suppliers  and  licensors.
# The Material  is protected  by worldwide  copyright and trade secret laws and
# treaty provisions.  No part of the Material  may  be used, copied, reproduced,
# modified, published, uploaded, posted, transmitted, distributed, or disclosed
# in any way without Intel's prior express written permission. No license under
# any  patent,  copyright, trade secret or other intellectual property right is
# granted  to  or conferred upon you by disclosure or delivery of the Materials,
# either expressly, by implication, inducement, estoppel or otherwise.
# Any license under such intellectual property rights must be express and
# approved by Intel in writing.
from ..access import NodeAccess
from ..import settings
from namednodes import access as _access
from .general import (
    OfflineMissingNode,
    AccessOffline,
    AccessStoredValues,
    _StoredValuesFormat
)


class AccessSnapshot(AccessStoredValues):
    """
    Base class for offline accesses. Overrides get_access_info to display
    meaningful data.
    """
    # this is kept directly in target info similarly to offline mode
    # this is kept within the snapshot specific info
    _snapshot_key = "_snapshot"
    # these are already inside snapshot and dont need to be "redeclared" as Private
    _delayed_reads = "delayed_reads"
    _all_reads_delayed = "all_reads_delayed"
    _capture_time = False


class AccessSnapshotCapture(AccessSnapshot):
    """Replicates the default accesses"""

    methods = AccessSnapshot.methods + ['delayed_read']
    original_class = None  # derived classes fill this in

    @classmethod
    def delayed_read(cls, node):
        """Registers a read to be sent when a freeze is requested"""
        node.component.origin.target_info. \
            setdefault(cls._snapshot_key, {}). \
            setdefault(cls._delayed_reads, []). \
            append((node.component.path, node.nodepath))

    @classmethod
    def read(cls, node, *args, **kwargs):
        """
        """
        if (node.component.origin.target_info[cls._snapshot_key][cls._all_reads_delayed] and
                not cls._capture_time):
            cls.delayed_read(node)
            # not sure what to actually return here for a delayed read..
            return 0

        # read from original class
        if cls.original_class is None:
            raise RuntimeError("this class must be derived and original_class filed in before using")
        value = cls.original_class.read(node, *args, **kwargs)
        # we add our value since are are going to call recursively to our children
        node._value = value
        cls._write_cache(node, value, include_children=True)
        return value

    @classmethod
    def write(cls, node, value, *args, **kwargs):
        """
        """
        if node.component.origin.target_info[cls._snapshot_key][cls._all_reads_delayed] and not cls._capture_time:
            raise RuntimeError("Write method not allowed if delayed reads are being forced")
        if cls.original_class is None:
            raise RuntimeError("this class must be derived and original_class filed in before using")
        return cls.original_class.write(node, value, *args, **kwargs)

    @classmethod
    def flush(cls, node, *args, **kwargs):
        if node.component.origin.target_info[cls._snapshot_key][cls._all_reads_delayed] and not cls._capture_time:
            raise RuntimeError("Flush method not allowed if delayed reads are being forced")
        access_class_name = node.access_class_name
        if cls.original_class is None:
            raise RuntimeError("this class must be derived and original_class filed in before using")
        return cls.original_class.flush(node, *args, **kwargs)

    @classmethod
    def read_write(cls, node, mask, value, *args, **kwargs):
        if node.component.origin.target_info[cls._snapshot_key][cls._all_reads_delayed] and not cls._capture_time:
            raise RuntimeError("read_write method not allowed if delayed reads are being forced")
        access_class_name = node.access_class_name
        if cls.original_class is None:
            raise RuntimeError("this class must be derived and original_class filed in before using")
        return cls.original_class.read_write(node, mask, value, *args, **kwargs)

    @classmethod
    def is_access_available(cls, component, *args, **kwargs):
        if cls.original_class is None:
            raise RuntimeError("this class must be derived and original_class filed in before using")
        return cls.original_class.is_access_available(component)

    @classmethod
    def get_access_info(cls, node, **kwargs):
        if cls.original_class is None:
            raise RuntimeError("this class must be derived and original_class filed in before using")
        return cls.original_class.get_access_info(node, **kwargs)

    @classmethod
    def is_available(cls, node, *args, **kwargs):
        if cls.original_class is None:
            raise RuntimeError("this class must be derived and original_class filed in before using")
        node_available = cls.original_class.is_available(node, *args, **kwargs)
        return node_available

    @classmethod
    def is_available_cache(cls, node):
        return node.component.target_info. \
            setdefault('is_available_cache', {}). \
            setdefault(cls.original_class.__name__, {})


class AccessSnapshotFrozen(AccessSnapshot):
    """
    Similar to offline, reads what is available in the cache
    """

    @classmethod
    def read(cls, node):
        """
        Read from the frozen value ()
        """
        value = cls._read_cache(node)
        # was not able to get a value back
        if value != -1:
            return value
        # compute a better default
        raise OfflineMissingNode("Snapshot: node missing data.\n\tfullpath: {}".format(node.path), node.path)

    @classmethod
    def write(cls, node, value):
        """
        Writes are not supported after frozen is called
        """
        raise RuntimeError("Snapshot: Unsupported operation: Write")

    @classmethod
    def is_available(cls, node):
        constant_dict = node.component.origin.target_info.get(cls._cache_name, None)
        if constant_dict is None:
            constant_dict = cls._init_cache(node)
        node_path = node.target_info.get("aliases_original_path", None)
        component_path = node.component.path
        if node_path is None:
            node_path = node.nodepath
        else:
            # turn to relative path...
            node_path = node_path.replace(component_path + ".", "")

        return constant_dict.get(component_path, {}).get(node_path, KeyError) is not KeyError
