
# INTEL CONFIDENTIAL
# Copyright 2014 2018 Intel Corporation
#
# The source code  contained or  described herein and  all documents related to
# the source code  ("Material") are owned by Intel Corporation or its suppliers
# or licensors.  Title to the  Material  remains with  Intel Corporation or its
# suppliers  and licensors. The Material contains trade secrets and proprietary
# and  confidential  information  of  Intel  or  its  suppliers  and  licensors.
# The Material  is protected  by worldwide  copyright and trade secret laws and
# treaty provisions.  No part of the Material  may  be used, copied, reproduced,
# modified, published, uploaded, posted, transmitted, distributed, or disclosed
# in any way without Intel's prior express written permission. No license under
# any  patent,  copyright, trade secret or other intellectual property right is
# granted  to  or conferred upon you by disclosure or delivery of the Materials,
# either expressly, by implication, inducement, estoppel or otherwise.
# Any license under such intellectual property rights must be express and
# approved by Intel in writing.



from __future__ import absolute_import
from __future__ import division
from __future__ import print_function

from ..access import NodeAccess
from ..utils._py2to3 import *
from .. import settings


class AccessRegister(NodeAccess):
    @classmethod
    def read_write(cls, node, mask, value):
        """
        Special read_write function needed for registers due to the possibility
        of having fields that are write-1-to-clear.

        If no rw1c is present then the NodeAccess.read_write is called
        """
        if node.definition.info.get('w1c_present', False) is False:
            return super(AccessRegister, cls).read_write(node, mask, value)
        else:
            regval = cls.read(node)
            # apply w1c mask
            regval = regval & ~(node.definition.info.get('w1c_mask',0))
            # clear out bits we are going to update
            regval = regval & ~(mask)
            # or in actual data
            regval = regval | (value & mask)
            # now issue the actual write
            return cls.write(node, regval)

    @classmethod
    def flush(cls, node, **kwargs):
        """Special flush function needed for registers due the possibility
        of having fields that are wite-1-to-clear

        If no rw1c is present then the NamedNodeValue.flush is called
        """
        if node.definition.info.get('w1c_present', False) is False:
            return super(AccessRegister, cls).flush(node, **kwargs)
        else:
            if node.stored_value is None:
                raise RuntimeError("flush called, but no data stored for this node")
            # PythonSv backwards compatibility that's why no "_" here
            skip_read = kwargs.get("skipRead",False)
            # if no stored mask, then just write out
            # ...this means someone just called store on the main node
            if node.stored_mask is None:
                return cls.write( node, node.stored_value )
            else:
                # this means someone from a child queued up some information
                # three scenarios
                # - we have an existing value we should apply our mask to
                # - we should do a read-modify-write to the register
                # - we should just write to the register (that's handled earlier)
                # if we have a stored mask, then use read-modify-write
                if node._value is None and skip_read is False:
                    return cls.read_write(node, node.stored_mask, node.stored_value )
                else:
                    newval = node._value or 0 # make sure 0 is 2nd here
                    newval = newval & ~(node.definition.info.get("w1c_mask")) # clear out w1c bits
                    newval = newval & (~node.stored_mask) # clear out old value
                    newval = newval | node.stored_value        # OR in new one...
                    return cls.write(node, newval)




class AccessField(NodeAccess):
    """Access that all register fields should use for getting their value. Requires
    the bits in the parent to be contiguous
    """
    #: requires the info to have numbits and lowerbit to use when accessing the parent
    requires = ['numbits','lowerbit']

    @classmethod
    def update_value(cls,node):
        """
        Get the value from our parent (which will cause a read if it does not have the value)
        then update our value to be based off our parent using upperbit and lowerbit
        """
        # this should use parents cached value if it has one, and if it does not,
        # then it will force a read
        number = long(node.parent)
        # if a bitdata, keep it a bitdata
        # not so sure that this is a good idea...bitdata's for fields are dangerous due to their
        #if isinstance(number,BitData):
        #    ourval = number[node.definition.info.lowerbit:node.definition.info.upperbit]
        # otherwise we need to get the value
        #else:
        numbits = node.numbits
        lower = node.lowerbit
        ourval = (number >> lower) & ( ((1<<(numbits)) -1) )
        return ourval

    @classmethod
    def write(cls,node,value):
        """
        Using our upper/lower bit calculate our portion of the parents value and a mask
        to represent our bits. Then request the parent to do a read-modify-write
        """
        numbits = node.numbits
        lower = node.lowerbit
        # ~ because we are masking out
        field_mask = (1 << numbits) -1
        register_mask = field_mask << lower
        # we have to call long here due to BitData not supporting shifting...there's gotta
        # be a better way, but cant do an isinstance on DAL's bitdata
        write_value = long(value & field_mask) << lower
        # bounds check, make sure we do > 0 because BitData evaluates to true if it has a bit length
        if (value > 0) and ((~field_mask) & value)>0:
            raise ValueError("This field does not support that many bits being written")
        # used read-modify-write in case parent has a trick for speed...
        # if w1c not present, then nothing special
        return node.parent.read_write(register_mask, write_value)

    @classmethod
    def read(cls, node):
        """
        Cause the parent to do a read and then update our value using the parents latest value
        """
        node.parent.read()
        # update our value
        return cls.update_value(node)

    @classmethod
    def store(cls, node, value, mask=None):
        """
        Store value to later be flushed

        Args:
            node : NamedNodeValue passed in
            value : Value to store for a future flush
            mask : optional mask to use alongside value for eventual flush

        Value is stored in the parent (a RegisterValue)'s "stored_value" attribute. Along
        with update the stored_mask to repsent this field
        """
        if node.parent.stored_mask is None:
            node.parent.stored_mask = 0
        if node.parent.stored_value is None:
            node.parent.stored_value = 0
        upper = node.upperbit
        lower = node.lowerbit
        field_mask = ((1<<(upper-lower+1)) -1)
        if mask is not None:
            assert mask<=field_mask, "Too many bits in custom mask for field"
            field_mask = mask
        # make sure value written is valid for this field
        # do this before the mask check because it may be fewer bits anyway and
        if (value > 0) and ((~field_mask) & value) > 0:
            raise ValueError("This field (or mask) does not support that many bits being written")
        reg_mask = ((field_mask) << lower)
        # perform AND to make sure we didn't get bogus value
        value &= field_mask
        # now put date in to parent's info...
        node.parent.stored_mask |= reg_mask
        # Clear older field stored value if any
        node.parent.stored_value &= ((1 << node.parent.numbits) - 1) ^ reg_mask
        # Save new field value
        node.parent.stored_value |= (value << lower)

    @classmethod
    def flush(cls, node):
        """Field flush really just flushes the queued value at the Register"""
        return node.parent.flush()


