
# INTEL CONFIDENTIAL
# Copyright 2014 2018 Intel Corporation
#
# The source code  contained or  described herein and  all documents related to
# the source code  ("Material") are owned by Intel Corporation or its suppliers
# or licensors.  Title to the  Material  remains with  Intel Corporation or its
# suppliers  and licensors. The Material contains trade secrets and proprietary
# and  confidential  information  of  Intel  or  its  suppliers  and  licensors.
# The Material  is protected  by worldwide  copyright and trade secret laws and
# treaty provisions.  No part of the Material  may  be used, copied, reproduced,
# modified, published, uploaded, posted, transmitted, distributed, or disclosed
# in any way without Intel's prior express written permission. No license under
# any  patent,  copyright, trade secret or other intellectual property right is
# granted  to  or conferred upon you by disclosure or delivery of the Materials,
# either expressly, by implication, inducement, estoppel or otherwise.
# Any license under such intellectual property rights must be express and
# approved by Intel in writing.





"""
Module containing the pcicfg access implementation.
"""

from __future__ import absolute_import
from __future__ import division
from __future__ import print_function

from .register import AccessRegister
from ..utils.bitmanip import getbits, setbits
from ..utils._py2to3 import *
try:
    from svtools.common import baseaccess
except ImportError:
    baseaccess = None

__all__ = [
    "AccessPcieCfg",
    "AccessPcieMultiBus",
    "AccessMmio",
    "AccessMem",
    "AccessPcieBar",
    ]

class AccessPcieCfg(AccessRegister):
    """
    Performs a pcicfg access based on the required pci attributes that
    will be gathered from either the node or component. With precedence
    being given to the node.

    Note:
        Currently this requires PythonSv common module
    """
    #: Requires these parameters from the register for access
    requires = [
        'pci_offset',
        'numbits',
        'pci_segment',
        'pci_bus',
        'pci_device',
        'pci_function',
        ]
    requires_from_target = []
    defaults = {'pci_segment': 0}

    @classmethod
    def read(cls, node):
        """
        Perform a pcicfg read.

        Args:
            node (NamedNodeValue): The node that we are trying to get
                the data from.
        """
        return cls._access(node)

    @classmethod
    def write(cls, node, value):
        """
        Perform a pcicfg write.

        Args:
            node (NamedNodeValue): The node that we are trying to write
                the data to.
            value (object): The data to write to the node.
        """
        return cls._access(node, value)

    @classmethod
    def _access(cls, node, value=None):
        """
        Perform a pcicfg write/read.

        Args:
            node (NamedNodeValue): The node that we are trying to write
                the data to/read data from.
            value (object): The data to write to the node, if None a read
                is implied.
        """
        # use our requires to get the needed info
        access_info = cls.get_access_info(node)
        pci_size = access_info['numbits']//8
        return cls.pciecfg(
            access_info['pci_segment'],
            access_info['pci_bus'],
            access_info['pci_device'],
            access_info['pci_function'],
            access_info['pci_offset'],
            pci_size,
            value
            )

    @classmethod
    def pciecfg(cls, pci_segment, pci_bus, pci_device, pci_function, pci_offset, pci_size, value):
        """
        Used by the various access classes here to perform a pcie cfg transactions

        Args:
            pci_segment (long): The pci segment number (often this can be just 0)
            pci_bus (long): The pci bus number.
            pci_device (long): The pci device number.
            pci_function (long) : The pci function number.
            pci_offset (long) : The register offset, in bytes.
            size (long) : The register size, in bytes.
            value (long) : The value to write to the register, if None a read
                is implied.
        """
        # DO NOT add 8 to this list. if 8 is desired, then
        # a new class or some other new construct needs to be added
        aligned = pci_size in [1, 2, 4]
        aligned = aligned and (pci_offset % pci_size) == 0
        assert baseaccess is not None, "common module required"
        base = baseaccess.getglobalbase()
        if aligned:
            # take the short cut for most registers:
            return base.pcicfg(
                pci_bus,
                pci_device,
                pci_function,
                pci_offset,
                pci_size,
                value,
                segment=pci_segment,
                )
        else:
            # non-aligned go through more code
            return pciecfg(
                pci_segment,
                pci_bus,
                pci_device,
                pci_function,
                pci_offset,
                pci_size,
                value,
                )


class AccessPcieMultiBus(AccessPcieCfg):
    """
    This is based off pciecfg but supports components that may have
    registers across multiple busses

    See Also:
        - :py:class:`~namednodes.accesses.AccessPcieCfg`

    """
    requires = [
        'pci_segment',
        'pci_internal_bus',
        'pci_device',
        'pci_function',
        'pci_offset',
        'numbits',
        ]
    # must get a mapping of internal bus to real bus
    # key=pci_internal_bus
    # value=pci_bus
    requires_from_target = [
        'pci_bus_map',
    ]
    access_offset_key = "pci_offset"

    @classmethod
    def _access(cls, node, value=None):
        """
        Perform a pcicfg write/read.

        Args:
            node (NamedNodeValue): The node that we are trying to write
                the data to/read data from.
            value (object): The data to write to the node, if None a read
                is implied.
        """
        # use our requires to get the needed info
        access_info = cls.get_access_info(node)
        # get the pci_internal_bus from the pci_bus_map, return None if missing:
        bus_map = access_info['pci_bus_map']
        if bus_map is LookupError:
            raise LookupError("'pci_bus_map' missing from component")
        internal_bus = access_info['pci_internal_bus']
        if internal_bus is LookupError:
            raise LookupError("pci_internal_bus' missing from node/register")
        pci_bus = bus_map.get(internal_bus, None)
        if pci_bus is None:
            raise ValueError("pci_bus_map is missing our internal bus: %s"%
                             access_info['pci_internal_bus'])
        pci_size = access_info['numbits']//8
        return cls.pciecfg(
            access_info['pci_segment'],
            pci_bus,
            access_info['pci_device'],
            access_info['pci_function'],
            access_info['pci_offset'],
            pci_size,
            value
            )

class AccessPcieCfg64(AccessPcieCfg):
    """
    Performs a pcicfg access based on the required pci attributes that
    will be gathered from either the node or component.

    Note:
        - This version assumes 64 bit regs are legal vs. the original
          assumes 64bit needs to be broken in to two accesses

    Note:
        -
        - Currently this requires PythonSv common module
    """

    @classmethod
    def pciecfg(cls, pci_segment, pci_bus, pci_device, pci_function, pci_offset, pci_size, value):
        """
        Used by the various access classes here to perform a pcie cfg transactions, requires
        that pci_offset and pci_size are aligned

        Args:
            pci_segment (long): The pci segment number (often this can be just 0)
            pci_bus (long): The pci bus number.
            pci_device (long): The pci device number.
            pci_function (long) : The pci function number.
            pci_offset (long) : The register offset, in bytes.
            size (long) : The register size, in bytes.
            value (long) : The value to write to the register, if None a read
                is implied.
        """
        # DO NOT add 8 to this list. if 8 is desired, then
        # a new class or some other new construct needs to be added
        aligned = pci_size in [1, 2, 4, 8]
        aligned = aligned and (pci_offset % pci_size) == 0
        assert baseaccess is not None, "common module required"
        base = baseaccess.getglobalbase()
        if aligned:
            return base.pcicfg(
                pci_bus,
                pci_device,
                pci_function,
                pci_offset,
                pci_size,
                value,
                segment=pci_segment,
                )
        else:
            # non-aligned go through more code
            return pciecfg(
                pci_segment,
                pci_bus,
                pci_device,
                pci_function,
                pci_offset,
                pci_size,
                value,
                )

class AccessPcieMultiBus64(AccessPcieMultiBus):
    """
    This is based off pciecfg but supports components that may have
    registers across multiple busses.

    Note:
        - This version assumes 64 bit regs are legal vs. the original
          assumes 64bit needs to be broken in to two accesses

    See Also:
        - :py:class:`~namednodes.accesses.Access64PcieCfg`

    """
    pciecfg = AccessPcieCfg64.pciecfg


def pciecfg(pci_segment, pci_bus, pci_device, pci_function, pci_offset, pci_size, value):
    """
    Used by the various access classes here to perform a pcie cfg transation

    Args:
        pci_segment (long): The pci segment number (often this can be just 0)
        pci_bus (long): The pci bus number.
        pci_device (long): The pci device number.
        pci_function (long) : The pci function number.
        pci_offset (long) : The register offset, in bytes.
        size (long) : The register size, in bytes.
        value (long) : The value to write to the register, if None a read
            is implied.
    """
    # must be an oddball case
    # we can handle some of these odd cases of crossing a word boundary
    offset = pci_offset
    numbytes = pci_size
    lowerbit = 0
    upperbit = numbytes * 8
    if (offset & 0x3) == 1 and numbytes > 1:
        offset -= 1
        lowerbit += 8
        upperbit += 8
        numbytes += 1
    if (offset & 0x3) == 2 and numbytes > 2:
        offset -= 2
        lowerbit += 16
        upperbit += 16
        numbytes += 2
    if (offset & 0x3) == 3 and numbytes > 1:
        offset -= 3
        lowerbit += 24
        upperbit += 24
        numbytes += 3
    if numbytes == 3:  # just read an extra byte if we need to
        numbytes = numbytes + 1
    #
    # start with reads needed
    #
    assert baseaccess is not None, "common module required"
    base = baseaccess.getglobalbase()
    if numbytes <= 4:
        regval = base.pcicfg(
            pci_bus,
            pci_device,
            pci_function,
            offset,  # use local offset, not the address attribute
            numbytes,  # use local numbytes and not reginfo.size
            segment=pci_segment,
        )
    else:
        # this part handles 'oversized' config registers
        # some variables we'll need for these big registers
        numdwords = 0
        regval = 0
        tmpoffset = offset
        tmpbytes = numbytes
        # always read in 4 byte chunks to be safe
        while tmpbytes > 0:
            val = base.pcicfg(
                pci_bus,
                pci_device,
                pci_function,
                tmpoffset,
                4,
                segment = pci_segment,
            )
            regval |= long(val) << (32 * numdwords)
            numdwords += 1
            tmpbytes -= 4
            tmpoffset += 4
    # if we were just doing a read, then we're done
    if value is None:
        return getbits(regval, upperbit, lowerbit)
    else:
        # must be a write, set the needed bits:
        regval = setbits(regval, upperbit, lowerbit, value)
        if numbytes <= 4:
            base.pcicfg(
                pci_bus,
                pci_device,
                pci_function,
                offset,
                numbytes,
                regval,
                segment=pci_segment,
            )
        # and begin writing out to pcicfg space
        else:
            numdwords = 0
            tmpoffset = offset
            tmpbytes = numbytes
            while tmpbytes > 0:
                # these are the 4 bytes we care about right now
                val = (regval >> (32 * numdwords)) & 0xFFFFFFFF
                if tmpbytes >= 4:
                    # as long as there's more than 4 use this, write 4
                    base.pcicfg(
                        pci_bus,
                        pci_device,
                        pci_function,
                        tmpoffset,
                        4,
                        val,
                        segment=pci_segment,
                    )
                else:
                    # we're on the last write, how do we handle it?
                    if tmpbytes == 3:
                        # that's not valid, we'll do 4 anyway
                        base.pcicfg(
                            pci_bus,
                            pci_device,
                            pci_function,
                            tmpoffset,
                            4,
                            val,
                            segment=pci_segment,
                        )
                    else:
                        # other values should be o.k.  so just write
                        # the register
                        base.pcicfg(
                            pci_bus,
                            pci_device,
                            pci_function,
                            tmpoffset,
                            tmpbytes,
                            val,
                            segment=pci_segment,
                        )
                numdwords += 1
                tmpbytes -= 4
                tmpoffset += 4
        return


class AccessMmio(AccessRegister):
    """
    Performs an access to a pcie bar that is on an io region.
    """
    requires = [
        'pcibar_name',
        'pcibar_offset',
        ]
    requires_from_target = []

    bar_invalid_values = [
        0xFFFFFFF0,
        0xFFFFFFFFFFFFFFF0,
        0
        ]

    @classmethod
    def read(cls, node):
        """
        Perform a mmio read.

        Args:
            node (NamedNodeValue): The node that we are trying to get
                the data from.
        """
        return cls._access(node, None)

    @classmethod
    def write(cls, node, value):
        """
        Perform a mmio write.

        Args:
            node (NamedNodeValue): The node that we are trying to write
                the data to.
            value (object): The data to write to the node.
        """
        return cls._access(node, value)

    @classmethod
    def _access(cls, node, value=None):
        """
        Perform a mmio write/read.

        Args:
            node (NamedNodeValue): The node that we are trying to write
                the data to/read data from.
            value (object): The data to write to the node, if None a read
                is implied.
        """
        access_info = cls.get_access_info(node)
        pcibar_offset = access_info.get('pcibar_offset', None)
        if pcibar_offset is None:
            raise ValueError("Invalid pcibar_offset value, cannot be None")

        pcibar_size = node.numbits
        if pcibar_size is not None:
            # convert to bytes unit
            pcibar_size //= 8
        else:
            raise ValueError("Invalid numbits value, cannot be None")

        formula_value = node.target_info.get("cached_formula_value", None)
        if not formula_value:
            formula_name = access_info.get('pcibar_name', None)
            if formula_name is None:
                raise ValueError("Invalid pcibar_name value, cannot be None")
            formula_value = node.component.formulas.lookup_value(formula_name)

        if formula_value & ~(0xF) in cls.bar_invalid_values:
            raise ValueError("Invalid BAR value: 0x{0:08X}".format(formula_value))
            #return (1 << (node.numbits)) - 1

        return cls.mmio(formula_value, pcibar_offset, pcibar_size, value)

    @classmethod
    def io(cls, iobase, offset, size, value=None):
        """
        Read from our address and read/write the data for the
        specified upper/lower bits.

        Args:
            iobase (long): The IO base address.
            offset (long): The offset to apply when performing the
                read/write.
            numbits (long): The size, in bytes, of the IO area
                to read/write.
            value (object): The data to write to the node, if None a read
                is implied.
        """
        assert baseaccess is not None, "common module required"
        base = baseaccess.getglobalbase()
        return base.io(iobase + offset, size, value)

    @classmethod
    def mmio(cls, mmbase, offset, size, value=None):
        """
        Read from our address and read/write the data for the
        specified upper/lower bits.

        Args:
            mmbase (long): The memory base address.
            offset (long): The offset to apply when performing the
                read/write.
            numbits (long): The size, in bytes, of the memory area
                to read/write.
            value (object): The data to write to the node, if None a read
                is implied.
        """
        # From PythonSv:
        # I know this function looks complicated as heck, its all b/c
        # the cspecs dont honor the rules of config space, (not crossing
        # 2 byte boundaries, <=4 bytes, etc...
        assert baseaccess is not None, "common module required"
        base = baseaccess.getglobalbase()
        numbytes = size
        # check for already aligned cases and do what is required
        if offset % size == 0 and numbytes <= 8:
            return base.mem(mmbase + offset, numbytes, value)
        # ok, must not be an aligned size, so we will need more code.....
        # we can handle some of these odd cases of crossing a word boundary
        aligned_offset = offset
        if ((offset & 0x3 == 1 and numbytes == 2) or
                ((numbytes == 3) and (offset % 4) == 1)):
            aligned_offset -= 1
            numbytes += 1
        if numbytes == 3:
            # just read an extra byte if we need to
            numbytes = numbytes + 1

        # now begin doing the reads we need
        if numbytes <= 4:
            regval = base.mem(mmbase + aligned_offset, numbytes)
        else:
            # this part handles 'oversized' config registers
            # some variables we'll need for these big registers
            numdwords = 0
            regval = 0
            tmpoffset = aligned_offset
            tmpbytes = numbytes
            # always read in 4 byte chunks to be safe
            while tmpbytes > 0:
                val = base.mem(mmbase + tmpoffset, 4)
                regval |= (long(val) << (32 * numdwords))
                numdwords += 1
                tmpbytes -= 4
                tmpoffset += 4
        if value is None:
            # perform a read
            return getbits(regval, numbytes * 8, (offset - aligned_offset) * 8)
        else:
            # perform a write
            regval = setbits(
                regval, numbytes * 8, (offset - aligned_offset) * 8, value
            )
            if numbytes <= 4:
                base.mem(mmbase + offset, numbytes, regval)
            else:
                numdwords = 0
                tmpoffset = offset
                tmpbytes = numbytes
                while tmpbytes > 0:
                    # these are the 4 bytes we care about right now
                    val = (regval >> (32 * numdwords)) & 0xFFFFFFFF
                    if tmpbytes >= 4:
                        # as long as there's more than 4 use this, write 4
                        base.mem(mmbase + tmpoffset, 4, val)
                    else:
                        # we're on the last write, how do we handle it?
                        if tmpbytes == 3:
                            # that's not valid, we'll do 4 anyway
                            base.mem(mmbase + tmpoffset, 4, val)
                        else:
                            # other values should be o.k. so just write
                            # the register
                            base.mem(mmbase + tmpoffset, tmpbytes, val)
                    numdwords += 1
                    tmpbytes -= 4
                    tmpoffset += 4
            return

    @classmethod
    def is_available(cls, node):
        available = node.definition.info.get('available', True)
        if not available:
            return False
        formula_name = node.definition.info.get('pcibar_name', None)
        available_cache = cls.is_available_cache(node)
        # can't save this to cache becaue it is an error condition in the programming
        if formula_name is None:
            return False
        # we've already checked and it is available
        available = available_cache.get(formula_name, None)
        if available is not None:
            return available
        try:
            # this really only saves ONE extra read on the first register that got the
            # is available call...but...we'll take it....
            formula_value = node.component.formulas.lookup_value(formula_name)
            if formula_value & ~(0xF) in cls.bar_invalid_values:
                return False
            node.target_info['cached_formula_value'] = formula_value
            available_cache[formula_name] = True
        except:
            # something about reading the formula failed here, dont log this register
            # if you change the code above, be sure to turn off the blind/try except during testing..
            available_cache[formula_name] = False
        return available_cache[formula_name]

class AccessMem(AccessMmio):
    """
    Performs an access to a pcie bar that is on a mem region.
    """
    requires = [
        'mem_address',
        'numbits'
        ]
    requires_from_target = []

    @classmethod
    def read(cls, node):
        """
        Perform a mmio read.

        Args:
            node (NamedNodeValue): The node that we are trying to get
                the data from.
        """
        return cls._access(node, None)

    @classmethod
    def write(cls, node, value):
        """
        Perform a mmio write.

        Args:
            node (NamedNodeValue): The node that we are trying to write
                the data to.
            value (object): The data to write to the node.
        """
        return cls._access(node, value)

    @classmethod
    def _access(cls, node, value=None):
        """
        Perform a mem write/read.

        Args:
            node (NamedNodeValue): The node that we are trying to write
                the data to/read data from.
            value (object): The data to write to the node, if None a read
                is implied.
        """
        mem_address = node.definition.info.get('mem_address', None)
        if mem_address is None:
            raise ValueError("Invalid mem_address value, cannot be None")

        mem_size = node.numbits
        if mem_size is not None:
            # convert to bytes unit
            mem_size //= 8
        else:
            raise ValueError("Invalid numbits value, cannot be None")

        return cls.mmio(0, mem_address, mem_size, value)


    @classmethod
    def is_available(cls, node):
        # original is_available, not the same as mmio
        return node.definition.info.get('available', True)


class AccessPcieBar(AccessMmio):
    """
    Performs an access to a pcie bar.
    """
    requires = [
        'pcibar_name',
        'pcibar_offset',
        'numbits'
        ]
    requires_from_target = []

    @classmethod
    def read(cls, node):
        """
        Perform a pcie bar read.

        Args:
            node (NamedNodeValue): The node that we are trying to get
                the data from.
        """
        return cls._access(node, None)

    @classmethod
    def write(cls, node, value):
        """
        Perform a pcie bar write.

        Args:
            node (NamedNodeValue): The node that we are trying to write
                the data to.
            value (object): The data to write to the node.
        """
        return cls._access(node, value)

    @classmethod
    def _access(cls, node, value=None):
        """
        Perform a pcie bar write/read.

        Args:
            node (NamedNodeValue): The node that we are trying to write
                the data to/read data from.
            value (object): The data to write to the node, if None a read
                is implied.
        """
        pcibar_offset = node.definition.info.get('pcibar_offset', None)
        if pcibar_offset is None:
            raise ValueError("Invalid pcibar_offset value, cannot be None")

        pcibar_size = node.numbits
        if pcibar_size is not None:
            # convert to bytes unit
            pcibar_size //= 8
        else:
            raise ValueError("Invalid numbits value, cannot be None")

        formula_value = node.target_info.get("cached_formula_value", None)
        if not formula_value:
            formula_name = node.definition.info.get('pcibar_name', None)
            if formula_name is None:
                raise ValueError("Invalid pcibar_name value, cannot be None")
            formula_value = node.component.formulas.lookup_value(formula_name)

        if formula_value & 1:
            # must be an IO address
            iobase = formula_value & 0xFFFFFFFC
            return cls.io(iobase, pcibar_offset, pcibar_size, value)
        else:
            # must be a memory mapped address
            mmbase = formula_value & 0xFFFFFFFFFFFFFFF0
            # make sure mmbase is legitimate and not all F's, if all F's,
            # then don't go reading from random memory locations...
            if mmbase & ~(0xF) in cls.bar_invalid_values:
                raise ValueError("Invalid BAR value: 0x{0:08X}".format(formula_value))
                #return (1 << pcibar_size) - 1

            return cls.mmio(mmbase, pcibar_offset, pcibar_size, value)
