
# INTEL CONFIDENTIAL
# Copyright 2014 2018 Intel Corporation
#
# The source code  contained or  described herein and  all documents related to
# the source code  ("Material") are owned by Intel Corporation or its suppliers
# or licensors.  Title to the  Material  remains with  Intel Corporation or its
# suppliers  and licensors. The Material contains trade secrets and proprietary
# and  confidential  information  of  Intel  or  its  suppliers  and  licensors.
# The Material  is protected  by worldwide  copyright and trade secret laws and
# treaty provisions.  No part of the Material  may  be used, copied, reproduced,
# modified, published, uploaded, posted, transmitted, distributed, or disclosed
# in any way without Intel's prior express written permission. No license under
# any  patent,  copyright, trade secret or other intellectual property right is
# granted  to  or conferred upon you by disclosure or delivery of the Materials,
# either expressly, by implication, inducement, estoppel or otherwise.
# Any license under such intellectual property rights must be express and
# approved by Intel in writing.



"""
Access classes used by ijtag components
"""
import ipccli
from .register import AccessRegister

class IJtagAccess(AccessRegister):
    """
    The IJtag Access relies on the component definition having a "sib_tree". A sib_tree is a
    component hierarchy used just for reflecting the underlying sib hierarchy. It is not
    used in the traditional component sense and is just needed for easily walking/storing data

    In the future this may be redone so that the sib_tree is a more simple dictionary
    """
    requires = [
        'sib_name', # immediate parent sib that needs to be enabled (along with this sib) in order to view fields
        # an 'optional' one we want to support here would be 'mux_pos' to be 'msb' or 'lsb' of its children
        'chain_ir', # need to make sure this counts as 'provided' if the component's target_info has it
        # TBD: whether sib_tree belongs in this list
    ]
    requires_from_target = [
        'device'
    ]

    @classmethod
    def read(cls, node):
        return cls.access(node, value=None)

    @classmethod
    def write(cls, node, value):
        return cls.access(node, value)

    @classmethod
    def read_write(cls, node, mask, value):
        return cls.access(node, value, mask)

    @classmethod
    def access(cls, node, value, write_mask=None):
        ipc = ipccli.baseaccess()
        # create list of parent objects that starts with highest one in the hiearchy
        sib_tree = node.component.definition.info['sib_tree']
        device = cls.lookup_info(node, "device")
        chain_ir = cls.lookup_info(node, "chain_ir")
        mux_pos = 'lsb' # could grab this from object
        sibs = []
        psib_name = node.info['sib_name']
        assert 'sibs' in sib_tree, "Sib Tree Dictionary missing 'sibs' key"
        psib = sib_tree['sibs'].get(psib_name, None)
        assert psib is not None, "Sib Tree missing the required sib: %s for register: %s"%(sib_name, node.name)
        sibs.append(psib)
        while True:
            psib_name = psib.get('parent_sib', None)
            if psib_name is None:
                break
            # KEY here is we are getting definition objects for performance vs. creating value objects
            # that we will never really get values for
            psib = sib_tree['sibs'].get(psib_name, None)
            assert psib is not None, "%s 's parent sib (%s) is missing from sib tree"%(psib['name'], psib['parent_sib'])
            sibs.append(psib)
        # now reverse the list to be in order from top parent first
        sibs.reverse()
        # bit mask to write in to enable sibs
        serialize_mask = 0
        deserialize_mask = 0
        # number of bits at top level of a fully collapsed chain
        numbits = sib_tree['numbits']
        prev_numbits = 0
        bit_offset = 0
        for pnode in sibs:
            # bitpos of this sib to modify
            bit_offset = bit_offset + pnode['bit_offset']
            # set serialize_mask and advance bit offset
            if mux_pos == 'lsb':
                serialize_mask |= (1<<bit_offset)
                deserialize_mask = serialize_mask # for LSB these are the same
                bit_offset += 1
            elif mux_pos == 'msb':
                serialize_mask |= (1 << bit_offset)
                deserialize_mask |= (1 << pnode['numbits'])
                # next offset starts at size of new register added + 1 bit was added for sib
                bit_offset += (pnode['numbits'] + 1)
            else:
                raise AssertionError("unexpected mux pos: %s"%mux_pos)
            #irdrscanrmw(device, instruction, writeData, maskData, bitCount=None, returnData=True):
            ipc.irdrscanrmw(device, chain_ir, serialize_mask, serialize_mask, numbits, returnData=False) # dont need data back
            # save off current number of bits, so that the next sib knows where to offset from
            prev_numbits = numbits
            # increment numbits
            numbits += pnode['numbits']

        startbit = bit_offset # bitoffset has already been incremented
        endbit = startbit + pnode['numbits'] - 1

        # same as below, but reads only, then expects clear to happen later
        # if value is None:
        #     #### DO READ
        #     #### THIS could be optimzied in the read to go ahead and do the deserialization
        #     data = ipc.irdrscanreplace(device, chain_ir, numbits)
        # else:
        #     ### DO WRITE
        #     ipc.irdrscanrmw(device, chain_ir, value<<startbit, mask<<startbit, numbits)

        ### NOW put the chain back simply write 0's back where we wrote 1's
        if value is None:
            # write 0 to mask bits AND return all the data
            data = ipc.irdrscanrmw(device, chain_ir, 0, deserialize_mask, numbits)
        else:
            if write_mask is not None:
                all_mask = (write_mask<<startbit) | deserialize_mask
            else:
                mask_numbits = endbit - startbit + 1
                local_mask = ((1 << (mask_numbits)) - 1)
                all_mask = (local_mask<<startbit) | deserialize_mask
            # in case value is a BitData/BundleMap, we need to make a bitdata of the correct size
            write_value = ipc.BitData(numbits, 0)
            write_value[startbit:startbit+numbits-1] = value
            ipc.irdrscanrmw(device, chain_ir, write_value, all_mask, numbits)

        ### all done, return our data
        #print("Startbit: %d"%startbit)
        #print("Endbit: %d"%endbit)
        if value is None:
            return data[startbit:endbit]
        else:
            # was a write
            return None


class IJtagAccessWriteOnly(AccessRegister):
    """
    The IJtag Access relies on the component definition having a "sib_tree". A sib_tree is a
    component hierarchy used just for reflecting the underlying sib hierarchy. It is not
    used in the traditional component sense and is just needed for easily walking/storing data

    In the future this may be redone so that the sib_tree is a more simple dictionary
    
    This access is bundle-friendly but only provides a write capability
    """
    requires = [
        'sib_name', # immediate parent sib that needs to be enabled (along with this sib) in order to view fields
        # an 'optional' one we want to support here would be 'mux_pos' to be 'msb' or 'lsb' of its children
        'chain_ir', # need to make sure this counts as 'provided' if the component's target_info has it
        # TBD: whether sib_tree belongs in this list
    ]
    requires_from_target = [
        'device'
    ]

    @classmethod
    def read(cls, node):
        raise NotImplementedError("Read functionality is not supported in IJtagAccessWriteOnly class")

    @classmethod
    def write(cls, node, value):
        return cls.access(node, value)

    @classmethod
    def read_write(cls, node, mask, value):
        return cls.access(node, value, mask)

    @classmethod
    def access(cls, node, value, write_mask=None):
        if value is None:
            raise NotImplementedError("Read functionality is not supported in IJtagAccessWriteOnly class")
        ipc = ipccli.baseaccess()
        # create list of parent objects that starts with highest one in the hiearchy
        sib_tree = node.component.definition.info['sib_tree']
        device = cls.lookup_info(node, "device")
        chain_ir = cls.lookup_info(node, "chain_ir")
        mux_pos = 'lsb' # could grab this from object
        sibs = []
        psib_name = node.info['sib_name']
        assert 'sibs' in sib_tree, "Sib Tree Dictionary missing 'sibs' key"
        psib = sib_tree['sibs'].get(psib_name, None)
        assert psib is not None, "Sib Tree missing the required sib: %s for register: %s"%(sib_name, node.name)
        sibs.append(psib)
        while True:
            psib_name = psib.get('parent_sib', None)
            if psib_name is None:
                break
            # KEY here is we are getting definition objects for performance vs. creating value objects
            # that we will never really get values for
            psib = sib_tree['sibs'].get(psib_name, None)
            assert psib is not None, "%s 's parent sib (%s) is missing from sib tree"%(psib['name'], psib['parent_sib'])
            sibs.append(psib)
        # now reverse the list to be in order from top parent first
        sibs.reverse()
        # bit mask to write in to enable sibs
        serialize_mask = 0
        deserialize_mask = 0
        # number of bits at top level of a fully collapsed chain
        numbits = sib_tree['numbits']
        prev_numbits = 0
        bit_offset = 0
        for pnode in sibs:
            # bitpos of this sib to modify
            bit_offset = bit_offset + pnode['bit_offset']
            # set serialize_mask and advance bit offset
            if mux_pos == 'lsb':
                serialize_mask |= (1<<bit_offset)
                deserialize_mask = serialize_mask # for LSB these are the same
                bit_offset += 1
            elif mux_pos == 'msb':
                serialize_mask |= (1 << bit_offset)
                deserialize_mask |= (1 << pnode['numbits'])
                # next offset starts at size of new register added + 1 bit was added for sib
                bit_offset += (pnode['numbits'] + 1)
            else:
                raise AssertionError("unexpected mux pos: %s"%mux_pos)
            #irdrscanrmw(device, instruction, writeData, maskData, bitCount=None, returnData=True):
            #ipc.irdrscanrmw(device, chain_ir, serialize_mask, serialize_mask, numbits, returnData=False) # dont need data back
            ipc.irdrscan(device, chain_ir, numbits, serialize_mask)
            # save off current number of bits, so that the next sib knows where to offset from
            prev_numbits = numbits
            # increment numbits
            numbits += pnode['numbits']

        startbit = bit_offset # bitoffset has already been incremented
        endbit = startbit + pnode['numbits'] - 1

        # same as below, but reads only, then expects clear to happen later
        # if value is None:
        #     #### DO READ
        #     #### THIS could be optimzied in the read to go ahead and do the deserialization
        #     data = ipc.irdrscanreplace(device, chain_ir, numbits)
        # else:
        #     ### DO WRITE
        #     ipc.irdrscanrmw(device, chain_ir, value<<startbit, mask<<startbit, numbits)


        if write_mask is not None:
            all_mask = (write_mask<<startbit) | deserialize_mask
        else:
            mask_numbits = endbit - startbit + 1
            local_mask = ((1 << (mask_numbits)) - 1)
            all_mask = (local_mask<<startbit) | deserialize_mask
        # in case value is a BitData/BundleMap, we need to make a bitdata of the correct size
        write_value = ipc.BitData(numbits, 0)
        write_value[startbit:startbit+numbits-1] = ipc.BitData(numbits, value)
        #ipc.irdrscanrmw(device, chain_ir, write_value, all_mask, numbits)
        ipc.irdrscan(device, chain_ir, numbits, write_value)
        ### all done, return our data
        #print("Startbit: %d"%startbit)
        #print("Endbit: %d"%endbit)
        # was a write
        return None


            
            
def create_sib_tree(numbits):
    """
    Create a sib tree object/lass. While this does return
    a dictionary do not manually add/modify it
    """
    tree = {}
    tree['numbits'] = numbits
    tree['sibs'] = {}
    return tree

def add_sib_info(sib_tree, sib_name, bit_offset, numbits, parent_sib, **kwargs):
    """
    Helper function for creating sib dictionary
    so that it is clear what is needed for the sib tree to work
    """
    # needed in creating the collateral were sometimes we come back and "fix" the tree
    ignore_errors = kwargs.pop("ignore_errors", False)
    if parent_sib is not None and not ignore_errors:
        assert parent_sib in sib_tree['sibs'],"'%s' missing from sib_tree"%parent_sib
    sib_tree['sibs'][sib_name] = dict(name=sib_name,
                            bit_offset=bit_offset,
                            numbits=numbits,
                            parent_sib=parent_sib
                            )
