
# INTEL CONFIDENTIAL
# Copyright 2014 2018 Intel Corporation
#
# The source code  contained or  described herein and  all documents related to
# the source code  ("Material") are owned by Intel Corporation or its suppliers
# or licensors.  Title to the  Material  remains with  Intel Corporation or its
# suppliers  and licensors. The Material contains trade secrets and proprietary
# and  confidential  information  of  Intel  or  its  suppliers  and  licensors.
# The Material  is protected  by worldwide  copyright and trade secret laws and
# treaty provisions.  No part of the Material  may  be used, copied, reproduced,
# modified, published, uploaded, posted, transmitted, distributed, or disclosed
# in any way without Intel's prior express written permission. No license under
# any  patent,  copyright, trade secret or other intellectual property right is
# granted  to  or conferred upon you by disclosure or delivery of the Materials,
# either expressly, by implication, inducement, estoppel or otherwise.
# Any license under such intellectual property rights must be express and
# approved by Intel in writing.




from __future__ import absolute_import
from __future__ import division
from __future__ import print_function

from ..access import NodeAccess
from ..errors import OfflineMissingNode
from ..nodes import NamedNodeValue
from ipccli import BitData
from namednodes import settings
from namednodes import access as _access


__all__ = [
         "AccessParentRead",
         "AccessConstant",
         "AccessConstantReadOnly",
         "AccessStoredValues",
         "AccessRegisterStoredValues",
        ]


class _StoredValuesFormat:
    fullpath = 1  # no longer used
    path_tuple = 2  # current default
    nested_dict = 3  # alpha/prototyping


# this is named similar to node access to somewhat signify its importance
# this should be the primary access class that 'typical' register accesses
# should inherit from
class AccessParentRead(NodeAccess):
    """
    used by any node that does not set its value directly and instead
    relies on the parent to sets it value
    """
    #:
    requires = []

    @classmethod
    def read(cls,node):
        node.parent.read()
        return 0

class AccessConstant(NodeAccess):
    """When using nodes to simply display a constant value

    Note:
        The node value is taken from the definition, not the value
        object

    """
    #: definition must have
    requires = ['constant']

    @classmethod
    def read(cls, node):
        """Read from some stored constant for the register"""
        if 'constant' in node.target_info:
            constant = node.target_info['constant']
        elif 'constant' in node.definition.info:
            constant = node.definition.info['constant']  
        else:
            raise RuntimeError("Database must have constant attribute")
        return constant

    @classmethod
    def write(cls, node, value):
        """Write from some stored constant for the register

        Note:

            This writes to the definition so all RegisterValues
            referencing this definition will get the new value

        """
        if 'constant' in node.target_info:
            node.target_info['constant'] = value
        else:
            node.definition.info['constant'] = value
            
class AccessConstantReadOnly(AccessConstant):
    @classmethod
    def write(cls,node,value):
        """This method is a stub, since the access is read-only
        """
        raise RuntimeError("This is a read-only constant node, you cannot update the value")

class AccessOffline(NodeAccess):
    """
    Base class for offline accesses. Overrides get_access_info to display
    meaningful data.
    """
    _cache_name = "stored_values"
    _cache_format = "stored_values_format"

    @classmethod
    def get_access_info(cls, node, **kwargs):
        """
        Get the properties in a node required to perform an access.

        Args:
            node (NamedNodeValue): The node to get the properties from.

        Returns:
            A dictionary with the node's access properties. If a key's value is
            missing then it will be given the value of *LookupError*
        """
        access_class_name = node.access_class_name
        # see if "previous access mapping" exists, and if so, use that
        # if not, then use the master one...
        previous_mapping = getattr(node.component, "_previous_access_mapping", None)
        if previous_mapping is None:
            previous_mapping = _access.NodeAccessManager
        access_class = previous_mapping[access_class_name]
        # not sure of a better way around this right now, but we have to
        # special case the default access and look up the underlying access that we
        # will end up mapping to, otherwise we end up in recursion
        if access_class_name == "DefaultNodeAccess":
            actual_class_name = access_class.priority_access_class_name(node)
            access_class = previous_mapping[actual_class_name]

        return access_class.get_access_info(node)


##### add the read to offline ?
##### add an AccessStoredValues no default

class AccessStoredValues(AccessOffline):
    """
    Used to keep a constant value per path to the node. This is
    useful for example code or for snapshotting all the current
    values.

    Note::
       The node value is taken from the definition, but uses
       the parent components path to determine the value.

    Note::
        If the node does not have a stored value, 'default' will
        be attempted, if that does not exist, -1 is returned

    Example::
        >>> component.node.write( value )
        >>> # results in:
        >>> component.node.definition.info[ component.path ] = value


    """

    @classmethod
    def _convert_tuple_to_dict(cls, comp):
        """convert the tuple caching format to the dict format"""
        # original format is (component, register): value
        values_cache = comp.target_info.get(cls._cache_name)
        new_cache = {}
        for (cpath, npath), value in values_cache.items():
            if cpath not in new_cache:
                new_cache[cpath] = {}
            new_cache[cpath][npath] = dict(value=value)
        comp.target_info[cls._cache_name] = new_cache

    @classmethod
    def _init_cache(cls, component, default_data=None):
        default_data = default_data or {}
        constant_dict = component.origin.target_info.setdefault(cls._cache_name, default_data)
        component.origin.target_info[cls._cache_format] = _StoredValuesFormat.nested_dict
        return constant_dict

    @classmethod
    def _read_cache(cls, node):
        constant_dict = node.component.origin.target_info.get(cls._cache_name, None)
        if constant_dict is None:
            constant_dict = cls._init_cache(node.component)
        node_path = node.target_info.get("aliases_original_path", None)
        component_path = node.component.path
        if node_path is None:
            node_path = node.nodepath
        else:
            # turn to relative path...
            node_path = node_path.replace(component_path+".", "")

        full_node_path = node.path
        # a few versions over the generations
        value = constant_dict.get(full_node_path, KeyError)
        if value is not KeyError:
            return value
        value = constant_dict.get((component_path, node_path), KeyError)
        if value is not KeyError:
            return value
        # assume latest format
        value = constant_dict.get(component_path, {}).get(node_path, {}).get('value', -1)
        return value

    @classmethod
    def _write_cache(cls, node, value, include_children=False):
        # make sure it exists and create it if it does not
        constant_dict = node.component.origin.target_info.get(cls._cache_name, None)
        if constant_dict is None:
            constant_dict = cls._init_cache(node.component)

        # Be aware of alias nodes...i admit that I don't know if this is adequately tested yet
        node_path = node.target_info.get("aliases_original_path", None)
        if node_path is None:
            node_path = node.nodepath
        # NOTE: we actually don't check existing format, we just use this, that seems to be ok
        # since format 2 and format 3 are discoverable during the read....
        constant_dict.setdefault(node.component.path, {})[node_path] = dict(value=value)
        # note: could trigger a read if used improperly by the caller
        if include_children:
            comp_dict = constant_dict.setdefault(node.component.path, {})
            for child in node.walk_nodes():
                comp_dict[child.nodepath] = dict(value=child.get_value())

    @classmethod
    def read(cls, node):
        """Read from some stored constant for the register"""
        # get dictionary with all the values
        value = cls._read_cache(node)
        if value == -1:
            default = node.definition.info.get("default", -1)
            return default
        else:
            # no clue why this would happen, but make sure we always return
            # a number
            if value is None:
                return -1
            else:
                return value

    @classmethod
    def write(cls, node, value):
        """Write from some stored constant for the register, stored using
        the path we used to get to the register
        """
        # if we get a node for the value, we have to convert that to an actual value
        # otherwise we could get in to some circular references
        if isinstance(value, NamedNodeValue):
            value = value.get_value()

        # bounds check if we can
        numbits = node.target_info.get('numbits', None)
        if numbits is None:
            numbits = node.definition.info.get('numbits', None)
        if numbits is not None:
            # negative can have special meeting, so don't bounds check it
            if value & ~((1<<numbits)-1) and value>0:
                raise ValueError("register does not support that many bits to be written to")

        cls._write_cache(node, value)

# this does NOT inherit from register because there's no need
# to have read-modify support for w1c bits when we are in an offline mode
class AccessRegisterStoredValues(AccessStoredValues):
    """
    Used to keep a constant value per path to the register. This is
    useful for example code or for snapshotting all the current
    values.

    This is different than AccessStoredValue since it tries to look
    up the defaults of children nodes (assuming they are fields) for the
    overall value of the register


    Note::
        If the node does not have a stored value, it will create a default
        from its children nodes assuming they are fields

    """
    @classmethod
    def read(cls, node):
        """Read from some stored constant for the register"""
        value = AccessStoredValues.read(node)
        # was not able to get a value back
        if value != -1:
            return value
        # compute a better default
        return node.getdefault()

