
# INTEL CONFIDENTIAL
# Copyright 2014 2018 Intel Corporation
#
# The source code  contained or  described herein and  all documents related to
# the source code  ("Material") are owned by Intel Corporation or its suppliers
# or licensors.  Title to the  Material  remains with  Intel Corporation or its
# suppliers  and licensors. The Material contains trade secrets and proprietary
# and  confidential  information  of  Intel  or  its  suppliers  and  licensors.
# The Material  is protected  by worldwide  copyright and trade secret laws and
# treaty provisions.  No part of the Material  may  be used, copied, reproduced,
# modified, published, uploaded, posted, transmitted, distributed, or disclosed
# in any way without Intel's prior express written permission. No license under
# any  patent,  copyright, trade secret or other intellectual property right is
# granted  to  or conferred upon you by disclosure or delivery of the Materials,
# either expressly, by implication, inducement, estoppel or otherwise.
# Any license under such intellectual property rights must be express and
# approved by Intel in writing.





"""
Module containing all core accesses.
"""

from __future__ import absolute_import
from __future__ import division
from __future__ import print_function

from .register import AccessRegister
from ..preconditions import GpcHaltRequired, IpcHaltRequired, LookupInfoCache
from ..utils.bitmanip import bits
try:
    from svtools.common import baseaccess
except ImportError:
    baseaccess = None

__all__ = [
    "AccessMsr",
    "AccessMsrWithGpcHalt",
    ]

class AccessMsr(AccessRegister):
    """
    Performs an MSR read/write.

    MSR stands for Model Specific Registers.

    MSR definition from the Intel Acronyms dictionary:

    Model-specific registers are control registers provided by processor
    implementations to provide system software with features that are
    provided on specific processor implementations, but not others.
    MSRs are used for performance monitoring, debugging, testability and
    program execution tracing, but also to enable and disable certain
    model-specific features of the processor implementation.
    """
    requires = ["msr_offset"]
    requires_from_target = [
        # order is important, and msr_did should take precedence
        ("msr_did", "did"),  # this means that we can have either of these
        ]

    @classmethod
    def read(cls, node):
        """
        Perform an MSR read.

        Args:
            node (NamedNodeValue): The node that we are trying to get
                the data from.
        """
        return cls._access(node)

    @classmethod
    def write(cls, node, value):
        """
        Perform an MSR write.

        Args:
            node (NamedNodeValue): The node that we are trying to write
                the data to.
            value (object): The data to write to the node.
        """
        return cls._access(node, value)

    @classmethod
    def is_access_available(cls, component):
        """
        Check if it is currently *safe* to read from a component
        using MSR.

        Args:
            component (NamedComponent): Component which we are trying
                to check availability on.
        """
        if not super(AccessMsr, cls).is_access_available(component):
            return False

        msr_did = cls.lookup_info(None, ("msr_did", "did",),component=component)
        # can not dump MSR's when we are running
        assert baseaccess is not None, "common module required"
        base = baseaccess.getglobalbase()
        # if isrunning available, base it on that
        isrunning = getattr(base, "isrunning", None)
        if isrunning:
            return not isrunning(msr_did)
        else:
            # without is running assume that the access is available
            return True

    @classmethod
    def _access(cls, node, value=None):
        """
        Perform an MSR write/read.

        Args:
            node (NamedNodeValue): The node that we are trying to write
                the data to/read data from.
            value (object): The data to write to the node, if None a read
                is implied.
        """
        msr_offset = node.definition.info.get("msr_offset", None)
        if msr_offset is None:
            raise ValueError("Invalid msr_offset value, cannot be None")

        msr_did = cls.lookup_info(node, ("msr_did", "did",))

        assert baseaccess is not None, "common module required"
        base = baseaccess.getglobalbase()
        return base.msr(msr_offset, value, msr_did)


class AccessMsrWithGpcHalt(AccessMsr):
    """
    Similar to AccessMsr, but will execute a Halt for GPC cores before doing
    the access. This works across all baseaccesses
    """
    preconditions = [GpcHaltRequired, LookupInfoCache]

    @classmethod
    def is_access_available(cls, component):
        # skip MSR parent access since it requires target to be halted
        return AccessRegister.is_access_available(component)

class AccessMsrWithIpcHalt(AccessMsr):
    """
    Similar to AccessMsr, but will execute a Halt using "did" or "msr_did"
    found in the component.

    Note:
        This works only works on OpenIPC
    """
    preconditions = [ IpcHaltRequired ]

    @classmethod
    def is_access_available(cls, component):
        # skip MSR parent access since it requires target to be halted
        return AccessRegister.is_access_available(component)


