
# INTEL CONFIDENTIAL
# Copyright 2014 2018 Intel Corporation
#
# The source code  contained or  described herein and  all documents related to
# the source code  ("Material") are owned by Intel Corporation or its suppliers
# or licensors.  Title to the  Material  remains with  Intel Corporation or its
# suppliers  and licensors. The Material contains trade secrets and proprietary
# and  confidential  information  of  Intel  or  its  suppliers  and  licensors.
# The Material  is protected  by worldwide  copyright and trade secret laws and
# treaty provisions.  No part of the Material  may  be used, copied, reproduced,
# modified, published, uploaded, posted, transmitted, distributed, or disclosed
# in any way without Intel's prior express written permission. No license under
# any  patent,  copyright, trade secret or other intellectual property right is
# granted  to  or conferred upon you by disclosure or delivery of the Materials,
# either expressly, by implication, inducement, estoppel or otherwise.
# Any license under such intellectual property rights must be express and
# approved by Intel in writing.

"""
Access classes used by cabist components
"""
from math import ceil
import re
#import ipccli
from ipccli import BitData
from .register import AccessRegister
from namednodes.comp import NamedComponent
from .tap import AccessTapNoReplaceNoWrData

_debug = False

class CABIST_OPCODES:
    OP_NOP     = 0x0
    OP_WRITE   = 0x1
    OP_READ    = 0x2

class AccessCabistIndirect(AccessRegister):
    """
    The AccessCabistIndirect relies on the component definition having the following parameters               
        - config_path       - Path to CABIST CONFIG register
        - write_data_path   - Path to CABIST WRITE_DATA register or field
        - return_data_path  - Path to CABIST RETURN_DATA register or field
        
    Node itself should have the following parameters:
        - target_module    - Address of the target module
        - indirect_address - Inidrect address of the register
        - numbits          - Number of bits in a register
        
    """
    requires = [
        "target_module",      # Address of the target module
        "indirect_address",   # Inidrect address of the register
        "numbits",            # Number of bits in a register
        "config_path",        # Path to CABIST CONFIG register
        "write_data_path",    # Path to CABIST WRITE_DATA register or field
        "return_data_path",   # Path to CABIST RETURN_DATA register or field    
    ]
    
    bundle_friendly = True # Will make tap register reads bundle friendly
    
    
    @classmethod
    def read(cls, node):
        return cls.access(node, value=None)

    @classmethod
    def write(cls, node, value):
        return cls.access(node, value)  
    @classmethod
    def _get_config_value(cls, node, opcode):
        target_module            = cls.lookup_info(node,'target_module')
        indirect_address         = cls.lookup_info(node,'indirect_address')
        value = BitData(32,0)
        value[7:0]   = indirect_address
        value[10:8]  = target_module
        value[12:11] = opcode
        return value
        
        
    @classmethod
    def access(cls, node, value, mask=None):
        numbits                  = cls.lookup_info(node,'numbits')
        config_path              = cls.lookup_info(node,'config_path')
        write_data_path          = cls.lookup_info(node,'write_data_path')
        return_data_path         = cls.lookup_info(node,'return_data_path')
        
        if value is None:
            #read
            #TODO add device locker
            node.component.origin.get_by_path(config_path).write(cls._get_config_value(node,CABIST_OPCODES.OP_READ))
            robj = node.component.origin.get_by_path(return_data_path)
            if cls.bundle_friendly:
                robj.access_class = AccessTapNoReplaceNoWrData
            if _debug: print("{} -> 0x{:x}".format(node.path, robj.get_value()))
            return robj.get_value()
            #return BitData(32,0)
        else:            
            node.component.origin.get_by_path(config_path).write(cls._get_config_value(node,CABIST_OPCODES.OP_WRITE))
            node.component.origin.get_by_path(write_data_path).write(value & BitData(numbits, -1))
            if _debug: print("{} = 0x{:x}".format(node.path, value))
            return None
            
