
# INTEL CONFIDENTIAL
# Copyright 2014 2019 Intel Corporation
#
# The source code  contained or  described herein and  all documents related to
# the source code  ("Material") are owned by Intel Corporation or its suppliers
# or licensors.  Title to the  Material  remains with  Intel Corporation or its
# suppliers  and licensors. The Material contains trade secrets and proprietary
# and  confidential  information  of  Intel  or  its  suppliers  and  licensors.
# The Material  is protected  by worldwide  copyright and trade secret laws and
# treaty provisions.  No part of the Material  may  be used, copied, reproduced,
# modified, published, uploaded, posted, transmitted, distributed, or disclosed
# in any way without Intel's prior express written permission. No license under
# any  patent,  copyright, trade secret or other intellectual property right is
# granted  to  or conferred upon you by disclosure or delivery of the Materials,
# either expressly, by implication, inducement, estoppel or otherwise.
# Any license under such intellectual property rights must be express and
# approved by Intel in writing.



"""
The Access classes are what provides the Named Nodes with actual values.
Primarily there are read and write acccess at a minimum, but there may
be others.

Each NamedNodes is assigned multiples access via a string that describes that
access. Each of those accesses must be registered with the NodeAccessManager
during the plugin imports or during the discovery phase. When a NamedNode needs
to do the access, it will get the NodeAccess class from the Manager before
requesting the read/write method for that access.


.. autoclass:: NodeAccessManager
    :members:

.. autoclass:: NodeAccess
    :members:

"""
from __future__ import absolute_import
from __future__ import division
from __future__ import print_function

import functools
import numbers
import six
import copy


from collections import OrderedDict
from .logging import getLogger
from .utils._py2to3 import *
from .utils.ordereddict import odict
from .errors import OsxError

_LOG = getLogger("namednodes")
# key to expect in components for address maps
_OSX_MAP_KEY = "osx_address_maps"


class AvailabilityOptions:
    NotAvailable = 0
    Available = 1
    ChildrenOnly = 2
    NotAvailableTooManyFailures = 3

# Global tracker of ALL known accesses
NodeAccessManager = odict()

class _PreconditionedAccess(object):
    """
    For managing the preconditions of an access class

    Runs the required preconditions on enter and post conditions on exit,
    but throws the proper exceptions on exit
    """
    def __init__(self, node, access_class):
        """
        Args:
            node (obj) : node that we are running this for (needed by preconditions)
            access_class (obj) : class that we should check for preconditions and run
        """
        # These are purposefully two different parameters in case we want to run an access class
        # different than the current/default one
        self.node = node
        self.access_class = access_class
        self.preconditions = access_class.preconditions

    def __enter__(self):
        if self.preconditions is not None:
            for precondition in self.preconditions:
                precondition.run_precondition(self.node)

    def __exit__(self, exc_type, exc_val, exc_tb):
        # attempt to run post condtions regardless, but be aware that they may fail
        if self.preconditions is not None:
            precondition_failed = False
            for precondition in self.preconditions:
                try:
                    precondition.run_postcondition(self.node)
                except Exception as e2:
                    precondition_failed = True
                    _LOG.exception("Postcondition '%s' failed" % str(precondition))
                    if exc_type is not None:
                        _LOG.error("But before postcondition, register access failed")
                    # dont throw the exception so that other preconditions are sure to get unlocked
            if precondition_failed and exc_type is None:
                raise RuntimeError("One or more preconditions failed")
        if exc_type:
            six.reraise(exc_type, exc_val, exc_tb)


def _precondition_wrap(originalf):
    # wrap access function such that preconditions are always run
    if isinstance(originalf, classmethod):
        originalf = originalf.__func__

    @functools.wraps(originalf)
    def mynew_f(*args, **kwargs):
        # ALL access functions require node as first paramter
        # ALL access functions are passing have access class being pass in since
        # they are class methods
        window = _PreconditionedAccess(args[1], args[0])
        with window:
            return originalf(*args, **kwargs)

    return mynew_f


class NodeAccessTracker(type):
    """
    Developers should not inherit from this class, it is a meta-class and its
    purpose is to track all the plugins that have been created.
    """
    # dictionary of 'name':class for all plugins that have been created
    plugins = odict()
    def __new__(mcs, name, bases, atts):
        newcls = super(NodeAccessTracker, mcs).__new__(mcs, name, bases, atts)
        # no need to save a reference to this one since it is the base class
        # for our plugins
        if name != "NodeAccess":
            if name in NodeAccessManager:
                # if module is same as the old, no error, we are just reloading...
                if NodeAccessManager[name].__module__ != newcls.__module__:
                    _LOG.caller("NodeAccessManager")
                    _LOG.warning("{} was registered twice, once in {} and once in {}, please rename one of these classes".format(
                            name, NodeAccessManager[name].__module__, newcls.__module__,
                    ))
            # no need to store or create preconditions for wrapped classes
            if 'original_class' in atts and newcls.original_class is not None:
                return newcls
            NodeAccessManager[name] = newcls
        # make sure the accesses on classes are wrapped to always run the preconditions
        for access_method in newcls.requires_preconditions:
            # we dont need to wrap each level or class hiearchy, just new ones
            # that is why we check the __dict__ (or could have checks 'atts') instead of using getattr
            orig_func = newcls.__dict__.get(access_method, None)
            if orig_func is None:
                continue
            # save off the original funciton on the class in case we need it for debug
            setattr(newcls, "_"+access_method + "_raw", orig_func)
            setattr(newcls, access_method, classmethod(_precondition_wrap(orig_func)))

        return newcls

    def __getattr__(cls, item):
        if cls.original_class is not None:
            return getattr(cls.original_class, item)
        else:
            raise AttributeError("{} has not attribute {}".format(cls, item))

    def __setattr__(cls, item, value):
        if cls.original_class is not None:
            return setattr(cls.original_class, item, value)
        else:
            return type.__setattr__(cls, item, value)

@with_metaclass(NodeAccessTracker)
class NodeAccess(object):
    """
    Base class for accesses.
    """
    # List of information that the node definition is expected to have in
    # order for this class to successfully perform an access
    requires = []
    # List of information that the node's, or its parent, is expected to
    # have in order for this class to successfully perform an access when
    # in a system
    requires_from_target = []
    # Default values for information in requires and requires_from_target. To be
    # used if the node and node definition don't explicitly define a value.
    defaults = {}
    # list of additional methods that this Acccess Class makes available to
    # the node that may not be implemented in this base class
    # inheriting methods should use something like::
    #
    #      methods = [ ] + NodeAccess.methods
    #
    methods = [
        'read',
        'write',
        'read_write',
        'store',
        'flush',
        'lookup_info',
        'show_access_info',
        'get_access_info',
        'update_value',
        'is_available',
        'is_available_cache',
        'is_access_available_by_node'
        ]
    # list of methods that require preconitions to be run first
    requires_preconditions = [
        'read',
        'write',
        'read_write',
        'flush',
    ]
    # inheriting classes should use the syntax of something like this
    # so that they get obt
    # methods = [ ] + NodeAccess.methods

    # for cases where the access requires a pre-condition
    # this should be filled in as a list by classes that have one or more pre-conditions
    preconditions = None

    # used for gathering telemetry gathering
    _logged_read = False
    _logged_write = False
    _logged_read_write = False
    _logged_store = False
    _logged_flush = False

    # special case for knowing that a class will be hijacking/proxying
    # on top of anoher class
    original_class = None

    #: osx_map_offset_key is used for OSX collateral where we will add the map_offset to the value stored in this key before
    #: returning it for non accesses
    osx_map_offset_key = "map_offset"
    access_offset_key = None

    def __init__(self):
        raise Exception("Not intended for object creation")

    @classmethod
    def read(cls, node):
        """
        Must be implemented by inheriting classes to provide useful data when
        read occurrs.

        Args:
            node - NodeValue object that we are trying to return the data for

        Return:
            must return a value

        It is expected that the access functions will primarily use
        *node.definition.info* which contains the meta-data from the design
        XML, and the *node.component* which is a reference to the specific
        component that we are trying to do the access to.
        """
        raise Exception("Must be implemented")

    @classmethod
    def update_value(cls, node):
        """Returns a value (potentially cached) that represents the node

        Args:
            node - NodeValue object that we are trying to return the data for

        Return:
            must return a value

        This function is expected to return a value that represents data for
        the node. The primary difference in this vs. the read, is that this
        function *could* return a cached value, where-as if someone
        specifically calls read, it is expected to perform some access to make
        sure the latest value is being returned.

        Implementing this function is optional, and by default it will just
        return the value provided by the read function.
        """
        # default get_value is to just use read, but there is some flexibility
        # here in to what 'read' can be setup to do vs. get_value.
        # get_value could be used to return the cached
        # read could be used to force an update -- if so, the read will need to
        # manually set node._value
        # ... the key here is that the user called "read" where-as most of the
        # named_node code calls get_value
        data = node.read()
        if data is None:
            raise Exception("ImplementationError for Access '{0}', read must return a value".format(str(cls.__name__)))
        return data._value

    @classmethod
    def write(cls, node, value):
        """
        Must be implemented by inheriting classes to take a value and do a
        write to the silicon.

        Args:
            node (obj): NodeValue object that we are trying to return the data for
            value (number): value to write to the node
        """
        raise Exception("Must be implemented")

    @classmethod
    def read_write(cls, node, mask, value):
        """
        Recommended to be implemented by access methods that can improve
        performance of read-modify-writes

        Args:
            node (obj): NodeValue object that we are trying to return the data for
            mask (number): mask off these bits before we or in value
            value (number): value to or in (no checking is done to make sure this
                aligns with mask)

        """
        # we use node.read instead of cls.read so that the
        # read/write values end up in the logfile
        oldval = node.read().value
        newval = oldval & (~mask)  # clear out old value
        newval = newval | (value & mask)   # OR in new one...
        return node.write(newval)

    @classmethod
    def is_access_available(cls, component):
        """
        Used to determine whether it is currently *safe* to read from
        this component.

        Args:
            component (NamedComponent): Component on which we are trying to
                check availability.
        """
        # if components's definition info does not define available, then
        # assume the answer is "yes it is available"
        return component.definition.info.get("available", True)

    @classmethod
    def is_access_available_by_node(cls, node):
        """
        Used to determine whether component or access is currently safe to read from

        Args:
            node (obj): Node is passed in, but node, component is what is used
        """
        # we need to call the per component versions due to some legacy classes that implement that
        return cls.is_access_available(node.component)

    @classmethod
    def is_available(cls, node):
        """
        Used to determine whether it is currently *safe* to read from
        this node, it is primarily used as a hint when iterating through
        all the nodes for displaying the value. Some nodes should only be
        read when they are specifically requested.

        Args:
            node (NamedNodeValue): Node on which we are trying to check
                availability.
        """
        # if node's definition info does not define available, then assume
        # the answer is "yes it is available"
        return node.definition.info.get("available", True)

    @classmethod
    def is_available_cache(cls, node):
        """
        Returns a cache that can be used to do a check once per component, that
        can in turn impact the availability of other nodes
        """
        return node.component.target_info.\
                setdefault('is_available_cache',{}).\
                setdefault(cls.__name__,{})

    # This one's added for backwards compatibility
    isavailable = is_available

    @classmethod
    def lookup_info(cls, node, keys, **kwargs):
        """
        Look up a key in a node's info database and return it's associated
        value when key's been found.

        Args:
            node (NamedNodeValue): The node to perform the search on.
            keys (string, tuple, list): The key (or keys) we are looking for.
            recursive (boolean, optional): True for performing a recursive
                search for keys up the parent, False otherwise. Default is
                True.
            default (optional) : optional value to return if lookup_value does not find the
                specified key any the node or any component

        Returns:
            The value associated with the supplied key when found, otherwise
            it raises LookupError.

        The following is the flow that we follow to perform the search:
                1. check component value
                2. check component definition
                3. if recursive, then check parent components
                4. check node value
                5. check node definition

        Node can be passed in as None, as long as a component is passed in.

        If using LookupInfoCache precondition, the keys parameter has to be
        either of type string or tuple.
        """
        from .preconditions import LookupInfoCache
        # need to import NamedComponent and NamedComponentDefinition this way,
        # doing the import at the module level generates a circular import...
        from .comp import NamedComponent, NamedComponentDefinition
        # The following is the flow that we follow to perform the search:
        # 1. check component value
        # 2. check component definition
        # 3. check node value
        # 4. check node definition
        # if node.name.find("ace_ip_wac_policy") > -1:
        #     import pdb;pdb.set_trace()
        # using an arbitrary object to identify when a key is not present in a
        # component's target_info or definition info nor in node's definition
        # info
        value = LookupError
        component = kwargs.pop("component", None)
        recursive = kwargs.pop("recursive", True)
        # this flag is used to determine if we want to search for keys
        # only in components (True) or also include nodes in the search (False)
        component_only = kwargs.pop('component_only', False)
        add_to_cache = kwargs.pop('add_to_cache', False)
        # internal counter to keep track of where in the recursion hole
        # we are, used to determine if we want LookupInfoCache.NodeOnly as
        # the value to be added to a component's cache.
        recursion_count = kwargs.pop('recursion_count', 0)
        # use an error condition for default not being specified
        default_present = 'default' in kwargs
        default = kwargs.pop("default", LookupError)
        assert len(kwargs) == 0,\
            "Unknown arguments: '%s'" % ",".join(kwargs.keys())
        key_list = keys if isinstance(keys, (list, tuple)) else [keys]

        lookup_component = component or node.component

        # check if there is a lookup_info cache we should use
        lookup_cache = cls.preconditions \
            and LookupInfoCache in cls.preconditions \
            and LookupInfoCache.cache_available(lookup_component)

        if lookup_cache and not component_only:
            # seems we are good to check the keys in the cache
            add_to_cache = True

            # note that keys in the cache can be tuples
            value = LookupInfoCache.get_info(lookup_component, keys, default=LookupError)
            if value is LookupInfoCache.NodeOnly:
                for key in key_list:
                    # key value not in cache, try getting it from node
                    value = node.target_info.get(key, LookupError)
                    if value is not LookupError:
                        break

                    value = node.definition.info.get(key, LookupError)
                    if value is not LookupError:
                        break

            # if we didn't find the value in the current component's cache
            # then let code flow to the recursive call below.
            if value is not LookupError:
                return value
        # else: - not looking up in the cache

        if isinstance(lookup_component, NamedComponent):
            for key in key_list:
                value = lookup_component.target_info.get(key, value)
                if value is not LookupError:
                    break
            if value is LookupError:
                for key in key_list:
                    value = lookup_component.definition.info.get(key, value)
                    if value is not LookupError:
                        break
        elif isinstance(lookup_component, NamedComponentDefinition):
            for key in key_list:
                value = lookup_component.info.get(key, value)
                if value is not LookupError:
                    break

        # check the parent when we couldn't find the key in the component
        if value is LookupError and recursive and \
                lookup_component.parent is not None:
            kwargs = dict(default=default) if default_present else {}
            value = cls.lookup_info(node, keys, recursive=recursive,
                                    component=lookup_component.parent,
                                    component_only=component_only,
                                    add_to_cache=add_to_cache,
                                    recursion_count=recursion_count+1, **kwargs)

        # we are done looking key up in the components see if we have
        # a value that we should add to the cache
        if value is not LookupError and lookup_cache and add_to_cache:
            LookupInfoCache.put_info(lookup_component, keys, value)

        # we are back where we started, after the recursive calls (if any),
        # if we arrived here with a LookupInfoCache.NodeOnly value then reset
        # it to LookupError to *force* searching the node for the actual value
        if recursion_count == 0 and value is LookupInfoCache.NodeOnly:
            value = LookupError

        if not component_only:
            # let's not allow the node's definition to override the
            # component (value or definition)
            if value is LookupError and node is not None:
                for key in key_list:
                    value = node.target_info.get(key, value)
                    if value is not LookupError:
                        # add key to cache as NodeOnly
                        if lookup_cache and add_to_cache:
                            LookupInfoCache.put_info(lookup_component, keys,
                                                     LookupInfoCache.NodeOnly)
                        # return LookupInfoCache.NodeOnly as the value only if
                        # we are deep into the recursion hole so that we correctly
                        # *tell* other components in the way what value to add
                        # to their cache
                        value = value if recursion_count == 0 else LookupInfoCache.NodeOnly
                        break
            if value is LookupError and node is not None:
                for key in key_list:
                    value = node.definition.info.get(key, value)
                    if value is not LookupError:
                        # add key to cache as NodeOnly
                        if lookup_cache and add_to_cache:
                            LookupInfoCache.put_info(lookup_component, keys,
                                                     LookupInfoCache.NodeOnly)
                        # return LookupInfoCache.NodeOnly as the value only if
                        # we are deep into the recursion hole so that we correctly
                        # *tell* other components in the way what value to add
                        # to their cache
                        value = value if recursion_count == 0 else LookupInfoCache.NodeOnly
                        break
            # only raise exception if default was not provided
            # because we provide default as lookup_error when we are in recursive calls
            if value is LookupError and default is LookupError and not default_present:
                # we don't have to check the recursion for raising an error
                # because we are always passing the same node value on each
                # recursive call.
                raise LookupError("Couldn't find key(s): %s"%", ".join(key_list))
            elif value is LookupError:
                # couldn't find a value but a default was specified
                if lookup_cache and add_to_cache:
                    # specifying NodeOnly because other node might have other than the default
                    LookupInfoCache.put_info(lookup_component, keys, LookupInfoCache.NodeOnly)

                # return LookupInfoCache.NodeOnly as the value only if
                # we are deep into the recursion hole so that we correctly
                # *tell* other components in the way what value to add
                # to their cache
                return default if recursion_count == 0 else LookupInfoCache.NodeOnly

        return value

    @classmethod
    def _get_osx_info(cls, node):
        """Find and returns the appropriate OSX interfrace information if this node uses that,

        Returns:
            * dictionary of all interface info for this class if it is an osx node
            * None if not an osx node
            * raises an exception if node has some osx info, but the component is missing it

        """
        node_osx_access_info = node.definition.info.get("osx_accesses", None)
        node_osx_address_maps = node.definition.info.get("osx_address_maps", None)
        if node_osx_access_info is not None and node_osx_address_maps is not None:
            # try to get class name from node, if that doesn't exist, use this class name
            this_map = node_osx_access_info.get(cls.__name__, None)
            if this_map is None:
                this_map = node_osx_access_info.get(node.access_class_name, None)
            if this_map is None:
                raise RuntimeError("Couldn't find osx access info for {} or {}".format(cls.__name__, node.access_class_name))
            osx_address_map = this_map['osx_address_map']
            osx_interface = this_map['osx_interface']
            # get map info
            component_maps = node.component.target_info.get(_OSX_MAP_KEY, None)
            if component_maps is None:
                raise OsxError("node has a reference to an osx address map '{}', but parent componet is '{}' key ",
                    osx_address_map, _OSX_MAP_KEY
                )
            component_addr_map = component_maps.get(osx_address_map, None)
            if component_addr_map is None:
                raise OsxError("address map {} is missing from the parent component".format(osx_address_map))
            node_addr_map = node_osx_address_maps.get(osx_address_map)
            if node_addr_map is None:
                raise OsxError("address map {} is missing from the node info".format(osx_address_map))
            # get interface info
            component_interface = component_addr_map.get(osx_interface, None)
            if component_interface is None:
                raise OsxError(
                    "address map {} is missing interface {} from the parent component".format(
                        osx_address_map,
                        this_map['osx_interface']))
            # is this layer missing in the registers and is needed ??
            # node_interface = node_addr_map.get(this_map["osx_interface"], None)
            # if node_interface is None:
            #     raise OsxError(
            #         "address map {} is missing interface {} from the node infot".format(
            #             osx_address_map,
            #             osx_interface))
            # ??

            # create a copy of the interface info
            osx_info = copy.copy(component_interface)
            # update any node info we have
            if len(node.array_info):
                instance = node.array_info.get(node.definition.name)
                instance_info = node_osx_address_maps[osx_address_map][instance]
                osx_info.update(instance_info)
            else:
                osx_info.update(node_osx_address_maps[osx_address_map])
            return osx_info

        # no osx_access info so assume it is not an osx access and return None
        return None

    @classmethod
    def get_access_info(cls, node, **kwargs):
        """
        Get the properties in a node required to perform an access.

        Args:
            node (NamedNodeValue): The node to get the properties from.

        Returns:
            A dictionary with the node's access properties. If a key's value is missing
            then it will be given the value of *LookupError*
        """
        # return cached if there...
        if '_get_access_info' in node.target_info:
            return node.target_info['_get_access_info']

        # Need to include both requires and requires_from_target
        required_keys = list(cls.requires)
        required_keys.extend(cls.requires_from_target)
        component_only = kwargs.pop('component_only', False)

        # keep order for consistency
        access_properties = OrderedDict()

        # checks for and returns osx info if it is found
        osx_info = cls._get_osx_info(node)
        if osx_info:
            # be sure to cast and copy required key list because we will modify it
            for req_key in list(required_keys):
                key_list = req_key if isinstance(req_key, (list, tuple)) else [req_key]
                for key in key_list:
                    if key in osx_info:
                        access_properties[req_key] = osx_info[key]
                        required_keys.remove(req_key)
                    if key == cls.access_offset_key and cls.osx_map_offset_key in osx_info:
                        access_properties[req_key] += osx_info[cls.osx_map_offset_key]

        # check if there is a cache we should use
        from .preconditions import LookupInfoCache
        lookup_component = node.component
        lookup_cache = cls.preconditions \
            and LookupInfoCache in cls.preconditions \
            and LookupInfoCache.cache_available(lookup_component)


        if lookup_cache:
            # seems we are good to check the keys in the cache
            for req_key in required_keys:
                default = cls.defaults.get(req_key, LookupError)

                # note that keys in the cache can be tuples
                value = LookupInfoCache.get_info(lookup_component, req_key, default=LookupError)
                if value is LookupInfoCache.NodeOnly:
                    # key value not in cache, try getting it from node
                    key_list = req_key if isinstance(req_key, (list, tuple)) else [req_key]
                    for key in key_list:
                        value = node.target_info.get(key, value)
                        if value is not LookupInfoCache.NodeOnly:
                            break

                        value = node.definition.info.get(key, value)
                        if value is not LookupInfoCache.NodeOnly:
                            break

                # property not found in cache neither in node
                # let lookup_info find it and add it to the cache
                if value is LookupError:
                    value = cls.lookup_info(node, req_key,
                                            default=default,
                                            component_only=False,
                                            add_to_cache=True)
                elif value is LookupInfoCache.NodeOnly:
                    # couldn't find a value, using default if available
                    if default is LookupError:
                        key_list = req_key if isinstance(req_key, (list, tuple)) else [req_key]
                        raise LookupError("Couldn't find key(s): %s"%", ".join(key_list))
                    value = default

                access_properties[req_key] = value
        else:
            # don't have a cache, call lookup_info
            for key in required_keys:
                value = cls.lookup_info(node, key,
                                        default=cls.defaults.get(key, LookupError),
                                        component_only=component_only)
                access_properties[key] = value

        return access_properties

    @classmethod
    def show_access_info(cls, node):
        """
        Show the properties in a node required to perform an access.

        Args:
            node (NamedNodeValue): The node to get the properties from.
        """
        from .utils.ordereddict import expand_ordered_dict
        access_properties = expand_ordered_dict(cls.get_access_info(node))        
        for key, value in sorted(access_properties.items()):
            
            if isinstance(key, tuple):
                key = " / ".join(key)
            if isinstance(value, numbers.Integral):
                _LOG.result("%s: %#x " % (key, value))
            elif value is LookupError:
                _LOG.result("%s: <Missing> " % key)
            else:
                _LOG.result("%s: %s " % (key, value))

    @classmethod
    def check(cls, node):
        """
        Checks info to make sure that the node, and its parent, have all the
        required information for an access.

        Args:
            node (NamedNodeValue): The node to perform the check on.
        """
        missing = []
        info = cls.get_access_info(node)

        # Determine if parent component is a component definition
        is_definition = not hasattr(node.component, "definition")

        # Check for missing info
        for key, value in info.items():
            if value == LookupError:
                if is_definition and key not in cls.requires:
                    # info not expected to be in component definition
                    continue
                missing.append(key)

        return missing

    @classmethod
    def store(cls, node, value):
        """Stores a value in to the node's info object for later flushing"""
        # note for here and for Node code - we purposefully don't 'fast track'
        # the store/flush in case we need to customize the parameters later for
        # some reason
        node.stored_value = value

    @classmethod
    def flush(cls, node, **kwargs):
        """
        Flush node's stored value.
        """
        if node.stored_value is None:
            raise RuntimeError(
                                "flush called, but no data stored for this node,\n"
                                "make sure you keep a reference to the same\n"
                                "node that you called store from")
        # if there is a store mask, then use that as well
        skip_read = kwargs.get("skipRead", False)  # PythonSv backwards compatibility
        skip_read = kwargs.get("skip_read", skip_read)  # for pep8 consistency
        # if no stored mask, then just write out
        # ...this means someone just called store on the main node
        if node.stored_mask is None:
            return cls.write(node, node.stored_value)
        else:
            # this means someone from a child queued up some information
            # three scenarios
            # - we have an existing value we should apply our mask to
            # - we should do a read-modify-write to the register
            # - we should just write to the register (that's handled earlier)
            # if we have a stored mask, then use read-modify-write
            if node._value is None and skip_read is False:
                return cls.read_write(node, node.stored_mask, node.stored_value)
            else:
                # value is either set, or skipRead == True
                value = node._value or 0  # make sure 0 is 2nd here
                newval = value & (~node.stored_mask)  # clear out old value
                newval = newval | node.stored_value        # OR in new one...
                return cls.write(node, newval)


class DefaultNodeAccess(NodeAccess):
    methods = NodeAccess.methods + ['priority_access_class', 'priority_access_class_name']

    @classmethod
    def read(cls, node, *args, **kwargs):
        class_name = cls._get_first_available_access(node)
        if class_name:
            return node.component.access_classes[class_name].read(node, *args, **kwargs)
        else:
            raise RuntimeError("DefaultNodeAccess: No access method available for reads")

    @classmethod
    def update_value(cls, node, *args, **kwargs):
        """Returns a value (potentially cached) that represents the node

        Args:
            node - NodeValue object that we are trying to return the data for

        Return:
            must return a value

        This function is expected to return a value that represents data for
        the node. The primary difference in this vs. the read, is that this
        function *could* return a cached value, where-as if someone
        specifically calls read, it is expected to perform some access to make
        sure the latest value is being returned.

        Implementing this function is optional, and by default it will just
        return the value provided by the read function.
        """
        # default get_value is to just use read, but there is some flexibility
        # here in to what 'read' can be setup to do vs. get_value.
        # get_value could be used to return the cached
        # read could be used to force an update -- if so, the read will need to
        # manually set node._value
        # ... the key here is that the user called "read" where-as most of the
        # named_node code calls get_value
        class_name = cls._get_first_available_access(node)
        if class_name:
            return node.component.access_classes[class_name].update_value(node, *args, **kwargs)
        else:
            raise RuntimeError("DefaultNodeAccess: No access method available for reads")


    @classmethod
    def write(cls, node, value, *args, **kwargs):
        class_name = cls._get_first_available_access(node)
        if class_name:
            return node.component.access_classes[class_name].write(node, value, *args, **kwargs)
        else:
            raise RuntimeError("DefaultNodeAccess: No access method available for writes")


    @classmethod
    def read_write(cls, node, mask, value, *args, **kwargs):
        """
        Recommended to be implemented by access methods that can improve
        performance of read-modify-writes

        Args:
            node (obj): NodeValue object that we are trying to return the data for
            mask (number): mask off these bits before we or in value
            value (number): value to or in (no checking is done to make sure this
                aligns with mask)

        """
        class_name = cls._get_first_available_access(node)
        if class_name:            
            return node.component.access_classes[class_name].read_write(node, mask, value, *args, **kwargs)
        else:
            raise RuntimeError("DefaultNodeAccess: No access method available for read_write")

    @classmethod
    def store(cls, node, value, *args, **kwargs):
        """Stores a value in to the node's info object for later flushing"""
        # note for here and for Node code - we purposefully don't 'fast track'
        # the store/flush in case we need to customize the parameters later for
        # some reason
        class_name = cls._get_first_available_access(node)
        if class_name:
            return node.component.access_classes[class_name].store(node, value, *args, **kwargs)
        else:
            raise RuntimeError("DefaultNodeAccess: No access method available for store")

    @classmethod
    def flush(cls, node, *args, **kwargs):
        """
        Flush node's stored value.
        """
        class_name = cls._get_first_available_access(node)
        if class_name:
            return node.component.access_classes[class_name].flush(node, *args, **kwargs)
        else:
            raise RuntimeError("DefaultNodeAccess: No access method available")


    @classmethod
    def show_access_info(cls, node):
        """
        Show the properties in a node required to perform an access.

        Args:
            node (NamedNodeValue): The node to get the properties from.
        """        
        class_name = cls._get_first_available_access(node)
        if class_name:
            return node.component.access_classes[class_name].show_access_info(node)
        else:
            raise RuntimeError("DefaultNodeAccess: No access method available")        

    @classmethod
    def get_access_info(cls, node, *args, **kwargs):
        """
        Get the properties in a node required to perform an access.

        Args:
            node (NamedNodeValue): The node to get the properties from.

        Returns:
            A dictionary with the node's access properties. If a key's value is missing
            then it will be given the value of *LookupError*
        """        
        class_name = cls._get_first_available_access(node)
        if class_name:
            return node.component.access_classes[class_name].get_access_info(node, *args, **kwargs)
        else:
            raise RuntimeError("DefaultNodeAccess: No access method available")    

        
    @classmethod
    def priority_access_class(cls, node):
        class_name = cls._get_first_available_access(node)
        if class_name:            
            return node.component.access_classes[class_name]
        else:
            raise RuntimeError("DefaultNodeAccess: No access method available")        

    @classmethod
    def priority_access_class_name(cls, node):
        class_name = cls._get_first_available_access(node)
        if class_name:
            return class_name
        else:
            raise RuntimeError("DefaultNodeAccess: No access method available")      

    @classmethod
    def is_available(cls, node):        
        class_name = cls._get_first_available_access(node)
        if class_name:        
            return node.component.access_classes[class_name].is_available(node)
        else:
            return False

    # This one's added for backwards compatibility
    isavailable = is_available


    @classmethod
    def is_access_available_by_node(cls, node):
        class_name = cls._get_first_available_access(node)
        if class_name:
            return node.component.access_classes[class_name].is_access_available_by_node(node)
        else:
            return False

    @classmethod
    def is_available_cache(cls, node):
        """
        Returns a cache that can be used to do a check once per component, that
        can in turn impact the availability of other nodes
        """
        class_name = cls._get_first_available_access(node)
        if class_name:
            return node.component.access_classes[class_name].is_available_cache(node)
        else:
            return False

    @classmethod
    def _get_first_available_access(cls, node):
        """checks (in order) if the access is supported by the component
        and if that's the case, it returns the name of the class.
        Returns False otherwise."""
        try:
            access_priority = node.lookup_info('access_priority')
        except Exception as e:
            raise RuntimeError("DefaultAccess requires access_priority list definition")
        for accessp in access_priority:
            class_name = node.accesses.get(accessp, None)
            if class_name:
                if class_name in node.component.access_classes:
                    return class_name
        return None

    @classmethod
    def current_priority(cls, node):
        access_priority = cls._get_first_available_access(node)
        return access_priority




def register(name, access_class):
    """Register as a known access class"""
    # do not throw warning/error here it is likely on purpose that we are overwriting some class
    NodeAccessManager[name] = access_class


def unregister(name):
    """Un-Register as a known access class"""
    if name not in NodeAccessManager:
        return False
    del NodeAccessManager[name]
    return True
