
# INTEL CONFIDENTIAL
# Copyright 2018 Intel Corporation
#
# The source code  contained or  described herein and  all documents related to
# the source code  ("Material") are owned by Intel Corporation or its suppliers
# or licensors.  Title to the  Material  remains with  Intel Corporation or its
# suppliers  and licensors. The Material contains trade secrets and proprietary
# and  confidential  information  of  Intel  or  its  suppliers  and  licensors.
# The Material  is protected  by worldwide  copyright and trade secret laws and
# treaty provisions.  No part of the Material  may  be used, copied, reproduced,
# modified, published, uploaded, posted, transmitted, distributed, or disclosed
# in any way without Intel's prior express written permission. No license under
# any  patent,  copyright, trade secret or other intellectual property right is
# granted  to  or conferred upon you by disclosure or delivery of the Materials,
# either expressly, by implication, inducement, estoppel or otherwise.
# Any license under such intellectual property rights must be express and
# approved by Intel in writing.

"""Namednodes's precondition module

This moudle contains multiple types of precondition abstract base classes. This
is implemented such that any class that inherits from these classes will be a
singleton. To make this work the __new__ method must not be overriden.  A
precondition is designed to run a section of code(the precondition) before
something and another section of code(the postcondition) after something.
Nesting is handled such that precondition code is only ran once at the high
level call and any nested calls will be no ops. For nested postcondition code,
it is similar.

SystemPrecondition        - global precondition to be ran regardless of the component
DiscoveryItemPrecondition - to be used with discovery items for a precondition
                            that must be ran per discovey item
ComponentPrecondition     - to be used with standard components for a precondition
                            that must be ran per component

It is expected you will inherit from one of the base classes above and
implement the precondition and postcondition methods.

Example Usage:

class MyPrecondition(ComponentPrecondition):
    def precondition(self, component):
        # do something that needs to be done before doing something with the componet
    def postcondition(self, component):
        # do something that needs to be done after doing something with the componet

def do_something(component, some_arg):
    MyPrecondition.run_precondition(component)
    # do something with the component like read or write
    MyPrecondition.run_postcondition(component)


"""
from __future__ import absolute_import
from __future__ import division
from __future__ import print_function

from abc import ABCMeta, abstractmethod
import functools
from .logging import getLogger
from .access import NodeAccess as _NodeAccess
from .nodes import NamedNodeValue
from . import settings

import sys
import weakref

from six import add_metaclass, reraise

# for things that should be imported by precondition.py to be made available to users
__all__ = [
    "SystemPreconditionGen2",
    "DiscoveryItemPreconditionGen2",
    "ComponentPreconditionGen2",
    "RunWhenNeededPrecondition",
    "MultipleAccesses",
    "multiple_accesses"
]

_LOG = getLogger("preconditions")


@add_metaclass(ABCMeta)
class _PreconditionGen2(object):
    # expects weakrefs with multiple accesses instead of the original object
    save_weakrefs = False

    def _acquire_lock(self, component):
        raise NotImplementedError

    def _release_lock(self, component):
        raise NotImplementedError

    @classmethod
    def _init_info(cls, component):
        """
        Initialize info container for precondition.

        Args:
            component (NamedComponent) : The component where precondition info
                                         info is stored.
        """
        raise NotImplementedError

    @classmethod
    def _cleanup_info(cls, component):
        """
        Cleanup info container for precondition.

        Args:
            component (NamedComponent) : The component where precondition info
                                         is stored.
        """
        raise NotImplementedError

    @classmethod
    def precondition(cls, node_or_component):
        """
        This abstract method should contain the actual precondition code.
        """
        raise NotImplementedError

    @classmethod
    def postcondition(cls, node_or_component):
        """
        This abstract method should contain the actual postcondition code.
        This will run during the last
        """
        raise NotImplementedError

    @classmethod
    def put_info(cls, node_or_component, key, value):
        """
        Save information needed by post condition, code will be sure to 'clean' this
        after postcondition is run.
        """
        raise NotImplementedError


    @classmethod
    def get_info(cls, node_or_component, key, default=KeyError):
        """
        Get info needed by the precondition.
        """
        raise NotImplementedError


    @classmethod
    def run_precondition(cls, node_or_component):
        """
        This method is to be always called before access code is ran. This
        in turn will call the precondition method only if run_precondition
        hasn't been previously called without a run_postcondition also being
        called previously.

        Note:
            Derived classes should NOT override this, and should be overriding the precondition
            function

        """
        if settings.DEBUG:
            _LOG.caller("Precondtion.run_precondition")
            path = node_or_component.path if node_or_component is not None else 'None'
            _LOG.debugall("run_precondition for class '%s' on path: '%s'" % (cls.__name__, path))

        if isinstance(node_or_component, NamedNodeValue):
            component = node_or_component.component
        else:
            # assume it is a component
            component = node_or_component

        cls._init_info(component)

        if cls._acquire_lock(component):
            if settings.DEBUG:
                _LOG.debugall("run_precondition lock acquired - running precondition")
            # if MultipleAccesses is in use and we just acquired lock,
            # lock it one more time and register ourselves to be unlocked
            # when MultipleAccesses is done
            # we do this BEFORE calling the precondition in case the precondition manually calls OTHER preconditions...
            if MultipleAccesses.in_use:
                cls._acquire_lock(component)
                if cls.save_weakrefs:
                    MultipleAccesses._register_precondition(cls, weakref.ref(node_or_component))
                else:
                    MultipleAccesses._register_precondition(cls, node_or_component)
            try:
                cls.precondition(node_or_component)
            except:
                cls._release_lock(component)
                # if in side a multiple access, it is debatable as to whether we should remove
                # ourselves from the post condition flow since we do not know if it succeeded or not
                raise
        elif settings.DEBUG:
            _LOG.debugall("run_precondition lock was not acquired")

        if settings.DEBUG:
            path = node_or_component.path if node_or_component is not None else 'None'
            _LOG.debugall("run_precondition EXIT for class '%s' on path: '%s'"%(cls.__name__, path))



    @classmethod
    def run_postcondition(cls, node_or_component):
        """
        This method is to be always called after access code is ran. This in
        turn will call the postcondition method only if there are no
        outstanding calls to run_postcondition that there hasn't been a call to
        run_postcondition for.

        Note:
            Derived classes should NOT override this, and should be overriding the postcondition
            function
        """
        if settings.DEBUG:
            _LOG.caller("Precondtion.run_postcondition")
            path = node_or_component.path if node_or_component is not None else 'None'
            _LOG.debugall("run_postcondition ENTER for class '%s' on path: '%s'"%(cls.__name__, path))

        if isinstance(node_or_component, NamedNodeValue):
            component = node_or_component.component
        else:
            # assume it is a component
            component = node_or_component

        if cls._release_lock(component):
            if settings.DEBUG:
                _LOG.debugall("run_postcondition lock released - running postcondition")
            try:
                cls.postcondition(node_or_component)
            except:
                cls._cleanup_info(component)
                raise
            # make sure we remove cached precondition information....
            cls._cleanup_info(component)

        elif settings.DEBUG:
            _LOG.debugall("run_postcondition lock not released")

        if settings.DEBUG:
            path = node_or_component.path if node_or_component is not None else 'None'
            _LOG.debugall("run_postcondition EXIT for class '%s' on path: '%s'"%(cls.__name__, path))



    @classmethod
    def lookup_info(cls, node_or_component, keys, **kwargs):
        """
        This function uses the same flow as the NodeAccess.lookup_info to find
        any information needed by the precondition.
        """
        default_present = 'default' in kwargs
        default = kwargs.pop('default', LookupError)
        if isinstance(node_or_component, NamedNodeValue):
            try:
                value= _NodeAccess.lookup_info(node_or_component, keys, **kwargs)
            except LookupError:
                    value = LookupError
        else:
            # must be a component
            try:
                value= _NodeAccess.lookup_info(None, keys, component=node_or_component, **kwargs)
            except LookupError:
                    value = LookupError
        #Standard
        if value is LookupError and default is LookupError and not default_present:
            # we don't have to check the recursion for raising an error
            # because we are always passing the same node value on each
            # recursive call.
            raise LookupError("Couldn't find key(s): '{}' in lookup_info for precondition '{}'".format(
                "".join(keys),
                cls.__name__,
            ))
        elif value is LookupError:
            # couldn't find a value but a default was specified
            return default
        return value


class SystemPreconditionGen2(_PreconditionGen2):
    """
    Abstract base class of a system level precondition. The precondition, postcondition,
    _init_info, _cleanup_info, put_info and get_info methods must be implemented.
    """
    _locks = {}
    _info = {}

    # NOTE: NONE of the functions here should actually use the component as it is system
    # wide, this could show work even if component is None

    @classmethod
    def _acquire_lock(cls, component):
        preval = cls._locks.get(cls, 0)
        cls._locks[cls] = preval + 1
        first_lock = cls._locks[cls] == 1
        # return actual first_lock value so we know whether to run pre-condition
        return first_lock

    @classmethod
    def _release_lock(cls, component):
        if cls._locks.get(cls, 0) == 0:
            raise RuntimeError("precondition '{}' already unlocked".format(cls.__name__))
        cls._locks[cls] -= 1
        return cls._locks[cls] == 0

    @classmethod
    def _init_info(cls, component):
        # SystemPreconditionGent2 preconditions store info in _info class
        # storage therefore we ignore the component that was passed in.
        cls._info.setdefault(cls.__name__, {})

    @classmethod
    def _cleanup_info(cls, component):
        cls._info.pop(cls.__name__)

    # get/put info for system level preconditions need to make sure the information is globally shared

    @classmethod
    def put_info(cls, node_or_component, key, value):
        """
        Save information needed by post condition, code will be sure to 'clean' this
        after postcondition is run.
        """
        cls._info[cls.__name__][key] = value

    @classmethod
    def get_info(cls, node_or_component, key, default=KeyError):
        """
        Retrieve info needed by the system pre-condition.
        """
        value = cls._info[cls.__name__].get(key, default)
        if value is KeyError:
            raise KeyError(key)
        else:
            return value


class RunWhenNeededPrecondition(SystemPreconditionGen2):
    """
    This precondition runs at the beginning of EVERY node, but only *once*
    when a window is closed
    """
    save_weakrefs = True

    @classmethod
    def _acquire_lock(cls, node_or_component):
        # call acquire lock every if we have not run for this node
        # so that the correct unlock happens, but return True so that we run the precondition every time
        preval = cls._locks.get(cls, 0)
        cls._locks[cls] = preval + 1
        first_lock = cls._locks[cls] == 1
        return True



class DiscoveryItemPreconditionGen2(_PreconditionGen2):
    """
    Abstract base class of a discovery item level precondition. The precondition, postcondition,
    _init_info, _cleanup_info, put_info and get_info methods must be implemented.
    """
    @classmethod
    def _acquire_lock(cls, component):
        name = "precondition_%s" % (repr(cls))
        target_info = component.origin.target_info
        if name not in target_info:
            target_info[name] = 0
        target_info[name] += 1
        first_lock = target_info[name] == 1
        # return actual first_lock value so we know whether to run pre-condition
        return first_lock

    @classmethod
    def _release_lock(cls, component):
        name = "precondition_%s" % (repr(cls))
        target_info = component.origin.target_info
        if name not in target_info or target_info[name] == 0:
            raise RuntimeError("precondition '{}' already unlocked".format(cls.__name__))
        target_info[name] -= 1
        return target_info[name] == 0

    @classmethod
    def _init_info(cls, component):
        # DiscoveryItemPreconditionGen2 preconditions store info
        # in current component's origin component.
        component.origin.target_info.setdefault('preconditions', {}).setdefault(cls, {})

    @classmethod
    def _cleanup_info(cls, component):
        component.origin.target_info['preconditions'].pop(cls)

    @classmethod
    def put_info(cls, node_or_component, key, value):
        """
        Save information needed by post condition, code will be sure to 'clean' this
        after postcondition is run.
        """
        component = node_or_component.component if isinstance(node_or_component, NamedNodeValue) else node_or_component
        component.origin.target_info['preconditions'][cls][key]=value

    @classmethod
    def get_info(cls, node_or_component, key, default=KeyError):
        component = node_or_component.component if isinstance(node_or_component, NamedNodeValue) else node_or_component
        value = component.origin.target_info['preconditions'][cls].get(key, default)
        if value is KeyError:
            raise KeyError("{} missing in get_info call for precondition {}".format(
                key,
                cls.__name__
            ))
        else:
            return value


class ComponentPreconditionGen2(_PreconditionGen2):
    """
    Abstract base class of a component level precondition. The precondition, postcondition,
    _init_info, _cleanup_info, put_info and get_info methods must be implemented.
    """
    @classmethod
    def _acquire_lock(cls, component):
        #name = "precondition_%s_%s" % (self.__class__, id(self))
        name = "precondition_%s" % repr(cls)
        target_info = component.target_info
        if name not in target_info:
            target_info[name] = 0
        target_info[name] += 1
        first_lock = target_info[name] == 1
        # return actual first_lock value so we know whether to run pre-condition
        return first_lock

    @classmethod
    def _release_lock(cls, component):
        # name = "precondition_%s_%s" % (self.__class__, id(self))
        name = "precondition_%s" % repr(cls)
        target_info = component.target_info
        if name not in target_info or target_info[name] == 0:
            raise RuntimeError("precondition '{}' already unlocked".format(cls.__name__))
        target_info[name] -= 1
        return target_info[name] == 0

    @classmethod
    def _init_info(cls, component):
        # ComponentPreconditionGen2 preconditions store info in current component.
        component.target_info.setdefault('preconditions', {}).setdefault(cls, {})

    @classmethod
    def _cleanup_info(cls, component):
        component.target_info['preconditions'].pop(cls)

    @classmethod
    def put_info(cls, node_or_component, key, value):
        """
        Save information needed by post condition, code will be sure to 'clean' this
        after postcondition is run.
        """
        component = node_or_component.component if isinstance(node_or_component, NamedNodeValue) else node_or_component
        component.target_info['preconditions'][cls][key]=value

    @classmethod
    def get_info(cls, node_or_component, key, default=KeyError):
        component = node_or_component.component if isinstance(node_or_component, NamedNodeValue) else node_or_component
        value = component.target_info['preconditions'][cls].get(key, default)
        if value is KeyError:
            raise KeyError(key)
        else:
            return value


class MultipleAccesses(object):
    current = None
    __initialized = False
    # flag to know if we
    in_use = False
    _count = 0
    # this is a CLASS variable and will be used to track
    # ANY/ALL exit post conditions that we should do on exit
    _exit_post_conditions = []
    # THIS is an object variable and used to register
    # specific/additional pre-conditions to get run on exit of this
    # particular "multiple" object
    _additional_preconditions = None

    def __init__(self, *, silent=False, device_locker=True):
        """
        Args:
            device_locker : automatically add a system wide device locker precondition
            silent : whether to silence any exceptions that may occurr
        """
        self._silent = silent
        self._additional_preconditions = []
        if device_locker:
            from .preconditions.ipc import IpcSystemDeviceLocker
            self._additional_preconditions.append((IpcSystemDeviceLocker, None))

    def add_precondition(self, precondition, node_or_component):
        """Add an additional precondition to run, and the component to pass in as the parameter"""
        self._additional_preconditions.append((precondition, node_or_component))

    @classmethod
    def _register_precondition(cls, precondition, node_or_component):
        """
        register a _Precondition object that we should call on the last __exit__

        Args:
            condition (obj) : actual condition object

        Called by pre-condition lock code when MultipleAccesses is in use.
        """
        cls._exit_post_conditions.append((precondition, node_or_component))

    def __enter__(self):
        self.__class__.in_use = True
        for pre, comp in self._additional_preconditions:
            pre.run_precondition(comp)
        self.__class__._count += 1

    def __exit__(self, exc_type, exc_value, traceback):
        self.__class__._count -= 1
        # check for site wide precondtions that should be run
        # on exit if we are the last "multiple" to exit
        exception = None
        if self.__class__._count == 0:
            for pre, comp in reversed(self._exit_post_conditions):
                try:
                    pre.run_postcondition(comp)
                except Exception as e:
                    if not self._silent:
                        _LOG.exception("post condition failed...")
                    # exception should store the first one that fails..
                    exception = sys.exc_info() if exception is None else exception
                    # on exit, we need to clear our lists
            type(self)._exit_post_conditions = []
            type(self).in_use = False
        # run this particular objects post conditions
        # I feel like this *might* be a bug after amd that this may need to be under count==0
        for pre, node_or_component in reversed(self._additional_preconditions):
            try:
                pre.run_postcondition(node_or_component)
            except Exception as e:
                if not self._silent:
                    _LOG.exception("post condition failed...")
                exception = sys.exc_info() if exception is None else exception
        # if error occurred to cause exit, make sure it is raised
        if not self._silent and exc_type:
            reraise(exc_type, exc_value, traceback)
        if not self._silent and exception:
            # after all postconditions have had a chance to run, throw up any exceptions...
            reraise(*exception)


def multiple_accesses(f):
    """Function decorator for putting a multiple access wrapper around the function"""
    @functools.wraps(f)
    def wrapper(*args, **kwargs):
        with MultipleAccesses():
            return f(*args, **kwargs)
    return wrapper
