###############################################################################
# Copyright 2014 2017 Intel Corporation.
#
# The source code, information and material ("Material") contained herein is
# owned by Intel Corporation or its suppliers or licensors, and title to such
# Material remains with Intel Corporation or its suppliers or licensors. The
# Material contains proprietary information of Intel or its suppliers and
# licensors. The Material is protected by worldwide copyright laws and treaty
# provisions. No part of the Material may be used, copied, reproduced, modified,
# published, uploaded, posted, transmitted, distributed or disclosed in any way
# without Intel's prior express written permission.
#
# No license under any patent, copyright or other intellectual property rights
# in the Material is granted to or conferred upon you, either expressly, by
# implication, inducement, estoppel or otherwise. Any license under such
#
# intellectual property rights must be express and approved by Intel in writing.
# Unless otherwise agreed by Intel in writing, you may not remove or alter 
# this notice or any other notice embedded in Materials by Intel or Intel's 
# suppliers or licensors in any way.
###############################################################################

# ipython fixes
# If we are currently running ipython, then 
import sys 

if 'IPython' in sys.modules:
    import IPython
    ipython = IPython.get_ipython()
else:
    ipython = None
    
import re
ansi_escape = re.compile(r'''
    \x1B  # ESC
    (?:   # 7-bit C1 Fe (except CSI)
        [@-Z\\-_]
    |     # or [ for CSI, followed by a control sequence
        \[
        [0-?]*  # Parameter bytes
        [ -/]*  # Intermediate bytes
        [@-~]   # Final byte
    )
''', re.VERBOSE)


# nasty workaround for dal embedded not handling readline
import __main__
if __main__.__dict__.get("__fromembeddedpython__",False):
    readline = None
else:
    try:
        import readline
    except: 
        # a blind try-except is needed to support folks using ipccli in embedded environment
        readline = None

class StreamLog(object):
    def __init__(self, ostream, file = None):
        self.ostream = ostream
        if hasattr(ostream,"closed"):
            self.closed = ostream.closed # SG need to prevent exception with colorama
        self.logfile = file

    def set_file(self,file):
        # close any pre-existing and open file...
        if self.logfile and not self.logfile.closed:
            self.logfile.close()
        self.logfile = file

    def write(self, *args):
        if self.logfile:
            if(len(args)):
                msg = ansi_escape.sub('', args[0])
                args = [msg] + list(args[1:])
            self.logfile.write(*args)
        return self.ostream.write(*args)

    def flush(self):
        if self.logfile:
            self.logfile.flush()
        self.ostream.flush()

    # for any missing attributes, try the primary stream
    def __getattr__(self,attr):
        return getattr(self.ostream,attr)

class StdIoHook(object):
    # need to 
    singleton = None
    def __init__(self):
        self.ip_shell = None
        self.filename = ""
        self.file = None # no file open yet
        # used only in the ipython mode
        self._ip_stdout = None
        self._ip_stderr = None
        self._q_close = False # needed by Ipyton to queue closing of file after the post hook
        # this post happened is also used by our event display
        self._executing = True # to watch out for cases of exceptions where our post hook never runs
        # used by readline mode (not ipython)
        self._rl_stdout = None
        self._rl_stderr = None
        # use this to know if we have already registerd our hooks
        self._ipython_registered = False
        # create our overrides
        if sys.stdout is not None:
            self._rl_stdout = StreamLog(sys.stdout)
            if not (sys.stdout.__class__ == StreamLog):
                sys.stdout = self._rl_stdout
        if sys.stderr is not None:
            self._rl_stderr = StreamLog(sys.stderr)
            if not (sys.stderr.__class__ == StreamLog):
                sys.stderr = self._rl_stderr
        # this gives us our hook for logging the input:
        if sys.platform  == 'win32':
            if readline and not ipython:
                readline.rl.set_pre_input_hook( _pre_hook )
                readline.rl.logcmd = _logcmd
                if hasattr(readline.rl,"mode"): #if newer readline module we have to put the function in different place
                    readline.rl.mode.logcmd = _logcmd
                readline.parse_and_bind('Return: logcmd')

    def _ipython_check(self):
        if self._ipython_registered:
            return
        if 'IPython' in sys.modules:     
            import IPython
            ipython = IPython.get_ipython()            
            ipython.events.register( 'pre_run_cell', self._pre_execute )
            ipython.events.register( 'post_execute', self._post_execute )
            self._ipython_registered = True

    @classmethod
    def create(cls):
        if cls.singleton == None:
            cls.singleton = cls()
        return cls.singleton

    def is_cmd_executing(self):
        return self._executing

    def _pre_execute(self, *args, **kwargs):
        self._executing = True
        if self.file:
            #import pdb;pdb.set_trace()
            # write out the command before we execute it
            if len(args) == 0 and readline:
                # for ipython5
                self.file.write("%s %s\n" % (sys.ps1,readline.get_line_buffer()))
            elif len(args) > 0:
                # for ipython7
                self.file.write("%s %s\n" % (sys.ps1, args[0].raw_cell))
            else:
                # cant support if readline is not present on older pythons..
                pass

    def _post_execute(self,*args):
        self._executing = False

    def log(self,filename,accesstype="w"):
        self._ipython_check()
        if self.file:
            self.file.close()
        self.file = open( filename, accesstype)
        if readline: # if readline is available, log the command used to start the logging
            buffer = readline.get_line_buffer()
            if len(buffer)>0 and hasattr(sys,"ps1"):
                self.file.write("%s %s\n" % (sys.ps1,readline.get_line_buffer()))
        # stdxxx overrides have local file to set/not set
        if self._rl_stdout: self._rl_stdout.set_file(self.file)
        if self._rl_stderr: self._rl_stderr.set_file(self.file)

    def logfile(self):
        """returns file handle or None if no logging is currently being done"""
        return self.file

    def nolog(self):
        # stdxxx overrides have local file to set/not set

        if self.file and not self.file.closed:
            self.file.close()
            self.file = None
        if self._rl_stdout: self._rl_stdout.set_file(None)
        if self._rl_stderr: self._rl_stderr.set_file(None)


########################################################
# Need to see if we can convert to be part of the object
# global command since we can't have self
########################################################
def _logcmd(e):
    """
    This helper function is used to grab the command line input and log it
    to the file so that we don't get just the output of scripts
    """
    global hook
    # for when we are using readline, lets still log when someone
    # has hit the ENTER
    hook._executing = True
    if hook.file:
        ps1 = getattr(sys,"ps1",">>> ")
        hook.file.write("%s %s\n" % (ps1, readline.get_line_buffer()))
        hook.file.flush() # make sure what we have so far is in the log file
    return True

#
# Standalone function vs. part of StdIoHook due to the requirements of pyreadline
# used to specify that we are not executing anything currently and to make sure
# the file is up-to-date
def _pre_hook():
    global hook
    hook._executing = False
    if hook.file:
        hook.file.flush() # make sure what we have so far is in the log file

# THIS IS USED
hook = None
log = None
nolog = None
logfile = None
# used to initialize hook, log, nolog variables
# intent is that this can be called after starting ipython if this
# code gets run before Ipython is installed.
def init():
    global hook
    global log
    global nolog
    global logfile

    hook = StdIoHook()
    log = hook.log
    nolog = hook.nolog
    logfile = hook.logfile

init()

# If we ever want to restore orignal defaults, just use:
# sys.stdout = sys.__stdout__
# sys.stderr = sys.__stderr__

def get_hook():
    """Used to get the hook object"""
    global hook
    return hook

def test_simple():
    log("test_simple.log")
    print(" print ")
    #import logging
    from common import toolbox
    logger = toolbox.getLogger('test')
    #logger = logging.getLogger("test")
    logger.result(" logger ")
    nolog()

     
def test_ipython():
    import IPython
    IPython.start_ipython()


if __name__ == "__main__":
    test_ipython()
