###############################################################################
# Copyright 2014 2017 Intel Corporation.
#
# The source code, information and material ("Material") contained herein is
# owned by Intel Corporation or its suppliers or licensors, and title to such
# Material remains with Intel Corporation or its suppliers or licensors. The
# Material contains proprietary information of Intel or its suppliers and
# licensors. The Material is protected by worldwide copyright laws and treaty
# provisions. No part of the Material may be used, copied, reproduced, modified,
# published, uploaded, posted, transmitted, distributed or disclosed in any way
# without Intel's prior express written permission.
#
# No license under any patent, copyright or other intellectual property rights
# in the Material is granted to or conferred upon you, either expressly, by
# implication, inducement, estoppel or otherwise. Any license under such
#
# intellectual property rights must be express and approved by Intel in writing.
# Unless otherwise agreed by Intel in writing, you may not remove or alter 
# this notice or any other notice embedded in Materials by Intel or Intel's 
# suppliers or licensors in any way.
###############################################################################

import os
import sys
import argparse
import subprocess

class ParseParamsAction(argparse.Action):
    """
    Action for argparse to parse a comma-separated list of key/value pairs into a dictionary.
    """
    def __call__(self, parser, namespace, values, option_string=None):
        value = dict((k.strip(), v) for k, v in (p.split("=") for p in values.split(",")))
        setattr(namespace, self.dest, value)

# Note: This should stay in sync with default_settings in settings.py
parser = argparse.ArgumentParser(description="Launches an interactive IPC CLI session.")
parser.add_argument("--in-process", dest="OUT_OF_PROCESS", action="store_false", help="Connect to IPC API in the same process space.")
parser.set_defaults(OUT_OF_PROCESS=None)
parser.add_argument("--ipc-launch-server", dest="IPC_LAUNCH_SERVER", action="store_true", help="Force launch of IPC API server (instead of attaching to an existing server). Use --ipc-api-server to specify a port for the server.")
parser.set_defaults(IPC_LAUNCH_SERVER=None)
parser.add_argument("--ipc-api-server", dest="IPC_API_SERVER", action="store", help="URI for the IPC API server to connect to (e.g. \"domain.corp.com:1234\", \"10.2.14.13:1234\", \"[::1]:1234\"). Assumes that the server is already started and listening on that address. This option takes precendence over --in-process.")
parser.set_defaults(IPC_API_SERVER=None)
parser.add_argument("--ipc-config-file", dest="IPC_CONFIG_FILE", action="store", help="Path to the configuration file for configuring the IPC API server. When not specified, the server's default configuration is used.")
parser.set_defaults(IPC_CONFIG_FILE=None)
parser.add_argument("--ipc-config-name", dest="IPC_CONFIG_NAME", action="store", help="Name of the configuration for configuring the IPC API server. When not specified, the server's default configuration is used.")
parser.set_defaults(IPC_CONFIG_NAME=None)
parser.add_argument("--ipc-config-params", dest="IPC_CONFIG_PARAMS", action=ParseParamsAction, help="Configuration parameter values to apply to the selected configuration as key/value pairs separated by commas (e.g. \"TargetIpAddress=127.0.0.1,TargetPort=987\").")
parser.set_defaults(IPC_CONFIG_PARAMS=None)
parser.add_argument("--ipc-path", dest="IPC_PATH", action="store", help="Path to the IPC API implementation to connect to.")
parser.set_defaults(IPC_PATH=None)
parser.add_argument("--ipc-logging-preset", dest="IPC_LOGGING_PRESET", action="store", help="Sets the logging preset to the name specified after connecting (e.g. \"GeneralDebug\", \"Off\", \"All\")")
parser.set_defaults(IPC_LOGGING_PRESET=None)
parser.add_argument("--no-version-check", dest="VERSION_CHECK", action="store_false", help="Disable checking IPC API and implementation version before invoking newer APIs.")
parser.set_defaults(VERSION_CHECK=None)
parser.add_argument("--no-event-display", dest="EVENT_DISPLAY", action="store_false", help="Disable display of IPC API events.")
parser.set_defaults(EVENT_DISPLAY=None)
parser.add_argument("--no-event-timestamp", dest="EVENT_TIMESTAMP", action="store_false", help="Disable display of IPC API event timestamps.")
parser.set_defaults(EVENT_TIMESTAMP=None)
parser.add_argument("--display-wait-time", dest="DISPLAY_WAIT_TIME", action="store", help="Time to wait before displaying queued events.")
parser.set_defaults(DISPLAY_WAIT_TIME=None)
parser.add_argument("--event-wait-time", dest="EVENT_WAIT_TIME", action="store", help="Time to delay event display before checking for another event.")
parser.set_defaults(EVENT_WAIT_TIME=None)
parser.add_argument("--developer", dest="DEVELOPER", action="store_true", help="Enable developer mode.")
parser.set_defaults(DEVELOPER=None)
parser.add_argument("--prompt-display", dest="PROMPT_DISPLAY", action="store_true", help="Enable display of current status on the prompt.")
parser.set_defaults(PROMPT_DISPLAY=None)
parser.add_argument("--prompt-prefix", dest="PROMPT_PREFIX", action="store", help="Prefix before the prompt.")
parser.set_defaults(PROMPT_PREFIX=None)
parser.add_argument("--wizard", dest="WIZARD", action="store_true", help="Use an interactive wizard to configure IPC initialization.")
parser.set_defaults(WIZARD=None)

# Dynamic initialization arguments
parser.add_argument("--configs", dest="CONFIGS", nargs="*", action="store", help="Specifies a list of preset configuration scenarios to select; if no configurations are provided then fully automatic configuration is used. Example: --configs DCI")
parser.set_defaults(CONFIGS=None)
parser.add_argument("--show-configs", dest="SHOW_CONFIGS", action="store_true", help="Shows the available preset configuration scenarios.")
parser.set_defaults(SHOW_CONFIGS=None)
parser.add_argument("--probe", dest="PROBE", nargs="*", action="append", help="Selects the specified probe to be active in the current configuration. Optionally, a unique identifier can be specified to select a specific physical probe. This argument can be used multiple times to select multiple probes. The selected probe may be a probe connected to the host or might be a remote or virtual probe. Example: --probe CCA")
parser.set_defaults(PROBE=None)
parser.add_argument("--show-probes", dest="SHOW_PROBES", action="store_true", help="Shows the connected probes.")
parser.set_defaults(SHOW_PROBES=None)
parser.add_argument("--probe-tap-controllers", dest="PROBE_TAP_CONTROLLERS", nargs="*", action="append", help="Specifies which TAPs are expected on each chain for the most recently selected probe. Example: --probe-tap-controllers SKL_UC.A0")
parser.set_defaults(PROBE_TAP_CONTROLLERS=None)
parser.add_argument("--show-tap-controllers", dest="SHOW_TAP_CONTROLLERS", action="store_true", help="Shows the available TAP controllers.")
parser.set_defaults(SHOW_TAP_CONTROLLERS=None)
parser.add_argument("--global-configs", dest="GLOBAL_CONFIGS", action="store", help="Specifies one or more global device config values. Device config values are defined in comma-separated key-value pair form. Example: --global-configs 'TargetOs: Linux, TargetManager.DetectDevices: false'")
parser.set_defaults(GLOBAL_CONFIGS=None)
parser.add_argument("--probe-configs", dest="PROBE_CONFIGS", nargs="*", action="append", help="Specifies one or more device config values for the most recently selected probe. Device config values are defined in comma-separated key-value pair form. Example: --probe-configs 'Jtag.DisableAllInterfaces: true'")
parser.set_defaults(PROBE_CONFIGS=None)
parser.add_argument("--device-configs", dest="DEVICE_CONFIGS", nargs="*", action="store", help="Specifies one or more device config values for a particular device. Device config values are defined in comma-separated key-value pair form. Example: --device-configs GPC 'RunControl.ProbeModeEntryMode: Uncore'")
parser.set_defaults(DEVICE_CONFIGS=None)
parser.add_argument("--save-config", dest="SAVE_CONFIG", action="store_true", help="Saves the resulting configuration specified in the command-line arguments.")
parser.set_defaults(SAVE_CONFIG=None)
parser.add_argument("--load-config", dest="LOAD_CONFIG", action="store_true", help="Loads the last saved configuration.")
parser.set_defaults(LOAD_CONFIG=None)
parser.add_argument("--output-init-commands", dest="OUTPUT_INIT_COMMANDS", action="store_true", help="Prints the resulting IPC CLI commands built from the command-line arguments.")
parser.set_defaults(OUTPUT_INIT_COMMANDS=None)

_dynamic_initialization_settings = set(["CONFIGS", "PROBE", "PROBE_TAP_CONTROLLERS", "GLOBAL_CONFIGS", "PROBE_CONFIGS", "DEVICE_CONFIGS", "SAVE_CONFIG", "LOAD_CONFIG", "OUTPUT_INIT_COMMANDS"])

def _disable_ipccli_logging():
    import ipccli.cli_logging
    ipc_log = ipccli.cli_logging.getLogger("ipccli.ipc")
    ipc_log.propagate = False
    event_log = ipccli.cli_logging.getLogger("event_display")
    event_log.propagate = False

def _show_configs():
    _disable_ipccli_logging()
    import ipccli
    ipc = ipccli.baseaccess(dynamic_initialization=True)
    scenarios = ipc.init.get_available_scenarios()
    for scenario in scenarios:
        print(scenario)

def _show_probes():
    _disable_ipccli_logging()
    import ipccli
    ipc = ipccli.baseaccess(dynamic_initialization=True)
    probes = ipc.init.get_connected_probes()
    for probe in probes:
        print("{} {}".format(probe[0], probe[1]))

def _show_tap_controllers():
    _disable_ipccli_logging()
    import ipccli
    ipc = ipccli.baseaccess(dynamic_initialization=True)
    tap_controllers = ipc.init.get_supported_tap_controllers()
    devicetype_to_steppings = {}
    for tap_controller in tap_controllers:
        if len(tap_controller) > 2:
            devicetype = "{}-{}".format(tap_controller[0], tap_controller[2])
        else:
            devicetype = tap_controller[0]
        stepping = tap_controller[1]
        if devicetype_to_steppings.get(devicetype) is None:
            devicetype_to_steppings[devicetype] = []
        devicetype_to_steppings[devicetype].append(stepping)
    for devicetype, steppings in devicetype_to_steppings.items():
        print("{}.{}".format(devicetype, ",".join(steppings)))

def _parse_device_configs_string(string):
    config_name_value_pairs = []
    for config in string.split(","):
        (name, value) = config.split(":")
        config_name_value_pairs.append((name.strip(), value.strip()))
    return config_name_value_pairs

def _parse_tap_controller(tap_controller):
    (devicetype, stepping) = tap_controller.split(".")
    subtype = None
    if "-" in devicetype:
        (devicetype, subtype) = devicetype.split("-")
        return (devicetype, stepping, subtype)
    else:
        return (devicetype, stepping)

def _build_dynamic_initialization_commands(args, ipccli_commands):
    ipccli_commands.append("itp=ipc=ipccli.baseaccess(dynamic_initialization=True)")

    if args.LOAD_CONFIG:
        ipccli_commands.append("ipc.init.load_config()")

    # Set global configs
    if args.GLOBAL_CONFIGS is not None:
        config_name_value_pairs = _parse_device_configs_string(args.GLOBAL_CONFIGS)
        for name, value in config_name_value_pairs:
            ipccli_commands.append("ipc.config.root.{} = '{}'".format(name, value))

    # Select configuration scenarios
    if args.CONFIGS is not None:
        scenarios = []
        for scenario_entry in args.CONFIGS:
            for scenario in scenario_entry.split(","):
                scenarios.append(scenario)
        ipccli_commands.append("ipc.init.select_scenarios({})".format(scenarios))

    # Select probe(s)
    if args.PROBE is not None:
        probe_index = 0
        for probe_entry in args.PROBE:

            # Select the probe
            probe_type = probe_entry[0]
            if len(probe_entry) > 1:
                unique_identifier = probe_entry[1]
                ipccli_commands.append("selected_probe = ipc.init.select_probe('{}', '{}')".format(probe_type, unique_identifier))
            else:
                ipccli_commands.append("selected_probe = ipc.init.select_probe('{}')".format(probe_type))

            # Prespecify TAP controllers for the probe
            if args.PROBE_TAP_CONTROLLERS is not None and len(args.PROBE_TAP_CONTROLLERS) > probe_index:
                tap_network_entry = args.PROBE_TAP_CONTROLLERS[probe_index]
                jtag_chain_index = 0
                for jtag_chain_entry in tap_network_entry:
                    tap_controllers = []
                    for tap_controller in jtag_chain_entry.split(","):
                        tap_controllers.append(_parse_tap_controller(tap_controller))
                    ipccli_commands.append("ipc.init.prespecify_tap_controllers(selected_probe, {}, {})".format(jtag_chain_index, tap_controllers))
                    jtag_chain_index += 1

            # Set probe device configs
            if args.PROBE_CONFIGS is not None and len(args.PROBE_CONFIGS) > probe_index:
                probe_configs_entry = args.PROBE_CONFIGS[probe_index]
                config_name_value_pairs = _parse_device_configs_string(probe_configs_entry[0])
                for name, value in config_name_value_pairs:
                    ipccli_commands.append("selected_probe.config.{} = '{}'".format(name, value))
            probe_index += 1

    # If TAP controllers were prespecified without a selected probe
    elif args.PROBE_TAP_CONTROLLERS is not None and len(args.PROBE_TAP_CONTROLLERS) > 0:
        tap_network_entry = args.PROBE_TAP_CONTROLLERS[0]
        jtag_chain_index = 0
        for jtag_chain_entry in tap_network_entry:
            tap_controllers = []
            for tap_controller in jtag_chain_entry.split(","):
                tap_controllers.append(_parse_tap_controller(tap_controller))
            ipccli_commands.append("ipc.init.prespecify_tap_controllers(None, {}, {})".format(jtag_chain_index, tap_controllers))
            jtag_chain_index += 1

    # Prespecify device configs
    if args.DEVICE_CONFIGS is not None:
        i = 0
        while (i + 1) < len(args.DEVICE_CONFIGS):
            device_identifier = args.DEVICE_CONFIGS[i]
            config_name_value_pairs = _parse_device_configs_string(args.DEVICE_CONFIGS[i + 1])
            ipccli_commands.append("ipc.init.prespecify_device_configs('{}', {})".format(device_identifier, config_name_value_pairs))
            i += 2

    if args.SAVE_CONFIG:
        ipccli_commands.append("ipc.init.save_config()")

    ipccli_commands.append("ipc.init.finish()")

def main():
    ipccli_commands = []
    ipccli_commands.append("import ipccli")
    ipccli_commands.append("import ipccli.settings")

    # Parse command-line arguments and append the settings to the start commands
    args = parser.parse_args()

    if args.SHOW_CONFIGS:
        _show_configs()
        return
    elif args.SHOW_PROBES:
        _show_probes()
        return
    elif args.SHOW_TAP_CONTROLLERS:
        _show_tap_controllers()
        return

    use_dynamic_initialization = False
    for setting, value in vars(args).items():
        if value is not None:
            if setting in _dynamic_initialization_settings:
                use_dynamic_initialization = True
            else:
                if isinstance(value, str):
                    ipccli_commands.append("ipccli.settings.{}=\'{}\'".format(setting, value))
                else:
                    ipccli_commands.append("ipccli.settings.{}={}".format(setting, value))

    if use_dynamic_initialization:
        _build_dynamic_initialization_commands(args, ipccli_commands)
    else:
        ipccli_commands.append("itp=ipc=ipccli.baseaccess()")

    if args.OUTPUT_INIT_COMMANDS:
        for command in ipccli_commands:
            print(command)

    try:
        subprocess.call([sys.executable,
            "-m",
            "IPython",
            "-i",
            "-c",
            "\"" + ";".join(ipccli_commands) + "\""
            ])
    except KeyboardInterrupt:
        pass

if __name__ == "__main__":
    main()
