###############################################################################
# Copyright 2014 2017 Intel Corporation.
#
# The source code, information and material ("Material") contained herein is
# owned by Intel Corporation or its suppliers or licensors, and title to such
# Material remains with Intel Corporation or its suppliers or licensors. The
# Material contains proprietary information of Intel or its suppliers and
# licensors. The Material is protected by worldwide copyright laws and treaty
# provisions. No part of the Material may be used, copied, reproduced, modified,
# published, uploaded, posted, transmitted, distributed or disclosed in any way
# without Intel's prior express written permission.
#
# No license under any patent, copyright or other intellectual property rights
# in the Material is granted to or conferred upon you, either expressly, by
# implication, inducement, estoppel or otherwise. Any license under such
#
# intellectual property rights must be express and approved by Intel in writing.
# Unless otherwise agreed by Intel in writing, you may not remove or alter 
# this notice or any other notice embedded in Materials by Intel or Intel's 
# suppliers or licensors in any way.
###############################################################################

import py2ipc

class SharedMemory:
    """
    Manages a region of memory that is shared by the current process and the IPC API
    server process.

    If you are familiar with the IPC API specification, then this class can be best
    described as a wrapper around the AllocateSharedMemory(), GetSharedMemoryAddress(),
    and FreeSharedMemory() functions of the General service.

    Example:
        >>> with SharedMemory(size) as mem:
        ...     assert(len(mem) == size)
        ...     mem[0] = 'a'
        ...     print(mem[:12])
    """
    def __init__(self, size):
        self._size = size
        self._handle = 0
        self._address = 0
        self._buffer = None

    def __enter__(self):
        service = py2ipc.IPC_GetService("General")
        self._handle = service.AllocateSharedMemory(self._size)
        self._address = service.GetSharedMemoryAddress(self._handle)
        self._buffer = service.GetBufferFromAddress(self.address, self._size)
        return self

    def __exit__(self, exc_type, exc_value, traceback):
        service = py2ipc.IPC_GetService("General")
        service.FreeSharedMemory(self._handle)
        self._buffer = None # Prevent further access to the buffer since the memory does not exist anymore
        self._address = 0

    def __copy__(self):
        raise RuntimeError("This object cannot be copied")

    def __deepcopy__(self, dict):
        raise RuntimeError("This object cannot be copied")

    @property
    def address(self):
        """
        Returns the memory address where the shared memory region is located.
        This is provided for informational purposes only.
        Note that this address cannot be used outside of the current process space.
        """
        return self._address

    def __len__(self):
        return len(self._buffer)

    def __getitem__(self, index):
        if isinstance(index, slice):
            return self.__getslice__(index.start, index.stop)
        return self._buffer[index]

    def __setitem__(self, index, value):
        if isinstance(index, slice):
            self.__setslice__(index.start, index.stop, value)
            return
        self._buffer[index] = value

    def __getslice__(self, start, end):
        return self._buffer[start:end]

    def __setslice__(self, start, end, value):
        self._buffer[start:end] = value
