###############################################################################
# Copyright 2014 2017 Intel Corporation.
#
# The source code, information and material ("Material") contained herein is
# owned by Intel Corporation or its suppliers or licensors, and title to such
# Material remains with Intel Corporation or its suppliers or licensors. The
# Material contains proprietary information of Intel or its suppliers and
# licensors. The Material is protected by worldwide copyright laws and treaty
# provisions. No part of the Material may be used, copied, reproduced, modified,
# published, uploaded, posted, transmitted, distributed or disclosed in any way
# without Intel's prior express written permission.
#
# No license under any patent, copyright or other intellectual property rights
# in the Material is granted to or conferred upon you, either expressly, by
# implication, inducement, estoppel or otherwise. Any license under such
#
# intellectual property rights must be express and approved by Intel in writing.
# Unless otherwise agreed by Intel in writing, you may not remove or alter 
# this notice or any other notice embedded in Materials by Intel or Intel's 
# suppliers or licensors in any way.
###############################################################################

'''
This file is to overwrite the default settings
Imports the default settings file that contains all the settings for the bindings
Verify is called to verify that all the names are correct
'''
from __future__ import absolute_import
from __future__ import division
from __future__ import print_function
import sys
import os
_ENV_PREFIX = "IPCCLI"

if sys.platform == 'win32':
    _IPC_PATH = r"C:\Intel\OpenIPC\Bin"
else:
    _IPC_PATH = r"/usr/local/intel/openipc/Bin/"

class Settings(object):

    # LIST FORM
    default_settings = dict(
        OUT_OF_PROCESS = True,
        IPC_LAUNCH_SERVER = False, # Used to specify whether a new IPC API server should be forced (instead of attaching to an existing server)
                                   # IPC_API_SERVER can be a port number to force the server to start on a specify port number
        IPC_API_SERVER = None,     # The URI for the IPC API server to connect to (e.g. "domain.corp.com:1234", "10.2.14.13:1234", "[::1]:1234").
                                   # Assumes that the server is already started and listening on that address. This option takes precendence over OUT_OF_PROCESS
        IPC_CONFIG_FILE = None,    # The configuration file for configuring the IPC API server. When set to None, the server's default configuration is used
        IPC_CONFIG_NAME = None,    # The configuration name for configuring the IPC API server. When set to None, the server's default configuration is used
        IPC_CONFIG_PARAMS = None,  # The dictionary of configuration parameter values to apply to the selected configuration
        IPC_CONFIG_STALLS = None,  # The dictionary of stalls to apply for every connection to the IPC API server
        IPC_LOGGING_PRESET = None, # Sets the logging preset to the name specified after connecting (e.g. "GeneralDebug", "Off", "All")
        VERSION_CHECK = True,
        IPC_PATH = _IPC_PATH,
        EVENT_DISPLAY = True,
        EVENT_TIMESTAMP = True,
        DISPLAY_WAIT_TIME = 1,     # max time before we showed any queued events 
        EVENT_WAIT_TIME = 0.15,    # time to delay display before checking for another event
        DEVICELIST_DEVICETYPE_FILTER = ["VirtualJTAGScanChain"], 
        DEVELOPER = False,
        PROMPT_DISPLAY = False,
        PROMPT_PREFIX = "",
        # This is off because it did not work as reliably, you can turn on
        # for debug but do NOT file sightings against its accuracy
        PROMPT_DISPLAY_USE_EVENTS = False,
        WIZARD = False,
    )

    def __setattr__(self,attr,value):
        if attr not in self.default_settings:
            raise Exception("Unknown setting: %s"%attr)
        object.__setattr__(self, attr, value)

    def __getattr__(self,attr):
        if attr not in self.default_settings:
            raise AttributeError("Unknown setting: %s"%attr)
        try:
            # won't get called if setting was set
            env_var = _ENV_PREFIX+"_"+attr.upper()
            if env_var in os.environ:
                value = os.environ[env_var]
                # convert if we can...start with bool
                if value.lower() in [ 'true','false' ]:
                    value = value.lower() == 'true'
                else:
                    if value.count("."):
                        try:
                            value = float(value)
                        except:
                            pass
                    else:
                        try:
                            value = int( value, 0)
                        except ValueError:
                            pass
            else:
                # not in env variable, save default and return it
                value = self.default_settings[attr]
            # save the value we determined
            object.__setattr__(self, attr, value)
            return value
        except AttributeError as e:
            raise RuntimeError(e)

    def __dir__(self):
        return list(self.default_settings.keys())

    def reset(self):
        """reset settings back to default. This does not impact the fact that
        environment variables override the default"""
        for setting in self.default_settings:
            if setting in self.__dict__:
                del self.__dict__[setting]


import sys
# workaround for the fact that we are about to hijack our module
_ref = sys.modules[__name__]
# replace with our settings object
sys.modules[__name__] = Settings()

