###############################################################################
# Copyright 2014 2017 Intel Corporation.
#
# The source code, information and material ("Material") contained herein is
# owned by Intel Corporation or its suppliers or licensors, and title to such
# Material remains with Intel Corporation or its suppliers or licensors. The
# Material contains proprietary information of Intel or its suppliers and
# licensors. The Material is protected by worldwide copyright laws and treaty
# provisions. No part of the Material may be used, copied, reproduced, modified,
# published, uploaded, posted, transmitted, distributed or disclosed in any way
# without Intel's prior express written permission.
#
# No license under any patent, copyright or other intellectual property rights
# in the Material is granted to or conferred upon you, either expressly, by
# implication, inducement, estoppel or otherwise. Any license under such
#
# intellectual property rights must be express and approved by Intel in writing.
# Unless otherwise agreed by Intel in writing, you may not remove or alter
# this notice or any other notice embedded in Materials by Intel or Intel's
# suppliers or licensors in any way.
###############################################################################

from __future__ import absolute_import
import sys

from . import settings
from .base import baseaccess
from ._py2to3 import *

class Prompt(object):
    """for saving our prompt options"""
    halted     = "]"
    running    = "?"
    no_threads = "!"

    def __init__(self):
        self._connected = True 

    def get_prompt(self):
        # default is unknown/no threads
        prompt = self.no_threads
        try:
            base = baseaccess()
            # IF IPC make sure we are connected before running commands
            if base.environment == "ipc":
                import py2ipc
                if py2ipc.IPC_GetConnectionStatus() == "NotConnected":
                    prompt = self.no_threads
                    return prompt
            if len(base.threads) > 0:
                if base.cv.isrunning:
                    prompt = self.running
                else:
                    prompt  = self.halted
            else:
                # no threads
                prompt = self.no_threads
            # everytime we successfully make it here we update our connected variable
            self._connected = True
        except KeyboardInterrupt as k:
            # ignore these, but we must return a prompt
            return prompt
        except Exception as e:
            if str(e).count("Not_Supported"):
                return self.no_threads
            elif self._connected:
                # seems we may get this even when IPC is alive under
                # certain conditions like boot stall
                #print("")
                #print("The connection to IPC appears to have been lost")
                #print("")
                prompt = self.no_threads
            self._connected = False
        return prompt

####################################################################
# Currently requires baseaccess.cv.isrunning and base.threads
####################################################################
class ReadlinePrompt(bytes, Prompt):
    no_threads = Prompt.no_threads+">> "
    default = ">>> "

    @property
    def halted(self):
        return settings.PROMPT_PREFIX + Prompt.halted+">> "

    @property
    def running(self):
        return settings.PROMPT_PREFIX + Prompt.running+">> "

    @property
    def no_threads(self):
        return settings.PROMPT_PREFIX + Prompt.no_threads+">> "

    def __init__(self):
        self._connected = True
        self.restarts = 0
        self._bytearray = bytearray()

    def get_prompt(self):
        prompt = Prompt.get_prompt(self)
        while len(self._bytearray) > 0:
            self._bytearray.pop()
        # i've had to change what we are extending a few times
        # so doing this to save some future hassel
        self._bytearray.extend(prompt)
        return self

    def decode(self, encoding, decode):
        self.get_prompt()
        return self._bytearray.decode(encoding, decode)

    def __unicode__(self):
        self.get_prompt()
        return unicode(self._bytearray)

    def __str__(self):
        self.get_prompt()
        return str(self._bytearray)

    def __repr__(self):
        self.get_prompt()
        return self.__str__()

    def enable(self):
        # sometimes ps1 is not set yet
        self.default = getattr(sys,"ps1",">>> ")
        sys.ps1 = self

    def disable(self):
        sys.ps1 = self.default

class IPythonPrompt_v3(Prompt):
    #halted     = 'In [\\#]'+Prompt.halted+": "
    #running    = 'In [\\#]'+Prompt.running+": "
    #no_threads = 'In [\\#]'+Prompt.no_threads+": "
    # store default we should restore to if we get disabled
    default = 'In [\\#]: '

    @property
    def halted(self):
        return settings.PROMPT_PREFIX + 'In [\\#]'+Prompt.halted+": "

    @property
    def running(self):
        return settings.PROMPT_PREFIX + 'In [\\#]'+Prompt.running+": "

    @property
    def no_threads(self):
        return settings.PROMPT_PREFIX + 'In [\\#]'+Prompt.no_threads+": "

    def get_prompt(self):
        # this will cause Prompt to get OUR values for
        # halted/running/no_threads
        prompt = Prompt.get_prompt(self)
        ip = get_ipython()
        ip.prompt_manager.in_template = prompt
        for status in ['halted', 'running', 'no_threads']:
            if prompt == getattr(self,status):
                sys.ps1 = getattr(ReadlinePrompt, status)

    def disable(self):
        ip = get_ipython()
        ip.events.unregister("post_run_cell", self.get_prompt)
        # put back default
        ip.prompt_manager.in_template = _PROMPT.default

    def enable(self):
        ip = get_ipython()
        self.default = ip.prompt_manager.in_template
        ip = get_ipython()
        ip.events.register("post_run_cell", self.get_prompt)


class IPythonPrompt_v5(Prompt):
    # we have to postpone class creation to handle different IPython
    # versions
    prompt_class = None

    def get_prompt_class(self):
        if self.prompt_class is not None:
            return self.prompt_class
        from IPython.terminal.prompts import Prompts, Token
        import os
        # we use "new_self" to not confuse with actual self above
        class MyPrompt(Prompts):
            def __init__(new_self, ip):
                Prompts.__init__(new_self, ip)
                new_self.count = 0
                new_self.current_prompt = Prompt.no_threads
                ip.events.register("post_run_cell", new_self._post_execute)

            def in_prompt_tokens(new_self, cli=None):
                new_self.count += 1
                #self.current_prompt = Prompt.get_prompt(self)
                return [(Token.Prompt, settings.PROMPT_PREFIX + "In ["),
                        (Token.PromptNum, str(new_self.shell.execution_count)),
                        (Token.Prompt, ']%s: '%new_self.current_prompt)]
            def _post_execute(new_self):
                # run afer a cell has been run so we can update the prompt
                try:
                    new_self.current_prompt = Prompt.get_prompt(self)
                except:
                    # nice ugly blind try/except so that the prompt
                    # doesnt bring everything down
                    new_self.current_prompt = Prompt.no_threads

        type(self).prompt_class = MyPrompt
        return self.prompt_class

    def enable(self):
        ip = get_ipython()
        self.default = getattr(ip, "prompts", None)
        ip.prompts = self.get_prompt_class()(ip)

    def disable(self):
        post = getattr(get_ipython().prompts,"_post_execute", None)
        if post is not None:
            get_ipython().events.unregister("post_run_cell", post)

        default = get_ipython().prompts = self.default



_PROMPT = None

def enable():
    global _PROMPT
    try:
        __IPYTHON__
        import IPython
        if IPython.__version__[0] in ['3','4']:
            if not isinstance(_PROMPT,IPythonPrompt_v3):
                _PROMPT = IPythonPrompt_v3()
        elif IPython.__version__[0] == '5':
            if not isinstance(_PROMPT,IPythonPrompt_v5):
                _PROMPT = IPythonPrompt_v5()
    except NameError:
        # just use/assume readline, see if we need to create a prompt object
        if not isinstance(_PROMPT,ReadlinePrompt):
            _PROMPT = ReadlinePrompt()

    # only happens if IPython is used but on a version not supported
    if _PROMPT is not None:
        _PROMPT.enable()

def disable():
    if _PROMPT is not None:
        _PROMPT.disable()

