###############################################################################
# Copyright 2014 2017 Intel Corporation.
#
# The source code, information and material ("Material") contained herein is
# owned by Intel Corporation or its suppliers or licensors, and title to such
# Material remains with Intel Corporation or its suppliers or licensors. The
# Material contains proprietary information of Intel or its suppliers and
# licensors. The Material is protected by worldwide copyright laws and treaty
# provisions. No part of the Material may be used, copied, reproduced, modified,
# published, uploaded, posted, transmitted, distributed or disclosed in any way
# without Intel's prior express written permission.
#
# No license under any patent, copyright or other intellectual property rights
# in the Material is granted to or conferred upon you, either expressly, by
# implication, inducement, estoppel or otherwise. Any license under such
#
# intellectual property rights must be express and approved by Intel in writing.
# Unless otherwise agreed by Intel in writing, you may not remove or alter 
# this notice or any other notice embedded in Materials by Intel or Intel's 
# suppliers or licensors in any way.
###############################################################################

from __future__ import absolute_import

from . import cli_logging
from ._py2to3 import *
import logging
import weakref
from functools import partial,update_wrapper,wraps
import types

# node debug should be rare...
_log = cli_logging.getLogger("node")
#
# TODO: need to "protect" objects
#

#
# Node based Control Variables - should never be used directly
#


########################################################################################
# - Nodes are for matching functions to certain deviceTypes
# - Nodes should have a reference to the device they were created for
# - Devices should have a reference to their corresponding node
#
########################################################################################
class NodeBase(object):
    nodetype = "nodebase"
    _node_functions = None
    _node_instance_functions = None

    def __init__(self,device):
        # these are legacy ITPII things where the attributes 
        # were copied from the device
        self.name = device.alias
        self.did  = device.did
        self.device = device
        # add a weakref to the device to this node
        device.node = weakref.proxy(self)
        self._node_instance_functions = set()

    @classmethod
    def _add_did_command(cls, name, function_ptr):
        """
        Used to add a function to a node
        it requires that function_ptr be a function where the first parameter is a 'did'
        
        :param function_ptr: function to add
        :param name:        name to give to the new function
        
        """
        # we purposely do not use the actual function pointer
        # because of a client that replaces/overwrites a function temporarily
        # and then it puts it back
        # already present...
        fname = function_ptr.__name__
        # use passed in name not function name
        # not sure why this causes issues, but we have to always redo this add
        #if name in cls._node_functions:
        #    return
        cls._node_functions.add(name)
        fself = function_ptr.__self__
        def newf(self,*args,**kwargs):
            function = getattr(fself,fname)
            return function(self.device.did,*args,**kwargs)
        newf = update_wrapper(newf,function_ptr)
        setattr(cls,name,newf)

    @classmethod
    def _add_core_group(cls, coregroup):
        for attr_name in ['core', 'thread', 'module']:
            attr = coregroup.lower()+"_"+attr_name + "s"
            # specifying default value here is a trick to make sure lambda doesnt use last value
            # in the iteration
            setattr(cls, attr, property(lambda self, attr_name=attr_name: self._group_maker(find_match=attr_name, coregroup=coregroup)))

    def _add_did_command_to_instance(self, name, function_ptr):
        """
        Used to add a function to a node
        it requires that function_ptr be a function where the first parameter is a 'did'
        
        :param function_ptr: function to add
        :param name:        name to give to the new function
        
        """
        # we purposely do not use the actual function pointer
        # because of a client that replaces/overwrites a function temporarily
        # and then it puts it back
        fname = function_ptr.__name__
        fself = function_ptr.__self__
        self._node_instance_functions.add(name)
        def newf(self,*args,**kwargs):
            function = getattr(fself,fname)
            return function(self.device.did,*args,**kwargs)
        newf = update_wrapper(newf,function_ptr)
        setattr(self, name, types.MethodType(newf, self))

    def __setattr__(self,attr,value):
        #
        # This exists to protect users from clobbering functions in
        # case they think they are dealing with an attribute
        # or just forget the syntax
        #
        try:
            curr = getattr(self,attr)
            if isinstance(curr,(types.MethodType,
                                types.FunctionType,
                                types.LambdaType,
                                )):
                raise Exception("you cannot override existing methods")
            # else...assume it is ok to override for now
        except AttributeError:
            pass
        # if attribut error, then fine to add
        object.__setattr__(self,attr, value)

    def _group_maker(self, find_match, device=None, coregroup=None):
        childlist = []
        toplevel = device is None
        device = device or self.device
        for child in device.children:
            if child.nodetype == find_match:
                if getattr(child, "coregroup", None) is None:
                    childlist.append(child.node)
                elif coregroup is None:
                    childlist.append(child.node)
                elif coregroup == child.coregroup:
                    childlist.append(child.node)
                else:
                    #coregroup present on device and specified, but it doesn't match
                    pass
                #childlist.append(child.node)
            childlist.extend(self._group_maker(find_match, child, coregroup))
        if toplevel:
            return NodeGroup(childlist)
        else:
            return childlist

    unknowns = property(lambda self: self._group_maker("unknown"))
    #roots = property(lambda self: self._group_maker("root"))
    #domains = property(lambda self: self._group_maker("domain"))
    threads = property(lambda self: self._group_maker("thread", coregroup="GPC"))
    cores = property(lambda self: self._group_maker("core", coregroup="GPC"))
    coregroups = property(lambda self: self._group_maker("groupcore"))
    uncores = property(lambda self: self._group_maker("uncore"))
    chipsets = property(lambda self: self._group_maker("chipset"))
    boxes = property(lambda self: self._group_maker("box"))
    #dies = property(lambda self: self._group_maker("die"))
    #packages = property(lambda self: self._group_maker("package"))
    modules = property(lambda self: self._group_maker("module", coregroup="GPC"))
    #scanchains = property(lambda self: self._group_maker("scanchain"))
    #interfaceports = property(lambda self: self._group_maker("interfaceport"))
    #ic2buses = property(lambda self: self._group_maker("ic2bus"))
    #debugports = property(lambda self: self._group_maker("debugport"))
    #pinsinterfaces = property(lambda self: self._group_maker("pinsinterface"))
    virtual_scanchains = property(lambda self: self._group_maker("virtual_scanchain"))
    
    @property
    def children(self):
        childlist = []
        for child in self.device.children:
            childlist.append(child.node)
        return NodeGroup(childlist)
    
    @property
    def parents(self):
        parentlist = []
        for parent in self.device.parents:
            parentlist.append(parent.node)
        return NodeGroup(parentlist)
    
    @property
    def parent(self):
        return NodeGroup([self.device.parent.node])

    def _get_added_functions(self):
        return self._node_instance_functions.union(self._node_functions)

#
# Create One Node class for whatever the NodeType requested is    
#
_nodeTypes = {}
def GetNodeClass(nodetype):
    """
    Dynamically Creates (if has not been created) a new Node class that
    inherits from NodeBase. It is expected that commands will be added to the
    new node class that make the most sense for that new node
    """
    if nodetype in _nodeTypes:
        return _nodeTypes[nodetype]
        
    # new derived class but it should have its own name 
    new_node= type(nodetype+"_Node",
                   (NodeBase,),
                    dict(
                         nodetype=nodetype,
                         _node_functions=set(),
                         )
                    )
    new_node.nodetype = nodetype
    _nodeTypes[nodetype] = new_node
    return new_node
    
def GetNodeClasses():
    """Get all of the node classes
    Returns a dictionary of the form:
        nodetype=nodetypeClass 
    """
    # this public function is definitely used by some tools that
    # override the CLI
    return _nodeTypes

#############################################
# Used to call all a function on multiple nodes 
# that are similar       
#############################################
# We will probably need some more robust error checking
# for cases when a NodeGroup gets built when it should not have
class NodeGroup(object):
    def __init__(self,nodelist=None,pathlist=None):
        """
        nodelist - list of objects inside this group
        pathlist - list to describe the path of the objects in this group
        """
        # no sub provided start new list
        if nodelist==None: 
            nodelist = []
        # store either new list or one that was passed in
        self._nodelist = nodelist
        
        ##############################################################
        # if we did not get any paths, but we did get a nodelist, then
        # update pathlist to use the alias for the various nodes
        ##############################################################
        if pathlist==None:
            # store off alias for pathlist
            self._pathlist = []
            if len(nodelist)>0 and isinstance(nodelist[0],NodeBase):
                for n in nodelist:
                    self._pathlist.append( n.device.alias)
        else:
            self._pathlist = pathlist
            
        # other commands we should probably implement
        # cmds = ['append','count','extend','index','insert','pop','remove','reverse','sort']

    def count(self,value): return self._nodelist.count(value)
    def index(self,value): return self._nodelist.index(value)
             
    def append(self,node):
        """append a node object to the list of nodes in this group"""
        if not isinstance(node,NodeBase):
            raise TypeError("node Must be of a derivative of NodeBase")
        self._nodelist.append(node)
        self._pathlist.append(node.device.alias)
        
    def remove(self,node):
        """node object or a string with device alias for the node that should be removed"""
        if isinstance(node,str):
            node_alias = node
        elif isinstance(node,NodeBase):
            node_alias = node.device.alias
        else:
            raise ValueError("Unsupported node type for NodeGroup.remove") 
        for n,node in enumerate(self._nodelist):
            if node.device.alias == node_alias:
                break
        else:
            _log.debug("NodeGroup.remove called with {0} but not found".format(node))
            return
        # we should have a legit "n" here
        self._nodelist = self._nodelist[:n] + self._nodelist[n+1:]
        self._pathlist = self._pathlist[:n] + self._pathlist[n+1:]
        
    def __dir__(self):
        dirdata = []
        if len(self._nodelist)>0:
            dirdata = dir(self._nodelist[0])
        else:
            dirdata = []
        for node in self._nodelist[1:]:
            # reduce to intersection of these two
            dirdata = set( dir(node) ) & set( dirdata)        
        return list(dirdata)
        
    def __getattr__(self,attr):
        nextsublist = []
        nextpathlist = []
        if len(self._nodelist) == 0:
            raise AttributeError("cannot find attribute on empty group")
        for index,sub in enumerate(self._nodelist):
            val = getattr(sub,attr)
            # add next attribute to new group and keep up with path
            if type(val)==NodeGroup:
                for nindex,nsub in enumerate(val._nodelist):
                    nextsublist.append(nsub)
                    nextpathlist.append( val._pathlist[nindex] + "."+attr  )
            else:
                nextsublist.append( val )
                nextpathlist.append( self._pathlist[index] + "."+attr )
        # update our sub list
        nextGroup = NodeGroup( nodelist=nextsublist, pathlist = nextpathlist )
        return nextGroup
    
    def __setattr__(self,attr,value):
        # error checking?
        if attr.startswith("_"):
            return object.__setattr__(self,attr,value)
        
        for index,sub in enumerate(self._nodelist):
            setattr(sub,attr,value)
    
    def __getitem__(self,item):
        #
        # If it is a string, assume it is alias and return the appropriate device
        #
        if isinstance(item,basestring):
            for node in self._nodelist:
                device = getattr(node,"device",None)
                if device==None: continue
                if device.alias == item or device.alias == item.upper():
                    return node
            else: # yes, else on for means we didn't return/break
                print("Error could not find device with alias: {0}".format(item))
        else: # assume it is an int
            return self._nodelist[item]

    def __call__(self,*args,**kwargs):
        # CALL for each node type we should call the function we need and return list of results
        # for keeping our return codes
        retlist = []
        for mysub in self._nodelist:
            # special case search...just return results for the first one...
            try: 
                ret = mysub(*args,**kwargs)
            except Exception as e:
                ret = e
            retlist.append( ret )
        
        # if everything is None, then just return None
        # instead of a list of None
        count = 0
        for r in retlist:
            if r is None:
                count += 1
        if count == len(retlist):
            return
        
        return NodeGroup( retlist, self._pathlist )

    def __repr__(self):        
        if len(self._nodelist)==0:
            return "<Empty NodeGroup>"
        out = []
        for i in range(len(self._nodelist)):
            out.append("%s - %s"%(self._pathlist[i],repr(self._nodelist[i])))
        return "\n".join(out)

    def __str__(self):
        return repr(self)
        
    ##### List functions...
    def __len__(self):
        return len(self._nodelist)
        
    def __getslice__(self, i, j):
        if j<i:
            raise ValueError("Must specify slice with lower:upper")
        return NodeGroup( self._nodelist[i:j] )

    def __iter__(self):
        # reset this ?
        return self._nodelist.__iter__()

    def __lt__(self,other): 
        if type(other)!=list:
            for sub in self._nodelist:
                if not (sub<other):
                    return False
            return True
        else: # just do default behavior
            return self._nodelist<other

    def __le__(self,other): 
        if type(other)!=list:
            for sub in self._nodelist:
                if not (sub<=other):
                    return False
            return True
        else: # just do default behavior
            return self._nodelist<=other

    def __eq__(self,other):
        if type(other)!=list:
            for sub in self._nodelist:
                if not (sub==other):
                    return False
            return True
        else: # just do default behavior
            return self._nodelist==other

    def __ne__(self,other):
        # dont be foold in to thinking this is just the
        # "not of __eq__"
        if type(other)!=list:
            for sub in self._nodelist:
                if not (sub!=other):
                    return False
            return True
        else: # just do default behavior
            return self._nodelist!=other

    def __gt__(self,other):
        if type(other)!=list:
            for sub in self._nodelist:
                if not (sub>other):
                    return False
            return True
        else: # just do default behavior
            return self._nodelist>other

    def __ge__(self,other):
        if type(other)!=list:
            for sub in self._nodelist:
                if not (sub>=other):
                    return False
            return True
        else: # just do default behavior
            return self._nodelist>=other
        
        
        
        
        
