###############################################################################
# Copyright 2014 2017 Intel Corporation.
#
# The source code, information and material ("Material") contained herein is
# owned by Intel Corporation or its suppliers or licensors, and title to such
# Material remains with Intel Corporation or its suppliers or licensors. The
# Material contains proprietary information of Intel or its suppliers and
# licensors. The Material is protected by worldwide copyright laws and treaty
# provisions. No part of the Material may be used, copied, reproduced, modified,
# published, uploaded, posted, transmitted, distributed or disclosed in any way
# without Intel's prior express written permission.
#
# No license under any patent, copyright or other intellectual property rights
# in the Material is granted to or conferred upon you, either expressly, by
# implication, inducement, estoppel or otherwise. Any license under such
# intellectual property rights must be express and approved by Intel in writing.
#
# Unless otherwise agreed by Intel in writing, you may not remove or alter
# this notice or any other notice embedded in Materials by Intel or Intel's
# suppliers or licensors in any way.
###############################################################################

# when the master acquires resource, it is unavailable
# when a child/thread acquires the resource, other children can
# also acquire the resource
# master_acquire - needed before transport goes down
#       - sets resource available to false
# master_release - done when resource returns
#       - sets resource available to True
import threading
import time

DEBUG = False

class BlockForResource(object):
    # for each reqeusted resource return the same
    # blocking object
    instances = {}
    def __init__(self):
        # start at 0, which means NO one is using
        # and means that it is still available
        # ---
        # children increment/decriment count and can use it if it stays 0 or above
        # master decrements count and gets control if count becomes less than 0
        # this is set to 1, so that master cannot acquire the resource
        # until AFTER an initialize has been called
        self._count = 1
        self._initialized = False
        # yes, this should be FALSE so that we can know if the resource started
        # off as unavailalbe, but becomes available
        # between our time of registering the handler and initializing this object
        self._resource_available = False
        # used to notify children when resource available is false
        # this is used to notify/wake them up
        self._lock_for_count = threading.Condition()
        # needs to be any event so that any child thread can do the unlock
        self._lock_for_master = threading.Event()
        # make sure we start with clear
        self._lock_for_master.clear()
        # so that we can release waiting children without them polling the count
        self._lock_for_child =  threading.Event()

    @classmethod
    def get_blocker(cls, resource_name):
        """return a new or existing blocker based on the resource name"""
        blocker = cls.instances.get(resource_name, None)
        if blocker is None:
            blocker = cls()
            cls.instances[resource_name] = blocker
        return blocker

    def initialize(self, ownership):
        """
        Initialize who started off as owning the resource.

        This MUST be called or master will never be able to get the resource

        Args:
            ownership: "child" or "master"

        Returns:

        """
        # nothing to do if already initialized
        if self._initialized:
            return
        if ownership not in ['child','master']:
            raise ValueError("unexpected ownership: %s"%ownership)
        self._lock_for_count.acquire()
        # this could be simplified perhaps, but its nice to know all 4 possible
        # scenarios here...
        if ownership == "child":
            if self._count > 0:
                # master doesn't have a pending request yet, it is safe to set count to be correct
                # value of 0
                self._count = 0
                self._resource_available = True
            else:
                # master has a pending request to get the resource (but is currently blocked
                # due to count's default being 1...it is waiting for an awake event
                # we set to -1 so some child can't come grab the resource
                self.count = -1
                self._resource_available = False
        # master owned the resource when we registered handler
        else:
            # this means we recieved an "arriving" request between the time we got here
            # registered the handler and got here
            if self._count > 0 and self._resource_available:
                # reset the count to be appropriate
                self._count = 0
            else:
                # could be no event occurred, count will be >0 and resource_available = False
                #    this mean master keeps owning the resource
                # could be recieved both an arrive (release) AND an (acquire), count will == 0
                #    this means master is blocked and just need official ownership
                self._count = -1
        self._initialized = True
        self._lock_for_count.release()
        # it should safe to wake up any waiting masters
        self._lock_for_master.set()

    def master_acquire(self):
        """
        waits till no one else is using
        """
        # this could take multiple attempts depending on
        # how the locks get acquires
        if DEBUG: tnum = threading.current_thread().getName()
        if DEBUG: print("%s: master_acquire: lock_for_count: acquiring..."%tnum)
        self._lock_for_count.acquire()
        if DEBUG: print("%s: master_acquire: lock_for_count: acquired..."%tnum)
        # dont decrement if it happens to be already -1 (due to initialization)
        # yes...this puts code back to only supporting a single master
        if self._count >= 0:
            self._count -= 1
        master_locked = False
        while True:
            if DEBUG: print("%s: master_acquire:  count=%d..."%(tnum, self._count))
            # a child already has the resource, wait...
            if self._count >= 0:
                # make sure we dont own the master lock...
                #if master_locked:
                #    if DEBUG: print("Master: lock_for_master: releasing...")
                #    self._lock_for_master.release()
                #    master_locked = False
                # release lock count for children and wait for one
                # of them to tell us they are done
                if DEBUG: print("%s: master_acquire: lock_for_count: releasing..."%tnum)
                self._lock_for_count.release()
                #self._lock_for_count.wait()
                if DEBUG: print("%s: master_acquire: lock_for_master: acquiring..."%tnum)
                master_locked = self._lock_for_master.wait()
                if DEBUG: print("%s: master_acquire: lock_for_master: acquired..."%tnum)
                if DEBUG: print("%s: master_acquire: lock_for_count: acquiring..."%tnum)
                self._lock_for_count.acquire()
                if DEBUG: print("%s: master_acquire: lock_for_count: acquired..."%tnum)
                # go back and check again...
                continue
            else:
                # should be easy to get this since we own the count also...
                # master_locked = self._lock_for_master.acquire()
                # we own the lock count and it is -1 so no other child is waiting
                if DEBUG: print("%s: Master_acquire: count==%d, resource=False..."%(tnum, self._count))
                # leave lock count acuquired
                # looks like we safely have the resource...
                self._resource_available = False
                # release lock on count in case another master needs to also acquire
                self._lock_for_count.release()
                break
        # this should leave us with the master lock AND the lock for count
        return

    def master_release(self):
        """
        waits till no one else is using
        """
        # mark resource as available
        if DEBUG:
            tnum = threading.current_thread().getName()
        if DEBUG: print("%s: master_release: acquiring count..."%(tnum))
        self._lock_for_count.acquire()
        # if release is called, but we dont actually have the resource
        # this can happen due to handler being registered while we transport
        # was disabled
        if self._count >= 0:
            # make sure it is flagged as resource is available...
            self._resource_available = True
            self._lock_for_count.release()
            return
        self._count += 1 # should be 0 now...
        if DEBUG: print("%s: master_release: acquiried count: %d" % (tnum, self._count))
        if self._count>=0:
            self._resource_available = True
            # free the children to acquire the resource
            if DEBUG: print("%s: master_release: notifying and releasing lock"%tnum)
            #self._lock_for_master.release()
            self._lock_for_count.notifyAll()
            self._lock_for_count.release()
            # let any waiting children know that we are done...
            # self._lock_for_child.set()
        else:
            if DEBUG: print("%s: master_release: another master still has the resource"%tnum)
            self._lock_for_count.release()

    def child_acquire(self):
        """
        declare intent to use the resource
        """
        if DEBUG:
            tnum = threading.current_thread().getName()
        while True:
            if DEBUG: print("%s: child_acquire: lock_for_count acquiring...." % tnum)
            self._lock_for_count.acquire()
            if DEBUG:
                print("%s: child_acquire: lock_for_count: acquired..."%tnum)
                print("%s: child_acquire: count=%d"%(tnum, self._count))
            if self._count < 0:
                if DEBUG: print("%s: child_acquire: waiting for master to release......."%tnum)
                self._lock_for_count.wait() # <- when we return for this we have the lock....
                # we do release since we try to acquire at top of thread to know the count...and
                # for some reason the acquire didn't seem to behave like a re-entrant one...
                self._lock_for_count.release()
                if DEBUG: print("%s: child_acquire: received master release......." % tnum)
                #self._lock_for_count.release()
                continue
                # if DEBUG: print("Child_acquire: master released, will attempt to lock again...")
            elif self._count == 0:
                # we are the first to get the resource
                # lock out the master, only the first
                if DEBUG: print("%s: child_acquire: first child, taking master lock..."%tnum)
                #self._lock_for_master.acquire()
                self._lock_for_master.clear()
                self._count += 1
                #self._lock_for_count.release()
                self._lock_for_count.notify()
                self._lock_for_count.release()
                break
            else:
                if DEBUG: print("%s: child_acquire: next child incrementing count and releasing"%tnum)
                self._count += 1
                self._lock_for_count.release()
                break

    def child_release(self):
        """
        done with the source
        """
        if DEBUG:
            tnum = threading.current_thread().getName()
            print("%s: child_release: lock_for_count: acquiring..."%tnum)
        self._lock_for_count.acquire()
        if DEBUG: print("%s: child_release: lock_for_count: acquired..."%tnum)
        self._count -= 1
        if DEBUG: print("%s: child_release:count=%d"%(tnum, self._count))
        # we no more threads have the resource, release it
        if self._count <= 0:
            # if count <= 0 then we are last thread and there is at least one master waiting..
            if DEBUG: print("%s: child_release: Releasing master..."%tnum)
            self._lock_for_master.set()
        # now release the count...
        if DEBUG: print("%s: child_release: lock_for_count: releasing..."%tnum)
        self._lock_for_count.release()

