###############################################################################
# Copyright 2014 2017 Intel Corporation.
#
# The source code, information and material ("Material") contained herein is
# owned by Intel Corporation or its suppliers or licensors, and title to such
# Material remains with Intel Corporation or its suppliers or licensors. The
# Material contains proprietary information of Intel or its suppliers and
# licensors. The Material is protected by worldwide copyright laws and treaty
# provisions. No part of the Material may be used, copied, reproduced, modified,
# published, uploaded, posted, transmitted, distributed or disclosed in any way
# without Intel's prior express written permission.
#
# No license under any patent, copyright or other intellectual property rights
# in the Material is granted to or conferred upon you, either expressly, by
# implication, inducement, estoppel or otherwise. Any license under such
# intellectual property rights must be express and approved by Intel in writing.
#
# Unless otherwise agreed by Intel in writing, you may not remove or alter
# this notice or any other notice embedded in Materials by Intel or Intel's
# suppliers or licensors in any way.
###############################################################################

"""
Contains code related to some events that can cause trace and/or the probe to go away
"""
# dont import ipccli up here or we'll have a circular import
import py2ipc
from .reverse_semaphore import BlockForResource as _BlockForResource
import threading

_RESOURCE_NAME = "debugport_pm_"

def _pm_target_event_handler(info, self):
    print("handler called....")
    if info.eventType in ['Transport'] and info.currentState == "Leaving":
        resource = _BlockForResource.get_blocker(_RESOURCE_NAME + str(info.debugportId))
        resource.master_acquire()
    elif info.eventType in ['Transport'] and info.currentState == "Arriving":
        resource = _BlockForResource.get_blocker(_RESOURCE_NAME + str(info.debugportId))
        resource.master_release()
    # not an event for this handler

class BlockTransportFromLeaving():
    """
    This locker helps prevent cases where the debug port could be going away. The expected usage
    is something like:

    # debug port going away will not be allowed during this portion/with statement
    with BlockTransportFromLeaving():
        ipc.irdrscan()
        ...other critical code

    """
    # used to know if we need to subscribe our pm event handler
    _init_key = "blocker_registered"
    # keep count of instances using the handler, if no instances are using
    # the handler on exit, then unreigster it
    _lock = threading.RLock()
    _instance_count = 0

    def __init__ (self, dids=None):
        self.dids = dids
        # we have to wait till the "enter" to fill this in

    def __enter__(self):
        # if no other instances are inside the instance, then
        import ipccli
        ipc = ipccli.baseaccess()
        with self._lock:
            if self._instance_count == 0:
                ipc.events.SubscribeTargetEvents(_pm_target_event_handler, None)
                self.__class__._instance_count += 1

        did_list = []
        dids = self.dids
        dev_srv = py2ipc.IPC_GetService("Device")
        if dids is None:
            domains = dev_srv.GetTargetDomainDeviceIds()
            for domain in domains:
                dids = dev_srv.GetDescendantIdsForType(domain, "Debugport", False)
                did_list.extend([d.deviceId for d in dids])
            pass
        elif not isinstance(dids, (list,tuple)):
            did_list = [dids]
        else:
            did_list = dids


        self.resources = []
        for dbgport in did_list:
            # NEED to get this string from "events" file or some other file
            blocker = _BlockForResource.get_blocker(_RESOURCE_NAME + str(dbgport))
            enabled = dev_srv.GetDeviceProperty(dbgport, "enabled")
            if enabled in ['true','True', 1, True]:
                # print("initialized child as owner: %s"%enabled)
                blocker.initialize("child")
            else:
                # print("initialized master as owner")
                blocker.initialize("master")
            self.resources.append(blocker)
        if len(self.resources) == 0:
            raise RuntimeError("Missing a debug port?")
        # get semapore per did ?
        for resource in self.resources:
            resource.child_acquire()

    def __exit__(self, exc_type, exc_value, traceback):
        # get semapore per did ?
        import ipccli
        for resource in self.resources:
            resource.child_release()
            # DEBUG
            # import time;time.sleep(1)
            # print("Rcount: %d"%resource._count)
        return
        # keep in mind handler could be running right now...
        ipc = ipccli.baseaccess()
        with self._lock:
            self.__class__._instance_count -= 1
            if len(self._instance_count) < 0:
                raise RuntimeError("some thread did not exit cleanly")
            if self._instance_count == 0:
                ipc.events.UnsubscribeTargetEvents(_pm_target_event_handler)
                # remove any resource blockers that already exist....
                # that way we start fresh next time...
                for rname in list(_BlockForResource.instances.keys()):
                    if rname.startswith(_RESOURCE_NAME):
                        del _BlockForResource.instances[rname]

def test():
    import ipccli
    ipc = ipccli.baseaccess()
    ipc.SubscribeTargetEvents(_pm_target_event_handler, None)
    pass

"""
Scratchpad:
test...
ipc.devs.debugport0.device_action.CausePmcHandshake('S0i2_0Entry,PlugCycle(500)',)
ipc.devs.debugport0.device_action.CausePmcHandshake('S0ixToS0,PlugCycle(500)',)

import threading
t = threading.Thread(target=ipc.devs.debugport0.device_action.CausePmcHandshake, args=('S0i2_0Entry,PlugCycle(5000)',))
t.start()
time.sleep(0.5)
print("Enabled: %s"%str(ipc.debugports[0].device.isenabled))

"""

if __name__ == "__main__":
    test()

