###############################################################################
# Copyright 2018 Intel Corporation.
#
# The source code, information and material ("Material") contained herein is
# owned by Intel Corporation or its suppliers or licensors, and title to such
# Material remains with Intel Corporation or its suppliers or licensors. The
# Material contains proprietary information of Intel or its suppliers and
# licensors. The Material is protected by worldwide copyright laws and treaty
# provisions. No part of the Material may be used, copied, reproduced,
# modified, published, uploaded, posted, transmitted, distributed or disclosed
# in any way without Intel's prior express written permission.
#
# No license under any patent, copyright or other intellectual property rights
# in the Material is granted to or conferred upon you, either expressly, by
# implication, inducement, estoppel or otherwise. Any license under such
# intellectual property rights must be express and approved by Intel in
# writing.
#
# Unless otherwise agreed by Intel in writing, you may not remove or alter
# this notice or any other notice embedded in Materials by Intel or Intel's
# suppliers or licensors in any way.
###############################################################################
from __future__ import absolute_import, print_function

from collections import namedtuple, OrderedDict
from time import sleep

try:
    import readline
except ImportError:
    readline = None

import py2ipc

from .._py2to3 import *


class Completer(object):
    """ Auto complete class for readline to complete based off list of strings.
    """

    def __init__(self, list):
        self._list = list

    def complete(self, text, state):
        buffer = readline.get_line_buffer()

        if not buffer:
            return [c + ' ' for c in self._list][state]
        cmd = buffer.lower()
        results = [c + ' ' for c in self._list if c.lower().startswith(cmd)]
        results += [None]
        return results[state]


# region Common user input functions
_yes_values = ['y', 'yes', 'true', '1']
_no_values = ['n', 'no', 'false', '0']


def _ask_yes_no(question):
    """ Asks the user a yes or no question and returns the answer as a bool.
    """
    while True:
        answer = input('\n' + question + ' [y/n]: ').lower()
        if answer in _yes_values:
            return True
        if answer in _no_values:
            return False
        print('Invalid input.')


def _ask_multiple_choice_autocomplete(question, choices, cancel_option=None):
    """ Asks the user a multiple choice question using autocomplete and returns
    the answer. """
    if readline is not None:
        comp = Completer(choices)
        readline.parse_and_bind('tab: complete')
        readline.set_completer(comp.complete)
    question += ":"
    while True:
        answer = _ask_string(question, cancel_option).strip().lower()
        try:
            return next(x for x in choices if x.lower() == answer)
        except StopIteration:
            pass
        if answer == cancel_option.lower():
            return cancel_option
        if readline is not None:
            print('Invalid input. Try using <TAB> completion.')
        else:
            print('Invalid input.')


def _ask_multiple_choice(question, choices, cancel_option=None):
    """ Asks the user a multiple choice question and returns the answer. """

    # If the number of choices is too large for a single digit, we will
    # use the autocomplete version to prompt the user.
    if len(choices) >= 9 and readline != None:
        return _ask_multiple_choice_autocomplete(question, choices,
                                                 cancel_option)

    if cancel_option:
        choices_ext = [cancel_option] + choices
        start_index = 0
    else:
        choices_ext = choices
        start_index = 1

    while True:
        count = start_index
        question_str = ('\n' + question) if question else ''
        options = []
        list_pad = len(str(len(choices_ext)))
        for choice in choices_ext:
            question_str += '\n  {0:>{1}}) {2}'.format(count, list_pad, choice)
            options.append(str(count))
            count = count + 1
        question_str += '\nEnter Selection: '
        answer = input(question_str).lower().strip()
        try:
            return next(x for x in choices_ext if x.lower() == answer)
        except StopIteration:
            pass
        if answer in options:
            return choices_ext[int(answer) - start_index]
        print('Invalid input.')


def _ask_string(question, cancel_value=None):
    """ Gets user input, or the cancel_value if the user enters an empty string
    """
    result = input('\n' + question + ' ')
    return result if result else cancel_value


# endregion


class Wizard(object):
    """ This class guides the user through the dynamic initialization of the
    IPC software. """

    def __init__(self):
        self._init_service = py2ipc.IPC_GetService('Initialization')
        self._devicecontrol_service = py2ipc.IPC_GetService('DeviceControl')
        self._device_service = py2ipc.IPC_GetService('Device')
        self._total_selected_probes = []
        self._selected_scenarios = []

    def _ask_how_to_configure(self):
        saved_config_exists = \
            self._init_service.IsSavedConfigurationAvailable()

        options = ['Use saved config'] if saved_config_exists else []
        options += [
            'Use auto config', 'Use connected probes', 'Manually specify probes'
        ]

        answer = _ask_multiple_choice('How would you like to configure?',
                                      options)
        if answer == 'Use saved config':
            self._init_service.RestoreConfiguration()
        elif answer == 'Use connected probes':
            # If finding connected probes is fast enough, we can
            # do that first, then just list the connected ones here
            self._ask_select_connected_probes()
        elif answer == 'Manually specify probes':
            # If there are no connected probes, and no saved config
            # we could jump into manually specifying a probe
            self._ask_add_manual_probes(True)
        elif answer == 'Use auto config':
            self._ask_finish_or_revise_configuration(ask_add_probe=False)

    def _ask_select_connected_probes(self):
        connected_probes = self._init_service.GetConnectedProbes()

        if not connected_probes:
            print('\nNo connected probes found.')
            self._ask_add_manual_probes()
            return

        ask = True
        while ask:
            connected_probe_types = [x.type for x in connected_probes]
            print('Selected Probes: {}'.format([
                x.type for x in self._total_selected_probes
            ] if self._total_selected_probes else 'None'))
            result = _ask_multiple_choice(
                'Which probe would you like to select?', connected_probe_types,
                'Done')
            if result == 'Done':
                ask = False
            else:
                selected = next(
                    x for x in connected_probes if x.type == result)
                self._total_selected_probes.append(selected)
                connected_probes.remove(selected)
                self._ask_configure_probe(selected)
                if not connected_probes:
                    ask = False
        self._ask_add_manual_probes()

    def _ask_add_manual_probes(self, ask_first=False):
        self._ask_finish_or_revise_configuration(ask_add_probe=True, ask_add_probe_first=ask_first)

    def _ask_configure_probe(self, probe):
        debug_port_id = self._init_service.SelectProbe(probe)

        ask = True
        DeviceConfigDetails = namedtuple('DeviceConfigDetails',
                                         'name value description options')
        while ask:
            device_config_details = []
            config_names = \
                self._devicecontrol_service.GetDeviceConfigNames(
                    debug_port_id)
            config_names = [x for x in config_names if not x.startswith('_')]
            config_names.sort()
            for config_name in config_names:
                try:
                    config_value = self._devicecontrol_service.GetDeviceConfig(
                        debug_port_id, config_name)
                except py2ipc.IPC_Error:
                    config_value = ''
                config_description = \
                    self._devicecontrol_service.GetDeviceConfigDescription(
                        debug_port_id, config_name)
                config_options = \
                    self._devicecontrol_service.GetDeviceConfigOptions(
                        debug_port_id, config_name)
                device_config_details.append(
                    DeviceConfigDetails(config_name, config_value,
                                        config_description, config_options))

            if len(device_config_details) > 0:
                max_config_name_length = max(
                    [len(x.name) for x in device_config_details])
                print('\nProbe {}'.format(probe.uniqueIdentifier))
                for x in device_config_details:
                    print('  {0: <{width}} = {1}'.format(
                        x.name, x.value, width=max_config_name_length))
                options = [x.name for x in device_config_details]

                result = _ask_multiple_choice(
                    'Which probe setting would you '
                    'like to modify?', options, 'None')
                if result == 'None':
                    ask = False
                else:
                    item = next(
                        x for x in device_config_details if x.name == result)
                    self._ask_set_config(debug_port_id, item)
        try:
            self._init_service.InitializeProbe(debug_port_id)
        except py2ipc.IPC_Error as e:
            print("An Error occurred while initializing this probe. Error: " +
                  str(e))
            self._total_selected_probes.remove(probe)
        else:
            sleep(1)  # small wait for messages to be printed
            self._ask_configure_interfaces(debug_port_id)

    def _ask_configure_interfaces(self, debug_port_id):

        children = []

        children_ids = self._device_service.GetChildrenIds(debug_port_id)
        for id in children_ids:
            children.append(self._device_service.GetDevice(id))

        ask = len(children) > 0
        interface_options = [x.identifier for x in children]
        while ask:
            result = _ask_multiple_choice('Select an interface to configure',
                                          interface_options, 'Done')
            if result == 'Done':
                ask = False
            else:
                item = next(x for x in children if x.identifier == result)
                self._ask_configure_interface(item)
        self._ask_detect_taps(children)

    def _ask_configure_interface(self, interface):

        DeviceConfigDetails = namedtuple('DeviceConfigDetails',
                                         'name value description options')
        ask = True
        while ask:
            device_config_details = []
            config_names = \
                self._devicecontrol_service.GetDeviceConfigNames(
                    interface.deviceId)
            config_names = [x for x in config_names if not x.startswith('_')]
            config_names.sort()
            for config_name in config_names:
                try:
                    config_value = self._devicecontrol_service.GetDeviceConfig(
                        interface.deviceId, config_name)
                except py2ipc.IPC_Error:
                    config_value = ''
                config_description = \
                    self._devicecontrol_service.GetDeviceConfigDescription(
                        interface.deviceId, config_name)
                config_options = \
                    self._devicecontrol_service.GetDeviceConfigOptions(
                        interface.deviceId, config_name)
                device_config_details.append(
                    DeviceConfigDetails(config_name, config_value,
                                        config_description, config_options))

            if len(device_config_details) > 0:
                max_config_name_length = max(
                    [len(x[0]) for x in device_config_details])
                print('\nInterface {}'.format(interface.identifier))
                for x in device_config_details:
                    print('  {0: <{width}} = {1}'.format(
                        x.name, x.value, width=max_config_name_length))
                options = [x.name for x in device_config_details]

                result = _ask_multiple_choice(
                    'Select an interface setting to modify', options, 'None')
                if result == 'None':
                    ask = False
                else:
                    item = next(
                        x for x in device_config_details if x.name == result)
                    self._ask_set_config(interface.deviceId, item)

    def _tap_to_str(self, tap):
        if tap.deviceType:
            return '{}{}{}_{}'.format(tap.deviceType, '-' if tap.deviceSubType
                                      else '', tap.deviceSubType, tap.stepping)
        else:
            return 'UnknownTap_IR{}'.format(tap.irLength)

    def _ask_detect_taps(self, interfaces):
        result = _ask_yes_no('Would you like to specify tap networks?')

        jtag_chains = [
            x for x in interfaces if x.deviceType == 'JTAGScanChain'
        ]
        print()
        if result:
            self._ask_specify_tapnetwork(jtag_chains)

    def _ask_specify_tapnetwork(self, jtag_chains):
        tap_controller_options = \
            self._init_service.GetSupportedTapControllers()

        # Add Unknown taps options for various IR lengths
        for i in range(1, 12):
            unknown_tap = py2ipc.IPC_Types.IPC_TapController()
            unknown_tap._irLength = i
            tap_controller_options.append(unknown_tap)

        tap_options = [self._tap_to_str(x) for x in tap_controller_options]

        jtag_chain_names = [x.identifier for x in jtag_chains]

        jtag_chain_to_selected_taps = {x: [] for x in jtag_chains}

        ask = True
        while ask:
            print()
            for jtag_chain in jtag_chains:
                print(jtag_chain.identifier)
                for tap in jtag_chain_to_selected_taps[jtag_chain]:
                    print('  ' + self._tap_to_str(tap))

            result = _ask_multiple_choice(
                'Which jtag chain would you like to add a tap network to?',
                jtag_chain_names, 'Done')
            if result == 'Done':
                break

            jtag_chain = next(x for x in jtag_chains if x.identifier == result)
            selected_taps = jtag_chain_to_selected_taps[jtag_chain]

            result = _ask_multiple_choice(
                'Which tap network would you like to add to this chain?',
                tap_options, 'None')
            if result != 'None':
                selected_taps.append(
                    next(x for x in tap_controller_options
                         if result == self._tap_to_str(x)))

        for jtag_chain in jtag_chains:
            self._init_service.SpecifyTapControllers(jtag_chain, jtag_chain_to_selected_taps[jtag_chain])

    def _ask_set_config(self, deviceId, config):
        print('\n  {0} = {1}'.format(config.name, config.value))
        print('      {0}'.format(config.description))

        if len(config[3]) == 0:
            new_value = _ask_string('Set Value:', 'Cancel')
        else:
            options = config.options
            options.sort()
            new_value = _ask_multiple_choice('Select Value', options, 'Cancel')

        self._devicecontrol_service.SetDeviceConfig(deviceId, config.name,
                                                    new_value)

    def _ask_add_manual_probe(self):
        probe_types = self._init_service.GetProbeTypes()

        # Only allow one manually specified probe to be created per type.
        probe_types = [
            x for x in probe_types
            if x not in [y.type for y in self._total_selected_probes]
        ]
        if not probe_types:
            return

        probe_type_tree = OrderedDict()
        for item in probe_types:
            t = probe_type_tree
            for part in item.split('.'):
                t = t.setdefault(part, OrderedDict())

        current_level = probe_type_tree
        current_selection = ''

        while current_level:
            keys = list(current_level)
            if len(keys) == 1:
                result = keys[0]
            else:
                result = _ask_multiple_choice(
                    'Which{} probe type do you want to create?'.format(
                        current_selection),
                    keys, 'Cancel')
            if result == 'Cancel':
                return
            current_selection += ' ' + result
            current_level = current_level[result]
        
        current_selection = current_selection[1:]

        probe = py2ipc.IPC_Types.IPC_Probe()
        probe.type = result

        self._total_selected_probes.append(probe)
        self._ask_configure_probe(probe)

    def _ask_select_scenario(self):
        scenarios = self._init_service.GetAvailableScenarios()

        if not scenarios:
            print('\nNo scenarios found.')
            return

        scenarios = [s for s in scenarios if s not in self._selected_scenarios]

        ask = True
        while ask:
            print('Selected Scenarios: {}'.format([
                s for s in self._selected_scenarios
            ] if self._selected_scenarios else 'None'))
            result = _ask_multiple_choice(
                'Which scenario would you like to select?', scenarios,
                'Done')
            if result == 'Done':
                ask = False
            else:
                self._init_service.SelectScenario(result)
                self._selected_scenarios.append(result)
                scenarios.remove(result)
                if not scenarios:
                    ask = False

    def _ask_finish_or_revise_configuration(self, ask_add_probe=True, ask_add_probe_first=False):
        ask = True
        while ask:
            print('Selected Probes: {}'.format([
                x.type for x in self._total_selected_probes
            ] if self._total_selected_probes else 'None'))
            if ask_add_probe_first:
                self._ask_add_manual_probe()
                ask_add_probe_first = False
            else:
                options = [ 'Manually specify another probe' ] if ask_add_probe else []
                options += [
                    'Select use case scenario(s)',
                    'Finish and save configuration',
                    'Finish without saving configuration'
                ]
                result = _ask_multiple_choice("", options)
                if result == 'Manually specify another probe':
                    self._ask_add_manual_probe()
                elif result == 'Select use case scenario(s)':
                    self._ask_select_scenario()
                elif result == 'Finish and save configuration':
                    ask = False
                    self._init_service.SaveConfiguration()
                elif result == 'Finish without saving configuration':
                    ask = False

    def run(self):
        """ Runs the Dynamic Initialization process gathering user input and
        initialing with it."""

        print('\nRunning Dynamic Initialization. Press CTRL+C at any time '
              'to finish or restart initialization.')
        restart_initialization = True
        while restart_initialization:
            restart_initialization = False
            py2ipc.IPC_BeginDynamicInitialization()
            self._total_selected_probes = []
            try:
                try:
                    self._ask_how_to_configure()
                except EOFError:
                    # this improves the reliability of catching the keyboard
                    # interrupt
                    sleep(1)
            except KeyboardInterrupt:
                restart_initialization = self._ask_bail_option()

    def _ask_bail_option(self):
        options = [
            'Finish and save configuration',
            'Restart Initialization'
        ]
        print()
        result = _ask_multiple_choice(
            'Looks like you are trying to exit. '
            'How would you like to proceed?', options)
        if result == 'Finish and save configuration':
            self._init_service.SaveConfiguration()
        return result == 'Restart Initialization'


def run_wizard():
    """ Runs the interactive wizard for dynamic initialization.
        IPC_ConnectSingleton must be called before this,
        and IPC_FinishInitialization must be called after. """

    wizard = Wizard()
    wizard.run()
