###############################################################################
# Copyright 2014 2017 Intel Corporation.
#
# The source code, information and material ("Material") contained herein is
# owned by Intel Corporation or its suppliers or licensors, and title to such
# Material remains with Intel Corporation or its suppliers or licensors. The
# Material contains proprietary information of Intel or its suppliers and
# licensors. The Material is protected by worldwide copyright laws and treaty
# provisions. No part of the Material may be used, copied, reproduced, modified,
# published, uploaded, posted, transmitted, distributed or disclosed in any way
# without Intel's prior express written permission.
#
# No license under any patent, copyright or other intellectual property rights
# in the Material is granted to or conferred upon you, either expressly, by
# implication, inducement, estoppel or otherwise. Any license under such
# intellectual property rights must be express and approved by Intel in writing.
#
# Unless otherwise agreed by Intel in writing, you may not remove or alter
# this notice or any other notice embedded in Materials by Intel or Intel's
# suppliers or licensors in any way.
###############################################################################


import py2ipc
from .. import cli_logging
from ..bitdata import BitData
from .._console import _console_trim
from .ipc_commands import OperationWindow
from collections import OrderedDict

_log = cli_logging.getLogger("ipc")

# debug
#cli_logging.getManager().echo(True)
#cli_logging.getManager().level("ipc","DEBUGALL")

class StatePort(object):
    """
    This object will be used to expose the various state port functions that are exposed by the
    IPC API for this particular device:
    """
    def __init__(self,device):
        """device is the device object (not just the did)"""
        self.device = device
        self.stateport_names = []
        # make sure device is enabled
        if not getattr(device,"enabled", True):
            return 
        # create the various functions that the IPC API supports for this device
        statesrvc = py2ipc.IPC_GetService("StatePort")
        _log.debugall("State port node created for device : {0}".format(device.alias))
        if device.node.nodetype not in [ "TargetDomain", "LogicalRoot" ]:
            try:
                self.stateport_names = stateport_names = statesrvc.GetStatePorts( device.did )
            except py2ipc.IPC_Error as err:
                # seems the possible "ok errors" we get here are various and not consisteent, so just
                # log it for debug, but make sure the CLI does not break
                _log.debugall("Exception getting stateport names for %s, message: %s"%(device.alias, str(err)))
                self.stateport_names = stateport_names = []
                return
        else:
            self.stateport_names = stateport_names = []
        _log.debugall("State ports supported: {0}".format(str(stateport_names)))
        # for each state port we will need to create a new function based off
        # its description and parameters
        for stateport in stateport_names:
            description = statesrvc.GetStatePortDescription(device.did, stateport)
            fields = statesrvc.GetAddressFields( device.did, stateport )

            field_descriptions = {}
            for field in fields:
                field_descriptions[field.name] = statesrvc.GetAddressFieldDescription( device.did, stateport, field.name )
                _log.debugall("Device: {0}: StatePort {1}: Field: {2}, \n\tDescription={3}".format(
                               device.alias,
                               stateport,
                               field.name,
                               field_descriptions[field.name]))
            self._create_function(stateport,description,fields,field_descriptions)

    def _create_function(self,stateport,description,fields,descriptions):
        # set is quicker reference for field names
        fieldnames = [] # [f.name for f in fields])
        optional_fields = OrderedDict()
        for field in fields:
            fieldnames.append( field.name )
            if field.validDefaultValue:
                optional_fields[field.name] = field.defaultValue

        def newf(*args,**kwargs):
            ##### logging
            if _log.isEnabledFor(cli_logging.DEBUG):
                _log.caller()
                _log.debug("ENTER: stateport: generated function for stateport {0} on {1}".format(stateport,self.device.alias))
                _log.debugall("stateport {0}: raw args: {1}, raw_kwargs:".format(stateport,args,kwargs))

            statesrvc = py2ipc.IPC_GetService("StatePort")

            # for determining read or write...if not specified, must be read or
            # specified in the args list
            newValue = kwargs.pop("newValue",None)
            # if args > number of fields, assume that the last argument as
            # newValue to be written
            if len(args) == (len(fieldnames)+1):
                # cannot pop because args is a tuple
                newValue = args[-1]
                args = args[:-1]
            elif len(args) > len(fieldnames): # and therefore > + 1
                raise TypeError("Too many arguments supplied")

            # start of with whatever the defaults are
            final_kwargs = dict(optional_fields)

            # walk through args, add to kwargs assuming order is same as fields
            for a,argval in enumerate(args):
                final_kwargs[fields[a].name] = argval

            # update with kwargs given to us (should override defaults)
            final_kwargs.update( kwargs )

            # make sure are in the correct order for stateport call
            try:
                field_values = [ final_kwargs.pop(f.name) for f in fields ]
            except KeyError: # missing an arg, check all of them
                # we don't use set/difference because it screws up order
                missing = []
                for f in fieldnames:
                    if f not in final_kwargs:
                        missing.append(f)
                raise TypeError("Parameter count mismatch, missing arguments: {0}"
                                .format(" ,".join(missing)))

            if len(final_kwargs)>0:
                raise TypeError("Unknown arguments specified: %s"%str(list(final_kwargs.keys())))

            # log resulting argument list
            if _log.isEnabledFor(cli_logging.DEBUG):
                datastr = hex(newValue) if newValue != None else "None"
                field_strs = [ hex(val) for val in field_values ]
                _log.debug("stateport {0}: args: {1}, newValue = {2}"
                           .format(stateport,field_strs,datastr))

            # now call stateport
            if newValue == None: #read
                operation = statesrvc.Read( self.device.did, stateport, field_values)
                if not OperationWindow.isopen(): # flush operations is not in a window
                    operation.Flush()
                    operation.ErrorCheck()
                bd = BitData.CreateFromLong(None,operation, truncate=True, verify=False)
                if _log.isEnabledFor(cli_logging.DEBUG):
                    _log.debug("EXIT: stateport {0}: result = {1}"
                               .format(stateport,bd))
                return bd
            else: # newValue != None...write
                operation = statesrvc.Write( self.device.did, stateport, field_values, newValue)
                if not OperationWindow.isopen(): # flush operations is not in a window
                    operation.Flush()
                    operation.ErrorCheck()
                if _log.isEnabledFor(cli_logging.DEBUG): # show that we are exiting....
                    _log.debug("EXIT: stateport {0}".format(stateport))
                return None
            # turn in to bit data and return
            return
        #---------
        # end of the new function
        #---------
        # clean up some things on this new function so that the help is better
        newf.__name__ = newf.func_name = stateport
        # build docstring using fields list for order, and description dictionary
        docstr = [description,""] # start with stateport description
        # to build signature of arguments
        signature = [ "{0}(".format(stateport) ]
        newf.__fields__ =[]
        newf.__optional_fields__ =[]
        for field in fields:
            newf.__fields__.append( field.name )
            if not field.validDefaultValue: # no default
                signature.append(" {0},".format(field.name ))
            else:
                newf.__optional_fields__.append(field.name)
                signature.append(" {name}={value},"
                                 .format(name=field.name,
                                         value=field.defaultValue))
            docstr.append("    {0} : {1}".format(field.name, descriptions[field.name]))
        signature.append(" newValue=None)")
        # turn bits and pieces back to big string
        signature = "".join(signature)
        newf.__signature__ = signature
        # put signature first in our documentation
        docstr.insert( 0, signature +" \n")
        newf.__doc__ = newf.func_doc = "\n".join( docstr )
        # now add the new function to our class
        setattr(self,stateport,newf)


class BaseaccessStatePorts(object):
    """
    this object will modify the baseaccess and add StatePort objects to all the known device nodes
    """
    def __init__(self, base):
        self._base = base
        for node in base.devs:
            node.stateport = StatePort(node.device)
            if len(node.stateport.stateport_names) > 0:
                setattr(self, node.device.alias.lower(), node.stateport)


    def show(self):
        display = []
        max_alias = max_stateport_len = 0
        for node in self._base.devs:
            if len(node.stateport.stateport_names) == 0:
                continue
            alias = node.device.alias.lower()
            max_alias = max(max_alias, len(alias))
            for sport in node.stateport.stateport_names:
                max_stateport_len = max(max_stateport_len, len(sport))
                description = getattr(node.stateport, sport).__signature__
                display.append(dict(alias=alias, stateport=sport, description=description))

        # we must have a stateport
        if (max_stateport_len == 0) or (max_alias == 0):
            _log.result("No stateports found")
            return

        delimiter = " | "
        format_str = "{{alias:{alias_len}s}}{delimiter}{{stateport:{stateport_len}s}}{delimiter}".format(
            delimiter=delimiter,
            alias_len=max_alias,
            stateport_len=max_stateport_len
        )
        _log.result(format_str.format(
            alias="Alias",
            stateport="Stateport",
        ))
        _log.result("-"*79)
        len_format_str = max_alias + max_stateport_len + len(delimiter)*2
        description_len = 79-len_format_str
        for d in display:
            lines = _console_trim(d['description'], description_len)
            # should always have at least one line
            _log.result(format_str.format(**d)+lines[0])
            for l in lines[1:]:
                _log.result(" "*len_format_str + l)


