###############################################################################
# Copyright 2014 2017 Intel Corporation.
#
# The source code, information and material ("Material") contained herein is
# owned by Intel Corporation or its suppliers or licensors, and title to such
# Material remains with Intel Corporation or its suppliers or licensors. The
# Material contains proprietary information of Intel or its suppliers and
# licensors. The Material is protected by worldwide copyright laws and treaty
# provisions. No part of the Material may be used, copied, reproduced, modified,
# published, uploaded, posted, transmitted, distributed or disclosed in any way
# without Intel's prior express written permission.
#
# No license under any patent, copyright or other intellectual property rights
# in the Material is granted to or conferred upon you, either expressly, by
# implication, inducement, estoppel or otherwise. Any license under such
# intellectual property rights must be express and approved by Intel in writing.
#
# Unless otherwise agreed by Intel in writing, you may not remove or alter
# this notice or any other notice embedded in Materials by Intel or Intel's
# suppliers or licensors in any way.
###############################################################################

from __future__ import absolute_import
from .. import cli_logging
from .._py2to3 import *
import py2ipc

import contextlib

_log = cli_logging.getLogger("ipc")

class IpcCliStalls(object):
    """

    The IPC CLI "stalls" variables will expose the supported stalls
    for those particular devices. The various stalls will
    halt the platform when the specified event occurs.


    The "stalls" will be available at the global level and will
    apply to all the devices in the
    current domain. The stalls attribute will be available on all debugport nodes

    Example 1:

        >>> ipc.stalls.powerbreak = 1

    Example 2:

        >>> ipc.devs.debugport0.stalls.powerbreak = 1

    """
    stalls = []
    _did = None

    def __init__(self, did):
        self._did = did
        # get domain did for the breakall get/set functionality
        # we will use this service a lot of course, so save a reference
        break_service = py2ipc.IPC_GetService("Break")
        try:
            self.stalls = [b.lower() for b in break_service.GetSystemStalls(did)]
        except py2ipc.IPC_Error as e:
            # sometimes we get invalid device instead of an empty list
            if e.codestr == "Invalid_Device_ID":
                self.stalls = []
            elif e.codestr == "Not_Supported":
                self.stalls = []
            else:
                _log.exception("IPC gave an unexpected error for system stalls")
                # dont raise this...just keep going

    def __dir__(self):
        dirlist = list(self.__dict__.keys())
        dirlist.extend(dir(self.__class__))
        dirlist.extend(self.stalls)
        return dirlist

    def __getattr__(self, attr):
        # the only dynamic attributes we support here are in the stalls variable
        if attr not in self.stalls:
            raise AttributeError("Unknown attribute or stall {0}".format(attr))
        # supported stalls, get the value:
        return self._get_stall(attr)

    def __setattr__(self, attr, value):
        # the only dynamic attributes we support here are in the stalls variable
        if attr in self.stalls:
            self._set_stall(attr, value)
        elif attr in dir(self):
            object.__setattr__(self, attr, value)
        else:
            self._set_stall(attr, value)
            self.stalls.append(attr)

    def _get_stall(self, stall):
        break_service = py2ipc.IPC_GetService("Break")
        return break_service.GetSystemStall(self._did, stall)

    def _set_stall(self, stall, value):
        break_service = py2ipc.IPC_GetService("Break")
        return break_service.SetSystemStall(self._did, stall.upper(), value)

    @contextlib.contextmanager
    def context(self, **kwargs):
        oldvalues = {}
        for k in kwargs:
            if k not in self.stalls:
                raise ValueError("Unknown stall %s" % k)
            oldvalues[k] = self._get_stall(k)
        # now set the new values
        for k, v in kwargs.items():
            self._set_stall(k, v)
        # let calling code do its thing
        yield
        # restore values
        for k in kwargs:
            self._set_stall(k, oldvalues[k])
