###############################################################################
# Copyright 2014 2017 Intel Corporation.
#
# The source code, information and material ("Material") contained herein is
# owned by Intel Corporation or its suppliers or licensors, and title to such
# Material remains with Intel Corporation or its suppliers or licensors. The
# Material contains proprietary information of Intel or its suppliers and
# licensors. The Material is protected by worldwide copyright laws and treaty
# provisions. No part of the Material may be used, copied, reproduced, modified,
# published, uploaded, posted, transmitted, distributed or disclosed in any way
# without Intel's prior express written permission.
#
# No license under any patent, copyright or other intellectual property rights
# in the Material is granted to or conferred upon you, either expressly, by
# implication, inducement, estoppel or otherwise. Any license under such
# intellectual property rights must be express and approved by Intel in writing.
#
# Unless otherwise agreed by Intel in writing, you may not remove or alter
# this notice or any other notice embedded in Materials by Intel or Intel's
# suppliers or licensors in any way.
###############################################################################


import py2ipc
from .. import cli_logging
from .._console import _console_trim

_log = cli_logging.getLogger("ipc")

class Precondition(object):
    """
    This object will be used to expose the various preconditions that are exposed by the 
    IPC API for this particular device:
    """
    _frozen = False    
    def __init__(self,device):
        _preconditionsrvc = py2ipc.IPC_GetService("Precondition")
        """device is the device object (not just the did)"""
        self._frozen = False
        self.device = device
        if not getattr(device,"enabled", True):
            self.precondition_names = []

        # create the various preconditions that the IPC API supports for this device
        # if it is a TAP device
        _log.debugall("Precondition node created for device : {0}".format(device.alias))
        if device.node.nodetype not in [ "LogicalGroupDomain", "LogicalGroupCore", "Debugport", "JTAGScanChain", "VirtualJTAGScanChain", "PinsInterface", "I2CBus", "TargetDomain", "LogicalRoot" ]:
            try:
                self.precondition_names = _preconditionsrvc.GetPreconditions(device.did)
            except py2ipc.IPC_Error as err:
                # seems the possible "ok errors" we get here are various and not consisteent, so just
                # log it for debug, but make sure the CLI does not break
                _log.debugall("Exception getting preconditions for %s, message: %s"%(device.alias, str(err)))
                self.precondition_names = []
                return
        else:
            self.precondition_names = []
        self.precondition_names = [p.lower() for p in self.precondition_names]
        _log.debugall("Preconditions supported: {0}".format(str(self.precondition_names)))
        # for each precondition we will need to create a new function based off its description
        # and parameters
        for name in self.precondition_names:
            id = _preconditionsrvc.GetPreconditionId(device.did, name)
            setattr(self, name, PreconditionNode(device, name, id))

        # prevent future accidental overrrides
        self._frozen = True

    def __setattr__(self, attr, value):
        if self._frozen:
            raise RuntimeError("You cannot add/replace attributes after initialization")
        return object.__setattr__(self, attr, value)


class PreconditionNode(object):
    """
    This object represents a specific precondition on device, exposing its
    functionality.
    """
    _frozen = False
    def __init__(self,device,name,id):
        self._device = device
        self._name = name
        self._id = id
        self._preconditionsrvc = py2ipc.IPC_GetService("Precondition")
        self._frozen = True
        
    def require(self, state):
        """
        Requires that the precondition is in the given state.  The device must
        be locked before this function can be called.
        
        Args:
            state (bool, None) - The state to require that the precondition is
                                 in (True, False, or None).         
        """
        op = self._preconditionsrvc.Require(self._device.did, self._id, self._bool_to_state(state))
        op.Flush()
        
    @property
    def currentstate(self):
        """
        The current state that the precondition is in.
        """
        state = self._preconditionsrvc.GetCurrentState(self._device.did, self._id)
        return self._state_to_bool(state)
        
    @currentstate.setter
    def currentstate(self, state):
        state = self._bool_to_state(state)
        self._preconditionsrvc.SetCurrentState(self._device.did, self._id, state)
        
    @property
    def defaultstate(self):
        """
        The default state that the precondition returns to when outside of a lock.
        """
        state = self._preconditionsrvc.GetDefaultState(self._device.did, self._id)
        return self._state_to_bool(state)
    
    @defaultstate.setter
    def defaultstate(self, state):
        state = self._bool_to_state(state)
        self._preconditionsrvc.SetDefaultState(self._device.did, self._id, state)
        
    @property
    def suppressed(self):
        """
        Whether the precondition transition scans are skipped.
        """
        return self._preconditionsrvc.IsSuppressed(self._device.did, self._id)
        
    @suppressed.setter
    def suppressed(self, value):
        return self._preconditionsrvc.SetSuppressed(self._device.did, self._id, value)
        
    def __repr__(self):
        output  = "name             = {0}\n".format(str(self._name))
        output += "id               = {0}\n".format(str(self._id))
        output += "currentstate     = {0}\n".format(str(self.currentstate))
        output += "defaultstate     = {0}\n".format(str(self.defaultstate))
        output += "suppressed       = {0}".format(str(self.suppressed))
        return output
    
    def _bool_to_state(self, state):
        if state is None:
            return "Unknown"
        if bool(state):
            return "Met"
        else:
            return "NotMet"
            
    def _state_to_bool(self, state):
        if state == "Met":
            return True
        elif state == "NotMet":
            return False
        else:
            return None

    def __setattr__(self, attr, value):
        if self._frozen and attr not in dir(self):
            raise RuntimeError("You cannot add/replace attributes after initialization")
        return object.__setattr__(self, attr, value)

class BaseaccessPreconditions(object):
    """
    This object will modify the baseaccess and add DeviceDeviceAction objects
    to all the known device nodes
    """
    def __init__(self, base):
        # add the device actions to every node
        self._base = base

        for node in base.devs:
            node.precondition = Precondition(node.device)
            # if the node has actions, make it were we can reference them from here
            if len(node.precondition.precondition_names):
                setattr(self, node.device.alias.lower(), node.precondition)

    def show(self):
        """show a summary of all devices with device actions, and the action descriptions"""
        display = []
        max_alias = max_action = 0
        maxes = {
            'alias_len': len("Alias"),
            'pre_name_len': len("Precondition"),
            'current_state_len': len("Current"),
            'default_state_len': len("Default"),
            'suppressed_len': len("Suppressed"),
        }
        for node in self._base.devs:
            if len(node.precondition.precondition_names) == 0:
                continue
            alias = node.device.alias.lower()
            maxes['alias_len'] = max( maxes['alias_len'], len(alias))
            for pre_name in node.precondition.precondition_names:
                precondition = getattr(node.precondition, pre_name)
                maxes['pre_name_len'] = max(maxes['pre_name_len'], len(pre_name))
                display.append(dict(alias=alias,
                                    pre_name=pre_name,
                                    current_state=str(precondition.currentstate),
                                    default_state=str(precondition.defaultstate),
                                    suppressed=str(precondition.suppressed),
                                    )
                               )

        # we must have an action
        maxes['delimiter'] = " | "
        format_str = ("{{alias:{alias_len}s}}{delimiter}"
                      "{{pre_name:{pre_name_len}s}}{delimiter}"
                      "{{current_state:{current_state_len}s}}{delimiter}"
                      "{{default_state:{default_state_len}s}}{delimiter}"
                      "{{suppressed:{suppressed_len}s}}"
                      ).format(**maxes)
        #len_format_str = max_alias + max_action + len(delimiter)*2
        # description_len = 79-len_format_str
        _log.result(format_str.format(
            alias="Alias",
            pre_name="Precondition",
            current_state="Current",
            default_state="Default",
            suppressed="Suppressed",
        ))
        _log.result("-"*79)
        for d in display:
            # lines = _console_trim(d['description'], description_len)
            # should always have at least one line
            _log.result(format_str.format(**d))
            #for l in lines[1:]:
            #    _log.result(" "*len_format_str + l)
