###############################################################################
# Copyright 2014 2017 Intel Corporation.
#
# The source code, information and material ("Material") contained herein is
# owned by Intel Corporation or its suppliers or licensors, and title to such
# Material remains with Intel Corporation or its suppliers or licensors. The
# Material contains proprietary information of Intel or its suppliers and
# licensors. The Material is protected by worldwide copyright laws and treaty
# provisions. No part of the Material may be used, copied, reproduced, modified,
# published, uploaded, posted, transmitted, distributed or disclosed in any way
# without Intel's prior express written permission.
#
# No license under any patent, copyright or other intellectual property rights
# in the Material is granted to or conferred upon you, either expressly, by
# implication, inducement, estoppel or otherwise. Any license under such
# intellectual property rights must be express and approved by Intel in writing.
#
# Unless otherwise agreed by Intel in writing, you may not remove or alter
# this notice or any other notice embedded in Materials by Intel or Intel's
# suppliers or licensors in any way.
###############################################################################


# general python imports
import weakref
import re

# ipccli shared module imports
from .. import cli_logging
from ..node import GetNodeClass,GetNodeClasses,NodeGroup

import py2ipc

_log = cli_logging.getLogger("ipc")

_jtag_commands = [
        "autoscandr",
        "idcode",
        "drscan",
        "irscan",
        "irdrscan",
        "irdrscanreplace",
        "irdrscanrmw",
        "irdrscanverify",
        "jtag_shift",
        "jtag_goto_state",

    ]

_chain_commands = [
        "autoscandr",
        "drscan",
        "irscan",
        "rawirdrscan",
        "jtag_shift",
        "jtag_goto_state",
        "specifytopology"
    ]


_thread_commands = [
        "apicid",
        "arch_register",
        "asm",
        "asmmode",
        "brnew",
        "brchange",
        "brget",
        "brremove",
        "brenable",
        "brdisable",
        "break_type",
        "break_subtype",
        "break_address",
        "cpuid",
        "cpuid_eax",
        "cpuid_ebx",
        "cpuid_ecx",
        "cpuid_edx",
        "dport",
        "eval",
        "go",
        "halt",
        "invd",
        "io",
        "isrunning",
        "ishalted",
        "mem",
        "memblock",
        "memdump",
        "memsave",
        "memload",
        "msr",
        "operating_modes",
        "port",
        "processor_mode",
        "regs",
        "stepintoexception",
        "step",
        "thread_status",
        "xmregs",
        "wport",
        "wrsubpir",
        "readpdr",
        "wbinvd",
        ]

_interfaceport_commands = [
        "interfaceport_read",
        "interfaceport_write",
        "interfaceport_read_to_shared_memory",
        "interfaceport_write_from_shared_memory",
        "interfaceport_window",
    ]

_interfaceport_dma_commands = [
        "dmaconfigmethods",
        "dma",
        "dmasave",
        "dmaload",
        "dmaclear",
    ]

_memory_commands = [
        "dma",
        "dmasave",
        "dmaload",
    ]

_core_group_commands = [
        "interestingthreads",
        "keepprobemoderedirectionset",
        "keepprobemoderedirectioncleared",
        "stepintoexception",
        "halt",
        "go",
        "isrunning",
        "ishalted",
        "runcontrolenable",
        "runcontroldisable",
        "isruncontrolenabled",
        "step_mode",
    ]

_debug_port_commands = [
        "setplatformtype",
        "getplatformtype",
        "getsupportedplatformtypes"
    ]

#####################################
#
# This class will hold various nodes. The nodes are the same
# concept/object as "thread", "core",etc...but will be more specific
# and down to the level of device type 
# 
#####################################
class NodeContainer(object):
    # 'declaring' variables that we will define and use
    _nodes = None      # list of nodes
    _nodes_dict = None  # dictionary of nodes for fast lookup
    
    def __init__(self, base):
        # store a weakref to the base only, so that we can be properly
        # garbage collected when we are no longer needed
        self.base = weakref.proxy(base)
        ###########################################
        # Using our function mapping create the various nodes we need
        ###########################################
        self._create_nodes()
        
        
    def _add_functions_to_node(self,function_names, nodetype):
        # this recursive function is used to add functions to nodes of all children devices
        # step one is this particular function and node
        _log.debugall("ENTER/EXIT: _add_functions_to_node: %s", nodetype.__class__)
        for fname in function_names:
            if hasattr(self.base.cmds, fname):
                nodetype._add_did_command( fname, getattr(self.base.cmds,fname) )
                _log.debugall("_add_functions_to_node: adding %s to %s",fname, nodetype.__class__)
        _log.debugall("EXIT: _add_functions_to_node: %s",nodetype.__class__ )

    def _add_functions_to_node_itself(self,function_names, nodeobj):
        # this recursive function is used to add functions to nodes of all children devices
        # step one is this particular function and node
        _log.debugall("ENTER/EXIT: _add_functions_to_node_itself: %s", nodeobj.name)
        for fname in function_names:
            if hasattr(self.base.cmds, fname):
                nodeobj._add_did_command_to_instance( fname, getattr(self.base.cmds,fname) )
                _log.debugall("_add_functions_to_node_itself: adding %s to %s",fname, nodeobj.name)
        _log.debugall("EXIT: _add_functions_to_node_itself: %s",nodeobj.name )

    def add_node(self, device):
        _log.debugall("ENTER: DeviceContainer._create_nodes")

        ############### create general node object and add default functions
        nodetype = GetNodeClass(device.devicetype)
        # create new node no matter what
        nodeobj = nodetype(device)
        # ########## save as a known node and for fast looksup later
        # keep node list in order of devicelist, and dictionary for fast lookup
        self._nodes.append( nodeobj )
        self._nodes_dict[ device.alias.lower() ] = nodeobj

        # mappings are going to be hardcoded now...
        if getattr(device, "nodetype", None) == "thread":
            self._add_functions_to_node( _thread_commands, nodetype )
            # plus one function that gets renamed...
            nodetype._add_did_command("status", self.base.cmds.thread_status)
        # these checks are in addtion to the core group check above
        if device.scanchain != None: 
            self._add_functions_to_node( _jtag_commands, nodetype )
        elif getattr(device,"nodetype",None)=="scanchain":
            self._add_functions_to_node( _chain_commands, nodetype )
            nodetype._add_did_command("enable", self.base.cmds.enable_device)
            nodetype._add_did_command("disable", self.base.cmds.disable_device)
        elif device.devicetype == "InterfacePort":
            self._add_functions_to_node( _interfaceport_commands , nodetype )
            if self.base._version_check("OpenIPC", build=2174, error=False, revision=521295):
                dmaservice = py2ipc.IPC_GetService("Dma")
                methods = dmaservice.GetSupportedConfigurationAccessMethods(device.did)
                if len(methods) != 0:
                    self._add_functions_to_node_itself( _interfaceport_dma_commands , nodeobj )
        elif device.devicetype == "MemoryInterface":
            self._add_functions_to_node( _memory_commands , nodetype )
        elif device.devicetype == "LogicalGroupCore":
            self._add_functions_to_node( _core_group_commands, nodetype )
        elif device.devicetype == "Debugport":
            self._add_functions_to_node( _debug_port_commands, nodetype )


    def _create_nodes(self):
        # reset cache and list for nodes
        self._nodes = []
        self._nodes_dict = {}
        _log.debugall("ENTER: DeviceContainer._create_nodes")
 
        for device in self.base.devicelist:
            self.add_node(device)

        for nodeClass in GetNodeClasses().values():
            for cgroup in self.base.devicelist.coregroups:
                nodeClass._add_core_group(cgroup)

        _log.debugall("EXIT: DeviceContainer._create_nodes")

    def group_by(self, values=None, bywhat=None,**kwargs):
        """
        
        Searches through all the known devices and looks for
        an attribute specified in the 'bywhat' variable. Returns
        a Node Group with a list of all the nodes whose devices
        matched the search criteria.
        
        Args:
            values : this can be either a list of values 
                    or a regular expression that we will match each devices 'bywhat' against.
            bywhat : any device attribute to match against.
            kwargs : bywhat can be expressed via kwargs like so: devicetype=value.
            ignore_case : (optional) special kwarg that will be looked for
                          (default=True).


            
        Returns:
            A node group containing all the nodes whose devices matched the search criteria.

        See Also:
            :ref:`nodegroups-label`
         
        Example::
        
            >>> # Search for aliases that match exactly with what is in the specified list
            >>> threads = ipc.devs.group_by( ["SLM_C0_T0","SLM_C1_T0"] )
            
            >>> # Search for aliases that match the given regular expression
            >>> threads = ipc.devs.group_by( "SLM_C._T." )
            
            >>> # search for all device types that match the given regular expression
            >>> threads = ipc.devs.group_by( devicetype="SLM" )
        
        Raises:
            ValueError: if values is None and kwargs is empty
            ValueError: if no nodes with the given attributes were found
        """
        ignore_case = kwargs.pop("ignore_case", True)

        # will be populated with tuples containing "bywhat" and "values" pairs
        criteria = []

        # doing this way in case one day I add kwargs support, and I can tell if user
        # specifically set the bywhat or not 
        if len(kwargs)==0:
            if bywhat == None: 
                bywhat = "alias"
            if values==None:
                raise ValueError("Must specify values to search for")
            criteria.append((bywhat, values))
        else:
            while len(kwargs) > 0:
                bywhat,values = kwargs.popitem()
                criteria.append((bywhat, values))

        nodelist = []
        attrfound = False
        for node in self._nodes:
            match = False
            for bywhat,values in criteria:
                # check if this node has the attribute in question
                if not hasattr(node.device,bywhat):
                    match = False
                    break 
                attrfound = True               
                
                # get the attribute value
                attrvalue = getattr(node.device,bywhat)

                # check if the attribute is a match based on
                if isinstance(values,list):
                    match = attrvalue in values
                elif isinstance(values,str):
                    regex = re.compile(values, re.I if ignore_case else 0)
                    match = regex.search(attrvalue)
                else:
                    match = attrvalue == values

                # no need to check the remaining criteria if there was no match
                if not match:
                    break

            # add the node to the list if it was a match
            if match:
                nodelist.append(node)

        if not attrfound:
            raise ValueError("no devices found with attribute '{0}'".format(bywhat))                    
        return NodeGroup(nodelist)

    def group_by_aliases(self,aliases):
        print("!! This function may get removed, recommend using 'group_by' instead\n")
        return self.group_by(aliases)
                 
    def __getattr__(self,attr):
        if attr in self._nodes_dict:
            return self._nodes_dict[attr]
        else:
            raise AttributeError("Unknown attribute or node: {0}".format(attr))
        
    def __dir__(self):
        keys = list(self.__dict__.keys())
        keys.extend( dir(self.__class__) )
        keys.extend( self._nodes_dict.keys() )
        return keys
    
    ##################################################
    # Other functions in case folks iterate as if this was a devicelist
    ##################################################
    def __iter__(self):
        return iter(self._nodes)
    
    def __len__(self):
        return len(self._nodes)

