###############################################################################
# Copyright 2014 2021 Intel Corporation.
# 
# The source code, information and material ("Material") contained herein is
# owned by Intel Corporation or its suppliers or licensors, and title to such
# Material remains with Intel Corporation or its suppliers or licensors. The
# Material contains proprietary information of Intel or its suppliers and
# licensors. The Material is protected by worldwide copyright laws and treaty
# provisions. No part of the Material may be used, copied, reproduced, modified,
# published, uploaded, posted, transmitted, distributed or disclosed in any way
# without Intel's prior express written permission. 
#
# No license under any patent, copyright or other intellectual property rights 
# in the Material is granted to or conferred upon you, either expressly, by 
# implication, inducement, estoppel or otherwise. Any license under such 
# intellectual property rights must be express and approved by Intel in writing.
#
# Unless otherwise agreed by Intel in writing, you may not remove or alter 
# this notice or any other notice embedded in Materials by Intel or Intel's 
# suppliers or licensors in any way.
###############################################################################

import py2ipc  # @UnresolvedImport

class IpcInit(object):
    """
    Provides dynamic initialization functionality.
    """
    _base = None
    _initializing = False

    def __init__(self, base):
        self._initializing = True
        self._base = base
        self._base._begin_initialization()
        py2ipc.IPC_BeginDynamicInitialization()
        self._base._refresh_devicelist()

    def _verify_initializing(self):
        if not self._initializing:
            raise Exception("Operation only supported in dynamic initialization")

    def finish(self):
        """
        Finalizes dynamic initialization.
        """
        self._initializing = False
        self._base._finish_initialization()

    def get_available_scenarios(self):
        """
        Returns the available configuration preset scenarios.

        Note: This function can only be used after dynamic initialization has begun and before it has
              finished.

        Examples:

            >>> ipc = ipccli.baseaccess(dynamic_initialization=True)
            >>> ipc.init.get_available_scenarios()
            ['Auto', 'DCI', 'XDPA', ...]
        """
        self._verify_initializing()
        initialization_service = py2ipc.IPC_GetService("Initialization")
        return initialization_service.GetAvailableScenarios()
    
    def select_scenarios(self, scenarios):
        """
        Selects one or more configuration preset scenarios to apply to the active configuration.

        Note: This function can only be used after dynamic initialization has begun and before it has
              finished.

        Args:
            scenarios (list) : A list of configuration scenario presets to apply.

        Examples:

            >>> ipc = ipccli.baseaccess(dynamic_initialization=True)
            >>> ipc.init.select_scenarios(['DCI', 'NoDetectTaps'])
            >>> ipc.init.finish()
        """
        self._verify_initializing()
        initialization_service = py2ipc.IPC_GetService("Initialization")
        for scenario in scenarios:
            initialization_service.SelectScenario(scenario)

    def get_connected_probes(self):
        """
        Returns a list of all supported probes currently connected to the host.

        Each probe is represented as a tuple with the probe type and the probe's unique identifier (e.g.
        USB port path, serial number, etc).

        Note: This function can only be used after dynamic initialization has begun and before it has
              finished.

        Examples:

            >>> ipc = ipccli.baseaccess(dynamic_initialization=True)
            >>> ipc.init.get_connected_probes()
            [('XDPA', '13102644'), ('CCA', '1-3')]
        """
        self._verify_initializing()
        initialization_service = py2ipc.IPC_GetService("Initialization")
        probes = initialization_service.GetConnectedProbes()
        return [(p.type, p.uniqueIdentifier) for p in probes]

    def select_probe(self, probe_type, unique_identifier=None):
        """
        Selects the specified probe to be active in the current configuration. This function can
        be called multiple times to select multiple probes.
        
        A probe can be selected using the results of get_connected_probes() or alternatively by
        manually specifying a specific probe type expected to be present. Once a probe is selected,
        a debug port device is added to the device list which can be used to set configuration
        parameters on the probe before it is fully initialized.

        Note: This function can only be used after dynamic initialization has begun and before it has
              finished.

        Args:
            probe_type (str) : The probe type of the probe to select.
            unique_identifier (str) : The unique identifier of the probe to select (e.g. port path,
                                      serial number).

        Examples:

          Selecting a probe by probe type:
            >>> ipc = ipccli.baseaccess(dynamic_initialization=True)
            >>> ipc.init.select_probe('CCA')
            >>> ipc.init.finish()

          Selecting a probe by probe type and unique identifier (port path):
            >>> ipc = ipccli.baseaccess(dynamic_initialization=True)
            >>> ipc.init.select_probe('CCA', '1-3')
            >>> ipc.init.finish()

          Selecting a probe and changing configuration settings:
            >>> ipc = ipccli.baseaccess(dynamic_initialization=True)
            >>> probe = ipc.init.select_probe('CCA')
            >>> probe.config.Jtag.DisableAllInterfaces = 'true'
            >>> ipc.init.finish()
        """
        self._verify_initializing()
        initialization_service = py2ipc.IPC_GetService("Initialization")
        if isinstance(probe_type, tuple):
            if len(probe_type) > 1:
                unique_identifier = probe_type[1]
            probe_type = probe_type[0]
        probe = py2ipc.IPC_Types.IPC_Probe()
        probe.type = probe_type
        if unique_identifier is not None:
            probe.uniqueIdentifier = unique_identifier
        did = initialization_service.SelectProbe(probe)
        self._base._refresh_devicelist()
        return self._base.devicelist.findByDID(did).node

    def initialize_probe(self, debugport):
        """
        Initialize the probe of the specified debug port device.
    
        Once a probe is initialized, its interfaces will be added to the device list.

        Args:
            debugport (int) : The debug port device representing the probe to initialize.
        """
        self._verify_initializing()
        initialization_service = py2ipc.IPC_GetService("Initialization")
        did = self._base.cmds._device_convert(debugport)
        initialization_service.InitializeProbe(did)
        self._base._refresh_devicelist()

    def get_supported_tap_controllers(self):
        """
        Returns a list of all supported TAP controllers.

        Note: This function can only be used after dynamic initialization has begun and before it has
              finished.

        Examples:

            >>> ipc = ipccli.baseaccess(dynamic_initialization=True)
            >>> ipc.init.get_supported_tap_controllers()
            [('SKL_U_UC', 'A0'), ('SKL_U_UC', 'B0'), ...]
        """
        self._verify_initializing()
        initialization_service = py2ipc.IPC_GetService("Initialization")
        tap_controllers = initialization_service.GetSupportedTapControllers()
        results = []
        for t in tap_controllers:
            if t.deviceSubType != "":
                results.append((t.deviceType, t.stepping, t.deviceSubType))
            else:
                results.append((t.deviceType, t.stepping))
        return results

    def prespecify_tap_controllers(self, debugport, jtag_chain_index, tap_controllers):
        """
        Prespecifies which TAP controllers are expected to be present on the specified JTAG chain
        of a selected probe before the probe is initialized.
        
        Prespecifying TAP controllers does not disable TAP discovery. It does, however, provide a
        hint to the implementation to allow for non-default configuration settings to be applied
        before TAP discovery based on which TAPs are expected, which is might be necessary for
        certain use cases.

        Note: This function can only be used after dynamic initialization has begun and before it has
              finished.

        Args:
            debugport (int) : The debug port device representing the probe to prespecify TAP
                              controllers for.
            jtag_chain_index (int) : The index of the JTAG chain.
            tap_controllers (list of tuples) : The list of the TAP controllers to prespecify.

        Examples:

            >>> ipc = ipccli.baseaccess(dynamic_initialization=True)
            >>> probe = ipc.init.select_probe('CCA')
            >>> ipc.init.prespecify_tap_controllers(probe, 0, [('SKL_U_UC', 'A0')])
            >>> ipc.init.prespecify_tap_controllers(probe, 1, [('SKL_U_UC', 'A0')])
            >>> ipc.init.finish()
        """
        self._verify_initializing()
        if debugport is None:
            did = py2ipc.IPC_Types.IPC_INVALID_DEVICE_ID
        else:
            did = self._base.cmds._device_convert(debugport)
        initialization_service = py2ipc.IPC_GetService("Initialization")
        controllers = []
        for entry in tap_controllers:
            tap_controller = py2ipc.IPC_Types.IPC_TapController()
            tap_controller.deviceType = entry[0]
            tap_controller.stepping = entry[1]
            if len(entry) > 2 and entry[2] is not None:
                tap_controller.deviceSubType = entry[2]
            controllers.append(tap_controller)
        initialization_service.PrespecifyTapControllers(did, jtag_chain_index, controllers)

    def prespecify_device_configs(self, device_search_pattern, config_name_value_pairs):
        """
        Prespecifies device configuration settings to be applied when device(s) matching the search
        pattern are added to the device tree.
        
        No error will be returned immediately from this function if any of the provided device configs
        are invalid in any way. A warning message will be published if any of the prespecified device
        configs fail to apply for any reason.

        Note: This function can only be used after dynamic initialization has begun and before it has
              finished.

        Args:
            device_search_pattern (str) : The regex search pattern used to select which device(s) the
                                          provided configuration settings apply to based on device name.
            config_name_value_pairs (list of tuples) : The list of name/values of the device configs to
                                                       prespecify.

        Examples:

            >>> ipc = ipccli.baseaccess(dynamic_initialization=True)
            >>> ipc.init.prespecify_device_configs("JtagScanChain0", [('Jtag.TclkRate', '100000')])
            >>> ipc.init.finish()
        """
        self._verify_initializing()
        initialization_service = py2ipc.IPC_GetService("Initialization")
        initialization_service.PrespecifyDeviceConfigs(device_search_pattern, config_name_value_pairs)

    def save_config(self):
        """
        Saves the active configuration to disk.

        The active configuration consists of the steps taken during dynamic initialization (e.g. probe
        selection, TAP controller discovery/specification, etc) along with any device configs set.
        
        Note: The saved configuration is written to an opaque binary file stored in the user's home
              directory.
        """
        self._verify_initializing()
        initialization_service = py2ipc.IPC_GetService("Initialization")
        initialization_service.SaveConfiguration()

    def load_config(self):
        """
        Restores the saved configuration from disk created from the last successful call to save_config().
        
        Restoring a saved configuration is akin to "replaying" the steps taken in dynamic initialization
        before the configuration was originally saved.
        
        Warning: Saved configurations from a previous version of the IPC implementation may be incompatible
                 and unable to be restored.
        """
        self._verify_initializing()
        initialization_service = py2ipc.IPC_GetService("Initialization")
        initialization_service.RestoreConfiguration()

