###############################################################################
# Copyright 2014 2017 Intel Corporation.
#
# The source code, information and material ("Material") contained herein is
# owned by Intel Corporation or its suppliers or licensors, and title to such
# Material remains with Intel Corporation or its suppliers or licensors. The
# Material contains proprietary information of Intel or its suppliers and
# licensors. The Material is protected by worldwide copyright laws and treaty
# provisions. No part of the Material may be used, copied, reproduced, modified,
# published, uploaded, posted, transmitted, distributed or disclosed in any way
# without Intel's prior express written permission.
#
# No license under any patent, copyright or other intellectual property rights
# in the Material is granted to or conferred upon you, either expressly, by
# implication, inducement, estoppel or otherwise. Any license under such
# intellectual property rights must be express and approved by Intel in writing.
#
# Unless otherwise agreed by Intel in writing, you may not remove or alter
# this notice or any other notice embedded in Materials by Intel or Intel's
# suppliers or licensors in any way.
###############################################################################

from .. import cli_logging
import py2ipc
from ..breakpoint import IaThreadBreak
from ..address import Address
from threading import Timer, Event, Lock
from .. import settings
from .. import stdiolog
import sys
import time
import datetime

# nasty workaround for dal embedded not handling readline
import __main__
if __main__.__dict__.get("__fromembeddedpython__",False):
    readline = None
else:
    try:
        import readline
    except:
        # a blind try-except is needed to support folks using ipccli in embedded environment
        readline = None

_log = cli_logging.getLogger("event_display")

_TS_FORMAT = "%H:%M:%S.%f %Y-%m-%d"

##############################
# go on with display
##############################

_IPY = None
_IPY_INITIALIZED = False
_IPY_VER = 0
def _screenwrite(msg):
    # don't write screen messages if the event display logger is disabled
    if not _log.propagate:
        return
    # if first time, initialize IPY a
    if not _IPY_INITIALIZED:
        _ipy_init()
    if _IPY and _IPY_VER=='5':
        # ipy, must be ipython5
        # not needed for ipython >=7
        _IPY.pt_cli.run_in_terminal(lambda: sys.stdout.write(msg))
    elif _IPY and _IPY_VER=='7':
        # simple write works with ipython 7
        sys.stdout.write(msg)
    elif readline is not None and sys.platform == 'win32':
        readline.rl.console.write(msg)     
    else:        
        sys.stdout.write(msg)

def _ipy_init():
    global _IPY
    global _IPY_INITIALIZED
    global _IPY_VER
    _IPY_INITIALIZED = True
    try:
        __IPYTHON__
        import IPython
        _IPY_VER = IPython.__version__[0]
        if _IPY_VER in ['5','6', '7']:
            # not needed for ipython >=7
            _IPY = IPython.get_ipython()
    except NameError:
        pass

class MTMessagePrinter:

    def __init__(self):
        self.msgs = []
        self.msg_lock = Lock()
        self.last_message = 0

    def append_msg(self, msg, nowait=False):
        msg += "" if msg.endswith("\n") else "\n"
        with self.msg_lock:
            if len(self.msgs) == 0:
                self.msgs.append(msg)
                self.first_message = self.last_message = time.time()
                # if we have a wait time, spawn thread
                if settings.DISPLAY_WAIT_TIME != 0 and nowait==False:
                    thread = Timer(settings.EVENT_WAIT_TIME, self.mt_print_msgs, (''))
                    thread.start()
            else:
                self.msgs.append(msg)
                self.last_message = time.time()

        if settings.DISPLAY_WAIT_TIME == 0 or nowait:
            self.mt_print_msgs()

    def mt_print_msgs(self):
        # NEW 1
        events_message = None
        with self.msg_lock:
            t=time.time()
            # minimum display hit, show _something_ at least
            if (t-self.first_message)>=settings.DISPLAY_WAIT_TIME:
                pass
            elif (t-self.last_message)>=settings.EVENT_WAIT_TIME:
                # too long since last message, assume we are done
                pass
            else:
                # need to wait longer
                thread = Timer(settings.EVENT_WAIT_TIME, self.mt_print_msgs, (''))
                thread.start()
                return
            if self.msgs:
                events_message = "\n" + "".join(self.msgs)
                self.msgs = []
        
        if events_message:
            _screenwrite(events_message)
            if stdiolog.hook.file is not None:
                stdiolog.hook.file.write(events_message[1:])
            _log.info(events_message) 
            # only print prompt and update line if there are no other messages waiting
            # only works on pyreadline, if not on a newer IPython, then we need to redraw the prompt
            if (sys.platform == 'win32' and readline is not None and _IPY is None and \
                    not stdiolog.hook.is_cmd_executing()):
                readline.rl._print_prompt()
                readline.rl._update_line()

class Events:
    def __init__(self, base):
        self._ipc = base
        self.event = py2ipc.IPC_GetService("Event")
        self.memory = py2ipc.IPC_GetService("Memory")
        self._message_subscribed = False
        self._run_control_subscribed = False
        self._target_subscribed = False
        self._reconfig_subscribed = False
        self._systemstall_subscribed = False
        self._stateport_subscribed = False
        self._break_event_triggered = Event()
        self.mtprinter = MTMessagePrinter()

    @property
    def run_control_subscribed(self):
        """Returns whether run control events are currently subscribed to."""
        return self._run_control_subscribed
        
    @property
    def stateport_subscribed(self):
        """Returns whether state port events are currently subscribed to."""
        return self._stateport_subscribed

    @property
    def target_subscribed(self):
        """Returns whether target events are currently subscribed to."""
        return self._target_subscribed

    @property
    def reconfiguration_subscribed(self):
        """Returns whether reconfiguration events are currently subscribed to."""
        return self._reconfig_subscribed

    @property
    def message_subscribed(self):
        """Returns whether message events are currently subscribed to."""
        return self._message_subscribed

    @property
    def systemstall_subscribed(self):
        """Returns whether system stall events are currently subscribed to."""
        return self._systemstall_subscribed

    def SubscribeMessageEvents(self, function, data):
        """
        Subscribe to Message events. Message events come from the IPC and represent warning conditions and
        general notes of interest.

        Args:
            function: The function to call when Message events occur.
            data: A 64-bit value that is passed unchanged to the callback when an
                event occurs.  This could be a token or even a pointer to an
                internal structure.
        """
        self.event.SubscribeMessageEvents( function, data )

    def SubscribeRunControlEvents(self, function, data):
        """
        Subscribe to Run Control events. Run Control events are those events that occur whenever
        the target enters or exits probe mode.  All break events, from resetbreak to
        breakpoints, are handled as Run Control events.

        Args:
            function: The function to call when Run Control events occur.
            data: A 64-bit value that is passed unchanged to the callback when an
                event occurs.  This could be a token or even a pointer to an
                internal structure.
        """
        self.event.SubscribeRunControlEvents( function, data )
        
    def SubscribeStatePortEvents(self, function, data):
        """
        Subscribe to State Port events. State Port events are those events that occur whenever
        a supported State Port changes.

        Args:
            function: The function to call when a State Port events occur.
            data: A 64-bit value that is passed unchanged to the callback when an
                event occurs.  This could be a token or even a pointer to an
                internal structure.
        """
        self.event.SubscribeStatePortEvents( function, data )


    def SubscribeTargetEvents(self, function, data):
        """
        Subscribe to Target events. These events include Power, Reset, and Sleep, among others.
        These events do not include any kind of break events, which are handled as
        Run Control events.

        Args:
            function: The function to call when Target events occur.
            data: A 64-bit value that is passed unchanged to the callback when an
                event occurs.  This could be a token or even a pointer to an
                internal structure.
        """
        self.event.SubscribeTargetEvents( function, data )

    def SubscribeReconfigurationEvents(self, function, data):
        """
        Subscribe to Reconfiguration events.

        Args:
            function: The function to call when Reconfiguration events occur.
            data: A 64-bit value that is passed unchanged to the callback when an
                event occurs.  This could be a token or even a pointer to an
                internal structure.
        """
        self.event.SubscribeReconfigurationEvents( function, data )

    def SubscribeSystemStallEvents(self, function, data):
        """
        Subscribe to System Stall events.

        Args:
            function: The function to call when System Stall events occur.
            data: A 64-bit value that is passed unchanged to the callback when an
                event occurs.  This could be a token or even a pointer to an
                internal structure.
        """
        self.event.SubscribeSystemStallEvents( function, data )


    def UnsubscribeMessageEvents(self, function):
        """
        Unsubscribe from Message events.

        Args:
            function: The function pointer that was passed to the SubscribeMessageEvents() function.
        """
        self.event.UnsubscribeMessageEvents(function)

    def UnsubscribeRunControlEvents(self, function):
        """
        Unsubscribe from Run Control events.

        Args:
            function: The function pointer that was passed to the SubscribeRunControlEvents() function.
        """
        self.event.UnsubscribeRunControlEvents( function)
        
    def UnsubscribeStatePortEvents(self, function):
        """
        Unsubscribe from State Port events.

        Args:
            function: The function pointer that was passed to the SubscribeStatePortEvents() function.
        """
        self.event.UnsubscribeStatePortEvents(function)

    def UnsubscribeTargetEvents(self, function):
        """
        Unsubscribe from Target events.

        Args:
            function: The function pointer that was passed to the SubscribeTargetEvents() function.
        """
        self.event.UnsubscribeTargetEvents( function)

    def UnsubscribeReconfigurationEvents(self, function):
        """
        Unsubscribe from Reconfiguration events.

        Args:
            function: The function pointer that was passed to the SubscribeReconfigurationEvents() function.
        """
        self.event.UnsubscribeReconfigurationEvents(function)

    def UnsubscribeSystemStallEvents(self, function):
        """
        Unsubscribe from System Stall events.

        Args:
            function: The function pointer that was passed to the SubscribeSystemStallEvents() function.
        """
        self.event.UnsubscribeSystemStallEvents(function)

    def ignore_run_control(self, turnoff, warn = True):
        """
        Turns on and off the default Run Control event.

        Args:
            turnoff: True unsubscribes from the default Run Control event. False subscribes to the default Run Control event.
            warn: True (default) warns when an event type was already enabled/disabled.

        Returns:
            If turnoff = None, will return True for unsubscribed, False for subscribed.
        """
        if(turnoff == True):
            if(self._run_control_subscribed == True):
                self._run_control_subscribed = False
                self.UnsubscribeRunControlEvents( self._show_break_events)
            elif(warn == True):
                _log.debug("Warning: Run Control Event logging is already disabled")
        elif(turnoff == False):
            if(self._run_control_subscribed == False):
                self._run_control_subscribed = True
                self.SubscribeRunControlEvents( self._show_break_events, self)
            elif(warn == True):
                _log.debug("Warning: Run Control Event logging is already enabled")
        else:
            return not self._run_control_subscribed

    def ignore_target(self, turnoff, warn = True):
        """
        Turns on and off the default Target event.

        Args:
            turnoff: True unsubscribes from the default Target event. False subscribes to the default Target event.
            warn: True (default) warns when an event type was already enabled/disabled.

        Returns:
            If turnoff = None, will return True for unsubscribed, False for subscribed.
        """
        if(turnoff == True):
            if(self._target_subscribed == True):
                self._target_subscribed = False
                self.UnsubscribeTargetEvents( self._show_target_events)
            elif(warn == True):
                _log.debug("Warning: Target Event logging is already disabled")
        elif(turnoff == False):
            if(self._target_subscribed == False):
                self._target_subscribed = True
                self.SubscribeTargetEvents( self._show_target_events, self)
            elif(warn == True):
                _log.debug("Warning: Target Event logging is already enabled")
        else:
            return not self._target_subscribed

    def ignore_message(self, turnoff, warn = True):
        """
        Turns on and off the default Message event.

        Args:
            turnoff: True unsubscribes from the default Message event. False subscribes to the default Message event.
            warn: True (default) warns when an event type was already enabled/disabled.

        Returns:
            If turnoff = None, will return True for unsubscribed, False for subscribed.
        """
        if(turnoff == True):
            if(self._message_subscribed == True):
                self._message_subscribed = False
                self.UnsubscribeMessageEvents( self._show_message_events)
            elif(warn == True):
                _log.debug("Warning: Message Event logging is already disabled")
        elif(turnoff == False):
            if(self._message_subscribed == False):
                self._message_subscribed = True
                self.SubscribeMessageEvents( self._show_message_events, self)
            elif(warn == True):
                _log.debug("Warning: Message Event logging is already enabled")
        else:
            return not self._message_subscribed

    def ignore_stateport(self, turnoff, warn = True):
        """
        Turns on and off the default State Port event.

        Args:
            turnoff: True unsubscribes from the default State Port event. False subscribes to the default State Port event.
            warn: True (default) warns when an event type was already enabled/disabled.

        Returns:
            If turnoff = None, will return True for unsubscribed, False for subscribed.
        """
        if(turnoff == True):
            if(self._stateport_subscribed == True):
                self._stateport_subscribed = False
                self.UnsubscribeStatePortEvents(self._show_stateport_events)
            elif(warn == True):
                _log.debug("Warning: StatePort Event logging is already disabled")
        elif(turnoff == False):
            if(self._stateport_subscribed == False):
                self._stateport_subscribed = True
                self.SubscribeStatePortEvents( self._show_stateport_events, self)
            elif(warn == True):
                _log.debug("Warning: StatePort Event logging is already enabled")
        else:
            return not self._stateport_subscribed

    def ignore_reconfiguration(self, turnoff, warn = True):
        """
        Turns on and off the default Reconfiguration event handling.

        Args:
            turnoff: True unsubscribes from the default Reconfiguration event. False subscribes to the default Reconfiguration event.
            warn: True (default) warns when an event type was already enabled/disabled.

        Returns:
            If turnoff = None, will return True for unsubscribed, False for subscribed.
        """
        if(turnoff == True):
            if(self._reconfig_subscribed == True):
                self._reconfig_subscribed = False
                self.UnsubscribeReconfigurationEvents( self._reconfig_events)
            elif(warn == True):
                _log.debug("Warning: Target Event logging is already disabled")
        elif(turnoff == False):
            if(self._reconfig_subscribed == False):
                self._reconfig_subscribed = True
                self.SubscribeReconfigurationEvents( self._reconfig_events, self )
            elif(warn == True):
                _log.debug("Warning: Target Event logging is already enabled")
        else:
            return not self._reconfig_subscribed

    def ignore_systemstall(self, turnoff, warn = True):
        """
        Turns on and off the default System Stall event.

        Args:
            turnoff: True unsubscribes from the default System Stall event. False subscribes to the default System Stall event.
            warn: True (default) warns when an event type was already enabled/disabled.

        Returns:
            If turnoff = None, will return True for unsubscribed, False for subscribed.
        """
        if(turnoff == True):
            if(self._systemstall_subscribed == True):
                self._systemstall_subscribed = False
                self.UnsubscribeSystemStallEvents( self._show_system_stall_events)
            elif(warn == True):
                _log.debug("Warning: System Stall Event logging is already disabled")
        elif(turnoff == False):
            if(self._systemstall_subscribed == False):
                self._systemstall_subscribed = True
                self.SubscribeSystemStallEvents( self._show_system_stall_events, self)
            elif(warn == True):
                _log.debug("Warning: System Stall Event logging is already enabled")
        else:
            return not self._systemstall_subscribed


    def ignore_all(self, turnoff, warn = True):
        """
        Turns on and off the default Target, Run Control, and Message events.

        Args:
            turnoff: True unsubscribes from the default events. False subscribes to the default events.
            warn: True (default) warns when an event type was already enabled/disabled.

        Returns:
            None.
        """
        self.ignore_run_control(turnoff, warn)
        self.ignore_target(turnoff, warn)
        self.ignore_message(turnoff, warn)
        self.ignore_stateport(turnoff, warn)
        if self._ipc._version_check("OpenIPC", build=1880, error=False, revision=512361):
            self.ignore_systemstall(turnoff, warn)
        if turnoff == True:
            self.ignore_reconfiguration(True, warn)
        else:
            # need to also do a version check here
            if self._ipc._version_check("OpenIPC", build=383, revision=454380):
                self.ignore_reconfiguration(False, warn)

    def wait(self, timeout = None):
        """
        Suspend script execution until a Run Control Event has occurred on the target or
        the specified number of seconds have passed.

        Args:
            timeout: Number of seconds to wait.  If set to None, waits forever.

        Returns:
            If "timeout" is specified, return True if the wait returned before the
            timeout expired.  If timeout is not specified, always returns None.
            Returns immediately if it is in probe mode.

        Note:
            Event display must be enabled for this to work - ipc.events.ignore_all(False).

        """
        if not self._run_control_subscribed:
            raise RuntimeError("Cannot wait if Run Control is unsubscribed.")
        # since we are waiting for new halt/break event to occur, we reset this and wait for event
        # from someone calling the corresponding set
        self._break_event_triggered.clear()

        if not self._ipc.isrunning():
            return True

        if timeout == None:
            # we have to handle indefinite wait a little different so that CTRL-C will work
            # otherwise, and indefinite wait will prevent CTRL-C form working
            # this returns False when we hit timeout, and True when the flag was set
            while not (self._break_event_triggered.wait(999999)):
                pass
        else:
            event_hit = self._break_event_triggered.wait(timeout)
        if timeout is not None:
            return event_hit


    @staticmethod
    def _show_message_events(info, self):
        self.mtprinter.append_msg("{0} - {1}".format(info.eventType, info.message))
        
    @staticmethod
    def _show_stateport_events(info, self):
        self.mtprinter.append_msg("    [StatePort {} Event] - StatePort: {}, NewValue: {}".format(info.eventType, info.statePortName,  hex(info.value)))

    @staticmethod
    def _show_break_events(info, self):
        devobj = self._ipc.devicelist.findByDID( info.deviceId )
        # grab time stamp if settings support it
        if settings.EVENT_TIMESTAMP:
            timestamp = "-- "+datetime.datetime.today().strftime(_TS_FORMAT)
        else:
            timestamp = ""
        if info.eventType == "Resume":
            if devobj.devicetype == "LogicalGroupCore":
                self.mtprinter.append_msg("{indent}[{alias} core group] Resuming on {deviceCount} devices {time}".format(
                            indent=" "*4,
                            alias=devobj.alias,
                            deviceCount=info.deviceCount,
                            time=timestamp,
                            ))
            else:
                self.mtprinter.append_msg("{indent}[{alias}] Resuming {time}".format(
                            indent=" "*4,
                            alias=devobj.alias,
                            time=timestamp,
                            ))
        else:
            # only clear this for break events
            self._break_event_triggered.set()
            if devobj.devicetype == "LogicalGroupCore":
                self.mtprinter.append_msg("{indent}[{alias} core group] {type} on {deviceCount} devices {time}".format(
                            indent=" "*4,
                            alias=devobj.alias,
                            type=info.eventType,
                            deviceCount=info.deviceCount,
                            time=timestamp,
                            ))
            else:
                type = info.eventType
                if info.breakType:
                    if info.eventType == "Breakpoint":
                        if info.breakSubtype:
                            type = "{type} {subtype} breakpoint {name}".format(type=info.breakType, subtype=info.breakSubtype, name=info.breakName)
                        else:
                            type = "{type} breakpoint {name}".format(type=info.breakType, name=info.breakName)
                    elif info.eventType in ["Break","Halt"]:
                        if info.breakType and info.breakSubtype:
                            type = "{name} {sub} Break".format(name=info.breakType,sub=info.breakSubtype)
                        elif info.breakType:
                            type = "{name} Break".format(name=info.breakType) 
                        else:
                            type = "Break"
                # self.mtprinter.append_msg(str(IaThreadBreak(self._ipc, devobj, info.eventType)))
                address_str = self.memory.AddressToString(info.fullBreakAddress)
                self.mtprinter.append_msg("{indent}[{alias}] {type} at [{address:s}] {time}".format(
                            indent=" "*4,
                            alias=devobj.alias,
                            type=type,
                            address=address_str,
                            time=timestamp,
                            ))

    @staticmethod
    def _show_target_events(info, self):
        # len_longest_msg = 34 - minus timestamp, use this to keep messages nice
        if settings.EVENT_TIMESTAMP:
            timestamp = "-- "+datetime.datetime.today().strftime(_TS_FORMAT)
        else:
            timestamp = ""
        if getattr(info, "powerDomain", None):
            msg = "{0} : {1} : {2}".format(info.eventType, info.powerDomain, info.currentState)
        else:
            msg = "{0} : {1}".format(info.eventType, info.currentState)
        self.mtprinter.append_msg("TargetEvent: {msg:26s} {time}".format(
            msg=msg,
            time=timestamp))

    @staticmethod
    def _reconfig_events(info, self):
        if settings.EVENT_TIMESTAMP:
            timestamp = "-- "+datetime.datetime.today().strftime(_TS_FORMAT)
        else:
            timestamp = ""

        # len_longest_msg = 34 - minus timestamp, use this to keep messages nice

        if info.eventType == "Started":
            self.mtprinter.append_msg("{msg:34s} {time}".format(
                msg="ReconfigEvent: Started",
                time=timestamp))

        elif info.eventType == "Completed":
            if info.devicesHaveChanged:
                # Warn user we are updating devicelist
                self.mtprinter.append_msg("{msg:34s} {time}".format(
                    msg="ReconfigEvent: Updating devicelist",
                    time=timestamp))
                self._ipc._refresh_devicelist()
                # print complete
                self.mtprinter.append_msg("{msg:34s} {time}".format(
                    msg="ReconfigEvent: Complete",
                    time=timestamp))
            else:
                self.mtprinter.append_msg("ReconfigEvent: Complete {time}".format(time=timestamp))

        elif info.eventType == "INVALID":
            self.mtprinter.append_msg("Invalid Reconfiguration Event {time}".format(time=timestamp))

    @staticmethod
    def _show_system_stall_events(info, self):
        devobj = self._ipc.devicelist.findByDID( info.debugportId )

        if settings.EVENT_TIMESTAMP:
            timestamp = "-- "+datetime.datetime.today().strftime(_TS_FORMAT)
        else:
            timestamp = ""

        if info.eventType == "Stalled":
            self.mtprinter.append_msg("System Stall '{stallName}' was encountered on {device} {time}".format(
                stallName=info.stallName,
                device=devobj.alias,
                time=timestamp
                ))

        elif info.eventType == "Released":
            self.mtprinter.append_msg("System Stall '{stallName}' was released on {device} {time}".format(
                stallName=info.stallName,
                device=devobj.alias,
                time=timestamp
                ))

        elif info.eventType == "INVALID":
            self.mtprinter.append_msg("Invalid System Stall Event {time}".format(
                time=timestamp
                ))
