###############################################################################
# Copyright 2014 2017 Intel Corporation.
#
# The source code, information and material ("Material") contained herein is
# owned by Intel Corporation or its suppliers or licensors, and title to such
# Material remains with Intel Corporation or its suppliers or licensors. The
# Material contains proprietary information of Intel or its suppliers and
# licensors. The Material is protected by worldwide copyright laws and treaty
# provisions. No part of the Material may be used, copied, reproduced, modified,
# published, uploaded, posted, transmitted, distributed or disclosed in any way
# without Intel's prior express written permission.
#
# No license under any patent, copyright or other intellectual property rights
# in the Material is granted to or conferred upon you, either expressly, by
# implication, inducement, estoppel or otherwise. Any license under such
# intellectual property rights must be express and approved by Intel in writing.
#
# Unless otherwise agreed by Intel in writing, you may not remove or alter
# this notice or any other notice embedded in Materials by Intel or Intel's
# suppliers or licensors in any way.
###############################################################################


import py2ipc
from .. import cli_logging
from .._py2to3 import *
from .._console import _console_trim
import warnings
import time

_log = cli_logging.getLogger("ipc")

class DeviceConfigNode(object):
    """
    A device config node either represents a device config (leaf node) or a
    path node with child nodes (non-leaf node).
    """
    device = None
    name = None
    description = None
    _parent = None
    _nodes = None
    _names = None
    _is_leaf = True

    def __init__(self, device, name, parent=None):
        self.device = device
        self.name = name
        self._parent = parent
        self._nodes = {}
        self._names = []

    def show(self, max_name_len=None):
        """
        Show the configuration names and values under this configuration node.
        """
        max_name_len = max(max_name_len, self._compute_max_name_len())
        if self._is_leaf:
            try:
                value = "'{}'".format(self._get_value())
            except:
                value = "<write-only>"
            name = self._resolve_path()
            _log.result("{name:{max_name_len}s} = {value}".format(name=name, value=value, max_name_len=max_name_len))
        else:
            for node_name in self._names:
                self._nodes[node_name.lower()].show(max_name_len=max_name_len)

    def show_help(self, max_name_len=None):
        """
        Show the configuration names and descriptions under this configuration node.
        """
        max_name_len = max(max_name_len, self._compute_max_name_len())
        if self._is_leaf:
            len_format_str = max_name_len + 2 
            description_len = 99-len_format_str
            lines = _console_trim(self.description, description_len)
            name = self._resolve_path()
            _log.result("{name:{max_name_len}s} : {description}".format(name=name, description=lines[0], max_name_len=max_name_len))
            for l in lines[1:]:
                _log.result(" "*len_format_str + l)
        else:
            for node_name in self._names:
                self._nodes[node_name.lower()].show_help(max_name_len=max_name_len)

    def _create_or_get_node(self, node_name):
        node = self._nodes.get(node_name.lower())
        if not node:
            node = DeviceConfigNode(self.device, node_name, parent=self)
            self._nodes[node_name.lower()] = node
            self._is_leaf = False
            self._names.append(node_name)
            self._names.sort()
        return node

    def _resolve_path(self):
        node_names = [self.name]
        parent = self._parent
        while parent is not None:
            node_names.append(parent.name)
            parent = parent._parent
        return ".".join(reversed(node_names))

    def _get_value(self):
        dev_control = py2ipc.IPC_GetService("DeviceControl")
        value = dev_control.GetDeviceConfig(self.device.did, self._resolve_path())
        return value

    def _set_value(self, value):
        if not isinstance(value, basestring):
            raise TypeError("value for configs must be a string")
        dev_control = py2ipc.IPC_GetService("DeviceControl")
        dev_control.SetDeviceConfig(self.device.did, self._resolve_path(), value)

    def _compute_max_name_len(self):
        if self._is_leaf:
            name = self._resolve_path()
            max_name_len = len(name)
        else:
            max_name_len = 0
        for node in self._nodes.values():
            max_name_len = max(max_name_len, node._compute_max_name_len())
        return max_name_len

    def __getattr__(self, attr):
        node = self._nodes.get(attr.lower()) if self._nodes else None
        if node:
            if node._is_leaf:
                return node._get_value()
            else:
                return node
        else:
            return super(DeviceConfigNode, self).__getattribute__(attr)

    def __setattr__(self, attr, value):
        node = self._nodes.get(attr.lower()) if self._nodes else None
        if node:
            node._set_value(value)
        elif hasattr(self, attr):
            super(DeviceConfigNode, self).__setattr__(attr, value)
        else:
            config_name = "{}.{}".format(self._resolve_path(), attr)
            raise AttributeError("'{}' is not a valid config name".format(config_name))

    def __dir__(self):
        return list(self._names)

class DeviceConfigAccess(object):
    """
    Provides access to the device configs available on a device. This object will
    be added to each device node.
    """
    device = None
    names = None
    _nodes = None
    _name_to_node = None

    def __init__(self, device):
        self.device = device
        self.names = []
        self._nodes = {}
        self._name_to_node = {}

        if not getattr(device, "enabled", True):
            return

        try:
            dev_control = py2ipc.IPC_GetService("DeviceControl")
        except py2ipc.IPC_Error as err:
            if err.code in [py2ipc.IPC_Error_Codes.Not_Implemented,
                             py2ipc.IPC_Error_Codes.Not_Supported]:
                # not a known service, nothing we can do
                return
            else:
                raise

        try:
            names = dev_control.GetDeviceConfigNames(self.device.did)
            self.names = list(filter(lambda name: not name.startswith("_"), names))
            self.names.sort()
        except py2ipc.IPC_Error as err:
            # seems the possible "ok errors" we get here are various and not consistent, so just
            # log it for debug, but make sure the CLI does not break
            _log.debugall("Exception getting configs names for %s, message: %s"%(device.alias, str(err)))
            return

        for config_name in self.names:
            node_names = config_name.split(".")
            if node_names:
                node = self._create_or_get_root_node(node_names[0])
                for node_name in node_names[1:]:
                    node = node._create_or_get_node(node_name)
                self._name_to_node[config_name] = node

                try:
                    node.description = dev_control.GetDeviceConfigDescription(self.device.did, config_name)
                except:
                    pass
                if node.description is None or len(node.description) == 0:
                    node.description = "<no description>"

    def show(self):
        """
        Show the configuration names and values for the device.
        """
        if len(self.names) == 0:
            return
        max_name_len = max([len(name) for name in self.names])
        for name in self.names:
            try:
                value = "'{}'".format(self._name_to_node[name]._get_value())
            except:
                value = "<write-only>"
            _log.result("{name:{max_name_len}s} = {value}".format(name=name, value=value, max_name_len=max_name_len))

    def show_help(self):
        """
        Show the configuration names and descriptions for the device.
        """
        if len(self.names) == 0:
            return
        max_name_len = max([len(name) for name in self.names])
        for name in self.names:
            description = self._name_to_node[name].description
            len_format_str = max_name_len + 2 
            description_len = 99-len_format_str
            lines = _console_trim(description, description_len)
            _log.result("{name:{max_name_len}s} : {description}".format(name=name, description=lines[0], max_name_len=max_name_len))
            for l in lines[1:]:
                _log.result(" "*len_format_str + l)

    def show_modified(self):
        """
        Show the configuration names and values that have been modified for the device.
        """
        if len(self.names) == 0:
            return
        dev_control = py2ipc.IPC_GetService("DeviceControl")
        modified_configs = dev_control.GetModifiedDeviceConfigs(self.device.did)
        if len(modified_configs) == 0:
            return
        max_name_len = max([len(name) for name, mv, iv in modified_configs])
        for (name, modified_value, initial_value) in modified_configs:
            modified_value = "'{}'".format(modified_value)
            initial_value = "'{}'".format(initial_value)
            _log.result("{name:{max_name_len}s} = {modified_value} (initial: {initial_value})".format(name=name, modified_value=modified_value, initial_value=initial_value, max_name_len=max_name_len))

    def revert_modified(self):
        """
        Revert the configuration values that have been modified to their initial values for the device.
        """
        dev_control = py2ipc.IPC_GetService("DeviceControl")
        dev_control.RevertModifiedDeviceConfigs(self.device.did)

    def _create_or_get_root_node(self, node_name):
        node = self._nodes.get(node_name.lower())
        if not node:
            node = DeviceConfigNode(self.device, node_name)
            self._nodes[node_name.lower()] = node
        return node

    def __getattr__(self, attr):
        node = self._nodes.get(attr.lower())
        if node:
            if node._is_leaf:
                return node._get_value()
            else:
                return node
        else:
            return super(DeviceConfigAccess, self).__getattribute__(attr)

    def __setattr__(self, attr, value):
        node = self._nodes.get(attr.lower()) if self._nodes else None
        if node:
            node._set_value(value)
        elif hasattr(self, attr):
            super(DeviceConfigAccess, self).__setattr__(attr, value)
        else:
            raise AttributeError("'{}' is not a valid config name".format(attr))

    def __dir__(self):
        return list(self.names)

    def __getitem__(self, key):
        dev_control = py2ipc.IPC_GetService("DeviceControl")
        value = dev_control.GetDeviceConfig(self.device.did, key)
        return value

    def __setitem__(self, key, value):
        dev_control = py2ipc.IPC_GetService("DeviceControl")
        dev_control.SetDeviceConfig(self.device.did, key, value)

class BaseaccessDeviceConfig(object):
    def __init__(self, base):
        self._base = base
        for node in base.devs:
            node.config = DeviceConfigAccess(node.device)
            if len(node.config.names):
                setattr(self, node.device.alias.lower(), node.config)

    def show(self):
        """
        Show the configuration names and values for all devices.
        """
        max_name_len = 0
        for node in self._base.devs:
            if len(node.config.names) == 0:
                continue
            max_name_len = max(max([len(name) for name in node.config.names]), max_name_len)
        for node in self._base.devs:
            if len(node.config.names) == 0:
                continue
            alias = node.device.alias.lower()
            _log.result(alias + ":")
            for name in node.config.names:
                try:
                    value = "'{}'".format(node.config._name_to_node[name]._get_value())
                except:
                    value = "<write-only>"
                _log.result("  {name:{max_name_len}s} = {value}".format(name=name, value=value, max_name_len=max_name_len))
            _log.result("")

    def show_help(self):
        """
        Show the configuration names and descriptions for all devices.
        """
        max_name_len = 0
        for node in self._base.devs:
            if len(node.config.names) == 0:
                continue
            max_name_len = max(max([len(name) for name in node.config.names]), max_name_len)
        for node in self._base.devs:
            if len(node.config.names) == 0:
                continue
            alias = node.device.alias.lower()
            _log.result(alias + ":")
            for name in node.config.names:
                description = node.config._name_to_node[name].description
                len_format_str = max_name_len + 4 
                description_len = 99-len_format_str
                lines = _console_trim(description, description_len)
                _log.result("  {name:{max_name_len}s} : {description}".format(name=name, description=lines[0], max_name_len=max_name_len))
                for l in lines[1:]:
                    _log.result(" "*len_format_str + l)
            _log.result("")

    def show_modified(self):
        """
        Show the configuration names and values that have been modified for all devices.
        """
        max_name_len = 0
        node_to_modified_configs = {}
        dev_control = py2ipc.IPC_GetService("DeviceControl")
        for node in self._base.devs:
            if len(node.config.names) == 0:
                continue
            modified_configs = dev_control.GetModifiedDeviceConfigs(node.device.did)
            if len(modified_configs) == 0:
                continue
            node_to_modified_configs[node] = modified_configs
            max_name_len = max([len(name) for name, mv, iv in modified_configs] + [max_name_len])
        for node in self._base.devs:
            if len(node.config.names) == 0 or node not in node_to_modified_configs:
                continue
            alias = node.device.alias.lower()
            _log.result(alias + ":")
            for (name, modified_value, initial_value) in node_to_modified_configs[node]:
                modified_value = "'{}'".format(modified_value)
                initial_value = "'{}'".format(initial_value)
                _log.result("  {name:{max_name_len}s} = {modified_value} (initial: {initial_value})".format(name=name, modified_value=modified_value, initial_value=initial_value, max_name_len=max_name_len))
            _log.result("")

    def revert_modified(self):
        """
        Revert the configuration values that have been modified to their initial values for all devices.
        """
        dev_control = py2ipc.IPC_GetService("DeviceControl")
        for node in self._base.devs:
            dev_control.RevertModifiedDeviceConfigs(node.device.did)

class DeviceDeviceConfig(object):
    """
    This object will be added to each Ipc Device to display its device actions
    """
    #: ipc device object
    device = None
    #: list of config names (strings)
    config_names = None
    #: list of visible config names (removing the ones with _)
    _all_config_names = None
    #: maps sanitized configuration names to their unsanitized counterparts
    _sanitized_config_names = { }

    def __init__(self, device):
        """
        For exposing device configurations  that the OpenIPC supports for the specified device

        Args:
            device: ipc device object

        """
        # if new parameters added, they must be part of the class
        self.device = device
        # do not set config_names to be != to None until we know for
        # sure it has some or not, setting config_names freezes
        # this component so nothing more can be set

        # if device not enabled, just return
        if not getattr(device,"enabled",True):
            self._all_config_names = []
            self.config_names = []
            return

        try:
            dev_control = py2ipc.IPC_GetService("DeviceControl")
        except py2ipc.IPC_Error as err:
            if err.code in [py2ipc.IPC_Error_Codes.Not_Implemented,
                             py2ipc.IPC_Error_Codes.Not_Supported]:
                # not a known service, nothing we can do
                return
            else:
                raise
        # get the name of the various configs we support
        try:
            self._all_config_names = dev_control.GetDeviceConfigNames(self.device.did)
            # build a dictionary from sanitized configuration names to their
            # unsanitized counterparts
            for config_name in self._all_config_names:
                self._sanitized_config_names[self._sanitize_config_name(config_name)] = config_name
            # sanitize the configuration names
            self.config_names = [self._sanitize_config_name(x) for x in self._all_config_names if not x.startswith("_")]
        except py2ipc.IPC_Error as err:
            # seems the possible "ok errors" we get here are various and not consisteent, so just
            # log it for debug, but make sure the CLI does not break
            _log.debugall("Exception getting configs names for %s, message: %s"%(device.alias, str(err)))
            self._all_config_names = []
            self.config_names = []
            return            

    def __getattr__(self, attr):
        if attr in self._sanitized_config_names:
            return self._get_config(self._sanitized_config_names[attr])
        elif attr in self._sanitized_config_names.values():
            return self._get_config(attr)
        else:
            return super(DeviceDeviceConfig, self).__getattribute__(attr)

    def __setattr__(self, attr, value):
        if self.config_names and attr in self._sanitized_config_names:
            if not isinstance(value, basestring):
                raise TypeError("value for configs must be a strings")
            self._set_config(self._sanitized_config_names[attr], value)
        elif self.config_names and attr in self._sanitized_config_names.values():
            if not isinstance(value, basestring):
                raise TypeError("value for configs must be a strings")
            self._set_config(attr, value)
        elif attr not in dir(self):
            raise ValueError("Unknown Device Config Name: %s"%attr)
        else:
            super(DeviceDeviceConfig, self).__setattr__(attr, value)

    def __getitem__(self, key):
        # convert if we can, but if not there, then pass along to sanitzed config name
        key = self._sanitized_config_names.get(key, key)
        return self._get_config(key)

    def __setitem__(self, key, value):
        if not isinstance(value, basestring):
            raise TypeError("value for configs must be a strings")
        # convert if we can
        key = self._sanitized_config_names.get(key, key)
        return self._set_config(key, value)

    def __dir__(self):
        """give the specified config names as options"""
        if self.config_names is None:
            return dir(type(self))
        else:
            return self.config_names

    def _get_config(self, config_name):
        """
        get the value of the specified config_name from the OpenIPC DeviceControl service
        Args:
            config_name (str) : name of configuration to get the value for
        """
        warnings.warn(FutureWarning("\nIn 1.20xx.xxx versions the device_config will be removed and ipc.config or node.config should be used\n"
                                    "   This warning has been present for quite a while, we are introducing a sleep to make sure you see this\n"
                                    "   and change your code"))
        time.sleep(5)
        dev_control = py2ipc.IPC_GetService("DeviceControl")
        value = dev_control.GetDeviceConfig(self.device.did, config_name)
        if _log.isEnabledFor(cli_logging.DEBUG):
            _log.caller()
            _log.debug("ENTER/EXIT: GET for deviceconfig: {0} = {1} for device {2}"
                       .format(config_name, value, self.device.alias))
        return value

    def _set_config(self, config_name, value):
        """
        get the value of the specified config_name from the OpenIPC DeviceControl service
        Args:
            config_name (str) : name of configuration to get the value for
        """
        warnings.warn(FutureWarning("\nIn 1.20xx.xxx versions the device_config will be removed and ipc.config or node.config should be used\n"
                                    "   This warning has been present for quite a while, we are introducing a sleep to make sure you see this\n"
                                    "   and change your code"))
        time.sleep(5)
        dev_control = py2ipc.IPC_GetService("DeviceControl")
        dev_control.SetDeviceConfig(self.device.did, config_name, value)
        if _log.isEnabledFor(cli_logging.DEBUG):
            _log.caller()
            _log.debug("ENTER/EXIT: SET for deviceconfig: {0} = {1} for device {2}"
                       .format(config_name, value, self.device.alias))

    def _create_property(self, action, description):
        # this will be our function that executes the specified action
        def newf(value):
            ##### logging
            if _log.isEnabledFor(cli_logging.DEBUG):
                _log.caller()
                _log.debug("ENTER: deviceaction: generated function for action {0} on {1} input: {2}"
                           .format(action,self.device.alias, value))
            dev_control = py2ipc.IPC_GetService("DeviceControl")
            return dev_control.ExecuteDeviceAction(self.device.did, action, value)
        newf.func_name = newf.__name__ = action
        newf.description = newf.func_doc = newf.__doc__ = description
        setattr(self, action, newf)

    def _sanitize_config_name(self, config_name):
        """
        converts a configuration name to be compatible with the Python CLI
        """
        return config_name.lower().replace(".", "_")

class BaseaccessDeviceConfigDeprecated(object):
    """
    This object will modify the baseaccess and add DeviceDeviceAction objects
    to all the known device nodes
    """
    def __init__(self, base):
        # add the device actions to every node
        self._base = base
        for node in base.devs:
            node.device_config = DeviceDeviceConfig(node.device)
            if len(node.device_config.config_names):
                setattr(self, node.device.alias.lower(), node.device_config)

    def show(self, **kwargs):
        show_values = kwargs.pop("values", False)
        if len(kwargs) != 0:
            raise ValueError("Unsupported kwargs: %s"%(list(kwargs)))
        display = []
        max_alias = max_config = 0

        for node in self._base.devs:
            if len(node.device_config.config_names) == 0:
                continue
            alias = node.device.alias.lower()
            max_alias = max(max_alias, len(alias))
            for cfg in node.device_config.config_names:
                max_config = max(max_config, len(cfg))
                display_dict  = dict(alias=alias, config=cfg)
                if show_values:
                    display_dict['value'] = getattr(node.device_config, cfg)
                display.append(display_dict)
        # we must have an action
        delimiter = " - "
        format_str = "{{alias:{alias_len}s}}{delimiter}{{config:{config_len}s}}{delimiter}".format(
            delimiter=delimiter,
            alias_len=max_alias,
            config_len=max_config
        )
        if show_values:
            format_str += "{value}"
        for d in display:
            _log.result(format_str.format(**d))

    def show_values(self):
        return self.show(values=True)
