###############################################################################
# Copyright 2014 2017 Intel Corporation.
#
# The source code, information and material ("Material") contained herein is
# owned by Intel Corporation or its suppliers or licensors, and title to such
# Material remains with Intel Corporation or its suppliers or licensors. The
# Material contains proprietary information of Intel or its suppliers and
# licensors. The Material is protected by worldwide copyright laws and treaty
# provisions. No part of the Material may be used, copied, reproduced, modified,
# published, uploaded, posted, transmitted, distributed or disclosed in any way
# without Intel's prior express written permission.
#
# No license under any patent, copyright or other intellectual property rights
# in the Material is granted to or conferred upon you, either expressly, by
# implication, inducement, estoppel or otherwise. Any license under such
# intellectual property rights must be express and approved by Intel in writing.
#
# Unless otherwise agreed by Intel in writing, you may not remove or alter
# this notice or any other notice embedded in Materials by Intel or Intel's
# suppliers or licensors in any way.
###############################################################################


import py2ipc
from .. import cli_logging
from .._console import _console_trim

_log = cli_logging.getLogger("ipc")

class DeviceDeviceActions(object):
    """
    This object will be added to each Ipc Device to display its device actions
    """
    #: ipc device object
    device = None
    #: list of action names (strings)
    action_names = None

    def __init__(self, device):
        """
        For exposing device actions that the OpenIPC supports
        Args:
            device:

        Returns:

        """
        self.device = device
        self.action_names = []
        # if device is not enabled, then don't query IPC
        if not getattr(device,"enabled",True):
            return
        try:
            dev_control = py2ipc.IPC_GetService("DeviceControl")
        except py2ipc.IPC_Error as err:
            if err.code in [py2ipc.IPC_Error_Codes.Not_Implemented,
                             py2ipc.IPC_Error_Codes.Not_Supported]:
                # not a known service, nothing we can do
                return
            else:
                raise
        # get the name of the various actions we support
        try:
            self.action_names = dev_control.GetDeviceActionNames(self.device.did)
        except py2ipc.IPC_Error as err:
            # seems the possible "ok errors" we get here are various and not consisteent, so just
            # log it for debug, but make sure the CLI does not break
            _log.debugall("Exception getting device actions for %s, message: %s"%(device.alias, str(err)))
            return

        for action in self.action_names:
            description = dev_control.GetDeviceActionDescription(self.device.did, action)
            self._create_function(action, description)

    def _create_function(self, action_name, description):
        # this will be our function that executes the specified action
        def action(value):
            ##### logging
            if _log.isEnabledFor(cli_logging.DEBUG):
                _log.caller()
                _log.debug("ENTER: deviceaction: generated function for action {0} on {1} input: {2}"
                           .format(action_name,self.device.alias, value))
            dev_control = py2ipc.IPC_GetService("DeviceControl")
            return dev_control.ExecuteDeviceAction(self.device.did, action_name, value)
        action.func_name = action.__name__ = action_name
        action.description = action.func_doc = action.__doc__ = description
        setattr(self, action_name, action)

    def execute(self, action, value):
        """attempt to execute the specified action"""
        if _log.isEnabledFor(cli_logging.DEBUG):
            _log.caller()
            _log.debug("ENTER: execute: action {0} on {1} input: {2}"
                       .format(action,self.device.alias, value))
        dev_control = py2ipc.IPC_GetService("DeviceControl")
        return dev_control.ExecuteDeviceAction(self.device.did, action, value)


class BaseaccessDeviceActions(object):
    """
    This object will modify the baseaccess and add DeviceDeviceAction objects
    to all the known device nodes
    """
    def __init__(self, base):
        # add the device actions to every node
        self._base = base
        for node in base.devs:
            node.device_action = DeviceDeviceActions(node.device)
            # if the node has actions, make it were we can reference them from here
            if len(node.device_action.action_names):
                setattr(self, node.device.alias.lower(), node.device_action)

    def show(self):
        """show a summary of all devices with device actions, and the action descriptions"""
        display = []
        max_alias = max_action = 0
        for node in self._base.devs:
            if len(node.device_action.action_names) == 0:
                continue
            alias = node.device.alias.lower()
            max_alias = max(max_alias, len(alias))
            for action in node.device_action.action_names:
                max_action = max(max_action, len(action))
                description = getattr(node.device_action, action).description
                display.append(dict(alias=alias, action=action, description=description))
        # we must have an action
        delimiter = " - "
        format_str = "{{alias:{alias_len}s}}{delimiter}{{action:{action_len}s}}{delimiter}".format(
            delimiter=delimiter,
            alias_len=max_alias,
            action_len=max_action
        )
        len_format_str = max_alias + max_action + len(delimiter)*2
        description_len = 79-len_format_str
        for d in display:
            lines = _console_trim(d['description'], description_len)
            # should always have at least one line
            _log.result(format_str.format(**d)+lines[0])
            for l in lines[1:]:
                _log.result(" "*len_format_str + l)

    def show_actions(self):
        import warnings
        warnings.warn("use ipc.device_action.show() instead...",Warning)
        return self.show()
