###############################################################################
# Copyright 2014 2017 Intel Corporation.
#
# The source code, information and material ("Material") contained herein is
# owned by Intel Corporation or its suppliers or licensors, and title to such
# Material remains with Intel Corporation or its suppliers or licensors. The
# Material contains proprietary information of Intel or its suppliers and
# licensors. The Material is protected by worldwide copyright laws and treaty
# provisions. No part of the Material may be used, copied, reproduced, modified,
# published, uploaded, posted, transmitted, distributed or disclosed in any way
# without Intel's prior express written permission.
#
# No license under any patent, copyright or other intellectual property rights
# in the Material is granted to or conferred upon you, either expressly, by
# implication, inducement, estoppel or otherwise. Any license under such
# intellectual property rights must be express and approved by Intel in writing.
#
# Unless otherwise agreed by Intel in writing, you may not remove or alter
# this notice or any other notice embedded in Materials by Intel or Intel's
# suppliers or licensors in any way.
###############################################################################

from __future__ import absolute_import
import weakref
import os
import copy
import traceback
import warnings
import threading
import timeit
import collections

from .._py2to3 import *
from .. import base_env
from .. import cli_logging
from .ipc_breakpoint import IpcCliBreakpoint
from ..address import Address
from ..breakpoint import IaThreadBreak
from ..node import NodeGroup
from . import ipc_breaks


# for deprecation warning
class JtagLockerWarning(Warning):
    pass
class I2CLockerWarning(Warning):
    pass


_log = cli_logging.getLogger("ipc")

####################################
#
# TODO: Need to revist how device is handled as first parameter
#        of breakpoint functions, should I really be specifiying "None"
#        or should we just for device to specified and let 
#        user specify None for it
#
####################################

################################################################################
# RULES for this file. Obey!!!!
# 1. first parameter for every function in the command object must be "device" <- that exactly
# 2. all calls in to the IPC should use the variable name "did" so that it is 
#    obvious that the device was converted to a number
# 3. We do not cache the services during init on purpose that way the whole 
#    environment will continue to work unless a non-existant/supported service 
#    is requested
##############################################################################
# May one day switch to a decorator, but for leaving as explicit call for now

#############################################################################
#
# Developer Q & A:
#
# Q: When (and why) to use _log.isEnabledFor(cli_logging.DEBUG) before logging call ?
# A: #1 : if evaluating the parameters would have a side effect (like displaying
#         the results of a function may flush an operation receipt)
#    #2 : if you need to do lots of formating before calling the log, then we
#         want to maximize performance and not do those things all the time
#
#



################################################################################
# All commands listed here should begin with deviceId as the first parameter
################################################################################
import py2ipc  # @UnresolvedImport
from ..bitdata import BitData
from ..shared_memory import SharedMemory
from ..device import Device
from ..node import NodeBase
from .. import bitdata
from ..asm import ASMList
from ..asm import ASMDecode
from ..getpass import getpass
from . import reverse_semaphore

##########################################################################################
# Used to convert eval legacy parameters to valid IPC_AddressTypes enum values
##########################################################################################
_eval_to_address_conversion = dict(linear="Linear",
                                   physical="Physical",
                                   guestphysical="GuestPhysical",
                                   logical_twofield="TwoFieldLogical",
                                   virtual="TwoFieldLogical",
                                   logical_threefield="ThreeFieldLogical",
                                   )

class Commands(base_env.Commands):
    def __init__(self,baseaccess):
        # key things needed for implementing the various commands
        self.base = weakref.proxy(baseaccess)
        self.devicelist = weakref.proxy(self.base.devicelist)
        self._asmmode = '32Bit' # see the asmmode function for details
        self._msvsmon_process = None
        self._party_screening_message_printed = False
        
    def _device_convert(self,device):
        # putting this as high up since it should be the most common
        if isinstance(device,(int,long)):
            # if number it is either real did or an index in to devicelist
            if device < 0x1000: # device is number and less than 0x1000, treat it as 0 based.
                if device<0 or device >= len(self.devicelist):
                    raise ValueError("Invalid device id: {0}".format(device))
                did = self.devicelist[device].did
            else: # otherwise assume it is correct did
                did = device
        elif isinstance(device,basestring):
            # convert str/alias or object in to a did
            did = self.devicelist[device].did                
        elif isinstance(device,Device):
            did = device.did
        elif isinstance(device,NodeBase):
            did = device.device.did
        else: # even if it is not a number, assume it can be treated like one
            did = device            
                        
        if _log.isEnabledFor(cli_logging.DEBUG):
            # only log if we did some conversion
            if did!=device:
                devicestr = self._device_to_devicestr(device)
                didstr = hex(did) # should always be number  
                _log.debug('Device {0} converted to did: {1}'.format(devicestr,didstr))
            
        return did
    
    def _device_to_devicestr(self,device):
        """used by logging functions for displaying device in log messages"""
        if isinstance(device,basestring):
            devicestr = device
        elif isinstance(device,Device):
            devicestr = "Device({0})".format(device.alias)
        elif isinstance(device,NodeBase):
            devicestr = "Node({0})".format(device.device.alias)            
        elif isinstance(device,(int,long)): 
            # if number just convert to hex
            devicestr = hex(device)
        else: 
            # otherwise assume it can just be converted to a string
            devicestr = str(device) 
        return devicestr
    
    
    def _convert_debug_port_i2c(self,debugport):
        """used by functions where the ITP supported debug port as the first parameter"""
        if isinstance(debugport,(long,int)) and debugport > 0x1000:
            return debugport
        elif isinstance(debugport,Device):
            did = debugport.did
        elif isinstance(debugport,NodeBase):
            did = debugport.device.did
        else:
            # assuming we got a number/did < 0x1000 and that it means
            # debugport #...convert this to the correct did
            devsrv = py2ipc.IPC_GetService("Device")
            debugports = devsrv.GetDescendantIdsForType(self.base._domain, "Debugport", True)
            if debugport>=len(debugports):
                raise ValueError("Specified Debugport {0} does not exist".format(debugport))
            # get the specific debug port...then get the specific i2c device
            dp = debugports[debugport]
            i2c = devsrv.GetDescendantIdsForType(dp, "I2CBus", True)
            if len(i2c) == 0:
                raise ValueError("IC2Bus device not present for debugport {0}".format(debugport))
            did = i2c[0]

        if _log.isEnabledFor(cli_logging.DEBUG):
            # only log if we did some conversion
            if did!=debugport:
                devicestr = str(debugport) # not sure about this yet...
                didstr = hex(did) # should always be number  
                _log.debug('Debugport {0} converted to did: {1}'.format(devicestr,didstr))
                
        return did
    
    def _convert_debug_port(self,debugport):
        """used by functions where the ITP supported debug port as the first parameter"""
        if isinstance(debugport,(long,int)) and debugport > 0x1000:
            return debugport
        elif isinstance(debugport,Device):
            did = debugport.did
        elif isinstance(debugport,NodeBase):
            did = debugport.device.did
        else:
            # assuming we got a number/did < 0x1000 and that it means
            # debugport #...convert this to the correct did
            devsrv = py2ipc.IPC_GetService("Device")
            debugports = devsrv.GetDescendantIdsForType(self.base._domain, "Debugport", True)
            if debugport == None: #then return them all
                did = debugports
            else:
                if debugport>=len(debugports):
                    raise ValueError("Specified Debugport {0} does not exist".format(debugport))
                # get the specific debug port...then get the specific i2c device
                did = debugports[debugport]

        if _log.isEnabledFor(cli_logging.DEBUG):
            # only log if we did some conversion
            if did!=debugport:
                devicestr = str(debugport) # not sure about this yet...
                didstr = hex(did) # should always be number  
                _log.debug('Debugport {0} converted to did: {1}'.format(devicestr,didstr))
                
        return did
    
    def _convert_debugport_to_chains(self, debugport):
        """
        Given the specified "debugport" return a list of JtagScanChain did's
        
        used by functions that support debugport as a parameter but need JtagScanChain's as did's
        
        Args:
            debugport (int) - de
            
        Returns:
            A list of did's that are scan chain devices
            
        """
        if isinstance(debugport, (long,int)) and debugport<0:
            raise ValueError("debugport must be >= 0")
        # first get debug port did...now if debugport>0, then it assumes we got a scanchain
        # and that would be fine...
        # if we got a real IPC device ID, then assume it is debugport
        if isinstance(debugport,(long,int)) and debugport > 0x1000:
            return [ debugport ]
        elif isinstance(debugport,Device):
            dids = [ debugport.did ]
        elif isinstance(debugport,NodeBase):
            dids= [ debugport.device.did ]
        elif not isinstance(debugport, (long, int, type(None))):
            raise ValueError("not a valid value for debugport, must be int or device object")
        else:
            # must have gotten 0 or None
            devsrv = py2ipc.IPC_GetService("Device")
            if debugport == None: #then return them all
                dids = devsrv.GetDescendantIdsForType(self.base._domain, "JTAGScanChain", True)
            else:
                # we got a number that actually references debug ports and we need to get scan chains       
                debugports = devsrv.GetDescendantIdsForType(self.base._domain, "Debugport", True)
                if debugport>len(debugports):
                    raise ValueError("invalid debugport value, must be:  >= 0 and <= {0}".format(len(debugports)-1))
                debugport = debugports[debugport]         
                dids = debugport.GetDescendantIdsForType("JTAGScanChain", True)

        if _log.isEnabledFor(cli_logging.DEBUG):
            devicestr = str(debugport) # not sure about this yet...
            dids_str = [hex(d) for d in dids] # should always be number  
            _log.debug('Debugport {0} converted to jtag scan chain dids: {1}'.format(devicestr,dids_str))
        
        return dids

    def msr(self,device, address, newValue = None):
        """

        Return or change a value in a model-specific register
        (MSR) via register address.
        
        The msr command utilizes the "mov" and "readmsr" instructions submitted
        to the Probe Instruction Register (PIR) and requires the use of valid
        indices as well.

        Args:
          device (int) : device to access the msr on. Not needed if using 
                         threads[0].msr command.
          address (int) : the register address of the msr.
          newValue (int): optional value to write to the MSR (if None, then a read is performed).

        Returns:
          if newValue==None, then the current value of the msr is returned.
      
        Note: 
            MSR access requires the target processor to be halted.
        
        Examples:
          >>> ipc.msr(186)                  # view the msr 186 from all threads
          >>> ipc.msr(186, 0xff)            # change the msr 186 across all threads        
          >>> ipc.threads[0].msr(186, 0xff)  # change the msr 186 on thread 0

        **IPC services required:** Register
        """
        ##### logging
        if _log.isEnabledFor(cli_logging.DEBUG):
            devicestr = self._device_to_devicestr(device)
            datastr = hex(newValue) if newValue != None else "None"
            addressstr = hex(address) # guess this would only be number?
            _log.caller()            
            _log.debug("ENTER: msr(device={0},address={1},newValue={2})".\
                       format(devicestr,addressstr,datastr))
        ##### Required for every function:
        did = self._device_convert(device)
        ####      
        regsrvc = py2ipc.IPC_GetService("Register")
        if newValue is None:
            operation = regsrvc.ReadMsr(did, address)                    
            if not OperationWindow.isopen(): # flush operations is not in a window
                operation.Flush()
                operation.ErrorCheck()
            bd = BitData.CreateFromLong(None, operation, truncate=True, verify=False)
            if _log.isEnabledFor(cli_logging.DEBUG):
                _log.debug("EXIT : msr : result = {0}".format(bd))            
            return bd
        else:
            # It's a write
            operation = regsrvc.WriteMsr(did, address, newValue)
            if not OperationWindow.isopen(): # flush operations is not in a window
                operation.Flush()
                operation.ErrorCheck()
            if _log.isEnabledFor(cli_logging.DEBUG):
                _log.debug("EXIT : msr")
                
    
    def apicid(self, device):
        """

        Reads an 8-bit Advanced Programmable Interrupt Controller  
        Identification(APICID) for a given thread device.
        
        This function reads the APICID base at MSR address:0x1B, then computes 
        the memory address to read the final 8-bit APIC ID value  for a given 
        thread.
        
        Args:
          device (int) : device to access msr on. Not needed if using 
                         threads[0].apicid command

        Returns:
          Returns a bitdata with the value retrieved.
     
        Example:
          >>> ipc.threads[0].apicid()

        **IPC services required:** Register
        """
        ##### logging
        if _log.isEnabledFor(cli_logging.DEBUG):
            devicestr = self._device_to_devicestr(device)
            _log.caller()            
            _log.debug("ENTER: apicid(device={0})".format(devicestr))
        ##### Required for every function:
        did = self._device_convert(device)
        ####
        regsrvc = py2ipc.IPC_GetService("Register")
        operation = regsrvc.ReadAPICIDValue(did)
        if not OperationWindow.isopen(): # flush operations is not in a window
            operation.Flush()
            operation.ErrorCheck()
        bd = BitData(None,operation)
        if _log.isEnabledFor(cli_logging.DEBUG):
            _log.debug("EXIT : apicid : result = {0}".format(bd))            
        return bd
            
    def invd(self, device):
        '''
                
        Invalidates the IA32 processor cache.
       
        Args:
            device (int): the did or alias of a thread where the invalidate
                          cache should be executed
            
        Examples:
            >>> ipc.threads[0].invd()
            >>> ipc.threads.invd()

        **IPC services required:** Memory
        '''
        ##### logging
        if _log.isEnabledFor(cli_logging.DEBUG):
            devicestr = self._device_to_devicestr(device)
            _log.caller()            
            _log.debug("ENTER: invd(device={0})".\
                       format(devicestr))
        ##### Required for every function:
        did = self._device_convert(device)
        ####
        memsrvc = py2ipc.IPC_GetService("Memory")
        operation = memsrvc.InvalidateCache(did,False)
        if not OperationWindow.isopen(): # flush operations is not in a window
            operation.Flush()
            operation.ErrorCheck()
        if _log.isEnabledFor(cli_logging.DEBUG):
            _log.debug("EXIT : invd")        
        
    def wbinvd(self, device):
        '''
        
        Write back modified data from the IA32 caches to main memory and invalidates the caches.
        
        Args:
            device (int): the did or alias of a thread where the wbinvd
                          should be executed.
            
        Example:
            >>> ipc.threads[0].wbinvd()
            >>> ipc.threads.wbinvd()

        **IPC services required:** Memory
        '''        
        if _log.isEnabledFor(cli_logging.DEBUG):
            devicestr = self._device_to_devicestr(device)
            _log.caller()            
            _log.debug("ENTER: wbinvd(device={0})".\
                       format(devicestr))
        ##### Required for every function:
        did = self._device_convert(device)
        #### 
        memsrvc = py2ipc.IPC_GetService("Memory")
        operation = memsrvc.InvalidateCache(did,True)
        if not OperationWindow.isopen(): # flush operations is not in a window
            operation.Flush()
            operation.ErrorCheck()
        if _log.isEnabledFor(cli_logging.DEBUG):
            _log.debug("EXIT : wbinvd")        
        

    def io(self, device, bitsize, portNumber, newValue = None):
        """
        
        Retrieves or changes the contents of an IA32 I/O Port.
        
        This function is used by the various port/wport/dport commands
        where the bitsize is part of the function name. Here, bitsize
        is a parameter.
        
        Args:
            device (int): the did or alias of the device to execute the 
                          in/out instruction on (not needed if using
                          from a node object).
            bitsize (int) : 8,16, or 32 - number of bits to read/write during
                            the in/out instruction.
            portNumber (int) : the port number in the target processor's I/O space,
                            a value between 0x0 and 0xFFFF (or 0xFFFFFFFF if using 32bits).
            newValue (int,None) : value to write to the specified port (or *None* if
                            a read is being requested).
           
        Raises:
             TypeError: if the specified port is not a number
             ValueError: if the port number is not greater than 0
        
        **IPC services required:** Memory     
        """
        
        if _log.isEnabledFor(cli_logging.DEBUG):
            devicestr = self._device_to_devicestr(device)
            newvaluestr = hex(newValue) if newValue != None else "None"
            _log.caller()
            _log.debug("ENTER: io(device={0},bitsize={1},portNumber={2},newValue={3})".\
                       format(devicestr, bitsize, hex(portNumber), newvaluestr))
        ##### Required for every function:
        did = self._device_convert(device)
        ###
        memsrvc = py2ipc.IPC_GetService("Memory")
        # try to make a number, if it is not through a better exception than what we would get later
        if not isinstance(portNumber,(int,long,BitData)):
            raise TypeError("Port Number must be of a number type")
        if portNumber<=0:
            raise ValueError("portNumber must be greater than 0")
        if newValue is None:
            operation = memsrvc.ReadIO( did, portNumber, bitsize>>3 )
            if not OperationWindow.isopen(): # flush operations is not in a window
                operation.Flush()
                operation.ErrorCheck()
            bd = BitData(None, operation )
            if _log.isEnabledFor(cli_logging.DEBUG):
                _log.debug("EXIT : io : result = {0}".format(bd))            
            return bd    
        else:
            operation = memsrvc.WriteIO(did,portNumber,bitsize>>3, newValue )
            if not OperationWindow.isopen(): # flush operations is not in a window
                operation.Flush()
                operation.ErrorCheck()
            if _log.isEnabledFor(cli_logging.DEBUG):
                _log.debug("EXIT : io ")            
            return
        
    def port(self,device,portNumber,newValue=None):
        """
        
        Retrieve or change the contents of an 8-bit IA32 I/O port.
        
        Args:
            device (int): the did or alias of the device to execute the 
                          in/out instruction on (not needed if using
                          from a node object).
            portNumber (int):  A port number in the target processor I/O space. 
                    The port number value exists in the range 0x00 to 0xff.
            newValue (int, None) : value to write to the specified port (or *None* if
                            a read is being requested).
             
        Examples:
          >>> ipc.threads[0].port(0x80, 0x12)        
          >>> ipc.threads[0].port(0x80)
          0x12

        **IPC services required:** Memory

        """        
        return self.io(device,8,portNumber,newValue)
    
    def wport(self,device,portNumber,newValue=None):
        """
        
        Retrieve or change the contents of an 16-bit IA32 I/O port.
        
        Args:
            device (int): the did or alias of the device to execute the 
                          in/out instruction on (not needed if using
                          from a node object).
            portNumber (int):  A port number in the target processor I/O space. 
                    The port number value exists in the range 0x00 to 0xffff.
            newValue (int): optional value to write to the port (if None, then a read is performed).
             
        Examples:
          >>> ipc.threads[0].wport(0x80, 0x1234)        
          >>> ipc.threads[0].wport(0x80)
          0x1234

        **IPC services required:** Memory

        """        
        return self.io(device,16,portNumber,newValue)
    
    def dport(self,device,portNumber,newValue=None):
        """
        
        Retrieve or change the contents of a 32-bit IA32 I/O port.
        
        Args:
            device (int): the did or alias of the device to execute the 
                          in/out instruction on, not needed if using
                          from a node object.
            portNumber (int):  A port number in the target processor I/O space. 
                    The port number value exists in the range 0x00 to 0xffffffff.
            newValue (int): optional value to write to the port (if None, then a read is performed).
             
        Examples:
          >>> ipc.threads[0].dport(0x80, 0x1234)        
          >>> ipc.threads[0].dport(0x80)
          0x1234

        **IPC services required:** Memory

        """
        return self.io(device,32,portNumber,newValue)    

    def dmaconfigmethods(self, device):
        """
        
        Retrieve the supported configuration access methods for direct memory
        accesses on the specified device.
        
        Args:
          device (int) : The did or alias of the device to retrieve the direct
                         memory access configuration methods for, not needed
                         if using from a node object.
             
        Examples:
          >>> ipc.devs.dci_usb_dma.dmaconfigmethods()
          [ 'TapBased' ]

        **IPC services required:** Dma

        """
        if _log.isEnabledFor(cli_logging.DEBUG):
            devicestr = self._device_to_devicestr(device)
            _log.caller()
            _log.debug("ENTER: dmaconfigmethods(device={0})".format(devicestr))

        self.base._version_check("OpenIPC", build=2174, revision=521295)

        did = self._device_convert(device)

        dmasrvc = py2ipc.IPC_GetService("Dma")
        result = dmasrvc.GetSupportedConfigurationAccessMethods(did)

        if _log.isEnabledFor(cli_logging.DEBUG):
            _log.debug("EXIT : dmaconfigmethods : result = {0}".format(result))                        
        return result
            
    def dma(self, device, address, byteCount, bytes=None, configMethod=None):
        """

        Read from or write to target memory using direct memory access.

        Args:
          device (int)        : The did or alias of the device to perform the
                                direct memory access on, not needed if using
                                from a node object.
          address (str)       : Address expressed as a string or an ipccli.Address
                                object; must be a physical address.
          byteCount (int)     : The length of the target memory range, in bytes.
          bytes (bytearray)   : The data to write, can be None to indicate a
                                read.
          configMethod (str)  : The name of the configuration access method to
                                use, can be None to use the default configuration
                                method.
             
        Returns:
          If bytes=None, then a bytearray object containing the value read
          from memory is returned.
        
        Examples:
          >>> ipc.devs.dci_usb_dma.dma("0x100P", 4, 0xdeadbeef)
          >>> ipc.devs.dci_usb_dma.dma("0x100P", 4)
          bytearray(b'\\xef\\xbe\\xad\\xde')

        Raises:
             ValueError: if the byte count does not match the size of the bytearray
        
        **IPC services required:** Dma

        """
        if _log.isEnabledFor(cli_logging.DEBUG):
            devicestr = self._device_to_devicestr(device)
            bytesstr = str(bytes) if bytes != None else "None"
            _log.caller()
            _log.debug("ENTER: dma(device={0},address={1},byteCount={2},bytes={3},configMethod={4})".\
                       format(devicestr, address, byteCount, bytesstr, configMethod))

        self.base._version_check("OpenIPC", build=2174, revision=521295)

        did = self._device_convert(device)
        devobj = self.devicelist.findByDID(did)

        dmasrvc = py2ipc.IPC_GetService("Dma")

        if not isinstance(address, py2ipc.IPC_Address):
            address = self._parse_address(devobj, address, 'ds')

        if configMethod is None:
            configMethod = dmasrvc.GetDefaultConfigurationAccessMethod(did)

        if bytes is None:
            result = dmasrvc.ReadMemory(did, configMethod, address, byteCount)
        else:
            if isinstance(bytes, (int, long)):
                bytes = self._int_to_bytearray(byteCount, bytes)
            elif isinstance(bytes, list):
                bytes = bytearray(bytes)

            if len(bytes) != byteCount:
                raise ValueError("Byte count {0} does not match the size of the provided bytes (len(bytes) = {1})".format(byteCount, len(bytes)))

            result = dmasrvc.WriteMemory(did, configMethod, address, bytes)

        if _log.isEnabledFor(cli_logging.DEBUG):
            _log.debug("EXIT : dma")                        

        return result

    def dmasave(self, device, filename, address, byteCount, overwrite=False, fileType="binary", saveOptions=None, configMethod=None):
        """

        Read from target memory using direct memory access and save to a file.

        Args:
          device (int)        : The did or alias of the device to perform the
                                direct memory access on, not needed if using
                                from a node object.
          filename (str)      : Path of the file where the data is to be stored.
          address (str)       : Address expressed as a string or an ipccli.Address
                                object; must be a physical address.
          byteCount (int)     : The length of the target memory range, in bytes.
          overwrite (bool)    : (optional, default=False) Set to True to overwrite the file 
                                if it exists; otherwise, an error is raised.
          fileType (str)      : The type of file being saved.
          saveOptions (str)   : Options for controlling save behavior.
          configMethod (str)  : The name of the configuration access method to
                                use, can be None to use the default configuration
                                method.

        Returns:
            The number of bytes successfully read from target memory and saved
            to the file.

        Example:
          >>> ipc.devs.dci_usb_dma.dmasave("data.bin", "0x100P", 256)
          256

        Raises:
             OSError: if the file already exists and overwrite is False
        
        **IPC services required:** Dma

        """
        if _log.isEnabledFor(cli_logging.DEBUG):
            devicestr = self._device_to_devicestr(device)
            _log.caller()
            _log.debug("ENTER: dmasave(device={0},filename={1},address={2},byteCount={3},overwrite={4},fileType={5},saveOptions={6},configMethod={7})".\
                       format(devicestr,filename,address, byteCount, overwrite, fileType, saveOptions, configMethod))

        self.base._version_check("OpenIPC", build=2174, revision=521295)

        did = self._device_convert(device)
        devobj = self.devicelist.findByDID(did)

        dmasrvc = py2ipc.IPC_GetService("Dma")

        if overwrite == False and os.path.exists(filename):
            raise OSError("File {0} already exists, and overwrite was not set to True".format(filename))

        if not isinstance(address, py2ipc.IPC_Address):
            address = self._parse_address(devobj, address, 'ds')

        if configMethod is None:
            configMethod = dmasrvc.GetDefaultConfigurationAccessMethod(did)

        result = dmasrvc.SaveMemory(did, configMethod, address, byteCount, filename, overwrite, fileType, saveOptions)

        if _log.isEnabledFor(cli_logging.DEBUG):
            _log.debug("EXIT : dmasave : result = {0}".format(result))                        
        return result

    def dmaload(self, device, filename, address, byteCount, fileType="binary", loadOptions=None, configMethod=None):
        """

        Write to target memory using direct memory access from data in a file.
        
        Args:
          device (int)        : The did or alias of the device to perform the
                                direct memory access on, not needed if using
                                from a node object.
          filename (str)      : Path of the file to load memory from.
          address (str)       : Address expressed as a string or an ipccli.Address
                                object; must be a physical address.
          byteCount (int)     : The length of the target memory range, in bytes.
          fileType (str)      : The type of file being loaded.
          loadOptions (str)   : Options for controlling load behavior.
          configMethod (str)  : The name of the configuration access method to
                                use, can be None to use the default configuration
                                method.

        Returns:
            The number of bytes successfully written to target memory from the file.
             
        Examples:
          >>> ipc.devs.dci_usb_dma.dmaload("data.bin", "0x100P", 256)
          256

        **IPC services required:** Dma

        """
        if _log.isEnabledFor(cli_logging.DEBUG):
            devicestr = self._device_to_devicestr(device)
            _log.caller()
            _log.debug("ENTER: dmaload(device={0},filename={1},address={2},byteCount={3},fileType={4},loadOptions={5},configMethod={6})".\
                       format(devicestr, filename, address, byteCount, fileType, loadOptions, configMethod))

        did = self._device_convert(device)
        devobj = self.devicelist.findByDID(did)

        dmasrvc = py2ipc.IPC_GetService("Dma")

        if not isinstance(address, py2ipc.IPC_Address):
            address = self._parse_address(devobj, address, 'ds')

        if configMethod is None:
            configMethod = dmasrvc.GetDefaultConfigurationAccessMethod(did)

        result = dmasrvc.LoadMemory(did, configMethod, address, byteCount, filename, fileType, loadOptions)

        if _log.isEnabledFor(cli_logging.DEBUG):
            _log.debug("EXIT : dmaload : result = {0}".format(result))                        
        return result

    def _int_to_bytearray(self, byteCount, value):
        import binascii
        fmt = '%%0%dx' % (byteCount * 2)
        return bytearray(binascii.unhexlify(fmt % value)[::-1])

    def dmaclear(self, device, force=False, configMethod=None):
        """

        Attempts to clear any errors in the DMA mechanism.

        Args:
          device (int)        : The did or alias of the device, not needed if
                                using from a node object.
          force (bool)        : (optional, default=False) Represents affirmation
                                from the caller that they consent to interrupt
                                any other ongoing DFx operations and potentially
                                power-cycle the platform.
          configMethod (str)  : The name of the configuration access method to
                                use, can be None to use the default configuration
                                method.

        Example:
          >>> ipc.devs.dci_usb_dma.dmaclear()

        **IPC services required:** Dma

        """
        if _log.isEnabledFor(cli_logging.DEBUG):
            devicestr = self._device_to_devicestr(device)
            _log.caller()
            _log.debug("ENTER: dmaclear(device={0},force={1},configMethod={2})".\
                       format(devicestr, force, configMethod))

        did = self._device_convert(device)
        devobj = self.devicelist.findByDID(did)

        dmasrvc = py2ipc.IPC_GetService("Dma")

        if configMethod is None:
            configMethod = dmasrvc.GetDefaultConfigurationAccessMethod(did)

        result = dmasrvc.ClearError(did, configMethod, force)

        if _log.isEnabledFor(cli_logging.DEBUG):
            _log.debug("EXIT : dmaclear")                        

    def mem(self, device, address, dataSize, newValue = None):
        """Reads from or writes data to the specified address.
        
        Args:
            device (int): the did or alias of the device to execute the 
                          memory instructions on (not needed if using
                          from a node object).
            address (str): the address expressed as a string or an ipccli.Address
                           object.
            dataSize (int): The number of bytes to read or write (1, 2, 4 or 8).
            newValue (int): if specified, the value to write to memory.
            
        Returns:
            if newValue=None, then a BitData object containing the value
            read from memory is returned.
        
        Examples:
             >>> ipc.threads[0].mem("0x100P", 2)
             >>> ipc.threads[0].mem("0x100P", 4, 0x12345678)

        Raises:
             TypeError: if the address is not a string or an ipccli.Address object
             ValueError: if the specified dataSize is not 1, 2, 4, or 8
        
        **IPC services required:** Memory

        """
        if _log.isEnabledFor(cli_logging.DEBUG):
            devicestr = self._device_to_devicestr(device)
            newvaluestr = hex(newValue) if newValue != None else "None"
            _log.caller()
            _log.debug("ENTER: mem(device={0},address={1},dataSize={2},newValue={3})".\
                       format(devicestr, address, hex(dataSize), newvaluestr))
        # version check needed
        self.base._version_check("OpenIPC", build=139, revision=454380)
        ##### Required for every function:
        did = self._device_convert(device)
        devobj = self.devicelist.findByDID(did)
        #### 
        memsrvc = py2ipc.IPC_GetService("Memory")
        if not isinstance(address,(Address,basestring,py2ipc.IPC_Address)):
            raise TypeError("address must be a string or an ipccli.Address object")
        if dataSize not in [1, 2, 4, 8]:
            raise ValueError("Invalid memory access size {0}".format(dataSize))
        # unless it is an IPC_Address (which is rare), then we need to convert
        if not isinstance(address,py2ipc.IPC_Address):
            address = self._parse_address(devobj, address, "ds")
            
        if newValue is None:
            operation = memsrvc.ReadMemoryWithAccessWidth(did, address, 1, dataSize)
            if not OperationWindow.isopen(): # flush operations is not in a window
                operation.Flush()
                operation.ErrorCheck()
            bd = BitData(None, operation)
            if _log.isEnabledFor(cli_logging.DEBUG):
                _log.debug("EXIT : mem : result = {0}".format(bd))                        
            return bd
            #return self.bitdata_service.extract_to_python_bitdata(bitdata)
        else:
            # count is single entry of 1, but we have to provide list as parameter, so give list with 1 entry
            operation = memsrvc.WriteMemoryWithAccessWidth(did, address, 1 ,dataSize, [newValue])
            if not OperationWindow.isopen(): # flush operations is not in a window
                operation.Flush()
                operation.ErrorCheck()    
            if _log.isEnabledFor(cli_logging.DEBUG):
                _log.debug("EXIT : mem")                        
            return

    def memblock(self, device, address, count, dataSize, *args):
        """
        
        Reads values from or writes values to memory by iteratively calling
        the mem command for the specified count. Returns
        the values as a list.
        
        Args:
            device (int): the did or alias of the device to execute the 
                          memory instructions on (not needed if using
                          from a node object).
            address (str): A string or address object indicating where to begin
                           the memory read at.
            count (int) : The number of times we call mem.
            dataSize (int) : The number of bytes to read per mem call (0, 1, 2, 4, or 8).
            args (list)    : If not empty, specifies one or more values to write
                            to memory. The specified data must match the count.
                            
        Note:
            This command is different from the legacy itp command.
            Address MUST be an address (not a count) and it returns 
            a list of bitdata objects.
        
        Returns:
            A list of size *count* where each element is a BitData whose width is *dataSize*.

        Note:
            dataSize = 0 writes the list of bytes (so each arg is a byte) in the most efficient manner possible.
                            
        Examples:
        
            >>> ipc.threads[0].memblock("0x100P", 4, 1)
            [32b] 0xC000EAB0
            >>> ipc.threads[0].memblock("0x100P", 1, 4)
            [32b] 0xC000EAB0
            >>> itp.threads[0].memblock("0x100P", 4, 1, [0xAA]*4)
            >>> itp.threads[0].memblock("0x100P", 4, 1, 0xAA, 0xAB, 0xAC, 0xAD)
            >>> itp.threads[0].memblock("0x100P", 4, 0, 0xAA, 0xAB, 0xAC, 0xAD)
            >>> ipc.threads[0].memblock("0x100P", 4, 1)
            [32b] 0xADACABAA

        Raises:
             ValueError: if the specified dataSize is not 1, 2, 4, or 8
             ValueError: if the specified count is not greater than 0
             ValueError: if the length of args does not match count
        
        **IPC services required:** Memory

        """
        if _log.isEnabledFor(cli_logging.DEBUG):
            devicestr = self._device_to_devicestr(device)
            _log.caller()
            _log.debug("ENTER: memblock(device={0},address={1},count={2},dataSize={3},args={4})".\
                       format(devicestr,address,count,dataSize,args))  
        # version check needed
        self.base._version_check("OpenIPC", build=139, revision=454380)
        did = self._device_convert(device)
        devobj = self.devicelist.findByDID(did)
        # conver our string address to IPC_Address
        if dataSize not in [0, 1, 2, 4, 8]:
            raise ValueError("Invalid memory access size {0}".format(dataSize))
        if count <= 0:
            raise ValueError("Must read/write at least one byte")
        if dataSize*count > (1024*1024): # 1MB -- not sure what cutoff should be
            # could we potentially break these up on some boundary to reduce the chance of hangs... ?
            _log.result("Warning: reading large amounts of memory will take a long time...")
        memsrvc = py2ipc.IPC_GetService("Memory")
        # in case it has a $
        address = self._parse_address(devobj, address, 'ds')
        # must be read
        if len(args)==0:
            if dataSize > 0:
                operation = memsrvc.ReadMemoryWithAccessWidth(did, address, count, dataSize)
            else:
                operation = memsrvc.ReadMemory(did, address, count)
            if not OperationWindow.isopen(): # flush operations is not in a window
                operation.Flush()
                operation.ErrorCheck()
            data = BitData(None, operation)
        else: 
            # some special cases to watch out for of single arg but count was higher...
            if len(args) == 1 and isinstance(args[0],list):
                args = args[0]
            elif len(args) == 1 and count>1: # also means args[0] is not a list...
                args = [args[0]]*count
            # at this point arglen should be same as count
            if len(args) != count: raise ValueError("Must specify same amount of data to match count")
            if dataSize>0:
                operation = memsrvc.WriteMemoryWithAccessWidth(did, address, len(args), dataSize, list(args))
            else:
                operation = memsrvc.WriteMemory(did, address, len(args), list(args))
            if not OperationWindow.isopen(): # flush operations is not in a window
                operation.Flush()
                operation.ErrorCheck()

            # end of write...result is None
            data = None
        if _log.isEnabledFor(cli_logging.DEBUG):
            _log.debug("EXIT: memblock : result = {0}".format(data))
            
        return data
    
    def _byte_to_ascii (self, byte):
        if byte > 0x20 and byte < 0x7F:
            return chr(byte)
        else:
            return '.'
    
    def memdump(self, device, address, count, dataSize):
        """
        
        Read values from memory by iteratively calling
        the mem command for the specified count and displays the
        values.
        
        Args:
            device (int): the did or alias of the device to execute the 
                          memory instructions on (not needed if using
                          from a node object).
            address (str): A string or address object of where to begin
                           the memory read at.
            count (int) : The number of times we call mem.
            dataSize (int) : The number of bytes to read per mem call (1, 2, 4, or 8).
                            
        Note:
            This command is different from the legacy itp command.
            Address MUST be an address, not a count.
        
        Examples:
        
            >>> ipc.threads[0].memdump("0x100P", 32 ,1)
            0000000000000100P: b0 ea 0 c0 b5 ea 0 c0 ba ea 00 c0 bf ea 00 c0
            0000000000000110P: c4 ea 0 c0 c9 ea 0 c0 ce ea 00 c0 d3 ea 00 c0
            >>> ipc.threads[0].memdump("0x100P", 16 , 4)
            0000000000000100P: c000eab0 c000eab5 c000eaba c000eabf
            0000000000000110P: c000eac4 c000eac9 c000eace c000ead3
            0000000000000120P: c000ead8 c000eadd c000eae2 c000eae7
            0000000000000130P: c000eaec c000eaf1 c000eaf6 c000eafb

        Raises:
             ValueError: if the specified dataSize is not 1, 2, 4, or 8
             Exception: if the specified address type is unknown
        
        **IPC services required:** Memory

        """
        if _log.isEnabledFor(cli_logging.DEBUG):
            devicestr = self._device_to_devicestr(device)
            _log.caller()
            _log.debug("ENTER: memdump(device={0},address={1},count={2},dataSize={3}".\
                       format(devicestr,address,count,dataSize))        
        did = self._device_convert(device)
        # conver our string address to IPC_Address
        if dataSize not in [1, 2, 4, 8]:
            raise ValueError("Invalid memory access size {0}".format(dataSize))
        memsrvc = py2ipc.IPC_GetService("Memory")
        if isinstance(address,Address):
            address = str(address)
        address = memsrvc.AddressFromString( address )
        # must be read
        bytesread = 0
        line = []
        ascii = []
        for i in range(count):
            if (bytesread%16)==0:
                # output line if it is not the first one
                if i != 0: 
                    line = " ".join(line)
                    # TODO: add ascii display support
                    _log.result(line)
                    line = []
                    ascii = []
                if address.addressType=="Physical":
                    line.append( "{0:016x}P:".format(address.offset))
                elif address.addressType=="Linear":
                    line.append( "{0:016x}L:".format(address.offset))
                elif address.addressType=="GuestPhysical":
                    line.append( "{0:016x}P:".format(address.offset))
                elif address.addressType=="OneFieldCodeLogical":
                    line.append( "CS{0:016x}:".format(address.offset))
                elif address.addressType=="OneFieldDataLogical":
                    line.append( "DS{0:016x}:".format(address.offset))
                elif address.addressType=="OneFieldStackLogical":
                    line.append( "SS{0:016x}:".format(address.offset))
                elif address.addressType=="TwoFieldLogical":
                    line.append( "{0:04x}:{1:016x}:".format(address.segsel,address.offset))
                elif address.addressType=="ThreeFieldLogical":
                    line.append( "{0:04x}:{1:04x}:{2:016x}:".format(
                                address.ldtsel,address.segsel,address.offset))
                else:
                    raise Exception("Unknown address type: {0}={1}".\
                                    format(address.addressType,address._addressType.value))                    
            bytesread += dataSize
            d = memsrvc.ReadMemoryWithAccessWidth(did, address, 1, dataSize)
            if dataSize==1:
                ascii.append(self._byte_to_ascii( d ))
            line.append( "{0:0{width}x}".format(long(d),width=(dataSize*2) ))
            address.offset += dataSize
        _log.result(" ".join(line))
        _log.debug("EXIT: memdump")
    
    def isrunning(self, device):
        """
            
        Reports whether the probe considers this thread to be in the running state or not.
        
        Args:
            device (int): the did or alias of the device to check run status on (not 
                          needed if using from a node object).              
        
        Returns:
            True/False (python boolean) as to whether the device is running.
            
        Example:
        
            >>> ipc.threads[0].isrunning()
            False

        **IPC services required:** RunControl
        """
        if _log.isEnabledFor(cli_logging.DEBUG):
            devicestr = self._device_to_devicestr(device)
            _log.caller()
            _log.debug("ENTER: isrunning(device={0})".format(devicestr))
        ##### Required for every function:
        did = self._device_convert(device)
        #### 
        runsrvc = py2ipc.IPC_GetService("RunControl")
        runstatus = runsrvc.GetRunStatus(did)
        _log.debug("isrunning : runstatus = {0}".format(runstatus))
        result = ("Running"== runstatus)
        _log.debug("EXIT: isrunning : result = {0}".format(result) )
        return result
    
    def ishalted(self, device):
        """
            
        Reports whether the probe considers this thread to be in the halted state or not.
        
        Args:
            device (int): the did or alias of the device to check run status on (not 
                          needed if using from a node object).            
        
        Returns:
            True/False (python boolean) as to whether the device is running.
            
        Example:
        
            >>> ipc.threads[0].isrunning()
            False
        
        **IPC services required:** RunControl

        """
        if _log.isEnabledFor(cli_logging.DEBUG):
            devicestr = self._device_to_devicestr(device)
            _log.caller()
            _log.debug("ENTER: ishalted(device={0})".format(devicestr))
        ##### Required for every function:
        did = self._device_convert(device)
        #### 
        runsrvc = py2ipc.IPC_GetService("RunControl")
        runstatus = runsrvc.GetRunStatus(did)
        _log.debug("ishalted : runstatus = {0}".format(runstatus))
        result = ("Halted"== runstatus)
        _log.debug("EXIT: ishalted : result = {0}".format(result) )
        return result    

    def break_type(self, device):
        """
        
        Returns the type of the last known break event for a given device.
        
        Args:
            device (int): the did or alias of the device to return the last break_type for
        
        Returns:
            a string containing the reason for the last known break event for 
            the specified device

        Example:
            >>> ipc.threads[0].break_type()

        **IPC services required:** RunControl
        """
        ##### logging
        if _log.isEnabledFor(cli_logging.DEBUG):
            devicestr = self._device_to_devicestr(device)
            _log.caller()
            _log.debug("ENTER: break_type(device={0})".format(devicestr))
        ##### Required for every function:
        did = self._device_convert(device)
        result = py2ipc.IPC_GetService("RunControl").GetBreakType(did)
        _log.debug("EXIT : break_type : result = {0}".format(result))
        return result

    def break_subtype(self, device):
        """
        
        Returns the subtype of the last known break event for the given device.

        Args:
            device (int): the did or alias of the device to return the last break_subtype for

        Returns:
            a string containing the sub-reason for the last known break event for
            the specified device

        Example:
            >>> ipc.threads[0].break_subtype()

        **IPC services required:** RunControl
        """
        ##### logging
        if _log.isEnabledFor(cli_logging.DEBUG):
            devicestr = self._device_to_devicestr(device)
            _log.caller()
            _log.debug("ENTER: break_subtype(device={0})".format(devicestr))
        ##### Required for every function:
        did = self._device_convert(device)
        result = py2ipc.IPC_GetService("RunControl").GetBreakSubtype(did)
        _log.debug("EXIT : break_subtype : result = {0}".format(result))
        return result

    def break_address(self, device):
        """
        
        Returns the address of the last known break event for the given device.

        Args:
            device (int): the did or alias of the device to return the last break address for

        Returns:
            a string containing the address for the last known break event for
            the specified device

        Example:
            >>> ipc.threads[0].break_address()

        **IPC services required:** RunControl, Memory
        """
        ##### logging
        if _log.isEnabledFor(cli_logging.DEBUG):
            devicestr = self._device_to_devicestr(device)
            _log.caller()
            _log.debug("ENTER: break_address(device={0})".format(devicestr))
        ##### Required for every function:
        did = self._device_convert(device)
        address = py2ipc.IPC_GetService("RunControl").GetBreakAddress(did)
        _log.debug("break_address : ipc_address : {0}".format(str(address)))
        # address is an IPC_Address type in this code
        addrstr =  py2ipc.IPC_GetService("Memory").AddressToString(address)
        _log.debug("break_address : ipc_address_str : {0}".format(str(addrstr)))
        # convert to a Python Address and put in your ASMDecode object
        pyaddr = Address(addrstr)
        _log.debug("EXIT : break_address : result = {0}".format(pyaddr))
        return pyaddr

    def interestingthreads(self, device, threads=None):
        """
        Sets or gets the list of interesting threads for the specified core group.

        Args:
            device (int): the did or alias of the core group device to get or
                          set the interesting threads for.
            threads (list,None): the list of interesting threads to set for
                                 the specified core group.

        Returns:
            a list of the interesting threads for the specified core group
            device.

        **IPC services required:** RunControlConfiguration
        """
        ##### logging
        if _log.isEnabledFor(cli_logging.DEBUG):
            devicestr = self._device_to_devicestr(device)
            _log.caller()
            _log.debug("ENTER: interestingthreads(device={0})".format(devicestr))
        ##### Required for every function:
        did = self._device_convert(device)
        runcontrolconfig = py2ipc.IPC_GetService("RunControlConfiguration")
        # set the interesting threads if the argument was specified
        if threads is not None:
            thread_ids = [self._device_convert(d) for d in threads]
            runcontrolconfig.SetInterestingThreads(did, thread_ids)
        # return the currently set interesting threads for the core group
        resultingthreads = [self.base.devicelist.findByDID(d.deviceId).node for d in runcontrolconfig.GetInterestingThreads(did)]
        _log.debug("EXIT : interestinghthreads : result = {0}".format(resultingthreads))
        return NodeGroup(resultingthreads)

    def keepprobemoderedirectionset(self, device, value=None):
        """
        Gets or sets the probe mode redirection state.

        Args:
            device (int): the did or alias of the core group device.
            value  (bool, None): whether to keep probe mode redirection enabled.
                          If the value is None, the current redirection state is returned.

        Returns:
            a boolean indicating whether probe mode redirection is kept enabled

        **IPC services required:** RunControlConfiguration
        """
        ##### logging
        if _log.isEnabledFor(cli_logging.DEBUG):
            devicestr = self._device_to_devicestr(device)
            _log.caller()
            _log.debug("ENTER: keepprobemoderedirectionset(device={0})".format(devicestr))
        ##### Required for every function:
        did = self._device_convert(device)
        runcontrolconfig = py2ipc.IPC_GetService("RunControlConfiguration")
        if value is not None:
            if value:
                mode = py2ipc.IPC_Types.IPC_ProbeModeRedirectionModes.Set
            else:
                mode = py2ipc.IPC_Types.IPC_ProbeModeRedirectionModes.Auto
            runcontrolconfig.SetProbeModeRedirectionMode(did, mode)
        mode = runcontrolconfig.GetProbeModeRedirectionMode(did)
        result = mode == "Set"
        _log.debug("EXIT : keepprobemoderedirectionset : result = {0}".format(result))
        return result

    def keepprobemoderedirectioncleared(self, device, value=None):
        """
        
        Gets or sets the probe mode redirection state.

        Args:
            device (int): the did or alias of the core group device.
            value  (bool, None): whether to keep probe mode redirection disabled.
                          If the value is None, the current redirection state is returned.

        Returns:
            a boolean indicating whether probe mode redirection is kept disabled.

        **IPC services required:** RunControlConfiguration
        """
        ##### logging
        if _log.isEnabledFor(cli_logging.DEBUG):
            devicestr = self._device_to_devicestr(device)
            _log.caller()
            _log.debug("ENTER: keepprobemoderedirectioncleared(device={0})".format(devicestr))
        ##### Required for every function:
        did = self._device_convert(device)
        runcontrolconfig = py2ipc.IPC_GetService("RunControlConfiguration")
        if value is not None:
            if value:
                mode = py2ipc.IPC_Types.IPC_ProbeModeRedirectionModes.Clear
            else:
                mode = py2ipc.IPC_Types.IPC_ProbeModeRedirectionModes.Auto
            runcontrolconfig.SetProbeModeRedirectionMode(did, mode)
        mode = runcontrolconfig.GetProbeModeRedirectionMode(did)
        result = mode == "Clear"
        _log.debug("EXIT : keepprobemoderedirectioncleared : result = {0}".format(result))
        return result


    def stepintoexception(self, device, value=None):
        """
        Whether to step in to exceptions

        Args:
            device (int/obj) : device to enable/disable this for
            value (bool) : value to set (or None to get the current value)

        Returns:
            if value is None, return the current setting

        """
        ##### logging
        if _log.isEnabledFor(cli_logging.DEBUG):
            devicestr = self._device_to_devicestr(device)
            _log.caller()
            _log.debug("ENTER: stepintoexception(device={0}, value={1})".format(devicestr,str(value)))
        ##### Required for every function:
        did = self._device_convert(device)
        runcontrolconfig = py2ipc.IPC_GetService("RunControlConfiguration")
        if value is not None:
            if value:
                runcontrolconfig.SetStepIntoExceptionEnabled(did, True)
            else:
                runcontrolconfig.SetStepIntoExceptionEnabled(did, False)
            _log.debug("EXIT : stepintoexception")
            return
        else:
            # value is not nont, return the current value
            result = runcontrolconfig.GetStepIntoExceptionEnabled(did)
            _log.debug("EXIT : stepintoexception : result = {0}".format(result))
            return result



    def arch_register(self, device, regname, newValue=None):
        """
        
        Reads or writes an architectural register from the specified device.
        
        Args:
            device (int): the did or alias of the device to read the requested 
                          architectural register from
                          
            regname (str) : string of register to read from
            
            newValue (int,None) : value to write to register, or *None* if a read
                                  is being requested
                                  
        Note:
            There's no guarantee on the bit size of the returned BitData object
            matching the exact size of the register that was read
                                  
        Example:
            >>> ipc.threads[0].arch_register("eax")
            [64b] 0x000000000000F124
            >>> ipc.threads[0].arch_register("ax",2)
            >>> ipc.threads[0].arch_register('al')
            [64b] 0x0000000000000002 
        
        Raises:
             TypeError: if register name is not a string

        **IPC services required:** Register

        """
        ##### logging
        if _log.isEnabledFor(cli_logging.DEBUG):
            devicestr = self._device_to_devicestr(device)
            newValueStr = hex(newValue) if newValue != None else "None"
            _log.caller()
            _log.debug("ENTER: arch_reg(device={0},regname={1},newValue={2})".\
                       format(devicestr,regname,newValueStr))
        ##### Required for every function:
        did = self._device_convert(device)
        if not isinstance(regname,basestring):
            raise TypeError("only strings supported for register name")

        regname = regname.lower() 
        
        regsrvc = py2ipc.IPC_GetService("Register")
        if newValue==None:
            operation = regsrvc.ReadRegister(did,regname)
            if not OperationWindow.isopen(): # flush operations is not in a window
                operation.Flush()
                operation.ErrorCheck()
            bd = BitData(None, operation)
            if _log.isEnabledFor(cli_logging.DEBUG):
                _log.debug("EXIT : arch_register : result = {0}".format(bd))            
            return bd
        else:
            operation = regsrvc.WriteRegister(did,regname,newValue,numbits=512)
            if not OperationWindow.isopen(): # flush operations is not in a window
                operation.Flush()
                operation.ErrorCheck()
            return None

    def regs(self, device, **kwargs):
        """
        Function for displaying architectural registers

        Args:
            Device
        Returns:
            None

        Note: this has no return code and is for showing only
        """
        extended = kwargs.pop("extended", False)
        ##### logging
        if _log.isEnabledFor(cli_logging.DEBUG):
            devicestr = self._device_to_devicestr(device)
            _log.caller()
            _log.debug("ENTER: regs(device={0}, extended={1})".format(devicestr, str(extended)))

        did = self._device_convert(device)
        device_obj = self.devicelist.findByDID(did)
        # these registers are needed in order to do the display
        if extended:
            resultClass = ArchExtendedRegsResults
        else:
            resultClass = ArchRegsResults
        missing_regs = set(resultClass.required_registers)- set(device_obj.node.state.regs._get_visible_regs())
        if len(missing_regs)>0:
            raise ValueError("This thread does not have all the need registers for this display function")

        reg_values = resultClass()
        for reg in list(resultClass.required_registers):
            reg_values[reg] = self.arch_register(did, reg)

        if _log.isEnabledFor(cli_logging.DEBUG):
            devicestr = self._device_to_devicestr(device)
            _log.caller()
            _log.debug("EXIT: show_arch_regs(device={0})".format(devicestr))

        return reg_values

    def xmregs(self, device):
        """
        Function for displaying architectural registers

        Args:
            Device
        Returns:
            None

        Note: this has no return code and is for showing only
        """
        return self.regs(device, extended=True)

    def idcode(self, device):
        """
        Return the processor boundary scan idcode for the specified device.

        Args:
            device (int): the did or alias of the device to run idcode on (not 
                          needed if using from a node object) 

        Returns:
            Return the idcode as a string.

        **Remarks:**
            Use the idcode command to display the processor boundary scan idcode
            for a device.  The returned value is always in hexadecimal regardless 
            of the setting of the base control variable. 

        Note:
            The idcode is fetched from the hardware every time this method 
            is called. A cached version of the idcode can be obtained for
            any device (ipc.devicelist[0].idcode).

        Example:
            >>> ipc.idcode(0)
        
        Raises:
             ValueError: if the specified device does not have an idcode

        **IPC services required:** JtagAccess
        """
        ##### logging
        if _log.isEnabledFor(cli_logging.DEBUG):
            devicestr = self._device_to_devicestr(device)
            _log.caller()
            _log.debug("ENTER: idcode(device={0})".format(devicestr))
        ##### Required for every function:
        did = self._device_convert(device)
        ####
        devobj = self.devicelist.findByDID(did)
        idcodeir = getattr(devobj,"idcodeir",None)
        if idcodeir is None:
            raise ValueError("The specified device does not have an idcode")
        idcode = self.irdrscan(did, idcodeir, 32)    
        ##### logging
        # only do this if logging enabled, otherwise we could cause operation receipt to flush
        if _log.isEnabledFor(cli_logging.DEBUG):
            _log.debug("EXIT : idcode result = {0}".format(idcode))
        #####
        return "0x{:08X}".format(idcode)
    
    def irdrscanreplace(self, device, instruction, bitCount, writeData = None ):
        """
        Perform a combined IR/DR scan to the specified device, passing in the
        specified instruction for the IR scan and the specified bit count for the
        DR scan.

        Args:
            device (int): the did or alias of the device to run idcode on (not 
                          needed if using from a node object).

            instruction (int) : The instruction to scan into the device.

            bitCount (int) : The number of bits to scan from the data register of the
                         designated device as selected by the current instruction
                         register handle.

            writeData (int) :  not actually written to device since replace puts the
                         original data back.
            
        Returns:
          A BitData object containing the bits that were read back.

        
        If the writeData is an array of values and the number and size of values
        are not sufficient to match the specified bitCount, ValueError is raised.

        If the writeData is an array of values and there are more values than
        needed for the specified bitCount, the remaining values are silently
        ignored.

        Examples:
            >>> ipc.irdrscanreplace(0,2,32)
            [32b] 0x1B28A013
            >>> ipc.irdrscanreplace(0, 2, 32, [0xFF] * 4 )
            [32b] 0x1B28A013

        **IPC services required:** JtagAccess

        """
        ##### logging
        if _log.isEnabledFor(cli_logging.DEBUG):
            devicestr = self._device_to_devicestr(device)
            if isinstance(writeData,int) or isinstance(writeData,long):  writeDataStr = hex(writeData)
            else:                                                        writeDataStr = str(writeData)
            _log.caller()
            _log.debug("ENTER: irdrscanreplace(device={0},instruction=0x{1:02x}, bitCount={2}, writeData={3})"\
                       .format(devicestr,instruction,bitCount,writeDataStr))
        ##### Required for every function:
        did = self._device_convert(device)
        ####
        jtagsrvc = py2ipc.IPC_GetService("JtagAccess")        
        if isinstance(writeData,list):
            newData = BitData(0,0)
            for d in writeData:
                if isinstance(d,(int,long)):
                    newData.Append( BitData(32,d) )
                elif isinstance(d,BitData):
                    newData.Append( d )
            writeData = newData
        operation = jtagsrvc.IrDrScanReplace( did, instruction, bitCount, writeData)
        if not OperationWindow.isopen(): # flush operations is not in a window
            operation.Flush()
            operation.ErrorCheck()
        bd = BitData(None, operation)
        ##### logging
        # only do this if logging enabled, otherwise we could cause operation receipt to flush
        if _log.isEnabledFor(cli_logging.DEBUG):
            _log.debug("EXIT : irdrscanreplace result = {0}".format(bd))
        #####
        return bd
        
    def irdrscan(self, device, instruction, bitCount, data=None, writeData=None, returnData=True ):
        """

        Perform a combined IR/DR scan to the specified device, passing in the
        specified instruction for the IR scan and the specified bit count for the
        DR scan.
            
        Note:
            The legacy tools had a readData parameter...that is not supported in the
            same way in the ipccli. If you specify only data, then it is
            assumed you wanted to do a write. If you specify both data
            and writeData, then it is assumed you are doing this because
            you have not ported your script to the IPC CLI yet.  This function
            will never 'fill in' a list specified in the data parameter.
        
        
        Args:
        
            device (int): the did or alias of the device (not 
                          needed if using from a node object).
        
            instruction (int) : The instruction to scan into the device.
        
            bitCount (int) : The number of bits to scan from the data register of the
                         designated device as selected by the current instruction
                         register handle.
                         
            data (int) : can specify this or writeData with a number or BitData object
                         to write to the device (see note about backwards compatibility).
        
            writeData (int) : a number or BitData object to write to the device.
            returnData (bool) : whether to return the data from the scan that was done.
        
        Returns:
            A BitData object containing the bits that were read back.

        **IPC services required:** JtagAccess

        This method combines an IR scan with a DR scan so as to eliminate any
        possibility of something else interfering with the scan.
        
        The bitCount parameter determines the number of bits that will be
        scanned. The writeData value must contain at least as many bits as
        are to be scanned.
        
        If the writeData is an array of values and the number and size of
        values are not sufficient to match the specified bitCount, a
        CommandsErrorException is raised.
        
        If the writeData is an array of values and there are more values
        than needed for the specified bitCount, the remaining values are
        silently ignored.
        
            
        **Example 1:**
          >>> ipc.irdrscan(0,2,32)
          [32b] 0x1B28A013
        
        **Example 2:**
          >>> ipc.irdrscan(0,2,32,writeData=0xFFFFFFFF)
          [32b] 0x1B28A013

        Raises:
             AssertionError: if both data and writeData are specified
            
        **IPC services required:** JtagAccess
        """
        ##### logging
        if _log.isEnabledFor(cli_logging.DEBUG):
            devicestr = self._device_to_devicestr(device)
            if isinstance(writeData,int) or isinstance(writeData,long):  writeDataStr = hex(writeData)
            else:                                                        writeDataStr = str(writeData)
            if isinstance(data,int) or isinstance(data,long):  dataStr = hex(data)
            else:                                              dataStr = str(data)
            _log.caller()
            _log.debug("ENTER: irdrscan(device={0},instruction=0x{1:02x}, bitCount={2}, data={3}, writeData={4})"\
                       .format(devicestr,instruction,bitCount,dataStr,writeDataStr))
        ##### Required for every function:
        did = self._device_convert(device)        # TODO: performance metric to know how to best handle
        ####
        if data != None and writeData !=None:
            msg = "Only one data parameter may be specified, readData no longer supported"
            _log.error(msg)
            raise AssertionError(msg)
        
        # if only data specified, assume it is writeData 
        writeData = data if (data != None and writeData==None) else writeData  
        jtagsrvc = py2ipc.IPC_GetService("JtagAccess")
        # if we get some odd list ?
        if isinstance(writeData,list):
            newData = BitData(0,0)
            for d in writeData:
                if isinstance(d,(int,long)):
                    newData.Append( BitData(32,d) )
                elif isinstance(d,BitData):
                    newData.Append( d )
            # now put back in to our data variable
            writeData = newData
        # if we get a bitdata, convert to a number just to be safe
        if isinstance(writeData,BitData):
            writeData = long(writeData)
        operation = jtagsrvc.IrDrScan( did, instruction, bitCount, writeData, returnData=returnData)
        if returnData:
            if not OperationWindow.isopen(): # flush operations is not in a window
                operation.Flush()
                operation.ErrorCheck()
            bd = BitData.CreateFromLong(bitCount, operation, truncate=True, verify=False)
            ##### logging
            # only do this if logging enabled, otherwise we could cause operation receipt to flush
            if _log.isEnabledFor(cli_logging.DEBUG):
                _log.debug("EXIT : irdrscan result = {0}".format(bd))
            #####
            return bd
        else:
            if _log.isEnabledFor(cli_logging.DEBUG):
                _log.debug("EXIT : irdrscan returnData = False")
    
    def rawirdrscan(self, device, irBitCount, irWriteData, drBitCount, drWriteData=None, returnData=True):
        """

        Write to the instruction registers of the devices on the specified JTAG scan chain and then
        write and read the data registers that the instruction register write enabled on the scan chain.

        Args:
            device (int) : Must be a scanchain device id.
            irBitCount (int) : The number of bits to scan from the instruction registers
                               on the selected JTAG scan chain.
            irWriteData (int) : A BitData or number to use for the JTAG instruction register.
            drBitCount (int) : The number of bits to scan from the data registers
                               on the selected JTAG scan chain.
            drWriteData (int) : A BitData or python number to use for writing in to the data registers
                                on the scan chain.
            returnData (bool) : whether to return the data from the scan that was done.

        **IPC services required:** JtagAccess

        Note:
            This function is not 100% backwards compatible with the legacy tool's version of this function.
            It does not take debugport and scanchain as the first two parameters due to device handling changes.

        Example:
            >>> ipc.rawirdrscan(0,8,2,32,0)
            [32b] 0x1B28A013

        **IPC services required:** JtagAccess
        """
        ##### logging
        if _log.isEnabledFor(cli_logging.DEBUG):
            devicestr = self._device_to_devicestr(device)
            if isinstance(irWriteData,(int,long)):  irWriteDataStr = hex(irWriteData)
            else:                                   irWriteDataStr = str(irWriteData)
            if isinstance(drWriteData,(int,long)):  drWriteDataStr = hex(drWriteData)
            else:                                   drWriteDataStr = str(drWriteData)
            _log.caller()
            _log.debug("ENTER: rawirdrscan(device={0},irBitCount={1}, irWriteData={2}, drBitCount={3}, drWriteData={4})"\
                       .format(devicestr,irBitCount,irBitCount,irWriteDataStr,drBitCount,drWriteDataStr))

        ##### Required for every function:
        did = self._device_convert(device)
        # should we force this? is it worth the performance hit to do this check? 
        # device = self.devicelist.findByDID(did)
        #if getattr(device,"nodetype","") != "scanchain": 
        #    raise Exception("Must pass a scan chain device for rawirdrscan commands")
        jtagsrvc = py2ipc.IPC_GetService("JtagAccess")
        jlock = self.base.device_locker(did)
        #if not jlock.lock_present():
        #    raise RuntimeError("JTAG lock must be present for this function to work")
        # only support returnData for new py2ipc
        with jlock:
            # IR
            jtagsrvc.Shift(did,"ShfIR",irBitCount,irWriteData, returnData=False)
            op = jtagsrvc.GoToState(did,"RTI",1,"COUNT") # 1=numClocks.
            # DR
            operation = jtagsrvc.Shift(did,"ShfDR",drBitCount, drWriteData, returnData=returnData)
            last_op = jtagsrvc.GoToState(did,"RTI",1,"COUNT") # 1=numClocks...TODO ?
            # don't need shitop.ErrorCheck() because this is within lock/unlock
        if returnData:
            if not OperationWindow.isopen(): # flush operations is not in a window
                last_op.Flush()
                last_op.ErrorCheck()
            bd = BitData.CreateFromLong(None, operation, truncate=True, verify=False)
            if _log.isEnabledFor(cli_logging.DEBUG):
                _log.debug("EXIT : rawirdrscan : result = {0}".format(bd))
            return bd
        else:
            if _log.isEnabledFor(cli_logging.DEBUG):
                _log.debug("EXIT : rawirdrscan : returnData=False")
            return 
  
    def irdrscanverify(self, device, instruction, bitCount, data=None, writeData = None ):
        """

        Perform an IR/DR scan, while verifying the length of the DR.
                    
        Args:
        
            device (int): the did or alias of the device (not 
                          needed if using from a node object).
        
            instruction (int) : The instruction to scan into the device.
        
            bitCount (int) : The number of bits to scan from the data register of the
                         designated device as selected by the current instruction
                         register handle.
                         
            data (int) : can specify this or writeData with a number or BitData object
                         to write to the device (see note about backwards compatibility).
        
            writeData (int) : a number or BitData object to write to the the device.
        
        Returns:
            A BitData object containing the bits that were read back.

        **Example 1:**
          >>> ipc.irdrscanverify(0,2,32)
          [32b] 0x1B28A013
        
        **Example 2:**
          >>> ipc.irdrscanverify(0,2,32,writeData=0xFFFFFFFF)
          [32b] 0x1B28A013
            
        Raises:
             AssertionError: if both data and writeData are specified
            
        **IPC services required:** JtagAccess
        """
        ##### logging
        if _log.isEnabledFor(cli_logging.DEBUG):
            devicestr = self._device_to_devicestr(device)
            if isinstance(writeData,int) or isinstance(writeData,long):  writeDataStr = hex(writeData)
            else:                                                        writeDataStr = str(writeData)
            if isinstance(data,int) or isinstance(data,long):  dataStr = hex(data)
            else:                                              dataStr = str(data)
            _log.caller()                     
            _log.debug("ENTER: irdrscanverify(device={0},instruction=0x{1:02x}, bitCount={2}, data={3}, writeData={4})"\
                       .format(devicestr,instruction,bitCount,dataStr,writeDataStr))
        ##### Required for every function:
        did = self._device_convert(device)        # TODO: performance metric to know how to best handle
        ####
        if data != None and writeData !=None:
            msg = "Only one data parameter may be specified, readData no longer supported"
            _log.error(msg)
            raise AssertionError(msg)
        
        # if only data specified, assume it is writeData 
        writeData = data if (data != None and writeData==None) else writeData  
        jtagsrvc = py2ipc.IPC_GetService("JtagAccess")
        # if we get some odd list ?
        if isinstance(writeData,list):
            newData = BitData(0,0)
            for d in writeData:
                if isinstance(d,(int,long)):
                    newData.Append( BitData(32,d) )
                elif isinstance(d,BitData):
                    newData.Append( d )
            # now put back in to our data variable
            writeData = newData
        # if we get a bitdata, convert to a number just to be safe
        if isinstance(writeData,BitData):
            writeData = long(writeData)
        operation = jtagsrvc.IrDrScanVerify( did, instruction, bitCount, writeData)
        if not OperationWindow.isopen(): # flush operations is not in a window
            operation.Flush()
            operation.ErrorCheck()
        bd = BitData.CreateFromLong(None, operation, truncate=True, verify=False)
        ##### logging
        # only do this if logging enabled, otherwise we could cause operation receipt to flush
        if _log.isEnabledFor(cli_logging.DEBUG):
            _log.debug("EXIT : irdrscanverify result = {0}".format(bd))
        #####
        return bd
 
    def irdrscanrmw(self, device, instruction, writeData, maskData, bitCount=None, returnData=True):
        """

        Perform a combined IR/DR scan to the specified device, which reads the
        original contents, modifies the bits specified in the mask with the
        write data, then writes the modified data back out.
            
        Args:
        
            device (int): the did or alias of the device (not 
                          needed if using from a node object).
        
            instruction (int) : The instruction to scan into the device.
                                
            writeData (int) : The data that should be written to the bits
                         specified by the mask.
        
            maskData (int) : What bits of the data will be modified.
            bitCount (int) : optional parameter to force specify the
                             number of bits in the scan.
            returnData (bool) : whether to return the data from the scan that was done (defaults to True).
        
        Note:
            The legacy tools required a BitData. We have added a new bitCount
            parameter that allows this function to be called without
            first creating the bitdata object or using it for the
            number of bits to scan instead of the bitdata size        
        
        Returns:
            the data from the scan, if returnData is True
            
        **Example 1:**
          >>> itp.irdrscanrmw(0, 2, BitData(32, 0x00000004), BitData(32, 0x0000000F))
          
        **Example 2:**
          >>> itp.irdrscanrmw(0, 2, 0x30, 0xC0, 32)

        Raises:
             TypeError: if bitCount is not specified and writeData is not a BitData object
            
        **IPC services required:** JtagAccess
            
        """
        ##### logging
        if _log.isEnabledFor(cli_logging.DEBUG):
            devicestr = self._device_to_devicestr(device)
            if isinstance(writeData,(int,long)):  writeDataStr = hex(writeData)
            else:                                 writeDataStr = str(writeData)
            if isinstance(maskData,(int,long)):  maskDataStr = hex(maskData)
            else:                                maskDataStr = str(maskData)
            _log.caller()
            _log.debug("ENTER: irdrscanrmw(device={0},instruction=0x{1:02x}, writeData={2}, maskData={3}, bitCount={4})"\
                       .format(devicestr,instruction,writeDataStr,maskDataStr,bitCount))
        ##### Required for every function:
        did = self._device_convert(device)        # TODO: performance metric to know how to best handle
        ####

        jtagsrvc = py2ipc.IPC_GetService("JtagAccess")
        # if bitCount is not specified, make sure we have a bitData
        if bitCount == None:
            if not isinstance(writeData,BitData):
                raise TypeError("If bitcount is not specified, then writeData must be a BitData object")
            bitCount = writeData.BitSize
        
        operation = jtagsrvc.IrDrScanReadModifyWrite( did, instruction, bitCount, writeData, maskData, returnData=returnData)
        if returnData:
            if not OperationWindow.isopen(): # flush operations is not in a window
                operation.Flush()
                operation.ErrorCheck()
            bd = BitData.CreateFromLong(None, operation, truncate=True, verify=False) # ?
            ##### logging
            # only do this if logging enabled, otherwise we could cause operation receipt to flush
            if _log.isEnabledFor(cli_logging.DEBUG):
                _log.debug("EXIT : irdrscanrmw result = {0}".format(bd))
            #####
            return bd 
        else:
            if _log.isEnabledFor(cli_logging.DEBUG):
                _log.debug("EXIT : irdrscanrmw returnData=False")
        
    def drscan(self, device, bitCount, data=None, writeData=None, returnData=True):
        """
        
        Read the data register of devices in the target boundary scan. This 
        function is not be used outside of a lock window; be sure to use device_locker
        in conjunction with this function.
        
        Note:
            The legacy tools had a readData parameter which is not supported in the
            same way in the ipccli. If you specify only data, then it is
            assumed you want to do a write. If you specify both data
            and writeData, an assertion will be thrown.  This function
            will never 'fill in' a list specified in the data parameter        
        
        Args:
        
            device (int): the did or alias of the device (not 
                          needed if using from a node object) 
            
            bitCount (int) : The number of bits to scan from the data register of the
                            designated device as selected by the current instruction
                            register handle.
                            
            data (int) : can specify this or writeData with a number or BitData object
                         to write to the device (see note about backwards compatibility)
        
            writeData (int) : a number or BitData object to write to the the device
            returnData (bool) : whether to return the data from the scan that was done (defaults to True)

        The irscan must be used with a lock acquired so that no operations occur
        between an irscan and any subsequent drscans.

        Example:
        
           >>> with ipc.device_locker() as lock:
           ...     # no other jtag operations from other windows or Trace32
           ...     # will interrupt the instructions in this block
           ...     itp.irscan(0, 2)
           ...     itp.drscan(32)

        Raises:
             AssertionError: if both data and writeData are specified
             RuntimeError: if there is no JTAG lock present
            
        **IPC services required:** JtagAccess

        See Also:
            - :py:meth:`~ipccli.ipc_env.ipc_commands.Commands.device_locker`
            - :py:meth:`~ipccli.ipc_env.ipc_commands.Commands.irscan`
        
        """
        ##### logging
        if _log.isEnabledFor(cli_logging.DEBUG):
            devicestr = self._device_to_devicestr(device)
            if isinstance(writeData,int) or isinstance(writeData,long):  writeDataStr = hex(writeData)
            else:                                                        writeDataStr = str(writeData)
            if isinstance(data,int) or isinstance(data,long):  dataStr = hex(data)
            else:                                              dataStr = str(data)
            _log.caller()
            _log.debug("ENTER: drscan(device={0},bitCount={1},data={2},writeData={3})".\
                       format(devicestr,bitCount,dataStr,writeDataStr))
        ##### Required for every function:
        did = self._device_convert(device)
        ####
        if data != None and writeData !=None:
            msg = "Only one data parameter may be specified, readData no longer supported"
            _log.error(msg)
            raise AssertionError(msg)
                        
        # if only data specified, assume it is writeData 
        writeData = data if (data and writeData==None) else writeData        
        jtagsrvc = py2ipc.IPC_GetService("JtagAccess")
        if isinstance(writeData,list):
            newData = BitData(0,0)
            for d in writeData:
                if isinstance(d,(int,long)):
                    newData.Append( BitData(32,d) )
                elif isinstance(d,BitData):
                    newData.Append( d )
            writeData = newData
        
        if not DeviceLocker.lock_present():
            raise RuntimeError("JTAG lock must be present for this function to work, please check the help on this function")
            
        shiftop = jtagsrvc.Shift(did,"ShfDR",bitCount, writeData, returnData=returnData)
        rti = jtagsrvc.GoToState(did,"RTI",1,"COUNT", returnData=returnData) # 1=numClocks...TODO ?
        # don't need shitop.ErrorCheck() because this is within lock/unlock
        if returnData:
            if not OperationWindow.isopen(): # flush operations is not in a window
                rti.Flush() # which should also flush shiftop
                rti.ErrorCheck() # make sure no errors on our shift
            bd = BitData.CreateFromLong(None, shiftop,  truncate=True, verify=False)
            if _log.isEnabledFor(cli_logging.DEBUG):
                _log.debug("EXIT : drscan : result = {0}".format(bd))
            return bd
        else:
            if _log.isEnabledFor(cli_logging.DEBUG):
                _log.debug("EXIT : drscan : returnData=False")
    
    def irscan(self, device, instruction, bitCount=None, returnData=True):
        """
        
        Write the designated instruction handle into the instruction register of
        the specified device.

        Args:
        
            device (int): the did or alias of the device (not 
                          needed if using from a node object).
                          
            instruction (int) : The instruction to scan into the device.
            bitCount (int) : The number of bits to scan into the device.
            returnData (bool) : whether to return the data from the scan that was done.

        Returns:
            the data from the shift operation.

        Use the irscan command with the drscan command to read from or write to
        the data register of a device on the target system boundary scan chain.
        The irscan command writes the designated instruction value into the
        instruction register of the specified device.

        The irscan must be used with a lock acquired so that no operations occur
        between an irscan and any subsequent drscans.

        Example:
        
           >>> with ipc.device_locker() as lock:
           ...     # no other jtag opperations from other windows or Trace32
           ...     # will interrupt the instructions in this block
           ...     itp.irscan(0, 2)
           ...     itp.drscan(32)

        Raises:
             RuntimeError: if there is no JTAG lock present
             ValueError: if the ir length is not specified when doing raw scans using the scan chain
            
        **IPC services required:** JtagAccess, JtagConfiguration
    
        See Also:
            - :py:meth:`~ipccli.ipc_env.ipc_commands.Commands.device_locker`
            - :py:meth:`~ipccli.ipc_env.ipc_commands.Commands.drscan`
                
        """
        ##### logging
        if _log.isEnabledFor(cli_logging.DEBUG):
            _log.caller()
            devicestr = self._device_to_devicestr(device)
            _log.debug("ENTER: irscan(device={0},instruction=0x{1:02x},bitCount={2})".\
                       format(devicestr,
                              instruction,
                              bitCount))
        ##### Required for every function:
        did = self._device_convert(device)
        ####
        jtagsrvc    = py2ipc.IPC_GetService("JtagAccess")
        jtagcfgsrvc = py2ipc.IPC_GetService("JtagConfiguration")
        
        if not DeviceLocker.lock_present():
            raise RuntimeError("JTAG lock must be present for this function to work, please check the help on this function")
            
        if bitCount == None:
            # should be this:
            #if devobj.nodetype == "scanchain":
            # but this workaround will have to exist until we have unique names for all devices
            if did & 0xFF000 == 0x12000: # this is a workaround until we get unique names for scan chains
                raise ValueError("when doing raw scans using the scan chain, you must specify the ir length")
            bitCount = jtagcfgsrvc.GetDeviceIrLength( did ) 
        operation = jtagsrvc.Shift(did,"ShfIR", bitCount, instruction, returnData=returnData)
        rti = jtagsrvc.GoToState(did,"RTI",1,"COUNT", returnData=returnData) # 1=numClocks.
        if returnData:
            # make sure flushed..
            if not OperationWindow.isopen(): # flush operations is not in a window
                rti.Flush()
                rti.ErrorCheck() # this should flag errors on either op
            # return any data from our shift
            bd = BitData.CreateFromLong(None, operation, truncate=True, verify=False)
            if _log.isEnabledFor(cli_logging.DEBUG):
                _log.debug("EXIT : irscan : result = {0}".format(bd))
            return bd
        else:
            if _log.isEnabledFor(cli_logging.DEBUG):
                _log.debug("EXIT : irscan ")
            
    def interfaceport_read(self, device, byte_count):
        """
        
        Reads data from an interface port device, returning a bytearray of the
        bytes read. This function is not to be used outside of an interface
        port window; be sure to use interfaceport_window in conjunction with
        this function.
        
        Note:
            If there is no data available then the returned bytearray will have
            zero length.
        
        Args:
        
            device (int): The did or alias of the device (not 
                          needed if using from a node object).
                          
            byte_count (int): The maximum number of bytes to read.
        Returns:
            a bytearray containing the data read.

        **IPC services required:** InterfacePort
        
        See Also:
            - :py:meth:`~ipccli.ipc_env.ipc_commands.Commands.interfaceport_window`

        """
        ##### logging
        if _log.isEnabledFor(cli_logging.DEBUG):
            _log.caller()
            _log.debug("ENTER: interfaceport_read(device=0x{0:08x},byte_count={1})".\
                       format(device,byte_count))
        ##### Required for every function:
        did = self._device_convert(device)
        
        interfaceport_srv = py2ipc.IPC_GetService("InterfacePort")
        data = interfaceport_srv.Read(did, byte_count);

        if _log.isEnabledFor(cli_logging.DEBUG):
            _log.debug("EXIT: interfaceport_read")
            
        return data
            
    def interfaceport_write(self, device, bytes):
        """
        
        Writes data to an interface port device, returning the number of bytes
        written. This function is not to be used outside of an interface port
        window; be sure to use interfaceport_window in conjunction with this
        function.
        
        Args:
        
            device (int): The did or alias of the device (not 
                          needed if using from a node object).
                          
            bytes (bytearray): The data to write.

        **IPC services required:** InterfacePort
        
        See Also:
            - :py:meth:`~ipccli.ipc_env.ipc_commands.Commands.interfaceport_window`

        """
        ##### logging
        if _log.isEnabledFor(cli_logging.DEBUG):
            _log.caller()
            _log.debug("ENTER: interfaceport_write(device=0x{0:08x},bytes={1})".\
                       format(device,bytes))
        ##### Required for every function:
        did = self._device_convert(device)
        
        interfaceport_srv = py2ipc.IPC_GetService("InterfacePort")
        bytes_written = interfaceport_srv.Write(did, bytes);

        if _log.isEnabledFor(cli_logging.DEBUG):
            _log.debug("EXIT: interfaceport_write")
            
        return bytes_written

    def interfaceport_read_to_shared_memory(self, device, byte_count, byte_offset, shared_memory):
        """
        
        Reads data from an interface port device into a pre-allocated shared memory region.
        This function is not to be used outside of an interface
        port window; be sure to use interfaceport_window in conjunction with
        this function.
        
        Args:
        
            device (int): The did or alias of the device (not 
                          needed if using from a node object).
                          
            byte_count (int): The maximum number of bytes to read.

            byte_offset (int): The offset into the shared memory region at which to start
                               storing data.

            shared_memory (int or SharedMemory): The handle corresponding to a shared memory
                                                 object, or the object itself.

        Returns:
            The number of bytes read from the interface port device

        **IPC services required:** InterfacePort
        
        See Also:
            - :py:meth:`~ipccli.ipc_env.ipc_commands.Commands.interfaceport_window`

        """
        ##### logging
        if _log.isEnabledFor(cli_logging.DEBUG):
            _log.caller()
            _log.debug("ENTER: interfaceport_read_to_shared_memory(device=0x{0:08x},byte_count={1},byte_offset={2})".\
                       format(device,byte_count,byte_offset))
        ##### Required for every function:
        did = self._device_convert(device)

        if isinstance(shared_memory, SharedMemory):
            handle = shared_memory._handle
        else:
            handle = shared_memory
        
        interfaceport_srv = py2ipc.IPC_GetService("InterfacePort")
        bytes_read = interfaceport_srv.ReadToSharedMemory(did, byte_count, byte_offset, handle);

        if _log.isEnabledFor(cli_logging.DEBUG):
            _log.debug("EXIT: interfaceport_read_to_shared_memory")
            
        return bytes_read
            
    def interfaceport_write_to_shared_memory(self, device, byte_count, byte_offset, shared_memory):
        """
        
        Writes data from a pre-allocated shared memory region to an interface port device,
        returning the number of bytes written. This function is not to be used outside of
        an interface port window; be sure to use interfaceport_window in conjunction with this
        function.
        
        Args:
        
            device (int): The did or alias of the device (not 
                          needed if using from a node object).
                          
            byte_count (int): The number of bytes to write.

            byte_offset (int): The offset into the shared memory region from which
                               the data should be fetched.

            shared_memory (int or SharedMemory): The handle corresponding to a shared memory
                                                 object, or the object itself.

        Returns:
            The number of bytes written to the interface port device.

        **IPC services required:** InterfacePort
        
        See Also:
            - :py:meth:`~ipccli.ipc_env.ipc_commands.Commands.interfaceport_window`

        """
        ##### logging
        if _log.isEnabledFor(cli_logging.DEBUG):
            _log.caller()
            _log.debug("ENTER: interfaceport_write_to_shared_memory(device=0x{0:08x},byte_count={1},byte_offset={2})".\
                       format(device,byte_count,byte_offset))
        ##### Required for every function:
        did = self._device_convert(device)

        if isinstance(shared_memory, SharedMemory):
            handle = shared_memory._handle
        else:
            handle = shared_memory
        
        interfaceport_srv = py2ipc.IPC_GetService("InterfacePort")
        bytes_written = interfaceport_srv.WriteToSharedMemory(did, byte_count, byte_offset, handle);

        if _log.isEnabledFor(cli_logging.DEBUG):
            _log.debug("EXIT: interfaceport_write_to_shared_memory")
            
        return bytes_written
        
    def interfaceport_window(self, device, filename=None, accessmode="read"):
        """
        This returns a context manager which opens/closes an interface port window
        for streaming data in a file or in memory to or from an interface port device.
        
        Args:
            device (int): The did or alias of the device (not 
                          needed if using from a node object).
            filename (string): name of a file to read from or write data to (defaults to None)
            accessmode (string): the mode of the interface port device access (defaults to "read")
        
        Use only within a 'with' block.
    
        Examples:
    
           >>> with dev.node.interfaceport_window() as window:
           ...     dev.node.interfaceport_read(...)
    
           >>> with dev.node.interfaceport_window(accessmode="write") as window:
           ...     dev.node.interfaceport_write(...)
                  
           >>> with dev.node.interfaceport_window(accessmode="readwrite") as window:
           ...     dev.node.interfaceport_write(...)
           ...     dev.node.interfaceport_read(...)
                  
           >>> with dev.node.interfaceport_window(filename="data.bin") as window:
           ...     # Wait until read is complete
                  
           >>> with dev.node.interfaceport_window(filename="data.bin", accessmode="write") as window:
           ...     # Wait until write is complete

        **IPC services required:** InterfacePort

        """
        did = self._device_convert(device)
        return InterfaceportWindow(did, filename, accessmode)
                
    def halt(self, device):
        """

        Will be used to stop the specified processor. If the processor is already halted,
        no exception/error will be given.

        Args:
        
            device (int): the did or alias of the device (not 
                          needed if using from a node object)        
        
        Raises: 
            An Exception will be raised if halt is not successful

        **IPC services required:** RunControl, RunControlConfiguration, Device
                
        See Also: 
            - :py:meth:`~ipccli.ipc_env.ipc_commands.Commands.go`
            - :py:meth:`~ipccli.ipc_env.ipc_commands.Commands.isrunning`
            - :py:meth:`~ipccli.ipc_env.ipc_commands.Commands.ishalted`
        
        """
        did = self._device_convert(device)
        runsrvc = py2ipc.IPC_GetService("RunControl")
        runconfigsrvc = py2ipc.IPC_GetService("RunControlConfiguration")
        devicesrvc = py2ipc.IPC_GetService("Device")
        device = self.devicelist.findByDID(did)

        # Remember if run control was disabled before the halt attempt
        if device.devicetype == "LogicalGroupCore":
            coregroupid = did
        else:
            coregroupid = devicesrvc.GetCoreGroupIdForDevice(did)
        run_control_was_disabled = not runconfigsrvc.IsRunControlEnabled(coregroupid)

        operation = runsrvc.Halt(did)
        if not OperationWindow.isopen(): # flush operations is not in a window
            operation.Flush()
            operation.ErrorCheck()

        # If run control was disabled and the halt attempt succeeded, then 
        # update the supported breaks on devices
        if run_control_was_disabled:
            self.base._add_breaks()

    def go(self, device):
        """
        

        Will be used to start the specified processor. If the processor is already executing,
        no exception/error will be given.
        
        Note:
            Legacy tools supported a "goTil" parameter, which is not supported

        Args:
        
            device (int): the did or alias of the device (not 
                          needed if using from a node object)        
        
        Returns: 
            An Exception will be raised if go is not successful

        **IPC services required:** RunControl
        
        See Also: 
            - :py:meth:`~ipccli.ipc_env.ipc_commands.Commands.halt`
            - :py:meth:`~ipccli.ipc_env.ipc_commands.Commands.isrunning`
            - :py:meth:`~ipccli.ipc_env.ipc_commands.Commands.ishalted`
                
        """
        did = self._device_convert(device)
        runsrvc = py2ipc.IPC_GetService("RunControl")
        operation = runsrvc.Go(did)
        if not OperationWindow.isopen(): # flush operations is not in a window
            operation.Flush()
            operation.ErrorCheck()
        
    def step(self, device,stepType="Into",steps=1):
        """
        
        Perform a single step of program execution.
        
        Args:
            device (int): the did or alias of the device (not 
                          needed if using from a node object)        
            stepType (str) : Type of step to perform; valid values are below.
            steps (int) : Number of steps to take before stopping (default is 1).
            
        Valid values for stepType are:
            - "Into"
            - "Over"
            - "Out"
            - "Branch"

        Example:
            >>> ipc.threads[0].step()
                [SLM_C0_T0]   Single STEP break at 0x0008:0x00000000000F2C94
            >>> ipc.threads[0].step("Over",2 ) # step over statements 2 times
                [SLM_C0_T0]   Single STEP break at 0x0008:0x00000000000F2C8E

        Raises:
             TypeError: if stepType is not a string
             ValueError: if stepType is not a valid value
            
        **IPC services required:** RunControl

        """
        did = self._device_convert(device)
        if not isinstance(stepType,basestring):
            raise TypeError("stepType should be string, see help for valid values")
        if stepType.lower() not in ["into","over","out","branch"]:
            raise ValueError("Invalide stepType {0}".format(stepType))
        # make sure case matches enum
        stepType = stepType[0].upper() + stepType[1:]
        runsrvc = py2ipc.IPC_GetService("RunControl")
        # i guess default step should be into ?
        operation = runsrvc.Step(did,stepType,steps)
        if not OperationWindow.isopen(): # flush operations is not in a window
            operation.Flush()
            operation.ErrorCheck()

    def step_mode(self, device=None, step_mode=None):
        """
        Set or return the current step mode for the core group specified

        Args:
            device: this must be a core group device
            step_mode: if None, return current setting, otherwise
                       set the step mode to use. Valid value sare:
                       "all", "targetted"

        Returns:
            "all" or "targetted" if step_mode is None

        """
        if device is not None:
            did = self._device_convert(device)
            devobj = self.devicelist.findByDID(did)
            if devobj.devicetype != "LogicalGroupCore":
                # then attempt to get the logical group core it belongs to
                core_group = getattr(devobj, "coregroup", None)
                if core_group == None:
                    raise ValueError("can not set step_mode since we cannot find the core group for this device")
                core_group = self.devicelist.get(core_group, None)
                if core_group is None:
                    raise RuntimeError("Could not find %s core group for device: %s"%(core_group, device))
                devobj = core_group
            # get underlying did for the core_group
            did = devobj.did
        else:
            # default to GPC core group?
            core_group = self.devicelist.get('GPC', None)
            if core_group is None:
                raise RuntimeError("Could not find GPC core group")
            did = core_group.did

        if _log.isEnabledFor(cli_logging.DEBUG):
            devicestr = self._device_to_devicestr(did)
            _log.caller()
            _log.debug("ENTER: step_mode(device={0},step_mode={1})".\
                       format(devicestr,str(step_mode)))

        runsrvc = py2ipc.IPC_GetService("RunControlConfiguration")
        # Getting the mode
        if step_mode is None:
            mode = runsrvc.GetStepMode(did)
            if mode == "ReleaseAllThreads":
                retvalue = "all"
            elif mode == "ReleaseTargetedThread":
                retvalue =  "targetted"
            else:
                raise ValueError("Unexpected step mode: %s"%step_mode)
            # returning value
            if _log.isEnabledFor(cli_logging.DEBUG):
                _log.debug("EXIT: step_mode : result = {0}".format(retvalue))
            return retvalue

        # setting the mode
        else:
            if step_mode == "all":
                mode = "ReleaseAllThreads"
            elif step_mode == "targetted":
                mode = "ReleaseTargetedThread"
            else:
                raise ValueError("Unexpected step mode: %s"%step_mode)
            runsrvc.SetStepMode(did, mode)
            # returning value
            if _log.isEnabledFor(cli_logging.DEBUG):
                _log.debug("EXIT: step_mode")

 
    def wrsubpir(self, device, instruction):
        """

        Submits a single instruction to the CPU for execution.

        Args:
        
            device (int): the did or alias of the device (not 
                          needed if using from a node object).
            instruction (str): the instruction opcodes (mnemonics) to execute.
        
        See Also: 
            - :py:meth:`~ipccli.ipc_env.ipc_commands.Commands.readpdr`
        
        **IPC services required:** RunControl
                
        """
        did = self._device_convert(device)
        runsrvc = py2ipc.IPC_GetService("RunControl")
        runsrvc.SubmitInstruction(did, instruction)

    def readpdr(self, device, offset=0):
        """

        Retrieve the result of an instruction previously submitted for
        execution.

        Args:
        
            device (int): the did or alias of the device (not 
                          needed if using from a node object).
            offset (int): the offset of the result to be retrieved.
        
        **IPC services required:** RunControl
                
        See Also: 
            - :py:meth:`~ipccli.ipc_env.ipc_commands.Commands.wrsubpir`
        
        """
        did = self._device_convert(device)
        runsrvc = py2ipc.IPC_GetService("RunControl")
        result = runsrvc.GetInstructionResult(did, offset)
        return result.value

                
    def cpuid(self,device, eax, ecx ):
        """
        
        Args:
            device (int): the did or alias of the device (not 
                          needed if using from a node object)
            eax (int) : 32bit bit value to put in eax for cpuid instruction
            ecx (int) : 32bit bit value to put in ecx for cpuid instruction
        
        Returns:
            Dictionary with  eax, ebx, edx, and ecx values. 
            
        Example:
            >>> ipc.threads[0].cpuid(0,0)
            {'eax': [64b] 0x000000000000000B,
             'edx': [64b] 0x0000000049656E69,
             'ebx': [64b] 0x00000000756E6547,
             'ecx': [64b] 0x000000006C65746E}
        Raises:
             IPC_Error: if not halted
                 
        **IPC services required:** Register

        """
        ##### logging
        if _log.isEnabledFor(cli_logging.DEBUG):
            devicestr = self._device_to_devicestr(device)
            _log.caller()
            _log.debug("ENTER: cpuid(device={0},eax={1},ecx={2})".\
                       format(devicestr,eax,ecx))
        ##### Required for every function:
        did = self._device_convert(device)
                
        regsrvc = py2ipc.IPC_GetService("Register")
        ret = {}
        # reg_out so that we don't collide with input params
        eax_out = regsrvc.ReadCPUIDValue(did,eax,ecx,'eax')
        ebx_out = regsrvc.ReadCPUIDValue(did,eax,ecx,'ebx')
        ecx_out = regsrvc.ReadCPUIDValue(did,eax,ecx,'ecx')
        edx_out = regsrvc.ReadCPUIDValue(did,eax,ecx,'edx')
        if not OperationWindow.isopen(): # flush operations is not in a window
            edx_out.Flush()
            # now check them all for errors
            eax_out.ErrorCheck()
            ebx_out.ErrorCheck()
            ecx_out.ErrorCheck()
            edx_out.ErrorCheck()
        ret['eax'] = BitData.CreateFromLong( None, eax_out, truncate=True, verify=False)
        ret['ebx'] = BitData.CreateFromLong( None, ebx_out, truncate=True, verify=False)
        ret['ecx'] = BitData.CreateFromLong( None, ecx_out, truncate=True, verify=False)
        ret['edx'] = BitData.CreateFromLong( None, edx_out, truncate=True, verify=False)
        if _log.isEnabledFor(cli_logging.DEBUG):
            _log.debug("EXIT: cpuid: result = {0}".format(str(ret)))        
        return ret    
        
    def cpuid_eax(self,device, eax = 1, ecx = 0):
        """
        
        Return the processor identification and feature information stored in the
        EAX register.
        
        Args:
            device (int): the did or alias of the device (not 
                          needed if using from a node object)        
            eax (int) : 32bit bit value to put in eax for cpuid instruction
            ecx (int) : 32bit bit value to put in ecx for cpuid instruction
            
        Returns:
            Value of eax after cpuid call
            
        Example:
            >>> ipc.threads[0].cpuid_eax(0,0)
            [64b] 0x000000000000000B        
            
        Raises:
             IPC_Error: if not halted

        **IPC services required:** Register
                        
        """
        ##### logging
        if _log.isEnabledFor(cli_logging.DEBUG):
            devicestr = self._device_to_devicestr(device)
            _log.caller()
            _log.debug("ENTER: cpuid_eax(device={0},eax={1},ecx={2})".\
                       format(devicestr,eax,ecx))
        ##### Required for every function:
        did = self._device_convert(device)
        regsrvc = py2ipc.IPC_GetService("Register")
        eax_out = regsrvc.ReadCPUIDValue(did,eax,ecx,'eax')
        if not OperationWindow.isopen(): # flush operations is not in a window
            eax_out.Flush()
        return BitData(None, eax_out )

    def cpuid_ebx(self,device, eax = 0, ecx = 0):
        """
        
        Return the processor identification and feature information stored in the
        EBX register.
        
        Args:
            device (int): the did or alias of the device (not 
                          needed if using from a node object)        
            eax (int) : 32bit bit value to put in eax for cpuid instruction
            ecx (int) : 32bit bit value to put in ecx for cpuid instruction
            
        Returns:
            Value of ebx after cpuid call

        Example:
            >>> ipc.threads[0].cpuid_ebx(0,0)
            [64b] 0x00000000756E6547
            
        Raises:
             IPC_Error: if not halted          

        **IPC services required:** Register

        """
        ##### logging
        if _log.isEnabledFor(cli_logging.DEBUG):
            devicestr = self._device_to_devicestr(device)
            _log.caller()
            _log.debug("ENTER: cpuid_ebx(device={0},eax={1},ecx={2})".\
                       format(devicestr,eax,ecx))
        ##### Required for every function:
        did = self._device_convert(device)
        regsrvc = py2ipc.IPC_GetService("Register")        
        ebx_out = regsrvc.ReadCPUIDValue(did,eax,ecx,'ebx')
        if not OperationWindow.isopen(): # flush operations is not in a window
            ebx_out.Flush()
        return BitData(None, ebx_out )

    def cpuid_ecx(self,device, eax = 0, ecx = 0):
        """
        
        Return the processor identification and feature information stored in the
        ECX register.
        
        Args:
            device (int): the did or alias of the device (not 
                          needed if using from a node object)        
            eax (int) : 32bit bit value to put in eax for cpuid instruction
            ecx (int) : 32bit bit value to put in ecx for cpuid instruction
            
        Returns:
            Value of ecx after cpuid call
            
        Example:
            >>> ipc.threads[0].cpuid_ecx(0,0)
            [64b] 0x000000006C65746E    
            
        Raises:
             IPC_Error: if not halted

        **IPC services required:** Register

        """
        ##### logging
        if _log.isEnabledFor(cli_logging.DEBUG):
            devicestr = self._device_to_devicestr(device)
            _log.caller()
            _log.debug("ENTER: cpuid_ecx(device={0},eax={1},ecx={2})".\
                       format(devicestr,eax,ecx))
        ##### Required for every function:
        did = self._device_convert(device)
        regsrvc = py2ipc.IPC_GetService("Register")        
        ecx_out = regsrvc.ReadCPUIDValue(did,eax,ecx,'ecx')
        if not OperationWindow.isopen(): # flush operations is not in a window
            ecx_out.Flush()
        return BitData(None, ecx_out )
    
    def cpuid_edx(self,device, eax = 0, ecx = 0):
        """
        
        Return the processor identification and feature information stored in the
        EDX register.
        
        Args:
            device (int): the did or alias of the device (not 
                          needed if using from a node object)  
            eax (int) : 32bit bit value to put in eax for cpuid instruction
            ecx (int) : 32bit bit value to put in ecx for cpuid instruction
            
        Returns:
            Value of edx after cpuid call
            
        Example:
            >>> ipc.threads[0].cpuid_edx(0,0)
            [64b] 0x0000000049656E69    
            
        Raises:
             IPC_Error: if not halted            
    
        **IPC services required:** Register

        """
        ##### logging
        if _log.isEnabledFor(cli_logging.DEBUG):
            devicestr = self._device_to_devicestr(device)
            _log.caller()
            _log.debug("ENTER: cpuid_edx(device={0},eax={1},ecx={2})".\
                       format(devicestr,eax,ecx))
        ##### Required for every function:
        did = self._device_convert(device)
        regsrvc = py2ipc.IPC_GetService("Register")        
        edx_out = regsrvc.ReadCPUIDValue(did,eax,ecx,'edx')
        if not OperationWindow.isopen(): # flush operations is not in a window
            edx_out.Flush()
        return BitData( None, edx_out )
            

    def _instruction_size(self, did):
        """
        Private function, future existing/api use not guaranteed

        Returns:
            '16Bit', '32Bit', or '64Bit'
        """
        proc_mode = self.processor_mode(did)
        proc_mode = [p.strip() for p in proc_mode.split(",")]
        if len(proc_mode)==1:
            # this means it was unknown
            return self._asmmode
        if proc_mode[0] == "SixtyFourBit":
            return '64Bit'
        elif proc_mode[1] == "Protected":
            # lookup csar.d
            csavalue = self.arch_register(did, "CSA")
            if (csavalue & 0x4000): # bit 14
                return '32Bit'
            else:
                return '16Bit'
        else:
            return '16Bit'


    def processor_mode(self, device):
        """
        Return the execution mode that the thread is in.

        Args:
            device (did) : the did of a thread device (not needed if using
                          from a node object).

        Returns:
            The execution mode, as a string.

        Valid Return values are:
            - "SixtyFourBit, "
            - "Compatibility, "
            - "Legacy, "

        Followed by one of:
            - "Virtual86"
            - "Protected"
            - "Real"

        Then possibly ", Paged"
        
        Example:
            >>> ipc.threads[0].processor_mode()
            'Legacy, Protected'        

        """
        ##### logging
        if _log.isEnabledFor(cli_logging.DEBUG):
            devicestr = self._device_to_devicestr(device)
            _log.caller()
            _log.debug("ENTER: processor_mode(device={0})".format(devicestr))
        ##### Required for every function:
        did = self._device_convert(device)
        devobj = self.devicelist.findByDID(did)
        #####
        try:
            cr0value = self.arch_register(did, "CR0")
            csavalue = self.arch_register(did, "CSA")
            if getattr(devobj, "coregroup", "GPC") == "GPC":
                efervalue = self.msr(did, 0xC0000080)
            else:
                efervalue = 0

            # check if L is set
            if (csavalue & 0x2000) == 0x2000:
                procmode = "SixtyFourBit, "
            # Check if the Long Mode Enable(EFER.LMA) bit is ON
            elif (efervalue & 0x400) == 0x400:
                procmode = "Compatibility, "
            else:
                procmode = "Legacy, "

            # Check if the protect enable(CR0.PE) bit is ON
            if (cr0value & 0x1) == 0x1:
                # Check if VM bit is ON in eflags register
                vm_value = self.arch_register(did, 'VM')
                vm_mode_str = 'Virtual86' if vm_value else 'Protected'
                procmode = procmode + vm_mode_str
            else:
                procmode = procmode + "Real"
            # Check if the paging enable(CR0.PG) bit is ON
            if (cr0value & 0x80000000) == 0x80000000:
                procmode = procmode + ", Paged"
        except py2ipc.IPC_Error as e:
            procmode = "Unknown"
            _log.debug("thread_status failed to get procmode:")
            _log.debug("--------------------")
            _log.debug("{0}".format(traceback.format_exc()))
            _log.debug("--------------------")
            # end if Halted
        if _log.isEnabledFor(cli_logging.DEBUG):
            _log.debug("EXIT : processor_mode : result = {0}".format(procmode))
        return procmode

    def thread_status(self,device):
        """
        
        Display the Processor's running state and what execution mode it is in.
        
        Args:
            device (int): the did or alias of the device (not 
                          needed if using from a node object).
        Example:
            >>> ipc.threads[0].thread_status()
            Status for       : SLM_C0_T0
              Processor      : Halted
              Processor mode : Legacy, Protected

        **IPC services required:** RunControl, Register

        """
        ##### logging
        if _log.isEnabledFor(cli_logging.DEBUG):
            devicestr = self._device_to_devicestr(device)
            _log.caller()
            _log.debug("ENTER: thread_status(device={0})".format(devicestr))
        ##### Required for every function:
        did = self._device_convert(device)
        devobj = self.devicelist.findByDID(did)
        #####
        devicesrvc = py2ipc.IPC_GetService("Device")
        runsrvc = py2ipc.IPC_GetService("RunControl")
        runconfigsrvc = py2ipc.IPC_GetService("RunControlConfiguration")
        runstatus = runsrvc.GetRunStatus( did )
        procmode = "Unavailable while {0}".format( runstatus )
        if runstatus=="Halted":
            coregroupid = devicesrvc.GetCoreGroupIdForDevice(did)
            if runconfigsrvc.IsRunControlEnabled(coregroupid):
                procmode = self.processor_mode(did)
            else:
                procmode = "Unavailable when run control is disabled"

        _log.result("Status for       : {0}".format(devobj.alias) )
        _log.result("  Processor      : {0}".format(runstatus) )
        _log.result("  Processor mode : {0}".format(procmode) )
        
        if _log.isEnabledFor(cli_logging.DEBUG):
            _log.debug("EXIT : thread_status")

    def runcontrolenable(self, device):
        """

        Enables run control functionality for the specified core group.
        
        Args:
            device (int): the did or alias of the core group device (not 
                          needed if using from a node object).
        Example:
            >>> ipc.devs.gpc.runcontrolenable()

        **IPC services required:** RunControlConfiguration
        """
        ##### logging
        if _log.isEnabledFor(cli_logging.DEBUG):
            devicestr = self._device_to_devicestr(device)
            _log.caller()
            _log.debug("ENTER: runcontrolenable(device={0})".format(devicestr))

        did = self._device_convert(device)
        runcontrolconfigsrvc = py2ipc.IPC_GetService("RunControlConfiguration")
        runcontrolconfigsrvc.SetRunControlEnabled(did, True)
        # if run-control enable is changed, then the breaks may have changed.
        self.base._add_breaks()
        
        if _log.isEnabledFor(cli_logging.DEBUG):
            _log.debug("EXIT : runcontrolenable")

    def runcontroldisable(self, device):
        """

        Disables run control functionality for the specified core group.

        Disabling run control functionality for a core group will suspend all
        target accesses supporting run control (including background tasks such
        as TAP status polling) and prevent run control operations from being
        invoked on any threads in the core group until re-enabled.
        
        Args:
            device (int): the did or alias of the core group device (not 
                          needed if using from a node object).
        Example:
            >>> ipc.devs.gpc.runcontroldisable()

        **IPC services required:** RunControlConfiguration
        """
        ##### logging
        if _log.isEnabledFor(cli_logging.DEBUG):
            devicestr = self._device_to_devicestr(device)
            _log.caller()
            _log.debug("ENTER: runcontroldisable(device={0})".format(devicestr))

        did = self._device_convert(device)
        runcontrolconfigsrvc = py2ipc.IPC_GetService("RunControlConfiguration")
        runcontrolconfigsrvc.SetRunControlEnabled(did, False)
        
        if _log.isEnabledFor(cli_logging.DEBUG):
            _log.debug("EXIT : runcontroldisable")

    def isruncontrolenabled(self, device):
        """

        Check whether run control functionality is enabled for the specified
        core group.
        
        Args:
            device (int): the did or alias of the core group device (not 
                          needed if using from a node object).

        Returns:
            A boolean value indicating whether run control is enabled.

        Example:
            >>> ipc.devs.gpc.isruncontrolenabled()
            True

        **IPC services required:** RunControlConfiguration
        """
        ##### logging
        if _log.isEnabledFor(cli_logging.DEBUG):
            devicestr = self._device_to_devicestr(device)
            _log.caller()
            _log.debug("ENTER: isruncontrolenabled(device={0})".format(devicestr))

        did = self._device_convert(device)
        runcontrolconfigsrvc = py2ipc.IPC_GetService("RunControlConfiguration")
        enabled = runcontrolconfigsrvc.IsRunControlEnabled(did)
        
        if _log.isEnabledFor(cli_logging.DEBUG):
            _log.debug("EXIT : isruncontrolenabled")

        return enabled

    def enable_device(self, device):
        """
        Enable the specified device.
        
        Args:
            device (int): the did or alias of the device to enable (not 
                          needed if using from a node object).

        Example:
            >>> ipc.devs.jtagscanchain0.enable()

        **IPC services required:** DeviceConfiguration
        """
        ##### logging
        if _log.isEnabledFor(cli_logging.DEBUG):
            devicestr = self._device_to_devicestr(device)
            _log.caller()
            _log.debug("ENTER: enable_device(device={0})".format(devicestr))

        did = self._device_convert(device)
        devicesrv = py2ipc.IPC_GetService("Device")
        devicesrv.SetDeviceEnabled(did, True)
        
        if _log.isEnabledFor(cli_logging.DEBUG):
            _log.debug("EXIT : enable_device")

    def disable_device(self, device):
        """
        Disable the specified device.
        
        Args:
            device (int): the did or alias of the device to disable (not 
                          needed if using from a node object).

        Example:
            >>> ipc.devs.jtagscanchain0.disable()

        **IPC services required:** DeviceConfiguration
        """
        ##### logging
        if _log.isEnabledFor(cli_logging.DEBUG):
            devicestr = self._device_to_devicestr(device)
            _log.caller()
            _log.debug("ENTER: disable_device(device={0})".format(devicestr))

        did = self._device_convert(device)
        devicesrv = py2ipc.IPC_GetService("Device")
        devicesrv.SetDeviceEnabled(did, False)
        
        if _log.isEnabledFor(cli_logging.DEBUG):
            _log.debug("EXIT : disable_device")



    def _getall_bps(self):
        bpsrvc = py2ipc.IPC_GetService("BreakpointManagement")
        bplist = []
        for bp in bpsrvc.GetBreakpoints( self.base._domain  ):
            bplist.append( IpcCliBreakpoint( bp ) ) 
        return bplist
    
    def brnew(self,device, address, breakType="exe", dbreg='Any', enable=True, breakpointId="", dataSize=1):
        """Creates a new breakpoint.
        
        Args:
            device (int) : do not specify when using via ipc.threads[0].brnew or when using via ipc.brnew
            address (string or ipccli.Address) : the address where the breakpoint is to be created
            breakType (str) :  A string containing the type of the break point to create (complete list below)
                              If *None* then defaults to 'exe global' type.
            dbreg (int) : Which debug register to use for the breakpoint information
                      (0 - 3).  If *None* then no dbreg override is used.
            enable (bool) :  Initial enable state for the new breakpoint (True or False).
                             Set to *None* defaults to True (enabled).
            breakpointId (str) : (optional) ID to use for the breakpoint.
                                 The BreakpointID must currently be a string.
                                 If this ID already exists, an error is raised.
                                 If *None*, a new breakpoint ID is created.
            dataSize (int) : dataSize must be one of 1, 2, 4, or 8
        
        Complete list of possible values for breakType::
                              
            "sw"         -- Software breakpoint
            "exe"        -- Hardware execution breakpoint
            "exe global" -- Hardware execution breakpoint (Global is across all tasks)
            "exe local"  -- Hardware execution breakpoint (Local is for current task)
            "acc"        -- Hardware memory access breakpoint
            "acc global" -- Hardware memory access breakpoint (Global is across all tasks)
            "acc local"  -- Hardware memory access breakpoint (Local is for current task)
            "wr"         -- Hardware memory write breakpoint
            "wr global"  -- Hardware memory write breakpoint (Global is across all tasks)
            "wr local"   -- Hardware memory write breakpoint (Local is for current task)
            "io"         -- Hardware I/O access breakpoint                   
            
            
        When used globally, then a new breakpoint is added to each new GPC thread. Note: if linear
        or physical is not specified, it is assumed the number is an offset for the CS selector. If $
        is specified, then the current instruction pointer is substituted in. See ipccli.Address 
        for additional address strings supported.

        Examples:

            >>> itp.threads[0].brnew("0x1000L")
            >>> itp.threads[0].brnew("0x100P", "io")
            >>> itp.threads[0].brnew("0x1200L", "exe global")
            >>> itp.threads[0].brnew("0x1200L", "exe global", enable = False)
            >>> # Create new breakpoint on thread 0 at address 0x1100P with specific breakpoint ID MyBrkpoint.
            >>> itp.threads[0].brnew("0x1100L", breakpointId="MyBrkpoint")

        Raises:
            Exception: if the address is not a string or an ipccli.Address object
            ValueError: if a breakpointId is specified when adding a new breakpoint to all threads
            ValueError: if dataSize is not 1,2,4, or 8
        
        **IPC services required:** Breakpoint, BreakpointManagement
        
        """
        if not isinstance(address,(Address,basestring)):
            raise Exception("address must be a string or ipccli.Address object")
        ##### logging
        if _log.isEnabledFor(cli_logging.DEBUG):
            devicestr = self._device_to_devicestr(device)
            _log.caller()
            _log.debug("ENTER: brnew(device={0},address={1},breakType={2},dbreg={3},enable={4},breakpointId={5},dataSize={6})".\
                        format(devicestr,address, breakType, dbreg, enable, breakpointId, dataSize))

        if device==None and breakpointId != "" and breakType != "sw":
            raise ValueError("can not specify breakpointId if you are adding a new breakpoint to all threads")

        if device==None:
            # then we do this for all threads:
            devicelist = []            
            for dev in self.devicelist:
                if dev.nodetype == 'thread' and getattr(dev,"coregroup","GPC") == "GPC" and getattr(dev,"isenabled",True):
                    devicelist.append(dev.did)
                    if self.base._version_check("OpenIPC", build=1744, error=False, revision=512361):
                        # special case for software breakpoints...we only do the first thread we find
                        if breakType == 'sw':
                            break
                    else:
                        # Break out regardless of the type of breakpoint. These older versions
                        # did not include a fix in the Lauterbach RunControl plugin, which
                        # resulted in odd behavior when a hardware breakpoint is created
                        # on more than one thread
                        break
        else:
            devicelist = [ self._device_convert(device) ]
            
        bplist = []
        for dev in devicelist:
            # make sure have address as expected, should be string at this point
            if isinstance(address,basestring):
                devobj = self.devicelist.findByDID(dev)
                address = self._parse_address(devobj, address, "cs", return_type="string")
            bp = IpcCliBreakpoint.new( dev, address, breakType, dbreg, enable, dataSize, name=breakpointId)
            # Let's not add the same breakpoint to the list more than once
            matched = filter(lambda x: x.breakpointId == bp.breakpointId, bplist)
            if next(iter(matched), None) is None:
                bplist.append( bp )
    
        _log.debug("EXIT: brnew")
        # if single device specified, return single bp
        if device==None:
            return bplist
        elif len(bplist) > 1 or len(bplist)==0: # not sure how this happens, but just in case
            return bplist
        else: # single device was specified so return the one new breakpoint created
            return bplist[0]
        
    def brget(self,device=None,breakpointId=None):
        """
        
        Gets the specified breakpoint, or a list of all breakpoints for the given device,
        if no breakpointId is specified.
        
        Args:
            device (int) : do not specify when using via ipc.threads[0].brget.
            breakpointId (int/str) : (optional) the name of ID of a breakpoint to retrieve
            
        Used to return all breakpoint for all GPC threads (if device = None and breakpointId = None)
        or to search all breakpoints for the specified name or breakpointId.
        
        Example:
        
            >>> ipc.brget() # returns ALL breakpoints
            >>> ipc.threads[0].breget() # returns ALL breakpoints on thread 0
            >>> ipc.threads[0].brget("#001") # returns the br #001 IF it exists on this thread

        Raises:
             Exception: if the specified breakpoint is not found
             TypeError: if the breakpoint is of an unknown type
        
        **IPC services required:** Breakpoint, BreakpointManagement

        """
        ##### logging
        if _log.isEnabledFor(cli_logging.DEBUG):
            devicestr = self._device_to_devicestr(device)
            _log.caller()
            _log.debug("ENTER: brget(device={0},breakpointId={1})".\
                        format(devicestr,breakpointId))
        # did = None if device = None, otherwise convert it
        did = self._device_convert(device) if device != None else None
        
        if isinstance(breakpointId,basestring):
            # if breakpoint ID is a string, then go through and search for it
            allbps = self._getall_bps()
            for bp in allbps:
                if (bp.name == breakpointId or bp.breakpointId==breakpointId) \
                    and \
                    ((bp.deviceId == did or did==None) or (bp.breakType == "sw")):
                    return bp
            else:
                raise Exception("breakpoint with name '{0}' not found".format(breakpointId))
        elif isinstance(breakpointId,(int,long)):
            # take the slightly faster route:
            bpsrvc = py2ipc.IPC_GetService("BreakpointManagement")
            try:
                ipcbp = bpsrvc.GetBreakpoint(breakpointId)
                bp = IpcCliBreakpoint( ipcbp )
            except py2ipc.IPC_Error as e:
                if e.code != 0x80020003: # not the error I expected ("breakpoint ID does not exist")
                    raise e
                # raise CLI exception so we have common exception for errors in this function
                raise Exception("breakpoint with breakpointId '{0}' not found".format(breakpointId))
            # if device was specified, make sure it matches (and it is not sw)
            if (did != None and bp.deviceId != did and bp.breakType != "sw"):               
                raise Exception("breakpoint with breakpointId '{0}' not found for this device".format(breakpointId))
            # turn in to a CLI breakpoint and return that.
            return bp
        elif breakpointId == None:
            # if not breakpoint specified, return them all for the specified device
            allbps = self._getall_bps()
            if device == None:
                return allbps
            else:
                # iterate through devices and return only breakpoints for the specified device
                bplist = []
                for bp in allbps:
                    if bp.deviceId == did or bp.breakType == 'sw':
                        bplist.append(bp)
                return bplist
        else:
            raise TypeError("Unknown type for breakpoint: {0},{1}".format(breakpointId,type(breakpointId)))
        
    def brremove(self,device=None,*breakpointIds):
        """
        
        Remove some or all breakpoints.

        Args:
            device (int) : do not specify, only use when accessing via ipc.cmds.brremove
            breakpointIds (ints or strings) : IDs of zero or more breakpoints to remove.
                       The BreakpointID can be a string or a number.
                       If no breakpoint IDs are specified, remove all
                       breakpoints.

        Example:
          >>> ipc.brremove() # remove all breakpoints
          >>> ipc.brremove(1, 3, 4) # remove breakpoints 1, 3, and 4
          >>> ipc.brremove("brkpoint1")  # remove breakpoint "brkpoint1"
          >>> ipc.brremove("BP1", "BP2", "BP3") #remove breakpoint "BP1","BP2","BP3"  
         
        Raises:
             Exception: if the specified breakpoint is not found
        
        **IPC services required:** Breakpoint, BreakpointManagement
             
        """
        # if breakpointId is not its already a breakpoint object, then use brget to find it
        if len(breakpointIds)==0:
            bps = self._getall_bps()
        else:
            bps = []
            for bp in breakpointIds:
                if isinstance( bp, IpcCliBreakpoint ):
                    pass
                else:
                    bp = self.brget(device,bp)
                bps.append(bp)
        # if list, remove them all
        for b in bps: b.remove()

    def brenable(self,device=None,*breakpointIds):
        """
        
        Enable some or all breakpoints.

        Args:
            device (int) : do not specify, only use when accessing via ipc.cmds.brenable
            breakpointIds (int, str) : IDs of zero or more breakpoints to enable.
                       The BreakpointID can be a string or a number.
                       If no breakpoint IDs are specified, enable all
                       breakpoints.

        Example:
          >>> ipc.brenable() # enable all breakpoints
          >>> ipc.brenable(1, 3, 4) # enable breakpoints 1, 3, and 4
          >>> ipc.brenable("brkpoint1")  # enable breakpoint "brkpoint1"
          >>> ipc.brenable("BP1", "BP2", "BP3") #enable breakpoint "BP1","BP2","BP3"        

        Raises:
             Exception: if the specified breakpoint is not found
        
        **IPC services required:** Breakpoint, BreakpointManagement

        """
        if len(breakpointIds)==0:
            bps = self._getall_bps()
        else:
            bps = []
            for bp in breakpointIds:
                if isinstance( bp, IpcCliBreakpoint ):
                    pass
                else:
                    bp = self.brget(device,bp)
                bps.append(bp)
        # if list, remove them all
        for b in bps: b.enable()
        
    def brdisable(self,device=None,*breakpointIds):
        """
        
        Disable some or all breakpoints.

        Args:
            device (int) : do not specify, only use when accessing via ipc.cmds.brdisable
            breakpointIds (int, str) : IDs of zero or more breakpoints to disable.
                       The BreakpointID can be a string or a number.
                       If no breakpoint IDs are specified, disable all
                       breakpoints.

        Example:
          >>> ipc.brdisable() # disable all breakpoints
          >>> ipc.brdisable(1, 3, 4) # disable breakpoints 1, 3, and 4
          >>> ipc.brdisable("brkpoint1")  # disable breakpoint "brkpoint1"
          >>> ipc.brdisable("BP1", "BP2", "BP3") #disable breakpoint "BP1","BP2","BP3"        
        
        Raises:
             Exception: if the specified breakpoint is not found
        
        **IPC services required:** Breakpoint, BreakpointManagement

        """
        if len(breakpointIds)==0:
            bps = self._getall_bps()
        else:
            bps = []
            for bp in breakpointIds:
                if isinstance( bp, IpcCliBreakpoint ):
                    pass
                else:
                    bp = self.brget(device,bp)
                bps.append(bp)
        # if list, remove them all
        for b in bps: b.disable()

        
    def brchange(self,device=None,breakpointId=None,**kwargs):
        """
        
        Change a breakpoint.
        
        Args:
            device (int) : do not specify, only use when accessing via ipc.cmds.brchange
            breakpointId (int, str) : the breakpoint to change.
                       The BreakpointID can be a string or a number.
            kwargs : must be used to specify what will be changed about the breakpoint. No defaults will be reset.
        
        **Valid args:**
            - address
            - breakType
            - dbreg
            - enable
            - dataSize

        Example:
            >>> ipc.brchange(1,Address="0x700")
        
        Raises:
            ValueError: if an invalid argument is specified

        **IPC services required:** Breakpoint, BreakpointManagement

        """
        # let this thing pass/fail based on whether breakpoint exists
        bpobj = self.brget(None,breakpointId)
        # check one at a time...
        kwargs['deviceId'] = device # slight tweak to device
        validargs = ['address','breakType','dbreg','enable','dataSize','deviceId']
        for arg in validargs:
            value = kwargs.pop(arg,None)
            if value != None:
                setattr(bpobj,arg,value)
        if len(kwargs)>0:
            raise ValueError("Unknown kwargs provided: {0}".format(str(kwargs)))
            
    def autoscandr(self,device,instruction,maxBitLength=65536):
        """

        Perform a combined IR/DR scan to the specified device, passing in the
        specified instruction for the IR scan and returning the appropriate amount
        of data by auto-discovering the length of the data to read.  Also returns
        the quality of the scan chain.
    
    
        Args:

          device (int,str) : Device ID or alias of the device.  Can also be a Node
                          object.
    
          instruction (int) : The instruction to scan into the device (an 8-bit handle).
    
          maxBitLength : Specifies the maximum bit length to assume for the scan
                          chain.  This number of bits are scanned in and a sentinel
                          handle is looked for.  maxBitLength must be less than or equal to 65536 (defaults to 65536).
    
        Returns:

            A tuple containing a BitData object which contains the bits that were read
            back and a string indicating the quality of the scan chain.
    
        The continuity string can contain one or more of the following, delimited
        by commas:

         - GOOD
         - OPEN
         - SHORT
         - INVERTED    
         - CORRUPTION
    
        Example:
          >>> (returnData, continuity) = ipc.autoscandr(0, 2)
    
        **IPC services required:** JtagDiagnostic

        """
        ##### logging
        if _log.isEnabledFor(cli_logging.DEBUG):
            devicestr = self._device_to_devicestr(device)
            maxbitstr = "{0:d}".format(maxBitLength) if maxBitLength != None else "None"
            _log.caller()
            _log.debug("ENTER: autoscandr(device={0},instruction={1},maxBitLength={2})".\
                        format(devicestr,instruction,maxbitstr))
        maxBitLength = 65536 if maxBitLength == None else maxBitLength
        # convert to did
        if not isinstance(maxBitLength, (int,long)): 
            raise TypeError("Wrong type for maxBitLength, must be an int")
        # convert to did
        did = self._device_convert(device)
        srvc = py2ipc.IPC_GetService("JtagDiagnostic")
        data,numbits,traits = srvc.CaptureDr(did,instruction,maxBitLength)
        # convert to python bitdata
        data = BitData(numbits,data)
        _log.debug("EXIT: autoscandr")
        return (data,traits)
                    
    def i2c_raw_read(self,device,idbyte,byteCount,options=0):
        """
        
        Performs a raw read on the specified I2C bus.
        
        Args:
            device (int) : The ID of the I2C bus to issue the transaction on.
            idbyte (int) : This is the very first "raw" byte issued on the bus, immediately
                           after a Start or Repeated Start event. The least signficant bit
                           will be blindly overwritten with a 1 to indicate a read operation.
                           In most simple cases, this byte indicates the 7-bit address of the
                           I2C slave on the target, shifted to the left by 1 bit. This may
                           also become part of a larger 10-bit, HS sequence, or special
                           control code. Please see an I2C specification for details.
            byteCount (int):  The number of bytes to read from the I2C bus (probably from a
                              targeted slave device on the bus).
            options (int): default=0, bit0 indicates a forced Stop between this and the next possible
                           scan within a Lock window. All other bits are currently reserved.
        Returns:
            A bit data object where the first byte is the slave ACK indication padded to a full byte, 
            and the other bytes of the bit data are the data bytes read.
            
        Note:
            This particular function requires the USER to check for 
            the ACK/NACK
            
        Raises:
            This does not raise NACK on error (unless the IPC API does due to some error)
            
        See Also:
            - :py:meth:`~ipccli.ipc_env.ipc_commands.Commands.device_locker`
            - :py:meth:`~ipccli.ipc_env.ipc_commands.Commands.i2c_raw_write`
            - :py:meth:`~ipccli.ipc_env.ipc_commands.Commands.i2cscan`            
            
        **IPC services required:** I2CAccess

        """
        ##### logging
        if _log.isEnabledFor(cli_logging.DEBUG):
            devicestr = self._device_to_devicestr(device)
            _log.caller()
            _log.debug("ENTER: i2c_rawread(device={0},idbyte=0x{1:x},byteCount={2},options={3})".\
                        format(devicestr,idbyte,byteCount,options))
        # sometimes we may still get old debugport number
        did = self._convert_debug_port_i2c(device)
        srvc = py2ipc.IPC_GetService("I2CAccess")
        # TODO: WORKAROUND - all i2c needs to be within lock for now
        with self.base.device_locker():
            operation = srvc.I2cRawRead(did,idbyte,byteCount,options)
        if not OperationWindow.isopen():
            operation.Flush()
            operation.ErrorCheck()
        data = BitData.CreateFromLong(None, operation, truncate=True, verify=False)
        if _log.isEnabledFor(cli_logging.DEBUG):
            _log.debug("EXIT : i2c_raw_read : result = {0}".format(data))
        return data
      
    def i2c_raw_write(self,device,idbyte,byte_data,options=0):
        """
        
        Performs a raw write on the specified I2C bus.

        Args:
            device (int) : The ID of the I2C bus to issue the transaction on.
            idbyte (int) : This is the very first "raw" byte issued on the bus, immediately
                           after a Start or Repeated Start event. The least signficant bit
                           will be blindly overwritten with a 0 to indicate a write operation.
                           In most simple cases, this byte indicates the 7-bit address of the
                           I2C slave on the target, shifted to the left by 1 bit. This may
                           also become part of a larger 10-bit, HS sequence, or special
                           control code. Please see an I2C specification for details.
            byte_data (list):  The bytes to write to the I2C bus.
            options (int): default=0, bit0 indicates a forced Stop between this and the next possible
                           scan within a Lock window. All other bits are currently reserved.
        Returns:
            None
            
        Note:
            This particular function requires the USER to check for 
            the ACK/NACK
            
        Raises:
            This does not raise NACK on error (unless the IPC API does due to some error)
            
        See Also:
            - :py:meth:`~ipccli.ipc_env.ipc_commands.Commands.device_locker`
            - :py:meth:`~ipccli.ipc_env.ipc_commands.Commands.i2c_raw_read`
            - :py:meth:`~ipccli.ipc_env.ipc_commands.Commands.i2cscan`
        
        **IPC services required:** I2CAccess
                    
        """
        ##### logging
        if _log.isEnabledFor(cli_logging.DEBUG):
            devicestr = self._device_to_devicestr(device)
            _log.caller()
            _log.debug("ENTER: i2c_raw_write(device={0},idbyte=0x{1:x},byte_data={2},options={3})".\
                        format(devicestr,idbyte,byte_data,options))
        # sometimes we may still get old debugport number
        did = self._convert_debug_port_i2c(device)              
        srvc = py2ipc.IPC_GetService("I2CAccess")
        # TODO: WORKAROUND - all i2c needs to be within lock for now
        with self.base.device_locker():
            operation = srvc.I2cRawWrite(did,idbyte,byte_data,options)
        if not OperationWindow.isopen():
            operation.Flush()
            operation.ErrorCheck()
        data = BitData.CreateFromLong(None, operation, truncate=True, verify=False)
        if _log.isEnabledFor(cli_logging.DEBUG):
            _log.debug("EXIT : i2c_raw_write : result = {0}".format(data))
        return data

    def i2cscan(self,device,addressData,writeData,readLength=0,hiSpeed=False):
        """
           
        Perform an I2C scan.

        Args:
            device (int) : The I2C deviceid or device object.
            addressData (int): The 7-bit BitData for the I2C slave address (10-bit not supported yet).
            writeData (int) : The BitData object containing the bits to write;
                              OR can be a list of numbers (each entry will be masked to 8 bits).
            readLength (int) :  The number of bits to read; should be 0 or a factor of 8
            hiSpeed (bool) : Boolean switch for high speed mode; not supported on some probes.
            
            
        Note:
            This function checks the ACK and returns only the data. This is to be backwards
            compatible with a legacy tool, so if multiple i2c's should be
            queued up, then use the i2c_raw_read/ic2_raw_write with an i2c lock.

        Returns:
            None (if readLength 0), or a BitData object containing the read data

        Example:
            >>> # to read a byte from a dimm eeprom (if it is on the i2c bus)
            >>> data = self.base.cmds.i2cscan(0, 0xA0>>1, [0], 2)
            
        Raises:
            ValueError: if writeData is not a list of bytes or a BitData object
            ValueError: if writeData is empty and readLength is 0
            ValueError: if readLength is not a multiple of 8
            IOError: if an ACK was not received for all writes
            IOError: if a NACK was received on a read

        See Also:
            - :py:meth:`~ipccli.ipc_env.ipc_commands.Commands.device_locker`
            - :py:meth:`~ipccli.ipc_env.ipc_commands.Commands.i2c_raw_read`
            - :py:meth:`~ipccli.ipc_env.ipc_commands.Commands.i2c_raw_write`
        
        **IPC services required:** I2CAccess
                                
        """
        ##### logging
        if _log.isEnabledFor(cli_logging.DEBUG):
            devicestr = self._device_to_devicestr(device)
            _log.caller()
            _log.debug("ENTER: i2cscan(device={0},addressData={1},writeData={2},readLength={3},hiSpeed={4})".\
                        format(devicestr,addressData,writeData,readLength,str(hiSpeed)))
            
        # sometimes we may still get old debugport number
        did = self._convert_debug_port_i2c(device)
        
        if isinstance(writeData,BitData):
            # convert to list of bytes
            writeData = writeData.ToRawBytes()
        elif isinstance(writeData,list):
            # as expected
            writeData = writeData
        elif writeData == None:
            # this is ok...
            pass
        else:
            raise ValueError("writeData must be list of bytes or a BitData object")
        if (writeData == None or writeData==[]) and  readLength == 0:
            raise ValueError("writeData empty and readLength==0, no i2c transaction to execute for that")
        if readLength > 0: # convert to bytes
            readBytes = readLength // 8
            if readLength % 8 > 0:
                raise ValueError("readLength must be a multiple of 8")
        # IPC API wants 8 bit address, if we were given a bitdata, then it
        # may have too few bits to support this << operation
        idbyte = long(addressData) << 1
        srvc = py2ipc.IPC_GetService("I2CAccess")
        wrbitdata = rdbitdata = None # default to no data returned
        # Must be within lock to get our repeated start
        with self.base.device_locker():
            # if we have write data, then do write first
            if writeData != [] and writeData != None:
                wroperation = srvc.I2cRawWrite(did,idbyte,writeData,options=0)
                wrbitdata = BitData(None, wroperation)
            if readLength > 0:
                rdoperation = srvc.I2cRawRead(did,idbyte,readBytes,options=0)
                rdbitdata = BitData(None, rdoperation)
        # make sure we recieved all acks on the write
        if wrbitdata != None:
            if wrbitdata.ToRawBytes() != [1]*(len(wrbitdata)//8):
                raise IOError("Did not get ACKs for all writes")
        if rdbitdata != None:
            # on read bit data the first byte is the ack
            ack = rdbitdata[0:7]
            if ack != 1:
                # TODO: need to change to correct exception type
                raise IOError("NACK received")
            rdbitdata = rdbitdata[8:]

        # logging only
        if _log.isEnabledFor(cli_logging.DEBUG):
            if rdbitdata:
                _log.debug("EXIT : i2cscan : result = {0}".format(rdbitdata))
            else:
                _log.debug("EXIT : i2cscan")
        return rdbitdata

    def getpin(self,device,pin):
        """
        
        Gets the specified pin singal state (asserted/deaserted) for the specified debugport device.
        
        Note: On some probes this **ALWAYS** reports *True* for output pins.
        
        Args:
          device (int) : This can be debug port # (for backwards compatibility) 
                         or the DID of the debug port device
          pin (int/str): This can be an integer value from 0-9 for the hook, or
                         one of the strings supported by the IPC.
                           
        Returns:
            True - if the hook is logically asserted.
            False - if the hook is logically deasserted.
        
        The various pins supported by this function may differ in active state (high or low). 
        for example, the HOOK pins are active-low signals (except HOOK0, the Obs pins are also 
        active-low signals. This command returns the *logical* value of the pin, not the 
        *electrical* value of the pin.
        
        Example:
            >>> ipc.getpin(0, 6) # get HOOK6 status from debug port 0
            >>> ipc.getpin(0, "HOOK0") # get HOOK0 status from debug port 0
            >>> ipc.getpin(0, "OBS_FN_A0") # get OBS_FN_A0 status from debug port 0
            >>> ipc.getpin(0, "OBS_DATA_D0") # get OBS_DATA_D0 status from debug port 0
        
        Raises:
            ValueError: if pin integer value provided is greater than 9
            IPC_Error: if pin string value is not a supported pin
            IPC_Error: if device is not a valid device
            
        **IPC services required:** InterfacePins
        
        """
        ##### logging
        if _log.isEnabledFor(cli_logging.DEBUG):
            devicestr = self._device_to_devicestr(device)
            _log.caller()
            _log.debug("ENTER: getpin(device={0},pin={1})".format(devicestr,pin))
        
        ipins = py2ipc.IPC_GetService("InterfacePins")
        
        if isinstance(pin,(int,long)):
            if pin > 9: raise ValueError("if hook is a number it must be less than or equal to 9")
            # convert to enum required by py2ipc
            pin = "HOOK{0}".format(pin) 
        # sometimes we may still get old debugport number
        did = self._convert_debug_port(device)
        # convert hook if needed
        
        pinstatus = ipins.GetPinState(did, pin)
        _log.debug("EXIT: getpin : result = {0}".format(pinstatus))
        # convert to True/False for if asserted or not
        return pinstatus == "Assert"
    
    def setpin(self,device,pin,state):
        """
        Holds the specified pin in the specified state.
        
        Args:
          device (int) : This can be debug port # (for backwards compatibility) 
                         or the DID of the debug port device
          pin (int/str): This can be an integer value from 0-9 for the hook, or
                         one of the strings supported by the IPC.
          state :        True (non-zero) means set the hook LOW;
                         False (zero) means set the hook HIGH.
        
        Returns:
            None
        
        In a typical configuration, HOOK0 is the power-good signal and HOOK6
        is the reset occurred signal.  These are considered "input" pins and
        generally shouldn't be asserted/deasserted.  However, not all pin
        configurations are the same so the setpin() command allows any pin signal
        to be set. The user is expected to understand the debug port's pin
        configuration before using this command.
        
        Note:  
            This command operates in terms of the _logical_ value, not the
            _electrical_ value. Passing in 1 or True means a logical assertion,
            which is electrically LOW. Passing in a 0 or False means a logical
            non-assertion, which means the signal will float to whatever state 
            (HIGH or LOW) the target naturally outputs.
        
        Example:
            >>> ipc.setpin(0, 1, True) # set HOOK1 LOW on all debug port 0
            >>> ipc.setpin(0, 1, False) # set HOOK1 HIGH on debug port 0
            >>> ipc.setpin(0, "HOOK0", True) # set HOOK1 LOW on all debug port 0
            >>> ipc.setpin(0, "HOOK0", False) # set HOOK1 HIGH on debug port 0
            >>> ipc.setpin(0, "OBS_FN_D0", 1) # set hook 1 LOW on debug port 0
        
        Raises:
            ValueError: if hook is an integer value greater than 9
            IPC_Error: if pin string value is not a supported pin
            IPC_Error: if device is not a valid device
            
        **IPC services required:** InterfacePins
        
        """
        ##### logging
        if _log.isEnabledFor(cli_logging.DEBUG):
            devicestr = self._device_to_devicestr(device)
            _log.caller()
            _log.debug("ENTER: setpin(device={0},pin={1},state={2})".format(devicestr,pin,state))
        
        # sometimes we may still get old debugport number
        did = self._convert_debug_port(device)
        
        # convert hook if needed
        if isinstance(pin,(int,long)):
            if pin > 9: raise ValueError("if hook is a number it must be less than or equal to 9")
            # convert to enum required by py2ipc
            hook = "HOOK{0}".format(pin)
        
        ipins = py2ipc.IPC_GetService("InterfacePins")
        
        # assertHook is logical, the integer value of the enum used by IPC API is electrical,
        # so True == "Asserted" == 0
        ipins.SetPinState(did, pin, not state)    
         
        _log.debug("EXIT: setpin")
    
    def pulsepin(self, device, pin, state, count):
        """
        
        Pulse any of the pins that are not input pins.
               
        Args:
          device (int) : This can be debug port # (for backwards compatibility) 
                         or the DID of the debug port device. If *None* then assert/
                         deassert the hook across all debug ports.
          pin (int/str): This can be an integer value from 0-9 for the hook, or
                         one of the strings supported by the IPC.
          state :        True (non-zero) means set the hook LOW;
                         False (zero) means set the hook HIGH.
          count (int) : An unsigned integer value that specifies the number of ns
                        increments to pulse the hook.
        
        Returns:
              None.
        
        Raises:
            ValueError: if pin integer value provided is greater than 9
            IPC_Error: if pin string value is not a supported pin
            IPC_Error: if device is not a valid device
            
        **IPC services required:** InterfacePins
        
        The pulsepin command asserts or deasserts any of the pins that are not input 
        pins on one or all debug ports for a specified length of time. Note that if a 
        particular pin is already asserted/deasserted, then this will result in a wait of 
        the specified length, then transition to the deasserted/asserted state.
        
        Example:
            >>> ipc.pulsepin(0, 0, 1, 16700000) # pulse asserted HOOK1 on debug port 0
            >>> ipc.pulsepin(0, "HOOK0", True, 167000000) # pulse asserted HOOK1 on debug port 0
               
        """
        ##### logging
        if _log.isEnabledFor(cli_logging.DEBUG):
            devicestr = self._device_to_devicestr(device)
            _log.caller()
            _log.debug("ENTER: pulsepin(device={0},pin={1},state={2},count={3})".format(devicestr,pin,state,count))
            
        # sometimes we may still get old debugport number
        did = self._convert_debug_port(device)
        
        # convert hook if needed
        if isinstance(pin,(int,long)):
            if pin > 9: raise ValueError("if hook is a number it must be less than or equal to 9")
            pin = "HOOK{0}".format(pin)
        
        ipins = py2ipc.IPC_GetService("InterfacePins")
        
        # state parameter logical assert value is 1, the integer value of the enum used by IPC API for 
        # asserted is 0.
        ipins.PulsePinState(did, pin, not state, count)    
        
        _log.debug("EXIT: pulsepin")
    
    def hookpins(self, device):
        """
        
        Gets all of the hook pins states (asserted/deaserted) for the specified debugport device.
        
        Note: On some probes this **ALWAYS** reports *True* for output pins.
        
        Args:
          device (int) : This can be debug port # (for backwards compatibility) 
                         or the DID of the debug port device
                           
        Returns:
            A formatted list cntaining the hook pins states.
            True - if the hook is logically asserted.
            False - if the hook is logically deasserted.
        
        The various pins supported by this function may differ in active state (high or low). 
        for example, the majority of HOOK pins are active low, but HOOK1 is active High.
        This command returns the *logical* value of the hook, not the *electrical* value of the hook.
        
        Example:
            >>> ipc.hookpins(0) # show all of the hook pins states on debug port 0
        
        
        Raises:
            ValueError: if pin integer value provided is greater than 9
            IPC_Error: if pin string value is not a supported pin
            IPC_Error: if device is not a valid device
            
        **IPC services required:** InterfacePins
        
        """
        results = {}
        for i in range(0,10):
            try:
                hook = "HOOK" + str(i)
                results[hook] = self.getpin(device,hook)
            except py2ipc.IPC_Error as e:
                if e.code == py2ipc.IPC_Error_Codes.InterfacePins_Pin_Not_Supported:
                    results[hook] = "N/A";
                else:
                    raise
        return ReportResults(results)
    
    def obspins(self, device):
        """
        
        Gets all of the obs pins states (asserted/deaserted) for the specified debugport device.
        
        Note: Not all probes support obs pins. Since Obs Pins are active high, electrical and logical
        values are the same.
        
        Args:
          device (int) : This can be debug port # (for backwards compatibility) 
                         or the DID of the debug port device
                           
        Returns:
            A formatted list cntaining the obs pins states.
            True - if the hook is electrically asserted.
            False - if the hook is electrically deasserted.
        
        This command returns the *electrical* value of the pin
        
        Example:
            >>> ipc.obspins(0) # show all of the obs pins states on debug port 0
        
        
        Raises:
            ValueError: if pin integer value provided is greater than 9
            IPC_Error: if pin string value is not a supported pin
            IPC_Error: if device is not a valid device
            
        **IPC services required:** InterfacePins
        
        """
        results = {}
        supported = True
        val = ""
        try:
            val = self.getpin(device,"OBS_DATA_D3")
        except py2ipc.IPC_Error as e:
            if e.code == py2ipc.IPC_Error_Codes.InterfacePins_Pin_Not_Supported:
                 print("\nObs Pins are not supported w/ the current probe type\n")
                 supported = False
        if supported:
            for obs in ("A", "B", "C", "D"):
                for fn in range(0,2):
                    name = "OBS_FN_" + obs + str(fn)
                    results[name] = self.getpin(device,name)
                for data in range(0,4):
                    name = "OBS_DATA_" + obs + str(data)
                    results[name] = self.getpin(device,name)
        return ReportResults(results)
    
    def holdhook(self,device,hook,assertHook):
        """
        Hold one of the eight ITP hook pins in the specified state.

        Args:
          device (int) : This can be debug port # (for backwards compatibility) 
                         or the DID of the debug port device. If *None* then assert/
                         deassert the hook across all debug ports.
          hook (int/str) : This can be an integer value from 0-9 for the hook, or
                           one of the strings supported by the IPC.
          assertHook : True (non-zero) means set the hook LOW;
                       False (zero) means set the hook HIGH.
    
        Returns:
            None
    
        In a typical configuration, hook 0 is the power-good signal and hook 6
        is the reset occurred signal.  These are considered "input" pins and
        generally shouldn't be asserted/deasserted.  However, not all hook
        configurations are the same so the holdhook() command allows any hook
        value. The user is expected to understand the debug port's hook
        configuration before using this command.
    
        Note:  
            The HOOK pins are active-low signals.  This command operates in
            terms of the _logical_ value, not the _electrical_ value.
            Passing in 1 or True means a logical assertion, which is electrically LOW.
            Passing in a 0 or False means a logical non-assertion, which means the signal
            will float to whatever state (HIGH or LOW) the target naturally outputs.
    
        Example:
            >>> ipc.holdhook(None, 1, True) # set hook 1 LOW on all debug ports
            >>> ipc.holdhook(0, 1, False) # set hook 1 HIGH on debug port 0
            >>> ipc.holdhook(0, 1, 1) # set hook 1 LOW on all debug ports
            >>> ipc.holdhook(0, 1, 0) # set hook 1 HIGH on all debug ports

        Raises:
            ValueError: if hook is an integer value greater than 9
            
        **IPC services required:** InterfacePins

        """
        ##### logging
        if _log.isEnabledFor(cli_logging.DEBUG):
            devicestr = self._device_to_devicestr(device)
            _log.caller()
            _log.debug("ENTER: holdhook(device={0},hook={1},assertHook={2})".format(devicestr,hook,assertHook))
            
        # sometimes we may still get old debugport number
        did = self._convert_debug_port(device)
        # convert to a list so we can walk through all debug ports if needed
        if not isinstance(did,list):
            did = [did]
        # convert hook if needed
        if isinstance(hook,(int,long)):
            if hook > 9: raise ValueError("if hook is a number it must be less than 7")
            # convert to enum required by py2ipc
            hook = "HOOK{0}".format(hook)
        
        ipins = py2ipc.IPC_GetService("InterfacePins")
        for d in did:
            try:
                # assertHook is logical, the integer value of the enum used by IPC API is electrical,
                # so True == "Asserted" == 0
                ipins.SetPinState(d, hook, not assertHook)    
            except py2ipc.IPC_Error as e:
                if e.code == py2ipc.IPC_Error_Codes.InterfacePins_Manual_Pin_Driving_Prevented:
                    raise py2ipc.IPC_Error(e.code, e.msg + " Use ipc.stalls.* for named stalls. Alternately, to enable hook pin drive, you could change platform type with ipc.setplatformtype() to one without a stall definition using that pin.")
                else:
                    raise

        _log.debug("EXIT: holdhook")


    def pulsehook(self, device, hook, assertHook, count):
        """
        
        Pulse any of the 8 hook pins that are not input pins.
               
        Args:
          device (int) : This can be debug port # (for backwards compatibility) 
                         or the DID of the debug port device. If *None* then assert/
                         deassert the hook across all debug ports.
          hook (int/str) : This can be a value of an integer 0-9 for the hook, or
                           one of the strings supported by the IPC.
          assertHook (bool) : True (non-zero) means sets the hook LOW;
                       False (zero) means set the hook HIGH.
          count (int) : An unsigned integer value that specifies the number of 10ns
                        increments to pulse the hook.
        
        Returns:
              None.

        Raises:
            ValueError: if hook is an integer value greater than 9
            
        **IPC services required:** InterfacePins
        
        The pulsehook command asserts or deasserts any of the 8  hook pins
        that are not input pins on one or all debug ports for a specified length
        of time. Note that if a particular pin is already asserted/deasserted,
        then this will result in a wait of the specified length, then transition
        to the deasserted/asserted state.
        
        Example:
            >>> ipc.pulsehook(0, 1, True, 16700000) # assert hook 1 on debug port 0
               
        """
        ##### logging
        if _log.isEnabledFor(cli_logging.DEBUG):
            devicestr = self._device_to_devicestr(device)
            _log.caller()
            _log.debug("ENTER: pulsehook(device={0},hook={1},assertHook={2},count={3})".format(devicestr,hook,assertHook,count))
            
        # sometimes we may still get old debugport number
        did = self._convert_debug_port(device)
        # convert to a list so we can walk through all debug ports if needed
        if not isinstance(did,list):
            did = [did]
        # convert hook if needed
        if isinstance(hook,(int,long)):
            if hook > 9: raise ValueError("if hook is a number it must be less than 7")
            # convert to enum required by py2ipc
            hook = "HOOK{0}".format(hook)
        
        ipins = py2ipc.IPC_GetService("InterfacePins")
        for d in did:
            # assertHook is logical, the integer value of the enum used by IPC API is electrical,
            # so True == "Asserted" == 0
            # count for this function is in 10ns (legacy) but IPC takes in nano-seconds
            ipins.PulsePinState(d, hook, not assertHook, count*10)    

        _log.debug("EXIT: pulsehook")
        

    def hookstatus(self,device,hook):
        """
        
        Read the status of any of the 8 hook pins.
    
        Note: On some probes this **ALWAYS** reports *True* for output pins.
        
        Args:
          device (int) : This can be debug port # (for backwards compatibility) 
                         or the DID of the debug port device
          hook (int/str) : This can be an integer value from 0-9 for the hook, or
                           one of the strings supported by the IPC
                           
        Returns:
            True - if the hook is logically asserted (electrically LOW).
            False - if the hook is logically deasserted (electrically HIGH).

        The HOOK pins are active-low signals.  This command returns the
        *logical* value of the hook, not the *electrical* value of the hook.

        Example:
            >>> ipc.hookstatus(0, 0) # get hook 0 status from debug port 0
        
        Raises:
            ValueError: if hook is an integer value greater than 9
            
        **IPC services required:** InterfacePins

        """
        ##### logging
        if _log.isEnabledFor(cli_logging.DEBUG):
            devicestr = self._device_to_devicestr(device)
            _log.caller()
            _log.debug("ENTER: hookstatus(device={0},hook={1})".format(devicestr,hook))
            
        # sometimes we may still get old debugport number
        did = self._convert_debug_port(device)
        # convert hook if needed
        if isinstance(hook,(int,long)):
            if hook > 9: raise ValueError("if hook is a number it must be less than 7")
            # convert to enum required by py2ipc
            hook = "HOOK{0}".format(hook)
        
        ipins = py2ipc.IPC_GetService("InterfacePins")
        hookstatus = ipins.GetPinState( did, hook)
        _log.debug("EXIT: hookstatus : result = {0}".format(hookstatus))
        # convert to True/False for if asserted or not
        return hookstatus == "Assert"

    def asm(self, device, address, *args, **kwargs):
        """
        
        Display memory as IA32 disassembled instructions or assembles IA32 instructions into memory.
    
        Args:
          device (int) : the did or alias of the device (not needed if using from a node object)

          address (str/int/Address) : Starting address to disassemble from or assemble to.
                     Can also be '$', indicating the current instruction pointer
                     for the thread.  
    
          *args   :  If the first argument is a number or is a string that does not
                     start with an alphabetic character, the argument is treated
                     as a "to" parameter (if number, it is the number of
                     instructions to work with, otherwise it is an end address).
                     Otherwise, all arguments are treated as strings containing
                     instructions to be assembled to memory.
    
        Returns:
          A list of ASMDecode objects. Each ASMDecode object has instruction, opcode, and address attributes. 
    
        Note:
          The asmmode control variable override is used only if the address
          provided to this command is linear or physical; otherwise, the current
          addressing mode takes precedence and dictates the mode to
          assemble/disassemble.
    
        Example:
        
          >>> itp.threads[0].asm("0x1000P")     # disassemble single instruction from address 0x1000
          >>> itp.threads[0].asm("$")           # disassemble single instruction from current instruction pointer address.
          >>> itp.threads[0].asm("0x1000P", "nop", "jmp $") # assemble two instructions to address 0x1000
          >>> itp.threads[0].asm("0x1000P", 3) # disassemble 3 instructions starting from address 0x1000
          0x0000000000001000P      3a30             cmp dh, byte ptr [eax]
          0x0000000000001002P      3230             xor dh, byte ptr [eax]
          0x0000000000001004P      3030             xor byte ptr [eax], dh
          >>> itp.threads[0].asm("0x1000P", byteCount=10)  # disassemble up to 10 bytes
          0x0000000000001000P      3a30             cmp dh, byte ptr [eax]
          0x0000000000001002P      3230             xor dh, byte ptr [eax]
          0x0000000000001004P      3030             xor byte ptr [eax], dh
          >>> x = itp.threads[0].asm("0x1000P", 3)
          >>> x[0]
          <ipccli.asm.ASMDecode instance at 0x00000000562D6B48>
          >>> x[0].instruction
          'cmp dh, byte ptr [eax]'
          >>> x[0].opcode
          [0x3a,
           0x30]
          >>> x[0].address
          0x0000000000001000P
          
        Raises:
            TypeError: if the address is not a string or an ipccli.Address object
            ValueError: if asmcount is less than or equal to 0
            TypeError: if the argument was of an unexpected type
            TypeError: if the argument list contains a non-string argument

        **IPC services required:** AsmDasm, Memory
                  
        See Also:
            :py:meth:`asmmode`
                
        """
        ##### logging
        if _log.isEnabledFor(cli_logging.DEBUG):
            devicestr = self._device_to_devicestr(device)
            _log.caller()
            _log.debug("ENTER: asm(device={0},address={1},*args={2},**kwargs={3})".format(devicestr,address,args,kwargs))
        
        did = self._device_convert(device)
        devobj = self.devicelist.findByDID(did)
        byteCount = kwargs.pop("byteCount", None)
        if len(kwargs) != 0:
            raise AssertionError("unexpected kwargs: %s"%kwargs.keys())
        # convert to string
        if not isinstance(address,(Address,basestring,py2ipc.IPC_Address)):
            raise TypeError("address must be a string or an ipccli.Address object")        
        # unless it is an IPC_Address (which is rare), then we need to convert
        if not isinstance(address,py2ipc.IPC_Address):
            address = self._parse_address(devobj, address, "cs")
        # should have an IPC address structure at this point
        asmsrvc = py2ipc.IPC_GetService("AsmDasm")
        memsrvc = py2ipc.IPC_GetService("Memory")
        # based on args...figure out what to do
        if len(args) == 0:
            action = 'dasm'
            asmcount = 1
        elif len(args) == 1:
            if isinstance(args[0],(int,long)):
                action = 'dasm'
                asmcount = args[0]
                if asmcount<=0:
                    raise ValueError("Invalid number for specifying bytes to decode: {0}".format(asmcount))
            elif isinstance(args[0],basestring):
                # complex case where it could be asm or address
                if args[0].startswith(("0x","0X")) and args[0].endswith("l","L","p","P"):
                    raise ValueError("Unsure if arg is for asm or dasm")
                else:
                    # assume it is an instruction
                    action = "asm"
                    bytedata = [ args[0] ]
            else:
                raise TypeError("Unexpected type for args: '{0}'".format(args[0]))
        else: # len(args)>=2:
            # make sure we got the correct args for asm...
            # get list of booleans as to whether we got the correct arg types
            args_are_strings = [isinstance(a,basestring) for a in args]
            if args_are_strings.count(True) != len(args):
                raise TypeError("Arg list contained a non-string argument, please see 'help' on this function")
            action = "asm"
            bytedata = args
            # make sure args are strings?

        if address.addressType in ['Physical','Linear']:
            asmmode = self._asmmode
        else:
            asmmode = self._instruction_size(did)
        _log.debug("asm : asmmode = {0}".format(asmmode))

        if action == 'dasm':
            opcodes = ASMList()
            if byteCount == None:
                for i in range(asmcount):
                    memdata = memsrvc.ReadMemory(did,address,16)
                    memdata = memdata.GetValueAsRawBytes()
                    instruction, numbytes = asmsrvc.Disassemble(did, asmmode, memdata, address.offset)
                    addrstr = memsrvc.AddressToString( address ) # address is an IPC_Address type in this code
                    pyaddr = Address(addrstr) # convert to a Python Address and put in your ASMDecode object
                    asmdecode = ASMDecode(memdata[:numbytes], pyaddr, instruction)
                    opcodes.append(asmdecode)
                    address.offset += numbytes
            else:  # numbytes specified
                # same loop really but exit on bytes_left = 0
                bytes_left = byteCount
                while bytes_left>0:
                    readsize = min(16, bytes_left)
                    memdata = memsrvc.ReadMemory(did,address, readsize)
                    memdata = memdata.GetValueAsRawBytes()
                    try:
                        instruction, numbytes = asmsrvc.Disassemble(did, asmmode, memdata, address.offset)
                    except py2ipc.IPC_Error as err:
                        if str(err).find("Not enough bytes")>0:
                            # if user specified very small amount, raise exception
                            if len(opcodes) == 0:
                                raise
                            else:
                                # otherwise, we just couldn't decode the last one in the number of bytes given
                                break
                        else:
                            # some other unexpected error
                            raise
                    addrstr = memsrvc.AddressToString( address ) # address is an IPC_Address type in this code
                    pyaddr = Address(addrstr) # convert to a Python Address and put in your ASMDecode object
                    asmdecode = ASMDecode(memdata[:numbytes], pyaddr, instruction)
                    opcodes.append(asmdecode)
                    address.offset += numbytes
                    bytes_left -= numbytes

            _log.debug("EXIT: asm : result = {0}".format(opcodes))   
            return opcodes
        elif action == "asm":
            for instruction in args:
                bytedata = asmsrvc.Assemble(did, asmmode, instruction, address.offset)
                _log.debug("asm : Assembling bytes for instruction {0}, at offset 0x{1:x}, result was = {2}".format(
                            instruction,address.offset,bytedata))
                for byte in bytedata:
                    operation = memsrvc.WriteMemory(did, address, 1, byte)
                    operation.Flush()
                    operation.ErrorCheck()
                    address.offset += 1
                
        else:
            raise AssertionError("Not sure how this code got reached")
        
    def asmmode(self,device=None, mode=None):
        """
        Used to set/get the mode of the processor. Used when using asm command with physical
        or linear addresses.
        
        Args:
          device (int) : the did or alias of the device (not needed if using from a node object)
          mode (str) : Force the mode to use for assembly/disassembly or if None is specified,
                        then return the current value for the asmmode
        
        **Valid mode values:**
            - '16Bit'
            - '32Bit'
            - '64Bit'
            - 'XBit' - for IA privileged instructions
        
        Raises:
            ValueError: if an invalid mode was specified
        """
        if _log.isEnabledFor(cli_logging.DEBUG):
            devicestr = self._device_to_devicestr(device)
            _log.caller()
            _log.debug("ENTER: asmmode(device={0},mode={1})".format(devicestr,mode))
        
        if mode==None:
            _log.debug("EXIT: asmmode: result = {0}".format(self._asmmode))
            return self._asmmode
        if mode.endswith("bit"): # be a little forgiving on this...
            mode = mode.replace("bit","Bit")
        # make sure it is a valid value
        if mode not in ['16Bit','32Bit','64Bit','XBit']:
            raise ValueError("Invalid mode '{0}' specified, see help for valid options".format(mode))
        self._asmmode = mode
        _log.debug("EXIT: asmmode: result = {0}".format(self._asmmode))

    def _parse_address(self, devobj, address, default_segment, return_type="ipc_address", ):
        """
        Used to parse address strings containing $ for the asm and 
        the mem commands. It is an error to call this with an address
        that is not a string
        
        Args:
            devobj (device) : Requires device object (not DID) to be passed in
            address (str) : String that we will be converting to an ipc address
            default_segment (str) : used to determine default segment to add when none is given
            returntype (str) : "ipc_address" or "string" - for specifying what kind of object to return
            
        Returns:
            This returns an py2ipc.IPC_Address object
        """
        ##### logging
        if _log.isEnabledFor(cli_logging.DEBUG):
            _log.caller()
            _log.debug("ENTER: _parse_address(devobj={0},address={1})".format(devobj.alias,address))
        
        did = devobj.did
        addrsrvc = py2ipc.IPC_GetService("Memory")

        if isinstance(address,Address):
            address = str(address)

        if address.count("$") == 0: # no $, simply convert from string
            # if the address didn't have any selectors or specify address type, then we are supposed to
            # use our default selector
            required = {":","P","p","l","L"} # a set
            for a in address:
                if a in required:
                    break
            else:
                # If the thread's architecture supports the provided segment register
                if default_segment in devobj.node.state.regs:
                    # none of our required address characters were found, add the segment
                    seg = self.arch_register(did, default_segment)
                    try:
                        # try to convert to int/hex depending on what was passed in
                        address = "0x{0:X}:0x{1:016X}".format(seg, int(address, 0))
                    except:
                        address = "0x{0:X}:{1}".format(seg, address)
                else:
                    # If the thread's architecture doesn't support a segment register then assume
                    # the address is linear
                    try:
                        # try to convert to int/hex depending on what was passed in
                        address = "0x{0:016X}L".format(int(address, 0))
                    except:
                        pass
            if return_type == "ipc_address":
                address = addrsrvc.AddressFromString(address)
            if _log.isEnabledFor(cli_logging.DEBUG):
                _log.debug("EXIT: _parse_address : result = {0}".format(address))
            return address

        # must have "$"
        if "cs" in devobj.node.state.regs:
            try:
                cs = self.arch_register(did,"cs")
            except py2ipc.IPC_Error as e:
                # ignore register invalid and try next register
                if e.code==py2ipc.IPC_Error_Codes.Register_Invalid:
                    raise ValueError("This core/thread does not support using the $ to determine the instruction pointer")
                else:
                    raise
        else:
            cs = None

        # attempt to determine an instruction pointer in this priority order
        regs = ['rip','eip','pp',"pc"]
        for r in regs:
            if r in devobj.node.state.regs:
                try:
                    ip = self.arch_register(did,r)
                    # success!
                    break
                except py2ipc.IPC_Error as e:
                    # ignore register invalid and try next register
                    if e.code==py2ipc.IPC_Error_Codes.Register_Invalid:
                        pass
                    else:
                        raise
        else:
            raise ValueError("This core/thread does not support using the $ to determine the instruction pointer")

        # for handling "+"
        if address.count("+"):
            sign = 1
            addrparts = address.split("+")
        elif address.count("-"):
            sign = -1
            addrparts = address.split("-")
        else:
            if address.strip() != "$":
                raise ValueError("Invalid format using $")
            addrparts = [address]
        if len(addrparts)>2:
            raise ValueError("Invalid address format specified: {0}".format(address))
        if len(addrparts) == 2:
            # conver the + number in to an integer
            addroffset = addrparts.pop()
            addroffset = sign * int(addroffset,0) # convert to decimal or hex, depending on the format
        else:
            addroffset = 0 
        # should leave us with addr
        addrstr = self._to_logical_or_linear_addr_string(cs, ip)
        _log.debug("_parse_address : $ converted to {0}".format(addrstr))
        if return_type == "string": 
            addrstr = self._to_logical_or_linear_addr_string(cs, ip+addroffset)
            _log.debug("EXIT: _parse_address : result = {0}".format(str(addrstr)))
            return addrstr
        else:
            addr = addrsrvc.AddressFromString(addrstr)
            # add any offset that was pass in
            addr.offset += addroffset
            _log.debug("EXIT: _parse_address : result = {0}".format(str(addr)))
        return addr

    def _to_logical_or_linear_addr_string(self, segment, offset):
        if segment is not None:
            return "0x{0:X}:0x{1:016X}".format(long(segment),long(offset))
        else:
            return "0x{0:016X}L".format(long(offset))

    def eval(self,device,expression,convertToType="linear"):
        """
        
        Convert an expression to the requested type.
        
        Args:
            device (int): the did or alias of the device (not 
                          needed if using from a node object)        
            expression (str) : expression to evaluate. Currently this is assumed
                               to be an address. "$" is supported along with "$+" or "$-",
                               but more complex uses of "$" are not. Other than "$", it
                               is expected that some string form of an address 
                               is being provided.
            convertToType (str) : a string that specifies what the expression should be 
                                  converted to.
                                  
        Supported values for convertToType:
            - 'linear' - Return the address as a linear address 
            - 'physical' - Return the address as a physical address
            - 'guestphysical' - Return the address as a "GPA" address
            - 'virtual' - for backwards compatibility, converts to a logical_twofield
            - 'logical_twofield' - name may be subject to change
            - 'logical_threefield' - name may be subject to change

        Returns:
            Returns the evaluated expression as a string

        Note:
            This does not do a general eval like legacy tools. It is primarily
            for converting address types. The convertToType variable is
            different than legacy tool's options.

        Raises:
            ValueError: if an invalid convertToType value was specified

        **IPC services required:** Memory
        
        """
        ##### logging
        if _log.isEnabledFor(cli_logging.DEBUG):
            devicestr = self._device_to_devicestr(device)            
            _log.caller()
            _log.debug("ENTER: eval(device={0},convertToType={1})".format(devicestr,convertToType))
            
        if convertToType not in _eval_to_address_conversion.keys():
            raise ValueError("Invalid convertToType value ({0})".format(convertToType))
        convertToType = _eval_to_address_conversion[convertToType]
        
        ##### Required for every function:
        did = self._device_convert(device)
        devobj = self.devicelist.findByDID( did )
        address = self._parse_address(devobj, expression, 'cs')
        
        memsrvc = py2ipc.IPC_GetService("Memory")
        ipcaddress = memsrvc.ConvertAddress(did, address, convertToType )
        # convert the address to a string
        addrstr = memsrvc.AddressToString( ipcaddress )
        return addrstr
    
    def resettap(self,device=None,resettype=None):
        """
        
        Reset the tap of the scan chains associated with this debugport or scanchain, or all scan chains.
        
        Args:
            device (int) : If None, then reset all scan chains.
            resettype (str) : type of tap reset to do, default is 'TLR_TRST'.
            
        Returns: 
            None
            
        Reset Types:
            - 'TLR_TRST' - Drive TMS high for at least 5 clocks, drive TRST pin when TLR is achieved, and then go to RTI.
            - 'TLR_ONLY'- Drive TMS high for at least 5 clocks, and then go to RTI.
            - 'TRST_ONLY' - Drive TRST pin, and then go to RTI.
            
        It is recommended to call resettap with device=None to reset all taps; otherwise, the JTAGScanChain device id must be specified.
        
        Note:
            Debug Port as a deviceid no longer works with this function; call with device=None instead.
        
        Raises:
            ValueError: if an invalid resettype value was specified

        **IPC services required:** JtagAccess

        """
        ##### logging
        if _log.isEnabledFor(cli_logging.DEBUG):
            devicestr = self._device_to_devicestr(device)            
            _log.caller()
            _log.debug("ENTER: resettap(device={0},resettype={1})".format(devicestr,resettype))
        
        # I didn't want to assume that case would NEVER matter, so yeah, I'm forcing this to be correct for now
        if isinstance(resettype,basestring):
            resettype = resettype.upper()
        if resettype not in [None,"TLR_TRST","TLR_ONLY","TRST_ONLY"]:
            raise ValueError("Unsupported reset type {0} (also make sure case matches)".format(resettype))
        
        if resettype == None:
            resettype = "TLR_TRST"
            
        dids = self._convert_debugport_to_chains(device)
        jtagsrvc = py2ipc.IPC_GetService("JtagAccess")
        window_open = OperationWindow.isopen() # flush operations is not in a window
        for did in dids:
            # this returns an operation receipt but we are not using/holding
            op = jtagsrvc.TapReset(did,resettype)
            if not window_open:
                op.Flush()
            
        _log.debug("EXIT: resettap")


    def jtag_shift(self,device,state,bitCount, value=None, returnData=True):
        """
        
        Provides the ability to specify a very detailed shift
        state as part of a larger set of jtag transactions.

        Args:
            device (int) : a JTAG scan device, tap device, or device
                           object (not needed if using from a node).
            state (str) : state to move the chain to.
            bitCount (int) : number of TCKs to shift for.
            value (int) : value to shift in during operation.
            returnData (bool) : whether to return the data from the scan that was done.
            
        Returns: 
            A BitData object
            
        Supported States:
            - "ShfDR"
            - "ShfIR"
               
        This function must be used within a lock window along
        with the jtag_goto_state function in order to build
        up a meaningful jtag transaction.
        
        The bitCount is relative to the device specified. If the device
        is a tap device, then the bitCount is for that particular device.
        If the device is a jtag chain device id, then the bitCount is
        the total number of bits shifted.
        
        Some of the shifts return meaningful data, and it is
        up to the caller of this function to track which
        operations can be treated as returning valid data.
        
        Note: 
            If you specify device < 0x1000, it is assumed that
            you meant the tap device and NOT the debug port #. To
            specify the scan chain, you have to use the actual jtag 
            scan chain did.
        
        Example:
            >>> dev = ipc.devicelist[0]
            >>> with ipc.device_locker():
            ...     ir=dev.node.jtag_shift("ShfIR", dev.irlength, dev.idcodeir )
            ...     dev.node.jtag_goto_state("RTI",1,"COUNT")
            ...     idcode = dev.node.jtag_shift("ShfDR", 32)
            ...     dev.node.jtag_goto_state("RTI",1,"COUNT")
            ...
            >>> print(idcode)
            [32b] 0x10A84013
            >>>        

        Raises:
            ValueError: if an invalid state value was specified
            RuntimeError: if a JTAG lock is not present

        **IPC services required:** JtagAccess

        """
        ##### logging
        if _log.isEnabledFor(cli_logging.DEBUG):
            devicestr = self._device_to_devicestr(device)
            _log.caller()
            _log.debug("ENTER: jtag_shift(device={0},state={1},bitCount={2},value={3})".\
                       format(devicestr,state,bitCount,value))
        ##### Required for every function:
        did = self._device_convert(device)
        ####
        if state not in ['ShfDR',"ShfIR"]:
            raise ValueError("Unsupported state '{0}', make sure case is correct".format(state))         
        jtagsrvc = py2ipc.IPC_GetService("JtagAccess")
        
        if not DeviceLocker.lock_present():
            raise RuntimeError("JTAG lock must be present for this function to work")
        
        shiftop = jtagsrvc.Shift(did,state,bitCount, value)
        # don't need shitop.ErrorCheck() because this is better be within lock/unlock
        bd = BitData(None, shiftop)
        if _log.isEnabledFor(cli_logging.DEBUG):
            _log.debug("EXIT : jtag_shift : result = {0}".format(bd))
        return bd
    
    def jtag_goto_state(self,device, state, numClocks, waitType):
        """
        
        Go to a particular JTAG state on a chain or device and wait
        for some specified time or condition.
        
        Args:
        
            device (int) : a JTAG scan device, tap device, or device
                           object (not needed if using from a node).
            state (str) : state to move the chain to.
            numClocks (int) : number of clocks to use as part of wait type.
            waitType (str) : whether to wait for TCK count and/or a trigger.
        
        Returns:
            None - though this may eventually return operation receipt or bitdata.
        
        Supported States:
            - TLR - Test logic reset state.
            - RTI - Run test/idle state.
            - SelDR - Select DR scan state.
            - CapDR - Capture DR state.
            - ShfDR - Shift DR state.
            - Ex1DR - Exit1 DR state.
            - PauDR - Pause DR state.
            - Ex2DR - Exit2 DR state.
            - UpdDR - Update DR state.
            - SelIR - Select IR scan state.
            - CapIR - Capture IR scan state.
            - ShfIR - Shift IR scan state.
            - Ex1IR - Exit1 IR state.
            - PauIR - Pause IR state.
            - Ex2IR - Exit2 IR state.
            - UpdIR - Update IR state.
            
        Supported Wait Types:
            - COUNT - Wait for a TCK count only.
            - COUNT_OR_TRIG - Wait for a TCK count or seeing a trigger.
            - COUNT_TRIGTO - Wait for a TCK count, and if you don't see a trigger in that period, then error (timeout).
            - COUNT_OR_TRIG_TRIGTO - Wait for a TCK count or seeing a trigger, and if you don't see a trigger in that period, then error (timeout).
            
        Example:
            >>> dev = ipc.devicelist[0]
            >>> with ipc.device_locker():
            ...     ir=dev.node.jtag_shift("ShfIR", dev.irlength, dev.idcodeir )
            ...     dev.node.jtag_goto_state("RTI",1,"COUNT")
            ...     idcode = dev.node.jtag_shift("ShfDR", 32)
            ...     dev.node.jtag_goto_state("RTI",1,"COUNT")
            ...
            >>> print(idcode)
            [32b] 0x10A84013
            >>>                

        Raises:
            ValueError: if an invalid state value was specified
            RuntimeError: if a JTAG lock is not present

        **IPC services required:** JtagAccess

        """
        ##### logging
        if _log.isEnabledFor(cli_logging.DEBUG):
            devicestr = self._device_to_devicestr(device)
            _log.caller()
            _log.debug("ENTER: jtag_shift(device={0},state={1},bitCount={2},value={3})".\
                       format(devicestr,state,numClocks,waitType))
        ##### Required for every function:
        did = self._device_convert(device)
        ####
        if state not in py2ipc.IPC_Types.IPC_JtagStates.keys():
            raise ValueError("Unsupported state '{0}' make sure case is correct".format(state))       
        
        if not DeviceLocker.lock_present():
            raise RuntimeError("JTAG lock must be present for this function to work")
                  
        jtagsrvc = py2ipc.IPC_GetService("JtagAccess")
        # I don't think the operation from goto state would ever have a value...
        operation = jtagsrvc.GoToState(did,state,numClocks,waitType, returnData=False)
        # don't need shitop.ErrorCheck() because this is better be within lock/unlock
        _log.debug("EXIT: jtag_goto_state")
        return

    def memsave(self,device,filename,address,addressOrCount,overwrite=False,fileType="binary",saveOptions=None):
        """

        Args:
            device (int): the did or alias of the device to execute the 
                          memory instructions on (not needed if using
                          from a node object).
            filename (str): Filename of the object file where the data is to be
                          stored.
            address (str): Address of the first byte to read from.
            addressOrCount (str/int): If a string, then this is the "to" address marking the
                          end of the memory block (inclusive). If a number, this is 
                          the length of the target memory range, in bytes.
            overwrite (bool) : (optional, default=False) set to True to overwrite the file 
                          if it exists; otherwise, an error is raised.
            fileType (str): valid options are 'binary', 'asciihex', and 'intelhex'.
            saveOptions (str) : options for controlling save behavior (probe specific).

        The saveOptions goes through no error checking and requires knowledge of the
        parameters that the probe supports.

        Returns:
            None.

        Example:
          >>> # Note usage of "raw" string notation (leading 'r') where backslashes
          >>> # are used in the filename:
          >>> itp.threads[0].memsave(r"c:\\temp\\file.bin","0x1000", "0x1100")
          >>> itp.threads[0].memsave( "c:/temp/file.bin","0x100P", 4, True)
          >>> itp.threads[0].memsave("c:\\\\temp\\\\file.bin","0x100P", "0x110P", True)
          >>> itp.threads[0].memsave("c:\\\\temp\\\\file.bin","0x100P", "0x110P", fileType="asciihex")

        Raises:
            ValueError: if fileType is not a string
            ValueError: if an invalid fileType was specified
            TypeError: if addressOrCount is not a string or a number
            OSError: if the file already exists and overwrite is not set to True

        **IPC services required:** Memory

        """
        ##### logging
        if _log.isEnabledFor(cli_logging.DEBUG):
            devicestr = self._device_to_devicestr(device)
            _log.caller()
            _log.debug("ENTER: memsave(device={0},filename={1},address={2},addressOrCount={3},overwrite={4},fileType={5})".\
                       format(devicestr,filename,address, addressOrCount, overwrite, fileType))
        ##### Version check needed
        self.base._version_check("OpenIPC", build=140, revision=454380) 
        ##### Required for every function:
        did = self._device_convert(device)
        devobj = self.devicelist.findByDID(did)
        ####
        if not isinstance(filename,basestring):
            raise TypeError("filename must be a string")
        if not isinstance(fileType,basestring):
            raise TypeError("fileType must be a string")
        valid_filetypes = ['binary','asciihex','intelhex','omf']
        if fileType.lower() not in valid_filetypes:
            raise ValueError("fileType must be one of: {0}".format(
                                ",".join(valid_filetypes)
                                ))
        #### from address ####
        # unless it is an IPC_Address (which is rare), then we need to convert
        if isinstance(address,py2ipc.IPC_Address):
            fromAddress = address
        else:
            fromAddress = self._parse_address(devobj, address, 'ds')
        #### to address ####
        if isinstance(addressOrCount,py2ipc.IPC_Address):
            toAddress = addressOrCount
        elif isinstance(addressOrCount,basestring):
            # if it is a string...convert it like we usually do
            toAddress = self._parse_address(devobj, addressOrCount, 'ds')
        elif isinstance(addressOrCount,(int,long)):
            toAddress = copy.copy(fromAddress)
            #toAddress = self._parse_address(devobj, address)
            toAddress.offset += (addressOrCount-1)
        else:
            raise TypeError("Unknown type for addressOrCount, must be a string or a number")

        # this does not work in all probes, so we will do check ourselves
        if overwrite==False and os.path.exists(filename):
            raise OSError("File {0} already exists, and overwrite was not set to True".format(filename))

        memsrvc = py2ipc.IPC_GetService("Memory")
        op = memsrvc.SaveMemory( did, filename, fromAddress, toAddress, overwrite, fileType, None)
        # workaround for now to leave these out...
        op.Flush()
        op.ErrorCheck()

    def memload(self,device,filename,address,addressOrCount=None,fileType="binary",loadOptions=None):
        """

        Fill a range of target memory with the contents of a host file,
        truncating or repeating the file contents as necessary.

        Args:
            device (int): the did or alias of the device to execute the 
                          memory instructions on (not needed if using
                          from a node object).
            filename (str): Filename of the object file where the data comes from.
            address (str) : Address of the first byte to write to.
            addressOrCount (int/str) : If a string, then this is the "to" address marking the
                          end of the memory block (inclusive). If a number, 
                          this is the length of the target memory
                          range, in bytes. If not specified or None, dictates to 
                          read the full contents of the file into memory.
            fileType (str) : see IPC API for documentation, depends on IPC API 
                          implementation.
            loadOptions (str) : see IPC API for documentation, depends on IPC API 
                          implementation.

        Returns:
          None.

        The loadOptions goes through no error checking and requires knowledge of the
        parameters that the IPC implementation supports.

        Example:
          >>> # Note usage of "raw" string notation (leading 'r') where backslashes
          >>> # are used in the filename:
          >>> itp.threads[0].memload(r"c:\\temp\\file.bin","0x100P")
          >>> itp.threads[0].memload( "c:/temp/file.bin","0x100P", 4)
          >>> itp.threads[0].memload(r"c:\\\\temp\\\\file.bin","0x100P", "0x110P")

        Raises:
            ValueError: if fileType is not a string
            ValueError: if an invalid fileType was specified
            TypeError: if addressOrCount is not a string or a number

        **IPC services required:** Memory

        """
        ##### logging
        if _log.isEnabledFor(cli_logging.DEBUG):
            devicestr = self._device_to_devicestr(device)
            _log.caller()
            _log.debug("ENTER: memsave(device={0},filename={1},address={2},addressOrCount={3},fileType={4})".\
                       format(devicestr,filename,address, addressOrCount, fileType))
        ##### Version check needed
        self.base._version_check("OpenIPC", build=140, revision=454380) 
        ##### Required for every function:
        did = self._device_convert(device)
        devobj = self.devicelist.findByDID(did)
        ####
        if not isinstance(fileType,basestring):
            raise TypeError("fileType must be a string")
        valid_filetypes = ['binary','elf','exe','coff','omf']
        if fileType.lower() not in valid_filetypes:
            raise ValueError("fileType must be one of: {0}".format(
                                ",".join(valid_filetypes)
                                ))
        #### from address ####
        # unless it is an IPC_Address (which is rare), then we need to convert
        if isinstance(address,py2ipc.IPC_Address):
            fromAddress = address
        else:
            fromAddress = self._parse_address(devobj, address, 'ds')
        #### to address ####
        if isinstance(addressOrCount,py2ipc.IPC_Address):
            toAddress = addressOrCount
        elif isinstance(addressOrCount,basestring):
            # if it is a string...convert it like we usually do
            toAddress = self._parse_address(devobj, addressOrCount, 'ds')
        elif isinstance(addressOrCount,(int,long)):
            toAddress = copy.copy(fromAddress)
            toAddress.offset += (addressOrCount-1)
        elif addressOrCount == None:
            toAddress = None # this gets translated to Null, which tells IPC to figure it out
        else:
            raise TypeError("Unknown type for addressOrCount, must be a string or a number")

        memsrvc = py2ipc.IPC_GetService("Memory")
        op = memsrvc.LoadMemory( did, filename, fromAddress, toAddress, fileType, loadOptions)
        # workaround for now to leave these out...
        op.Flush()
        op.ErrorCheck()

    def device_locker(self,device=None,queue=None):
        """    
        This returns a context manager which ensures that the jtag probe allows access only to the 
        thread within the jtag_lock's containing block. Use only within a 'with' block.
        
        Args:
            device (int) : Optional device id of the jtag interface to execute the lock on.
                        if None is provided (default), then all jtag chains will
                        have the lock called on them.
            queue (bool) : whether to queue up transactions during the lock so that they
                           get issued all at once.
    
        Example:
    
           >>> with ipc.device_locker() as lock:
           ...     # no other jtag operations from other windows or Trace32
           ...     # will interrupt the instructions in this block
           ...     itp.irdrscan(...) 
           ...     itp.irdrscan(...)
                
        """ 
        # it's not all that useful to log the caller here, that should be done in the enter
        if device is not None:
            did = self._device_convert(device)
        else:
            # by default, lock the whole domain
            did = self.base._domain
        return DeviceLocker(did,queue)

    def jtag_locker(self,device=None):
        """    
        This returns a context manager which ensures that the jtag probe allows access only to the 
        thread within the jtag_lock's containing block. Use only within a 'with' block.
        
        Args:
            device (int) : Optional device id of the jtag interface to execute the lock on.
                        if None is provided (default), then all jtag chains will
                        have the lock called on them.
        
        Note:
            jtag_locker is deprecated. device_locker should be used instead.
    
        Example:
    
           >>> with ipc.jtag_locker() as lock:
           ...     # no other jtag operations from other windows or Trace32
           ...     # will interrupt the instructions in this block
           ...     itp.irdrscan(...) 
           ...     itp.irdrscan(...)
                
        See Also:
            - :py:meth:`~ipccli.ipc_env.ipc_commands.Commands.device_locker`
        """ 
        # it's not all that useful to log the caller here, that should be done in the enter
        warnings.warn("jtag_locker is being deprecated...please use device_locker instead",JtagLockerWarning)
        return DeviceLocker(device)
    
    def i2c_locker(self,device=None):
        """
        This returns a context manager which ensures that the i2c bus allows access only to the 
        thread within the i2c_lock's containing block. It calls the py2ipc i2c Lock and Unlock
        API. The Lock will force repeated starts between the various SMBUS commands that
        are within the 'with' block.
        
        Args:
            device (int) : Optional device id of the i2c interface to execute the lock on.
                        if None is provided (default), then all i2c devices will
                        have the lock called on them.
        
        Use only within a 'with' block.
    
        Example:
    
           >>> with ipc.i2c_locker() as lock:
           ...     # a repeated stop will be inserted between these two
           ...     itp.i2c_raw_write(...)
           ...     itp.i2c_raw_read(...)

        """
        # it's not all that useful to log the caller here, that should be done in the enter
        warnings.warn("i2c_locker is being deprecated...please use device_locker instead",I2CLockerWarning)
        return DeviceLocker(device)

    def pulsepwrgood(self,device=None,timeOut=None,sleep=None,count=None):
        """
        Power-cycle the target using the specified debug port device. If the target
        is unpowered then power will be restored.
        
        Args:
            device (int): The did or alias of the device. If none is specified, 
                then the most appropriate debug port is used to power-cycle 
                the target.
            timeOut (int): In milliseconds, the amount of time to wait for the
                target to power down before aborting. If none is specified, then
                the value will automatically be determined.
            sleep (int): In milliseconds, the amount of time to wait after power
                loss before attempting to power the target system back on. If none
                is specified, then the value will automatically be determined.
            count (int): In nanoseconds, the pulse width of the pin assertion used
                to power the target system back up. If none is specified, then the
                value will automatically be determined.
            
        Returns:
            None.
    
        A target can be can be power-cycled by pulsing the power good signal
        exposed on the target through the debug port. There may also be
        alternate ways to accomplish the task.
    
        This function turns power off on the target, pauses a moment, then
        turns power back on.  This function uses the most appropriate way
        for power-cycling the target for the given device.  If no device
        is given (by not passing in anything), then the most appropriate 
        debug port and way is used to power-cycle the target.

        Note that the platform type of the debug port determines the algorithm
        and parameters used for power-cycling the target.
            
        Example:
            >>> ipc.pulsepwrgood()
            >>> ipc.pulsepwrgood(0x11000L)
            >>> ipc.pulsepwrgood(0)
            >>> ipc.pulsepwrgood(0, 10000, 6000, 16700000) # debug port 0, timeOut 10000, sleep 6000, count 16700000
            >>>        

        See Also:
            - :py:meth:`~ipccli.ipc_env.ipc_commands.Commands.setplatformtype`
            - :py:meth:`~ipccli.ipc_env.ipc_commands.Commands.getplatformtype`
            - :py:meth:`~ipccli.ipc_env.ipc_commands.Commands.getsupportedplatformtypes`

        **IPC services required:** PlatformControl

        """
        if _log.isEnabledFor(cli_logging.DEBUG):
            devicestr = self._device_to_devicestr(device)
            _log.debug("ENTER: pulsepwrgood(device={0})".format(devicestr))
        #if device is not specified, or set to None, set did to IPC_INVALID_DEVICE_ID
        #and the most appropriate debug port and way is used to power-cycle the target
        if(device == None):
            did = py2ipc.IPC_Types.IPC_INVALID_DEVICE_ID
        else:
            did = self._convert_debug_port(device)
        pcsrvc = py2ipc.IPC_GetService("PlatformControl")
        try:
            # If the caller provided values for the optional parameters then
            # temporarily modify the device configuration settings of the
            # debug port before calling PowerCycleTarget()
            powerofftimeout_orig = None
            poweroffdelay_orig = None
            poweronpulsewidth_orig = None
            if timeOut is not None or sleep is not None or count is not None:
                self.base._version_check("OpenIPC", build=1845, revision=614653)
                if OperationWindow.isopen():
                    raise RuntimeError("Specifying parameters to pulsepwrgood() within a lock window is not supported")
                debugportnode = self.devicelist.findByDID(did).node
                if timeOut is not None:
                    powerofftimeout_orig = debugportnode.config.PlatformControl.PowerOffTimeout
                    debugportnode.config.PlatformControl.PowerOffTimeout = str(timeOut)
                if sleep is not None:
                    poweroffdelay_orig = debugportnode.config.PlatformControl.PowerOffDelay
                    debugportnode.config.PlatformControl.PowerOffDelay = str(sleep)
                if count is not None:
                    poweronpulsewidth_orig = debugportnode.config.PlatformControl.PowerOnPulseWidth
                    debugportnode.config.PlatformControl.PowerOnPulseWidth = str(count)
            operation = pcsrvc.PowerCycleTarget(did)
            if not OperationWindow.isopen(): # flush operations is not in a window
                operation.Flush()
                operation.ErrorCheck()
        finally:
            if powerofftimeout_orig is not None:
                debugportnode.config.PlatformControl.PowerOffTimeout = powerofftimeout_orig
            if poweroffdelay_orig is not None:
                debugportnode.config.PlatformControl.PowerOffDelay = poweroffdelay_orig
            if poweronpulsewidth_orig is not None:
                debugportnode.config.PlatformControl.PowerOnPulseWidth = poweronpulsewidth_orig
        if _log.isEnabledFor(cli_logging.DEBUG):
            _log.debug("EXIT : pulsepwrgood")
    
    def resettarget(self,device=None):
        """
        Reset the target using the specified debug port device.
        
        Args:
            device (int): The did or alias of the debug port device. If none is
                specified, then the most appropriate debug port is used to reset
                the target.
            
        Returns:
            None.
            
        A target can be reset in a number of ways, depending on how the IPC
        is connected to the target and what is required to reset the target.
        For example, pulsing the reset pin on an XDP can often cause a target
        to go through a reset cycle. This function uses the most appropriate
        method for resetting the target for the given device. If no device
        is given (by not passing in anything), then the most appropriate 
        debug port is used to reset the target.

        Note that the platform type of the debug port determines the algorithm
        and mechanisms used for resetting the target.

        Example:
            >>> ipc.resettarget()
            >>> ipc.resettarget(0x11000L)
            >>>   

        See Also:
            - :py:meth:`~ipccli.ipc_env.ipc_commands.Commands.setplatformtype`
            - :py:meth:`~ipccli.ipc_env.ipc_commands.Commands.getplatformtype`
            - :py:meth:`~ipccli.ipc_env.ipc_commands.Commands.getsupportedplatformtypes`

        **IPC services required:** PlatformControl

        """
        if _log.isEnabledFor(cli_logging.DEBUG):
            devicestr = self._device_to_devicestr(device)
            _log.debug("ENTER: resettarget(device={0})".format(devicestr))
        #if device is not specified, or set to None, set did to IPC_INVALID_DEVICE_ID
        #and the most appropriate debug port and way is used to power-cycle the target
        if(device == None):
            did = py2ipc.IPC_Types.IPC_INVALID_DEVICE_ID
        else:
            did = self._convert_debug_port(device)
        pcsrvc = py2ipc.IPC_GetService("PlatformControl")
        operation = pcsrvc.ResetTarget(did)
        if not OperationWindow.isopen(): # flush operations is not in a window
            operation.Flush()
            operation.ErrorCheck()
        if _log.isEnabledFor(cli_logging.DEBUG):
            _log.debug("EXIT : resettarget")

    def power_status(self, device=None):
        """
        Returns the status of the target system power using the specified debug
        port device.
        
        Args:
            device (int): The did or alias of the debug port device. If none is
                specified, then the power status is called on each debug port.
            
        Returns:
            True/False (python boolean). If at least one debug port is off, then
            it is considered off.       

        Note that the platform type of the debug port determines the algorithm
        and mechanisms used for determining the power status of the target.

        Example:
            >>> ipc.power_status()
            >>> ipc.power_status(0x11000L)
            >>>   

        See Also:
            - :py:meth:`~ipccli.ipc_env.ipc_commands.Commands.setplatformtype`
            - :py:meth:`~ipccli.ipc_env.ipc_commands.Commands.getplatformtype`
            - :py:meth:`~ipccli.ipc_env.ipc_commands.Commands.getsupportedplatformtypes`

        **IPC services required:** PlatformControl, Device

        """
        if _log.isEnabledFor(cli_logging.DEBUG):
            devicestr = self._device_to_devicestr(device)
            _log.debug("ENTER: power_status(device={0})".format(devicestr))
        pcsrvc = py2ipc.IPC_GetService("PlatformControl")
        status = True
        
        if(device):
            did = self._convert_debug_port_i2c(device)
            status = pcsrvc.GetDebugPortPowerStatus(did)
        else:
            devsrv = py2ipc.IPC_GetService("Device")
            debugports = devsrv.GetDescendantIdsForType(self.base._domain, "Debugport", True)
            for did in debugports:
                try:
                    status = status and pcsrvc.GetDebugPortPowerStatus(did)
                except py2ipc.IPC_Error as err:
                    # assume server was already running if we get bind to port error
                    # and port was specified
                    if err.code != py2ipc.IPC_Error_Codes.Not_Supported:
                        raise

        if _log.isEnabledFor(cli_logging.DEBUG):
            _log.debug("EXIT : power_status")
        return status

    def operating_modes(self, device):
        """
        Returns a list of string of the operating modues the thread is currently operating in.

        An x86/64 processor can expect one or more of the following modes:
            Real, Protected, Virtual86, Compatibility, 64Bit, VmxRoot, VmxGuest, and Smm

        Unless the thread is in Smm, VmxRoot or VmxGuest mode -- only one operating mode
        would be returned. When the thread is in Smm, VmxRoot, or VmxGuest, more than
        one operating modes would be returned since for example the processor could be
        in Smm Real mode.

        Args:
            device (int): the did or alias of the device to execute the
                          return the operating modes for (not needed if using
                          from a node object).
        """
        if _log.isEnabledFor(cli_logging.DEBUG):
            devicestr = self._device_to_devicestr(device)
            _log.debug("ENTER: operating_modes(device={0})".format(devicestr))

        ##### Required for every function:
        did = self._device_convert(device)

        rcsrvc = py2ipc.IPC_GetService("RunControl")
        op_modes = rcsrvc.GetOperatingModes(did)

        if _log.isEnabledFor(cli_logging.DEBUG):
            _log.debug("EXIT : operating_modes : result = %s"%op_modes)

        return op_modes

    def specifytopology(self, device, topology):
        """
        
        Specifies a TAP topology for the given JTAG scanchain. For each TAP
        controller, the user can specify either:

            - Device type, sub type (optional), and stepping (optional)
            - The IR length of an irrelevant/unknown TAP

        If the caller wants to later prune it down to what is really there on
        the target, they can call ipc.forcereconfig() with the 'AdjustTaps' phase.

        Args:
            
            device (int): the did or alias of a JTAG scanchain device.
            topology (list): the TAP topology on the scanchain. This should
                             be a list where each entry is either a list of 3 strings
                             (device type, sub-type, and stepping), or an integer
                             indicating the IR length of an irrelevant/unknown TAP

        Examples:

            >>> ipc.devs.jtagscanchain0.specifytopology([["BXT_CLTAPC", "", "B0"]])
            >>> ipc.devs.jtagscanchain0.specifytopology([["SKL_U_UC", "", "D0"], 8, ["SPT", "H", "D0"]])

        **IPC services required:** Device
        """
        ##### logging
        if _log.isEnabledFor(cli_logging.DEBUG):
            devicestr = self._device_to_devicestr(device)
            _log.caller()
            _log.debug("ENTER: specifytopology(device={0}, topology={1})".\
                       format(devicestr, topology))

        self.base._version_check("OpenIPC", build=3033, revision=594042)
        from py2ipc.IPC_TapController import IPC_TapController

        if not isinstance(topology, list):
            raise TypeError("The given topology should be a list. Example: [['Device', '', ''], 12]")

        topo = []
        for entry in topology:
            new_topo = IPC_TapController()
            if isinstance(entry, list):
                if len(entry) != 3:
                    raise ValueError("The given list should contain 3 entries: device type, sub-type, and stepping. Example: ['Device', '', 'A0']")
                new_topo.deviceType = entry[0]
                new_topo.deviceSubType = entry[1]
                new_topo.stepping = entry[2]
            elif isinstance(entry, (int, long)):
                new_topo.irLength = entry
            else:
                raise TypeError("Each topology entry should be either a list of 3 strings, or an integer")
            topo.append(new_topo)

        dev_srv = py2ipc.IPC_GetService("Device")

        did = self._device_convert(device)
        dev_srv.SpecifyTopology(did, topo)

        _log.debug("EXIT: specifytopology")


    def setplatformtype(self, device, platformtype): 
        """
        Sets the platform type of a debug port.
        
        The platform type determines how platform control operations (e.g. power
        cycle, reset, and stalls) are implemented. It is determined on start-up
        either by the configuration of the IPC implementation or automatically
        detected based on the connected target.

        Args:
            device (int): The did or alias of the debug port (not needed if
                          using from a node object).
            platformtype (str): The platform type.

        Examples:
            >>> ipc.debugports[0].setplatformtype("Generic")

        See Also:
            - :py:meth:`~ipccli.ipc_env.ipc_commands.Commands.pulsepwrgood`
            - :py:meth:`~ipccli.ipc_env.ipc_commands.Commands.resettarget`
            - :py:meth:`~ipccli.ipc_env.ipc_commands.Commands.power_status`
        """
        ##### logging
        if _log.isEnabledFor(cli_logging.DEBUG):
            devicestr = self._device_to_devicestr(device)
            _log.caller()
            _log.debug("ENTER: setplatformtype(device={0},platformtype={1})".format(devicestr,platformtype))

        did = self._device_convert(device)
        debugport_device = self.devicelist.findByDID(did)
        debugport_device.node.config.PlatformControl.PlatformType = platformtype
        self.base._refresh_devicelist()

        _log.debug("EXIT: setplatformtype")

    def getplatformtype(self, device):
        """
        Gets the platform type of a debug port.

        The platform type determines how platform control operations (e.g. power
        cycle, reset, and stalls) are implemented. It is determined on start-up
        either by the configuration of the IPC implementation or automatically
        detected based on the connected target.

        Args:
            device (int): The did or alias of the debug port (not needed if
                          using from a node object).

        Examples:
            >>> ipc.debugports[0].getplatformtype()
            "Generic"

        See Also:
            - :py:meth:`~ipccli.ipc_env.ipc_commands.Commands.pulsepwrgood`
            - :py:meth:`~ipccli.ipc_env.ipc_commands.Commands.resettarget`
            - :py:meth:`~ipccli.ipc_env.ipc_commands.Commands.power_status`
        """
        ##### logging
        if _log.isEnabledFor(cli_logging.DEBUG):
            devicestr = self._device_to_devicestr(device)
            _log.caller()
            _log.debug("ENTER: getplatformtype(device={0})".format(devicestr))

        did = self._device_convert(device)
        debugport_device = self.devicelist.findByDID(did)
        platformtype = debugport_device.node.config.PlatformControl.PlatformType

        _log.debug("EXIT: getplatformtype : result = {0}".format(platformtype))
        return platformtype

    def getsupportedplatformtypes(self, device):
        """
        Gets the supported platform types of a debug port.

        Args:
            device (int): The did or alias of the debug port (not needed if
                          using from a node object).

        Examples:
            >>> ipc.debugports[0].getsupportedplatformtypes()
            ["Generic"]

        See Also:
            - :py:meth:`~ipccli.ipc_env.ipc_commands.Commands.pulsepwrgood`
            - :py:meth:`~ipccli.ipc_env.ipc_commands.Commands.resettarget`
            - :py:meth:`~ipccli.ipc_env.ipc_commands.Commands.power_status`
        """
        ##### logging
        if _log.isEnabledFor(cli_logging.DEBUG):
            devicestr = self._device_to_devicestr(device)
            _log.caller()
            _log.debug("ENTER: getsupportedplatformtype(device={0})".format(devicestr))

        did = self._device_convert(device)
        devicecontrol_srv = py2ipc.IPC_GetService("DeviceControl")
        platformtypes = devicecontrol_srv.GetDeviceConfigOptions(did, "PlatformControl.PlatformType")
        platformtypes.sort()

        _log.debug("EXIT: getsupportedplatformtypes : result = {0}".format(platformtypes))

        return platformtypes

    def block_transport_from_leaving(self, device=None):
        """
        On products where the probe may go away due to power management, this
        will prevent that from happening during the section of code within
        a with statement

        Currently this assumes transport is available when it is first run

        Args:
            device: this should be a debug port device

        Returns:
            context manager for prevening probe from leaving

        Example:

            >>> with ipc.block_transport_from_leaving():
            ...    ipc.irdrscan(0,2,32)
            ...    # ...other critical code...

        """
        from .probe_leaving_context import BlockTransportFromLeaving
        # it's not all that useful to log the caller here, that should be done in the enter
        if isinstance(device, collections.abc.Iterable):
            did = [self._device_convert(d) for d in device]
        elif device is not None:
            did = self._device_convert(device)
        else: # must be None
            did = None
        return BlockTransportFromLeaving(did)


    def perfreport(self, categories=["Version", "Host", "JTAG", "RunControl"], rc_test_thread=None):
        """
        Runs a performance check of the IPC implementation and prints the results.
        
        Args:
            categories (list): a list of the categories to include in the
                report; categories include: "Version", "Host", "JTAG", and
                "RunControl" (default is ["Version", "Host", "JTAG", "RunControl"]).
            rc_test_thread (thread): A thread node to be used during the Run Control tests. 
                If not specified the first thread in the device list is used. 
        Example:
            >>> ipc.perfreport()
            Host:
             OS                 : Windows-7-6.1.7601-SP1
             Processor          : Intel64 Family 6 Model 69 Stepping 1, GenuineIntel
             Thread Count       : 4
            JTAG:
             Debug Port 0:
              Chain 0:
               Minscan Latency  : 858us
               Thoughput        : 2.30Mtcks/sec
              Chain 1:
               Minscan Latency  : 825us
               Thoughput        : 3.80Mtcks/sec
            Run Control:
             GPC:
              IO Read           : 2286us
              MSR Read          : 3547us
              MSR Write         : 3069us
              Memory Latency    : 2519us
              Memory Throughput : 61KB/sec (using 1KB read)
              Single Step       : 109ms
            Version:
             IPC API Interface  : 2.0.14.0 (100031)
             IPC Implementation : 1.0.0.600 (OpenIPC:Main (rev 0))
             IPC-CLI            : 0.0
             IPC-CLI, SVN Rev   : 618205
             PY2IPC Bindings    : 0.0
            
            >>> results = ipc.perfreport()
            >>> results["JTAG"]["Debug Port 0"]["Chain 0"]["Minscan Latency"]
            858us
        """
        if _log.isEnabledFor(cli_logging.DEBUG):
            _log.debug("ENTER: perfreport(categories={0:s})".format(str(categories)))

        results = {}

        if "Version" in categories:
            results["Version"] = self.base.version()
            
        if "Host" in categories:
            results["Host"] = self._perfreport_host()

        if "JTAG" in categories:
            results["JTAG"] = self._perfreport_jtag()

        thread = rc_test_thread or (self.base.threads and self.base.threads[0])
        if "RunControl" in categories and thread:
            results["Run Control"] = self._perfreport_run_control(thread)

        if _log.isEnabledFor(cli_logging.DEBUG):
            _log.debug("EXIT: perfreport()")

        return ReportResults(results)

    def _perfreport_host(self):
        import platform
        import multiprocessing
        return ReportResults({
            "OS": platform.platform(),
            "Processor": platform.processor(),
            "Thread Count": multiprocessing.cpu_count()
        })

    def _perfreport_jtag(self):
        results = {}
        debugports = self.base.devs.group_by(nodetype="^Debugport$")
        for debugport in debugports:
            results["Debug Port {}".format(debugport.device.instanceid)] = self._perfreport_jtag_debugport(debugport)
        return ReportResults(results)

    def _perfreport_jtag_debugport(self, debugport):
        results = {}
        for chain in [child for child in debugport.children if child.nodetype == "JTAGScanChain"]:
            if len(chain.children) > 0:
                results["Chain {}".format(chain.device.instanceid)] = self._perfreport_jtag_chain(chain)
        return ReportResults(results)

    def _perfreport_jtag_chain(self, chain):
        tap = chain.children[0].device
        return ReportResults({
            "Minscan Latency": self._perfreport_jtag_minscan_latency(tap),
            "Thoughput": self._perfreport_jtag_throughput(tap)
        })

    def _perfreport_jtag_minscan_latency(self, device, iterations=1000):
        with self.device_locker():
            device.node.jtag_goto_state("TLR", 1, "COUNT")
            result = device.node.jtag_shift("ShfDR", 1)
            int(result) # Force the pending scan to flush

            start_time = timeit.default_timer()
            for i in range(iterations):
                result = device.node.jtag_shift("ShfDR", 1)
                int(result) # Force the pending scan to flush
            end_time = timeit.default_timer()
            total_time = end_time - start_time
            latency = int(total_time / iterations * 1000 * 1000)
            return TimedReportResult(latency, "us")

    def _perfreport_jtag_throughput(self, device, shiftcount=5000, shiftlength=90):
        with self.device_locker():
            device.node.jtag_goto_state("TLR", 1, "COUNT")
            result = device.node.jtag_shift("ShfDR", 1)
            int(result) # Force the pending scan to flush

            for i in range(shiftcount):
                a = device.node.jtag_shift("ShfDR", shiftlength)
                device.node.jtag_goto_state("RTI", 1, "COUNT")
            result = device.node.jtag_shift("ShfDR", 1)

            start_time = timeit.default_timer()
            int(result) # Force the pending scan to flush
            end_time = timeit.default_timer()
            total_time = end_time - start_time

            if total_time > 0:
                calculatedtclks = (shiftcount * (shiftlength + 5)) + 6
                throughput = calculatedtclks / total_time / 1000000
            else:
                throughput = 0

            return TimedReportResult(throughput, "Mtcks/sec")

    def _perfreport_run_control(self, thread):
        # Ignore run control events
        ignore_run_control_events = self.base.events.ignore_run_control(None)
        if not ignore_run_control_events:
            self.base.events.ignore_run_control(True)

        was_running = False

        results = {}

        try:
            # Halt the thread if needed
            if thread.isrunning():
                thread.halt()
                was_running = True

            results[thread.device.coregroup] = ReportResults({
                "Memory Latency": self._perfreport_memory_latency(thread),
                "Memory Throughput": self._perfreport_memory_throughput(thread),
                "Single Step": self._perfreport_single_step(thread),
                "IO Read": self._perfreport_io_read(thread),
                "MSR Read": self._perfreport_msr_read(thread),
                "MSR Write": self._perfreport_msr_write(thread)
            })
        except:
            raise
        finally:
            # Resume the thread if it was running
            if was_running:
                thread.go()

            # Stop ignoring run control events
            if not ignore_run_control_events:
                self.base.events.ignore_run_control(False)

        return ReportResults(results)

    def _perfreport_single_step(self, thread, iterations=20):
        start_time = timeit.default_timer()
        for i in range(iterations):
            thread.step()
        end_time = timeit.default_timer()
        total_time = end_time - start_time
        average_time = int(total_time / iterations * 1000)
        return TimedReportResult(average_time, "ms")

    def _perfreport_io_read(self, thread, iterations=100):
        start_time = timeit.default_timer()
        for i in range(iterations):
            thread.port(0x80)
        end_time = timeit.default_timer()
        total_time = end_time - start_time
        average_time = int(total_time / iterations * 1000 * 1000)
        return TimedReportResult(average_time, "us")

    def _perfreport_msr_read(self, thread, iterations=100):
        start_time = timeit.default_timer()
        for i in range(iterations):
            thread.msr(0x1D9)
        end_time = timeit.default_timer()
        total_time = end_time - start_time
        average_time = int(total_time / iterations * 1000 * 1000)
        return TimedReportResult(average_time, "us")

    def _perfreport_msr_write(self, thread, iterations=100):
        start_time = timeit.default_timer()
        for i in range(iterations):
            thread.msr(0x1D9, 0)
        end_time = timeit.default_timer()
        total_time = end_time - start_time
        average_time = int(total_time / iterations * 1000 * 1000)
        return TimedReportResult(average_time, "us")

    def _perfreport_memory_latency(self, thread, iterations=150):
        start_time = timeit.default_timer()
        for i in range(iterations):
            thread.memblock('0x10000P', 1, 8)
        end_time = timeit.default_timer()
        total_time = end_time - start_time
        latency = int(total_time / iterations * 1000 * 1000)
        return TimedReportResult(latency, "us")

    def _perfreport_memory_throughput(self, thread, iterations=100):
        thread.memblock('0x10000P', 1, 8)
        start_time = timeit.default_timer()
        for i in range(iterations):
            thread.memblock('0x10000P', 128, 8) # Read 1KB
        end_time = timeit.default_timer()
        total_time = end_time - start_time
        if total_time > 0:
            throughput = int(1.0 / (total_time / iterations))
        else:
            throughput = 0 
        return TimedReportResult(throughput, "KB/sec", "(using 1KB read)")

class ReportResults(dict):
    """
    Represents a table hierarchy of results intended for pretty-printing and querying.
    """

    def __init__(self, *args, **kwargs):
        dict.__init__(self, *args, **kwargs)

    def __str__(self):
        return self._to_string()

    def _to_string(self, depth=0):
        string = ""
        for k, v in sorted(self.items()):
            if isinstance(v, ReportResults):
                value = v._to_string(depth + 1)
                string += "{}{}:\n{}".format(" " * depth, k, value).expandtabs(20)
            else:
                string += "{}{}\t: {}\n".format(" " * depth, k, v).expandtabs(20)
        return string

    __repr__ = __str__

class ArchRegsResults(dict):
    required_registers =[
        'eax', 'ebx', 'ecx', 'edx',
        'esi', 'edi', 'ebp', 'esp', 'eip',
        'cs', 'ss', 'ds', 'es', 'fs', 'gs',
        'eflags',
    ]
    def __init__(self, *args, **kwargs):
        dict.__init__(self, *args, **kwargs)

    def __str__(self):
        return self._to_string()

    __repr__ = __str__

    def _to_string(self, depth=0):
        msg = []
        msg.append("eax={eax:08x}    ebx={ebx:08x}    ecx={ecx:08x}    edx={edx:08x}".format(**self))
        msg.append("esi={esi:08x}    edi={edi:08x}    ebp={ebp:08x}    esp={esp:08x}".format(**self))
        msg.append("eip={eip:08x}".format(**self))
        msg.append("cs={cs:04x}    ds={ds:04x}    es={es:04x}    fs={fs:04x}    gs={gs:04x}".format(**self))
        msg.append("eflags={eflags:08x}".format(**self))
        return "\n".join(msg)


class ArchExtendedRegsResults(dict):
    required_registers =[
        'rax', 'rbx', 'rcx', 'rdx',
        'rsi', 'rdi', 'rbp', 'rsp', 'rip',
        'r8', 'r9', 'r10','r11','r12','r13','r14',
        'cs', 'ss', 'ds', 'es', 'fs', 'gs',
        'rip',
        'eflags',
    ]
    def __init__(self, *args, **kwargs):
        dict.__init__(self, *args, **kwargs)

    def __str__(self):
        return self._to_string()

    __repr__ = __str__

    def _to_string(self, depth=0):
        msg = []
        msg.append("rax={rax:016x}    rbx={rbx:016x}    rcx={rcx:016x}    rdx={rdx:016x}".format(**self))
        msg.append("rsi={rsi:016x}    rdi={rdi:016x}    rbp={rbp:016x}    rsp={rsp:016x}".format(**self))
        msg.append("r8 ={r8:016x}    r9 ={r9:016x}    r10={r10:016x}    r11={r11:016x}".format(**self))
        msg.append("r12={r8:016x}    r13={r9:016x}    r14={r10:016x}    r15={r11:016x}".format(**self))
        msg.append("rip={rip:016x}".format(**self))
        msg.append("cs={cs:04x}    ds={ds:04x}    es={es:04x}    fs={fs:04x}    gs={gs:04x}".format(**self))
        msg.append("eflags={eflags:08x}".format(**self))
        return "\n".join(msg)


class TimedReportResult(object):
    """
    Represents a result in a ReportResults object containing a duration of time.
    """

    def __init__(self, value, unit, notes=None):
        self.value = value
        self.unit = unit
        self.notes = notes

    def __str__(self):
        if isinstance(self.value, float):
            string = "{:.2f}{}".format(self.value, self.unit)
        else:
            string = "{}{}".format(self.value, self.unit)

        if self.notes:
            string = "{} {}".format(string, self.notes)

        return string

    __repr__ = __str__

########################################################################################
# This is very similar to jtag locker...one might even recommend doing a meta class
# to create these various windows since they are so similar, but not today
# FYI: meta class would definitely be needed over inheritance since the classes
# keep counts instead of the objects...
########################################

class OperationWindow(object):
    """
    Use this window to have the CLI not force flushing of operations
    """
    threadid = "" # filled in during enter
    _window_count = {}
    _window_lock = threading.Lock() # I don't _think_ we need a re-entrant lock...
    def __init__(self):
        pass

    def __enter__(self):
        self.threadid = threading.currentThread().name
        with self._window_lock:
            if self.threadid not in self._window_count:
                self._window_count[self.threadid] = 0
            self._window_count[self.threadid] += 1
            _log.debug("OperationWindow opened for thread: %s, current window count is %d",
                       self.threadid,
                       self._window_count[self.threadid]
                       )
    
    @classmethod
    def isopen(cls):
        """isopen()

        For determining whether we are in a with statement that has an operation
        window active

        Returns:
            - True : we are within an operation window where operation receipts should
                   not be flushed
            - False : we are not within an operation window
        """
        # could speed this up by checking for thread count perhaps? some optimization
        # here around single threaded may be good
        count = cls._window_count.get( threading.currentThread().name, 0)
        return count

    def __exit__(self):
        if threading.currentThread().name != self.threadid:
            raise RuntimeError("!! Illegal use of Operation Windows!!\nYou entered lock on thread: '{0}' and exited on another thread: '{1}'".\
                    format(self.threadid,
                           threading.currentThread().name))

        self._window_count[self.threadid] -= 1
        _log.debug("OperationWindow closed for thread: %s, current window count is %d",
                    self.threadid,
                    self._window_count[self.threadid]
                    )

class DeviceLocker(object):
    '''
    This context manager ensures that the only this client has access to the
    specified device (whether it is I2C, jtag, or the debug port)

    Use only within a 'with' block.
    
    E.g.
    
    with ipc.device_locker() as lock:
        # no other jtag opperations from other windows or Trace32
        # will interrupt the instructions in this block
        ipc.irdrscan(...) 
        ipc.irdrscan(...)
    '''
    # class variable to keep with anytime we have entered a lock
    # we have to keep up with these per thread...
    _lock_count = {}
    _locker_lock = threading.Lock() # I don't _think_ we need a re-entrant lock...
    # in case user turns off queue
    _operation_window = None
    threadid = "" # filled in during enter
    def __init__ (self,did=None,queue=None):
        """
        device : Device to be locked (IPC AI did required for now), can be single device or list of devices
        queue : can be set to False to prevent queue'ing inside locks (default is equivalent to True)
        """
        # convert did ?        
        if isinstance(did,list):
            self.dids = [ did ]
        elif isinstance(did, (int,long) ):
            self.dids = [ did ]
        elif did == None:
            self.dids = []
        else:
            raise TypeError("Unknown type for did: type='{0:s}', value='{1:s}'".format(
                            type(did), did))
        if queue in [None,True]:
            self._operation_window = OperationWindow()        
 
    @classmethod
    def lock_present(cls):
        """Used to know if a lock has been queued"""
        # could speed this up by checking for thread count perhaps? some optimization
        # here around single threaded may be good
        count = cls._lock_count.get(threading.currentThread().name,0)
        return count > 0
       
    def __enter__(self):
        # if we are entering lock then enter operation window as well
        _log.caller()
        if self._operation_window:
            self._operation_window.__enter__()

        with self.__class__._locker_lock:
            self.threadid = threading.currentThread().name
            if self.threadid not in self._lock_count.keys():
                self._lock_count[self.threadid] = 0
            self._lock_count[self.threadid] += 1
            _log.debug("DeviceLocker: Lock entered for thread %s, lock count is %d",self.threadid,self._lock_count[self.threadid])
        # The IPC keeps up with a count, so it is fine for us to simply ensure we perform lock on
        # entry, and remove lock on exit
        dev_srv  = py2ipc.IPC_GetService("Device")
        # if no dids were provided then lock everything
        if self.dids == []:
            self.dids = dev_srv.GetTargetDomainDeviceIds()
            # we 'should' be able to lock the entire domain, but I'm seeing some issues with that
            # so also coding this workaround here for using scan chains...this code can be commented
            # out, but do not remove it
            #-----
            # domains = dev_srv.GetTargetDomainDeviceIds()
            #self.dids = []
            #for domain in domains:
            #    dids = dev_srv.GetDescendantIdsForType(domain, "JTAGScanChain", True)
            #    self.dids.extend(dids)
            #-----

        _log.debug("DeviceLocker: Locking: {0}".format( " ".join( map(hex, self.dids) ) ) )
        # maybe just always use LockMany ?
        op = dev_srv.LockMany( self.dids, -1)
        if self._operation_window == None:
            op.Flush()
        elif not self._operation_window.isopen():
            # we have an operation window and it is not open
            op.Flush()
            # else - we have an open operation window...no flushing

    def __exit__(self, exc_type, exc_value, traceback):
        # watchout: in dal compatibility the manualscans controlvar calls this manually with
        # None for the various parameters
        _log.caller()
        with self._locker_lock:
            if self.threadid != threading.currentThread().name:
                raise RuntimeError("!! Illegal use of threading and locks !!\nYou entered lock on thread: '{0}' and exited on another thread: '{1}'".\
                    format(self.threadid,threading.currentThread().name))
            self._lock_count[self.threadid] -= 1        

        _log.debug("DeviceLocker: Lock exited for thread %s, lock count is %d",self.threadid,self._lock_count[self.threadid])

        # if we are exiting lock...exit operation window as well
        if self._operation_window:
            self._operation_window.__exit__()

        dev_srv  = py2ipc.IPC_GetService("Device")
        _log.debug("DeviceLocker: Unlocking: {0}".format( " ".join( map(hex, self.dids) ) ) )
        op = dev_srv.Unlock()
        if self._operation_window==None or (not self._operation_window.isopen()):
            _log.debug("DeviceLocker: Flushing UnLock for thread %s",self.threadid)
            #stime = time.time()
            op.Flush()
            #etime = time.time()
            # _log.result("Flushing UnLock for thread %s, took: %0.04f",self.threadid,etime-stime)            
            op.ErrorCheck()  
            

class InterfaceportWindow:
    '''
    This context manager executes the interface port open and close window
    functions for performing read and/or write access from or to the given
    interface port.
        
    See Also:
        - :py:meth:`~ipccli.ipc_env.ipc_commands.Commands.interfaceport_window`
    '''

    def __init__ (self,did,filename=None,accessmode="read"):
        self.did = did
        self.filename = filename
        self.accessmode = accessmode
        
    def __enter__(self):        
        interfaceport_srv = py2ipc.IPC_GetService("InterfacePort")
        
        # Get the configuration for the given device
        configuration = interfaceport_srv.GetInterfacePortConfiguration(self.did)
        
        if self.filename is None:
            # Set the configuration to stream the resulting data directly in memory
            configuration.dataCaptureType = py2ipc.Service_InterfacePort.IPC_InterfacePortDataCaptureTypes.Stream
        else:
            # Set the configuration to stream the resulting data to file
            configuration.dataCaptureType = py2ipc.Service_InterfacePort.IPC_InterfacePortDataCaptureTypes.File
            configuration.fileName = self.filename
            
        if self.accessmode.lower() == "read":
            configuration.accessMode = py2ipc.Service_InterfacePort.IPC_InterfacePortAccessModes.Read
        elif self.accessmode.lower() == "write":
            configuration.accessMode = py2ipc.Service_InterfacePort.IPC_InterfacePortAccessModes.Write
        elif self.accessmode.lower() == "readwrite":
            if configuration.dataCaptureType == py2ipc.Service_InterfacePort.IPC_InterfacePortDataCaptureTypes.File:
                raise ValueError("Invalid access mode: {0} is not compatible with file streaming.".format(self.accessmode))
            configuration.accessMode = py2ipc.Service_InterfacePort.IPC_InterfacePortAccessModes.ReadWrite
        else:
            raise ValueError("Invalid access mode: {0}".format(self.accessmode))
        
        # Open the window using the configuration
        interfaceport_srv.OpenWindow(self.did, configuration)

    def __exit__(self, exc_type, exc_value, traceback):
        interfaceport_srv = py2ipc.IPC_GetService("InterfacePort")
        
        # Close the window
        interfaceport_srv.CloseWindow(self.did)

