###############################################################################
# Copyright 2014 2017 Intel Corporation.
#
# The source code, information and material ("Material") contained herein is
# owned by Intel Corporation or its suppliers or licensors, and title to such
# Material remains with Intel Corporation or its suppliers or licensors. The
# Material contains proprietary information of Intel or its suppliers and
# licensors. The Material is protected by worldwide copyright laws and treaty
# provisions. No part of the Material may be used, copied, reproduced, modified,
# published, uploaded, posted, transmitted, distributed or disclosed in any way
# without Intel's prior express written permission.
#
# No license under any patent, copyright or other intellectual property rights
# in the Material is granted to or conferred upon you, either expressly, by
# implication, inducement, estoppel or otherwise. Any license under such
# intellectual property rights must be express and approved by Intel in writing.
#
# Unless otherwise agreed by Intel in writing, you may not remove or alter
# this notice or any other notice embedded in Materials by Intel or Intel's
# suppliers or licensors in any way.
###############################################################################

from __future__ import absolute_import

from .. import cli_logging
from .._py2to3 import *
import py2ipc

_log = cli_logging.getLogger("ipc")


class IpcCliBreaks(object):
    """

    The IPC CLI "breaks" variables will expose the supported breaks
    for those particular devices. The various breaks will
    halt execution control when the specified event occurs and
    depending on breakall, one or more threads will enter probe mode.


    The "breaks" will be available at the global level and will
    apply to all the devices in the
    current domain. The breaks attribute will be available on any nodes
    that have one or more supported breaktypes.

    Example 1:

        >>> ipc.breaks.machinecheck = 1

    Example 2:

        >>> ipc.threads[0].breaks.machinecheck = 1

    Example 3:

        >>> ipc.breaks.breakall = 1


    Some of the supported breaks that may be here are:

        - generaldetect
        - init
        - machinecheck
        - reset
        - shutdown
        - smminto
        - smmentry
        - smmexit
        - vmentry
        - vmexit

    Note:
        LTB only supports a global setting. So setting *any* thread will result in all threads having
        the specified breaks value changed.

    See Also:
        :py:meth:`~IpcCliBreaks.breakall`

    """
    # breaks must be a list here for default as too many python tricks
    # are checking breaks attribute assuming it is a list
    # we will make it an object (instead of class) list during __init__
    breaks = []
    _did = None
    _domain = None
    _hide_breakall = False

    def __init__(self, device, hide_breakall):
        self._did = device.did
        self._hide_breakall = hide_breakall
        # get domain did for the breakall get/set functionality
        if device.nodetype == "domain":
            self._domain = device.did
        else:
            self._domain = device.target_domain

        if device.nodetype in ["thread", "groupcore", "domain", "debugport"]:
            # we will use this service a lot of course, so save a reference
            break_service = py2ipc.IPC_GetService("Break")
            runcontrolconfig_service = py2ipc.IPC_GetService("RunControlConfiguration")
            # if the device is a thread then check that run control is enabled before
            # querying the break conditions
            try:
                if device.nodetype in [ "thread" ]:
                    devsrvc = py2ipc.IPC_GetService("Device")
                    coregroup = devsrvc.GetCoreGroupIdForDevice(self._did)
                    if not runcontrolconfig_service.IsRunControlEnabled(coregroup):
                        self.breaks = []
                        return
                elif device.nodetype in [ "groupcore" ]:
                    if not runcontrolconfig_service.IsRunControlEnabled(self._did):
                        self.breaks = []
                        return
                self.breaks = [b.lower() for b in break_service.GetBreakConditions(self._did)]
            except py2ipc.IPC_Error as e:
                # sometimes we get invalid device instead of an empty list
                if e.codestr in ["Invalid_Device_ID", "Invalid_Device"]:
                    self.breaks = []
                elif e.codestr in ["Not_Supported", "RunControl_Disabled"]:
                    self.breaks = []
                else:
                    _log.exception("IPC gave an unexpected error for break conditions")
                    # dont raise this...just keep going
        else:
            self.breaks = []

    def __dir__(self):
        dirlist = list(self.__dict__.keys())
        dirlist.extend(dir(self.__class__))
        dirlist.extend(self.breaks)
        if self._hide_breakall:
            # don't 'show' breakall unless it is in the global breaks
            dirlist.remove('breakall')
        return dirlist

    def __getattr__(self, attr):
        # the only dynamic attributes we support here are in the breaks variable
        if attr not in self.breaks:
            raise AttributeError("Unknown attribute or break {0}".format(attr))
        # supported break, get the value:
        break_service = py2ipc.IPC_GetService("Break")
        return break_service.GetBreakCondition(self._did, attr)

    def __setattr__(self, attr, value):
        # the only dynamic attributes we support here are in the breaks variable
        if attr in self.breaks:
            break_service = py2ipc.IPC_GetService("Break")
            return break_service.SetBreakCondition(self._did, attr.upper(), value)
        elif attr in dir(self):
            object.__setattr__(self, attr, value)
        else:
            raise AttributeError("Unknown break: %s" % attr)

    @property
    def breakall(self):
        """This determines whether all threads or just the
        thread that hit the break event should stop execution.
        """
        bpsrvc = py2ipc.IPC_GetService("Breakpoint")
        enabled = bpsrvc.GetBreakAllSetting(self._domain)
        return enabled == "Enabled"

    @breakall.setter
    def breakall(self, value):
        bpsrvc = py2ipc.IPC_GetService("Breakpoint")
        if isinstance(value, basestring):
            if value.lower() in ['enabled', 'true', 'on']:
                value = True
            elif value.lower() in ['disabled', 'false', 'off']:
                value = False
            else:
                raise Exception("Unexpected value '{0}' for breakall setting, please use True/False".format(value))
        # if we get a poolean, then that will count as the 1 / 0 integer values
        # the SetBreakAllSetting function would normally take, so we don't have to convert
        # assume we are good at this point
        bpsrvc.SetBreakAllSetting(self._domain, value)


