###############################################################################
# Copyright 2014 2017 Intel Corporation.
#
# The source code, information and material ("Material") contained herein is
# owned by Intel Corporation or its suppliers or licensors, and title to such
# Material remains with Intel Corporation or its suppliers or licensors. The
# Material contains proprietary information of Intel or its suppliers and
# licensors. The Material is protected by worldwide copyright laws and treaty
# provisions. No part of the Material may be used, copied, reproduced, modified,
# published, uploaded, posted, transmitted, distributed or disclosed in any way
# without Intel's prior express written permission.
#
# No license under any patent, copyright or other intellectual property rights
# in the Material is granted to or conferred upon you, either expressly, by
# implication, inducement, estoppel or otherwise. Any license under such
# intellectual property rights must be express and approved by Intel in writing.
#
# Unless otherwise agreed by Intel in writing, you may not remove or alter
# this notice or any other notice embedded in Materials by Intel or Intel's
# suppliers or licensors in any way.
###############################################################################

from __future__ import absolute_import

from ..breakpoint import Breakpoint
from ..address import Address,AddressType
from .. import cli_logging
from ..datatypes import UserType
from .._py2to3 import *

_log = cli_logging.getLogger("ipc")

import py2ipc

#
#
#
# TODO: there's a slight issue here where someone setting the breakpoint object
# elsewhere does not get relfected in these breakpoint objects...
# we would need to add some "autoupdate" to make sure we have always have
# the latest data?

#########################
#
# Due to user confusion, any update to properties
# will immediately call the change as well. Therefore, the
# change function itself will NOT be exposed.
#
#
# Properties
#   - Every properties 'getattr' should call "_update" first
#   - Every properties 'setter' should call "_change" last (before returning)
#
"""
Plan
    - __init__ reqires an ipc_breakpoint id (?)
    - new...will have to convert all the args, create an ipc_breakpoint and THEN create
      a CliBreakpoint using the ID....

    
"""
# maps the old dal name to the IPC name
_btype_mapping = {
                    'sw'         : "SW",
                    'exe'        : "HW_Exec",
                    'acc'        : "HW_Access",
                    'wr'         : "HW_Write",
                    'io'         : "HW_IO",
                    'exe local'  : "HW_Exec",
                    'acc local'  : "HW_Access",
                    'wr local'   : "HW_Write",
                    'io local'   : "HW_IO",                             
                    'exe global' : "Global_HW_Exec",
                    'acc global' : "Global_HW_Access",
                    'wr global'  : "Global_HW_Write",
                    'io global'  : "Global_HW_IO",
                    #'exe global' : 'Default', # we need to spell out the "Default" in the IPC since that is an option there                          

            }
_btype_mapping_reverse = dict( (v,k) for k,v in _btype_mapping.items() if k not in ["exe", "acc", "wr", "io"])

# need to manually add this one
_btype_mapping_reverse['Default'] = 'exe global'
# all others assumed to be physical address
_linear_types = ['exe','acc','wr','exe local','acc local','wr local','exe global','acc global','wr global','sw']

#########################
class IpcCliBreakpoint(Breakpoint):
    address = None
    dbreg = None
    dataSize = None
    breakType = None
    enabled = None
    deviceId = None
    name = None
    # internally used variables
    _ipc_breakpoint = None
    _address = None
    """
    
    Attributes:
        - address
        - dbreg
        - dataSize
        - breakType
        - enabled
        - deviceId
        - name
    
    """
     
    def __init__(self,env_breakpoint):
        Breakpoint.__init__(self)
        # Commands function also uses the "_ipc_breakpoint" so that
        self._ipc_breakpoint = env_breakpoint
    
    @classmethod
    def new(cls, device, address, breakType=None, dbreg='Any', enable=True, dataSize=1, name="" ):
        """
        Creates a new breakpoint object and adds it to the known breakpoints
        
        Arguments:
          address (int,Address):  an address for the new breakpoint
          breakType (str):
          dbgreg (str,int): "Any" for any debug breakpoint or a number for a specific
          enable (bool): True/False on whether to enable the new breakpoint
          dataSize (int): dataSize must be one of 1, 2, 4, or 8
          name (str): Unique name to give this breakpoint

        Returns:
          The new breakpoint

        """
        # some checks first:
        btype = _btype_mapping.get(breakType,None)
        if btype==None:
            raise ValueError("Uknown breakType: {0}".format(breakType))
        if str(dataSize) not in "1248":
            raise ValueError("Size must be 1,2,4, or 8")
        if isinstance(dbreg,(int,long)):
            dbreg = str(dbreg)

        ipc_breakpoint = py2ipc.IPC_Breakpoint()
        ipc_breakpoint.threadDeviceId = device
        ipc_breakpoint.enabled = enable
        
        # Execution breakpoints provide a full breakpoint address
        if breakType in [ "sw", "exe", "exe local", "exe global" ]:
            address = Address(address)
            memsrvc = py2ipc.IPC_GetService("Memory")
            if address.Type in [AddressType.physical, AddressType.linear]:
                ipc_breakpoint.fullBreakpointAddress = memsrvc.AddressFromString(str(address))
                ipc_breakpoint.breakpointAddress = ipc_breakpoint.fullBreakpointAddress.offset
            else:
                addrstr = str(address)
                ipc_addr = memsrvc.AddressFromString(addrstr)
                newaddr = memsrvc.ConvertAddress(device, ipc_addr, "Linear")
                ipc_breakpoint.fullBreakpointAddress = newaddr
                ipc_breakpoint.breakpointAddress = newaddr.offset
        else:
            ipc_breakpoint.breakpointAddress = cls._set_addr_convert(device,breakType,address)
            
        ipc_breakpoint.breakpointName = name
        # verify size
        ipc_breakpoint.dataSize = str(dataSize)
        ipc_breakpoint.breakpointType = btype
        ipc_breakpoint.debugRegister = dbreg

        # Create the breakpoint in IPC
        bpsrvc = py2ipc.IPC_GetService("Breakpoint")
        bpsrvc.CreateBreakpoint( ipc_breakpoint )
        # now we can create the CLI one and return it
        newbp = cls(ipc_breakpoint)
        return newbp
    
    def remove(self):
        """
        Removes this breakpoint from the known breakpoints
        """
        bpsrvc = py2ipc.IPC_GetService("Breakpoint")
        bpsrvc.RemoveBreakpoint( self._ipc_breakpoint )
        
    def _change(self):
        """
        make sure the IPC API breakpoint matches our local settings
        users should not need directly. Setting the attributes
        of the breakpoint object should call this
        """
        bpsrvc = py2ipc.IPC_GetService("BreakpointManagement")
        bpsrvc.ChangeBreakpoint( self._ipc_breakpoint  )
        
    def enable(self):
        # I'm assuming this also updates self._ipc_breakpoint.enable
        bpsrvc = py2ipc.IPC_GetService("BreakpointManagement")
        bpsrvc.EnableBreakpoint( self._ipc_breakpoint.breakpointId  )
        
    def disable(self):
        # I'm assuming this also updates self._ipc_breakpoint.enable
        bpsrvc = py2ipc.IPC_GetService("BreakpointManagement")
        bpsrvc.DisableBreakpoint( self._ipc_breakpoint.breakpointId  )
        
    def _update(self):
        """
        makes sure that this breakpoint -- using breakpointId -- reflects
        the latest values in the IPC API
        """
        bpsrvc = py2ipc.IPC_GetService("BreakpointManagement")
        self._ipc_breakpoint = bpsrvc.GetBreakpoint( self._ipc_breakpoint.breakpointId )
                
    def __setattr__(self,attr,value):
        if attr in dir(self):
            object.__setattr__(self,attr,value)
        else:
            raise AttributeError("Not a valid attribute: {0}".format(attr))

    # we use this during constructor...so "self" is not allowed here
    @staticmethod
    def _set_addr_convert(deviceid,breakType,address):
        """
        takes an address string or CLI address object and converts to what is
        needed to set it in the IPC_Breakpoint structure
        
        Args:
            breakType (str) : CLI breakpoint type that the address will need to be used for
            address (str/obj): Address string or object that will be used in the breakpoint

        Returns:
            offset - returns the offset to pass in to the IPC_Breakpoint structure
        """
        if not isinstance(address, (basestring,Address)):
            raise TypeError("Address must be a string or Address object")
        if isinstance(address,basestring): # convert to address object
            address = Address(address)
        # flag to detrmine how to translate (if translating)
        needslinear = True if breakType in _linear_types else False
        # handle easy cases first?
        
        # go through our "needslinear cases first
        if needslinear:
            # needs linear and we have physical, no way to go back to linear
            if address.Type in [AddressType.physical, AddressType.io]: # no way can we get linear from physical address
                raise ValueError("Linear Address is needed for this breaktype: {0}".format(breakType))
            # needs linear and we have some some of the characteristics of that, so lets convert:
            elif needslinear and \
                address.Type in [AddressType.twofieldvirtual,
                               AddressType.threefieldvirtual,
                               AddressType.threefieldvirtual,
                               AddressType.onefieldcodevirtual,
                               AddressType.onefielddatavirtual,
                               AddressType.onefieldstackvirtual
                               ]: # we should be able to convert
                addrstr = str(address)
                memsrvc = py2ipc.IPC_GetService("Memory")
                ipc_addr = memsrvc.AddressFromString(addrstr)
                newaddr = memsrvc.ConvertAddress( deviceid,
                                                    ipc_addr,
                                                    "Linear"
                                                    )
                offset = newaddr.offset
                # end of needslinear and is a virtual address

            # assuming implied = linear
            elif needslinear and address.Type in [AddressType.linear, AddressType.implied]:
                # no conversion needed...value is the same as newaddr
                offset = address.Offset
            else:
                raise ValueError("Not sure how to handle breakpoint type {0} and AddressType: {1}".format(breakType,
                                                                                                          AddressType.Strings[address.Type]))
        # needs physical
        else:
            # needs physical and we have
            if address.Type in [AddressType.physical, AddressType.io]: # needs physical, and we have physical address, perfect!
                offset = address.Offset
            # needs physical and we have a linear...we should be able to convert right:
            else:
                addrstr = str(address)
                memsrvc = py2ipc.IPC_GetService("Memory")
                ipc_addr = memsrvc.AddressFromString(addrstr)
                newaddr = memsrvc.ConvertAddress( deviceid,
                                                    ipc_addr,
                                                    "Linear"
                                                    )
                offset = newaddr.offset
        return offset

    @property
    def address(self):
        ##############
        self._update()
        ##############
        if self._ipc_breakpoint.fullBreakpointAddress.is_valid():
            ipc_address = self._ipc_breakpoint.fullBreakpointAddress
            # only because we know that the cli and the openipc strings are the same
            return Address(
                address_type=str(ipc_address.addressType).lower(),
                offset=ipc_address.offset,
                table=ipc_address.ldtsel,
                segment=ipc_address.segsel,
            )
        else:
            # Convert based on the debug breakpoint type
            needslinear = True if self.breakType in _linear_types else False
            if needslinear:
                return Address(self._ipc_breakpoint.breakpointAddress, AddressType.linear)
            else:
                return Address(self._ipc_breakpoint.breakpointAddress, AddressType.physical)

    @address.setter
    def address(self, value):
        # IPC only cares about offset portion of this address
        offset = self._set_addr_convert(self.deviceId, self.breakType,value)
        self._ipc_breakpoint.breakpointAddress = offset
        ##############
        self._change()
        ##############
        
    @property
    def dbreg(self):
        ##############
        self._update()
        ##############
        # no conversion to int
        if self._ipc_breakpoint.debugRegister=="Any":
            return "Any"
        # otherwise convert to integer
        else:
            return int(self._ipc_breakpoint.debugRegister)

    @dbreg.setter
    def dbreg(self,value):
        # make sure we are setting to string
        self._ipc_breakpoint.debugRegister = str(value)
        ##############
        self._change()
        ##############
        
    @property
    def dataSize(self):
        ##############
        self._update()
        ##############
        return int(self._ipc_breakpoint.dataSize)
    
    @dataSize.setter
    def dataSize(self,value):
        # IPC enum must be string
        self._ipc_breakpoint.dataSize= str(value)
        ##############
        self._change()
        ##############
        
    @property
    def breakType(self):
        """
        Sets the type of breakpoint. Changing the breakpoint type may require
        changing the address in order for the breakpoint to work
        """
        ##############
        self._update()
        ##############
        return _btype_mapping_reverse.get(self._ipc_breakpoint.breakpointType)
    
    @breakType.setter
    def breakType(self,value):
        newvalue = _btype_mapping[ value ]
        self._ipc_breakpoint.breakpointType = newvalue
        ##############
        self._change()
        ##############
        
    @property
    def enabled(self):
        ##############
        self._update()
        ##############
        return self._ipc_breakpoint.enabled

    @enabled.setter
    def enabled(self,value):
        self._ipc_breakpoint.enabled = value
        ##############
        self._change()
        ##############
    
    @property
    def deviceId(self):
        ##############
        self._update()
        ##############
        return self._ipc_breakpoint.threadDeviceId
    
    @deviceId.setter
    def deviceId(self,value):
        self._ipc_breakpoint.threadDeviceId=value
        ##############
        self._change()
        ##############
 
    @property
    def name(self):
        ##############
        self._update()
        ##############
        return self._ipc_breakpoint.breakpointName

    @name.setter
    def name(self,value):
        self._ipc_breakpoint.breakpointName = value
        ##############
        self._change()
        ##############

    @property
    def breakpointId(self):
        ##############
        self._update()
        ##############
        return self._ipc_breakpoint.breakpointId

    @breakpointId.setter
    def breakpointId(self,value):
        self._ipc_breakpoint.breakpointId = value
        ##############
        self._change()
        ##############
