###############################################################################
# Copyright 2014 2017 Intel Corporation.
# 
# The source code, information and material ("Material") contained herein is
# owned by Intel Corporation or its suppliers or licensors, and title to such
# Material remains with Intel Corporation or its suppliers or licensors. The
# Material contains proprietary information of Intel or its suppliers and
# licensors. The Material is protected by worldwide copyright laws and treaty
# provisions. No part of the Material may be used, copied, reproduced, modified,
# published, uploaded, posted, transmitted, distributed or disclosed in any way
# without Intel's prior express written permission. 
#
# No license under any patent, copyright or other intellectual property rights 
# in the Material is granted to or conferred upon you, either expressly, by 
# implication, inducement, estoppel or otherwise. Any license under such 
# intellectual property rights must be express and approved by Intel in writing.
#
# Unless otherwise agreed by Intel in writing, you may not remove or alter 
# this notice or any other notice embedded in Materials by Intel or Intel's 
# suppliers or licensors in any way.
###############################################################################


# TODO - change prints to log calls
from __future__ import absolute_import
# standard Python modules
import weakref
import traceback
import os.path
import threading
import re
import time
import types
import uuid
import sys
# ipccli shared modules
from .._py2to3 import *
from .. import base_env

from ..device import Device, DeviceList
from ..breakpoint import IaThreadBreak
from ..node import GetNodeClass, GetNodeClasses, NodeGroup
from ..state import State
from ..arch_regs import Regs
from .. import bitdata
from .. import dal_compatibility
# used to control some of the behavior in this module
from .. import settings
from .. import stdiolog
# if we plan to do version checking, then do it at this import
# after this import, other modules may import py2ipc as well
if "IPC_PATH" not in os.environ:
    os.environ['IPC_PATH'] = settings.IPC_PATH
import py2ipc  # @UnresolvedImport

from . import ipc_events
from . import ipc_init
from . import ipc_breakpoint
from . import ipc_deviceactions
from . import ipc_precondition
from . import ipc_stateport
from . import ipc_deviceconfig
from . import ipc_stalls
from . import ipc_breaks
from . import ipc_wizard
from ..shared_memory import SharedMemory as _SharedMemory

###############
# at this point, it should be safe to set operation class in bitdata file
###############
import py2ipc.Service_OperationReceipt
bitdata._setOperationClass( py2ipc.Service_OperationReceipt.Operation )

# ipc_env modules
from .ipc_commands import Commands, ReportResults
from .ipc_node_container import NodeContainer

from .. import cli_logging
_log = cli_logging.getLogger("ipccli.ipc")

from . import openipc_logging

from .. import diagnostics

class IpcDevice(Device):
    # This will need to get filled in by the baseaccess when base is created
    _base = None
    # maps the ipc_device types to a "nodetype" variable which
    # we will use for creating nodes
    _map_devtype_to_nodetype = dict(
                                    Unknown="unknown",
                                    LogicalRoot="root",
                                    TargetDomain="domain",
                                    LogicalThread="thread",
                                    LogicalCore="core",
                                    LogicalGroupCore="groupcore",
                                    LogicalUncore="uncore",
                                    LogicalChipset="chipset",
                                    LogicalBox="box",
                                    LogicalDie="die",
                                    LogicalPackage="package",
                                    LogicalModule="module",
                                    JTAGScanChain="scanchain",
                                    InterfacePort="interfaceport",
                                    I2CBus="i2cbus",
                                    Debugport="debugport",
                                    PinsInterface="pinsinterface",
                                    VirtualJTAGScanChain="virtual_scanchain",
                                    MemoryInterface="memoryinterface"
                                    # others ?
                                    )

    def __init__(self, ipc_device, parent_did=None, targetdomain_device=None, debugport_device=None, scanchain_device=None):
        """
        Converts an "ipc_device" in to a CLI device
        """
        Device.__init__(self, ipc_device)
        # there are multiple "devicetypes"...if there's not a devicetype (lowercase), we will
        # be displaying the ipc deviceType (uppercase T), this makes devicelist more meaningful
        # if DeviceType (uppercase D) is specified as a property it will override
        self.devicetype = ipc_device.deviceType
        # very similarly...we will use the deviceType (lower case d) to determine how to make
        # our nodes later on
        self.nodetype = self._map_devtype_to_nodetype.get(ipc_device.deviceType,"missing")
        # mapping of ipc properties to ITP compatible ones
        # if we simply convert to lower case that should get most of the important ones
        # properties = ["DeviceType","Stepping","Icode","IdcodeIr","IrLength",]
        properties = ipc_device.GetDevicePropertyNames()
        properties = [x.lower() for x in properties]
        if 'instanceid' not in properties:
            properties.append("instanceid")
        props_to_request = properties
        if "nodetype" in props_to_request: # skip node type if IPC API exposes it
            props_to_request.remove("nodetype")
        if self._base._version_check("OpenIPC",build=2668,error=False, revision=568842):
            values = ipc_device.GetDeviceProperties(props_to_request)
            for prop, value in zip(props_to_request, values):
                setattr(self, prop, value)
        else:
            for prop in props_to_request:
                setattr(self, prop, ipc_device.GetDeviceProperty(prop))
        self.properties = properties
        # some of the general ones we must manually copy over
        self.did       = ipc_device.deviceId
        self.alias     = ipc_device.identifier
        self.target_domain = targetdomain_device.did if targetdomain_device != None else None
        self.debugport = debugport_device.did if debugport_device != None else None
        self.scanchain = scanchain_device.did if scanchain_device != None else None
        self._parent_did = parent_did

        # Convert to Numbers
        convert = ['socketid', 'coreid','threadid','idcode','idcodeir',
                   'packageid','dieid','irlength','instanceid' ]

        for attr in properties:
            if hasattr(self,attr):
                val = getattr(self,attr)
                if val == None: 
                    continue
                elif isinstance(val,basestring) and val.strip() == "":
                    # convert empty string to None
                    setattr(self,attr,None)
                # see if it likely a boolean 'thing'
                elif val in ['true','True','false','False']:
                    if val.lower() == 'true':
                        setattr(self,attr,True)
                    else:
                        setattr(self,attr,False)
                elif attr in convert or (attr.startswith("0x")): # a known integer thing
                    try:
                        setattr(self,attr, long(val,0))
                    except:
                        print("!IPC API ERROR!")
                        print('device: {0}, did: {1}'.format(self.alias,self.did))
                        print("attr: {0}, val: '{1}'".format(attr,val))
                        setattr(self,attr, 0)
        # one special case where we rename for backwards compatibility
        if hasattr(self,"enabled"):
            self.isenabled = self.enabled
        # make sure we dont end up with deviceType = None
        if self.devicetype == None:
            # if deviceType was a property, but it was empty, then this can end up None again
            # put it back to defaulting to deviceType..or if that is known, then use Unknown
            self.devicetype = ipc_device.deviceType or "Unknown"
        
    @property
    def children(self):
        childlist = []
        child_ids = self._env_device.GetChildrenIds()
        if self.nodetype == 'groupcore' and callable(getattr(self._env_device, 'GetDeviceIdsInCoreGroup', None)):
            child_ids += self._env_device.GetDeviceIdsInCoreGroup('LogicalCore', True)
            child_ids += self._env_device.GetDeviceIdsInCoreGroup('LogicalThread', True)

        for child in child_ids:
            # convert to cli device
            try: 
                # there are some devices we purposely do not add to the cli's devicelist 
                childdevice = self._base.devicelist.findByDID( child.deviceId )
            # we may get a keyerror or value error if the child is a device that we dont
            # show in the ipccli
            except KeyError:
                continue
            except ValueError:
                continue
            childlist.append( childdevice )
        return DeviceList(childlist)
    
    @property
    def parent(self):        
        if not self._parent_did:
            return None
        # convert to cli device
        try: 
            # there are some devices we purposely do not add to the cli's devicelist 
            return self._base.devicelist.findByDID( self._parent_did )
        # not sure which exception is the BKM, so expect either of them
        except KeyError:
            return None
        except ValueError:
            return None

    def __getattr__(self, attr):
        value = self._env_device.GetDeviceProperty(attr)
        if value == "":
            raise AttributeError("Unknown attribute: %s"%attr)
        return value

_tapport = None
# default instance id is for parent of 0 with empty dictionary..
_instanceid = None
_probetype = None
_packagenums = None
_nonjtag_devices = None

def _devicelist_globals_reset():
    global _tapport
    global _instanceid
    global _probetype
    global _packagenums
    global _nonjtag_devices

    _tapport = 0
    # default instance id is for parent of 0 with empty dictionary..
    _instanceid = {0:{}}
    _probetype = "unknown"
    _packagenums = -1
    _nonjtag_devices = []

def create_list_of_devices(ipc_device, devicelist, parent_device=None, targetdomain_device=None, debugport_device=None, scanchain_device=None):
    global _tapport
    global _instanceid
    global _packagenums
    postdevice = None
    # in order for backwards compatibility, the debugport and scanchain need to be put 
    # at the end instead of the beginning
    # ---
    # do not add to devicelist if the deviceType is not present
    # this way we do not add Domain, DebugPort or Scan Chain
    if ipc_device.identifier=="" or ipc_device.deviceId<0x1000:
        devlist = []
        _log.debug("Not adding deviceId 0x{0:x} with identifier {1} to devicelist".format(ipc_device.deviceId,ipc_device.identifier))
        # do not return, we still need to walk through children
    else:
        device = IpcDevice(ipc_device, parent_device, targetdomain_device, debugport_device, scanchain_device)

        if device.coreid != None and device.threadid==None:
            if not device.devicetype.endswith("_C"):
                device.devicetype += "_C"
        
        # TODO: WORKAROUND (has to be applied to all devices for now)
        # default packageid
        device.packageid = None
        if device.nodetype == "uncore":
            _packagenums += 1
            device.packageid = _packagenums
        if _packagenums < 0:
            device.packageid = None
        else:
            has_uncore = ipc_device.GetAncestorIdForType("LogicalUncore") # should get exactly one here
            if has_uncore:
                device.packageid = _packagenums

        # build instance id relative to parent...so create new count
        # dictionary for this device
        _instanceid[device.did] = {}
    
        # keep up with instance count of each devicetype we find
        if device.parent != None:
            parentdid = device.parent.did
        else:
            parentdid = 0

        if device.devicetype not in _instanceid[parentdid]:
            _instanceid[parentdid][device.devicetype] = 0
                
        # not sure this is really accurate yet...
        if device.nodetype not in ['thread','core']:
            _tapport += 1
        device.tapport = _tapport            
        
        #device.instanceid = _instanceid[parentdid][device.devicetype]
        # WORKAROUND where we set instanceid ourselves for now...
        #if device.nodetype=="uncore":
        #    # this used to be relative to the package...we have no package construct anymore
        #    # we must hardcode this vs the parent thing since our parent is the jtag chain itself
        #    device.instanceid = 0
        #else:
        #    device.instanceid = _instanceid[parentdid][device.devicetype]
        # if we don't have a unique identifier, invent it using devicetype and instanceid
        if device.alias == "":
            device.alias = device.devicetype + "{0}".format(device.instanceid)
        # update our count for this devicetype
        _instanceid[parentdid][device.devicetype]+=1
        if device.nodetype == "scanchain":
            scanchain_device = device
        elif device.nodetype == "debugport":
            debugport_device = device
        
        # should be good at this point, add it to devicelist
        if device.nodetype in ['scanchain'] or device.scanchain == None:
            _nonjtag_devices.append (device)
        else:
            devicelist.addDevice(device)

    # can not use the device.children because they have not been created yet
    for child in ipc_device.GetChildrenIds():
        create_list_of_devices(child, devicelist, ipc_device.deviceId, targetdomain_device, debugport_device, scanchain_device)
       
    return


class baseaccess(base_env.baseaccess):
    SharedMemory = _SharedMemory

    _pollingvalues = None
    _remote = None
    # this is required by many parts of the ipccli
    # it gets filled in with actual device later
    _domain = None
    _root = None
    # cli devices versions of the py2ipc devices
    _domain_dev = None
    _root_dev = None
    _device_tree_lock = threading.Lock()
    _initializing = False
    _using_new_api = False
        
    def __init__(self, remote=None, dynamic_initialization=False):
        """
        project - unused currently
        remote - for specifying remote connection 
        dynamic_initialization - whether to opt-in to dynamic initialization
        """
        ###################################################################
        # Internal - Variables used by this class:
        # all variables should be declared here or at the class level
        # and should not be created in a function without
        # be declared here
        ###################################################################
        # temporary workaround for polling
        self._pollingvalues = {}
        self._remote = remote
        # call the rest of the init...it is in a seperate function
        # to facilitate reconnect
        if dynamic_initialization:
            self._initializing = True
            self.init = ipc_init.IpcInit(self)
        else:
            self._init()

    def _get_client_name(self):
        # check for namednodes
        if ('namednodes' in sys.modules):
            return "pythonsv-ipccli"
        # check for any actually imported svtools, have to look for more than one due to namespace
        elif len([k for k in sys.modules if k.startswith('svtools')]) > 1:
            return "pythonsv-ipccli"
        elif len([k for k in sys.modules if k.startswith('pysvtools')]) > 1:
            return "pythonsv-ipccli"
        else:
            return "ipccli"

    def _init(self):
        self._begin_initialization()
        self._finish_initialization()

    def _begin_initialization(self):
        # this can be checked for times when reconnect has been called on this object
        self.init_id = uuid.uuid4()
        # this can be used by other code to know
        # if an init has been done (for folks that need to do things once per connection)
        self._init_status = {}
        # we can only handle one domain for now. This is
        # filled in by _refresh_devicelist
        self._domain = 1  
        ################################
        # Call base class - which will create devicelist, cmds
        # but we will fill in c
        ################################ 
        base_env.baseaccess.__init__(self)
        
        ####################################
        # Provide a link to the log manager for users
        ###################################
        self.logger = cli_logging.getManager()
        self.diagnostic = diagnostics.DiagnosticsManager(self)
       
        client_name = self._get_client_name()
        self._using_new_api = False
        if py2ipc.IPC_GetConnectionStatus()=="NotConnected":
            if settings.IPC_LAUNCH_SERVER:
                port = 0 if settings.IPC_API_SERVER is None else settings.IPC_API_SERVER
                try:
                    port = py2ipc.IPC_LaunchMultitonServerInstance(port)
                    _log.result("Started new IPC API server at port %d"%port)
                except py2ipc.IPC_Error as err:
                    # assume server was already running if we get bind to port error
                    # and port was specified
                    if err.code != py2ipc.IPC_Error_Codes.Cannot_Bind_To_Port:
                        raise
                    _log.result("Connecting to existing IPC API server at port %d"%port)
                py2ipc.IPC_ConnectWithURI(client_name, "127.0.0.1:%s"%port)
                self._using_new_api = True

            elif settings.IPC_API_SERVER is not None:
                _log.result("Connecting to existing IPC API server at '{0}'".format(settings.IPC_API_SERVER))
                py2ipc.IPC_ConnectWithURI(client_name, settings.IPC_API_SERVER)
                self._using_new_api = True
            elif settings.OUT_OF_PROCESS:
                _log.result("Connecting to IPC API....")
                try:
                    py2ipc.IPC_ConnectSingleton(client_name, py2ipc.IPCTypes.IPC_ServerHostingTypes.Out_of_Process)
                    self._using_new_api = True
                except py2ipc.IPC_Error as err:
                    if (err.code == py2ipc.IPC_Error_Codes.Not_Supported) or (err.code == py2ipc.IPC_Error_Codes.Not_Implemented):
                        py2ipc.IPC_Connect()
                    else:
                        raise err
            else:
                # a very subtle difference so we can tell that we are in process...
                # since this really isn't an option for the user we do not want to explicitly state it
                _log.result("Connecting to IPC API ....")
                try:
                    py2ipc.IPC_ConnectSingleton(client_name, py2ipc.IPCTypes.IPC_ServerHostingTypes.In_Process_Standalone)
                    self._using_new_api = True
                except py2ipc.IPC_Error as err:
                    if (err.code == py2ipc.IPC_Error_Codes.Not_Supported) or (err.code == py2ipc.IPC_Error_Codes.Not_Implemented):
                        py2ipc.IPC_ConnectInSameProcess()
                    else:
                        raise err

            if settings.IPC_LOGGING_PRESET:
                gen = py2ipc.IPC_GetService("General")
                gen.LoadLoggingPreset(settings.IPC_LOGGING_PRESET)

        ####################################################
        # Save IPC version info, there are sometimes version
        # dependencies that we need to track
        ####################################################
        self._get_ipc_version_info()

        #########################################
        # If the backend is OpenIPC then decorate the logger with additional type-sepcific methods
        #########################################
        
        if self._version_check('OpenIPC',error=False):
            self.logger.openipc_presetnames = openipc_logging.LoggingPresetNames
            self.logger.openipc_loadpreset = openipc_logging.LoadLoggingPreset
            self.logger.openipc_writemessage = openipc_logging.WriteLogMessage
            self.logger.openipc_flush = openipc_logging.FlushLogs
            self.logger.openipc_clear = openipc_logging.ClearLogs
            self.logger.openipc_archive = openipc_logging.ArchiveLogs
            self.logger.openipc_echo = types.MethodType(openipc_logging._openipc_echo, self.logger)
            self.logger.openipc_getloadedpreset = openipc_logging.GetLoadedLoggingPreset
            

        #########################################
        # Create devicelist early since other objects need
        #########################################
        IpcDevice._base = weakref.proxy(self)
        self.devicelist = DeviceList()
        #########################################
        # Supported commands that require device
        #########################################
        self.cmds = Commands(self)
        ##################################################
        # Copy Global Commands
        # - some commands MUST be available off itp object
        ###################################################
        self._copy_global_commands()        
        #########################################
        # Create our global control variables (requires global commands)
        #########################################
        self.cv = dal_compatibility.IpcGlobalControlVariables(self)
        ####################
        # Event registeration needs to be after we have created our devicelist
        # Here we subscribe merely to message events
        ####################
        self.events = ipc_events.Events(self)
        self.events.ignore_message(not settings.EVENT_DISPLAY, warn=False)
        #####################################
        # Finish initializing the IPC server
        #####################################
        if self._using_new_api:
            _log.result("Initializing IPC API....")
            try:
                if self._version_check("OpenIPC", build=2695, error=False, revision=571758):
                    import ipccli
                    py2ipc.IPC_SetClientInfo("version", str(ipccli.__version__))
                    py2ipc.IPC_SetClientInfo("ipccli_svn_rev", str(ipccli.__revision__))
                    py2ipc.IPC_SetClientInfo("py2ipc_bindings", str(py2ipc.__version__))

                if settings.IPC_CONFIG_FILE is not None:
                    _log.info("Using configuration file: '{0}'".format(settings.IPC_CONFIG_FILE))
                    py2ipc.IPC_SelectConfigurationByFileName(settings.IPC_CONFIG_FILE)
                elif settings.IPC_CONFIG_NAME is not None:
                    _log.info("Using configuration name: '{0}'".format(settings.IPC_CONFIG_NAME))
                    py2ipc.IPC_SelectConfigurationByName(settings.IPC_CONFIG_NAME)
                if settings.IPC_CONFIG_PARAMS is not None:
                    # If string was provided then convert it to a dict
                    if isinstance(settings.IPC_CONFIG_PARAMS, str):
                        settings.IPC_CONFIG_PARAMS = dict((k.strip(), v) for k, v in (p.split("=") for p in settings.IPC_CONFIG_PARAMS.split(",")))
                    _log.info("Using configuration parameters: '{0}'".format(settings.IPC_CONFIG_PARAMS))
                    py2ipc.IPC_SetConfigurationParameters(settings.IPC_CONFIG_PARAMS)
                if settings.WIZARD:
                    ipc_wizard.run_wizard()
                    settings.WIZARD = False
                if settings.IPC_CONFIG_STALLS is not None:
                    break_service = py2ipc.IPC_GetService("Break")
                    print("Using configuration stalls: '{0}'".format(str(settings.IPC_CONFIG_STALLS)))
                    _log.info("Using configuration stalls: '{0}'".format(str(settings.IPC_CONFIG_STALLS)))
                    for (stall,value) in settings.IPC_CONFIG_STALLS.items():
                        break_service.SetSystemStall(1, stall.upper(), int(value))
            except Exception as finalize:
                try:
                    py2ipc.IPC_Disconnect()
                except Exception as disconnect:
                    msg = str(finalize) + "\n" + str(disconnect)
                    _log.error("Both finalize and disconnect failed: "+msg)
                    raise finalize
                # raise the finalize error we got
                raise finalize

    def _finish_initialization(self):
        self._initializing = False
        if self._using_new_api:
            try:
                py2ipc.IPC_FinishInitialization()
            except Exception as finalize:
                try:
                    py2ipc.IPC_Disconnect()
                except Exception as disconnect:
                    msg = str(finalize) + "\n" + str(disconnect)
                    _log.error("Both finalize and disconnect failed: "+msg)
                    raise finalize
                # raise the finalize error we got
                raise finalize

        ######################################################
        # Devicelist and Node build up will have to
        # be redone anytime devicelist changes
        # all of that is encompassed in the reconfig function
        ######################################################
        self._refresh_devicelist()
        ######################################
        # Subscribe to the rest of the events
        ######################################
        self.events.ignore_all(not settings.EVENT_DISPLAY, warn=False)
        self.events.ignore_stateport(True)
        # make the wait command global for backwards compatibility
        if hasattr(self.events,"wait"):
            self.wait = self.events.wait
        # remove the "init" attribute now that initialization is finished
        if hasattr(self, "init"):
            delattr(self, "init")

    def _refresh_devicelist(self):
        with self._device_tree_lock:
            #########################################
            # Populate Device list
            #########################################
            self._create_devicelist()
            #########################################
            # create container for referencing various devices
            #########################################
            self.devs = NodeContainer(self)                
            #########################################
            # Now Build our Nodes
            #########################################
            self._create_nodes()
            #########################################
            # Add device configs
            #########################################
            self._add_configs()

            # Add higher-level functionality only if initialization is finished
            if not self._initializing:
                ###########################################
                # Add other servies that exist at top level
                # and on the nodes
                ###########################################
                self._add_services()

                #########################################
                # Create state nodes 
                #########################################
                self._add_node_capabilities()

                #########################################
                # Add arch Registers if we have threads
                #########################################
                self._create_arch_regs()

    def _get_ipc_version_info(self):
        """Save IPC version information so that other
        parts of the CLI can use"""
        import ipccli # do not import at global to reduce chance of recursive imports
        self._ipc_api_header = py2ipc.IPC_GetAPIVersion()
        gen = py2ipc.IPC_GetService("General")
        self._ipc_implementation =  gen.GetImplementationIdentifier()
        self._ipc_version        =  gen.GetImplementationVersion()
        cli = "IPC-CLI: {0}".format(ipccli.__version__)
        ipc = "{0} : {1}.{2}.{3}.{4}".format(
                                         self._ipc_implementation,
                                         self._ipc_version.major,
                                         self._ipc_version.minor,
                                         self._ipc_version.build,
                                         self._ipc_version.classification,
                                         )
        _log.result("{0}, {1}".format(cli,ipc))
        if self._ipc_implementation.startswith("DAL"):
            _log.result("! WARNING ! this is not a POR / validation configuration.")
            _log.result("  Use only for experimentation, not for validation.")

        
        # Attempt to determine the source revision that the IPC implementation  
        # was built from. Currently only supported by OpenIPC.
        self._ipc_implementation_rev = 0
        match = re.match(r"(?P<name>.+?):(?P<branch>.+?) \(rev (?P<rev>\d+)\)", 
                         self._ipc_implementation)
        if match:
            self._ipc_implementation = match.group("name").lower()
            self._ipc_implementation_rev = int(match.group("rev"))
        
        self._version_check("py2ipc",build=44)
        # not sure where else to put something like this
        # for older ipc api's, the sw breakpoint wants physical and not linear
        if self._ipc_api_header[0].build < 10 and self._ipc_api_header[0].major==2:
            ipc_breakpoint._linear_types.remove('sw')

    def _version_check(self, checkWhat, major=None, minor=None, build=None, error=True, **kargs):
        """
        checkWhat : "OpenIPC", "DAL", "IPC", "py2ipc"
        major : this is used when checking against IPC
        minor : this is used when checking against IPC
        build : this is used when checking against OpenIPC or DAL, this is the
                minimum needed for this function to work
        error : this is used to determine if an exception is raised if the version
                check fails.
        
        ..................
        Keyword Arguments:
        ..................
        revision : When specified, the revision of the IPC implementation is attempted
                   to be checked if the version information check fails.
        """
        # version checking is off
        if not settings.VERSION_CHECK:
            return True
        if checkWhat == "OpenIPC":
            if not self._ipc_implementation.startswith("openipc"):
                return True
            # If major is 0 then this is a test build.
            #if self._ipc_version.major == 0:
            #    return True
            # must be OpenIPC or main branch, so lets check version, just check build though
            if self._ipc_version.build == 0 or \
               self._ipc_version.build >= (build or 0):
                return True
            if "revision" in kargs and \
               self._ipc_implementation_rev >= kargs["revision"]:
                return True
            if error:
                raise Exception("This function requires OpenIPC build >= {0}".format(build))
            return False
        elif checkWhat == "py2ipc":
            # yeah, legacy thing for py2ipc, minor is compared against [1] for py2ipc, not [2]
            if minor is not None and py2ipc.__version__.version[1] != 0 and  py2ipc.__version__.version[1] < minor:
                if error:
                    raise Exception("This function requires py2ipc build >= {0}".format(build))
                else:
                    return False
        else:
            # don't error to user, but nothing else is supported here yet
            pass
        # passed version check
        return True
    
    def _create_devicelist(self):
        _devicelist_globals_reset()
        self.devicelist.empty()
        dev_srvc = py2ipc.IPC_GetService("Device")
        domains = dev_srvc.GetTargetDomainDeviceIds()
        # for now, we only support one domain
        self._domain = domains[0]
        # dal does not support root
        if self._ipc_implementation != "DAL":
            self._root = dev_srvc.GetDevice(0)
        # cli devices
        self._domain_dev = IpcDevice(self._domain)
        if self._root is not None:
            self._root_dev = IpcDevice(self._root)
        # create the rest of the devices
        create_list_of_devices(self._domain, self.devicelist, targetdomain_device=self._domain_dev)
        # now add in the non-jtag devices
        for dev in _nonjtag_devices:
            self.devicelist.addDevice(dev)

    def _create_nodes(self):
        """Walk through our devicelist, and use nodetype to build nodes"""
        # We need to always have these apparently, and default them to empty
        self.threads = []
        self.uncores = []
        self.boxes = []
        self.chipsets = []
        self.cores = []
        self.coregroups = []
        self.debugports = []
        # add the domain node
        self.devs.add_node(self._domain_dev)
        self.domains = [ self._domain_dev.node ]

        if self._root_dev is not None:
            self.devs.add_node(self._root_dev)
        
        try: self.domains  = self.devs.group_by("^domain$",'nodetype')
        except:
            errormsg = traceback.format_exc()
            _log.debug("No domains found: {0}".format( errormsg ) )

        try: self.debugports  = self.devs.group_by("^debugport$",'nodetype')
        except:
            errormsg = traceback.format_exc()
            _log.debug("No Debugports found: {0}".format( errormsg ) )

        try: self.uncores  = self.devs.group_by("^uncore$",'nodetype')
        except:
            errormsg = traceback.format_exc()
            _log.debug("No Uncores found: {0}".format( errormsg ) )

        try: self.boxes    = self.devs.group_by("^box$",'nodetype')
        except:
            errormsg = traceback.format_exc()
            _log.debug("No Boxes found: {0}".format( errormsg ) )

        try: self.chipsets = self.devs.group_by("^chipset$",'nodetype')
        except:
            errormsg = traceback.format_exc()
            _log.debug("No Chipsets found: {0}".format( errormsg ) )
        
        try: self.threads  = self.devs.group_by(nodetype="^thread$", isenabled=True)
        except:
            errormsg = traceback.format_exc()
            _log.debug("No Threads found: {0}".format( errormsg ) )
        
        try: self.cores  = self.devs.group_by(nodetype="^core$", isenabled=True)
        except:
            errormsg = traceback.format_exc()
            _log.debug("No Cores found: {0}".format( errormsg ) )

        try: self.coregroups  = self.devs.group_by(nodetype="^groupcore$", isenabled=True)
        except:
            errormsg = traceback.format_exc()
            _log.debug("No Coregroups found: {0}".format( errormsg ) )

        # walk through list backwards removing non-GPC threads
        for t in self.threads[::-1]:
            if getattr(t.device,"coregroup","GPC") != "GPC":
                self.threads.remove(t)

        # walk through list backwards removing non-GPC cores
        for t in self.cores[::-1]:
            if getattr(t.device,"coregroup","GPC") != "GPC":
                self.cores.remove(t)
        
        # to help other parts of the code know where to add state nodes to
        self._nodegroups = nodegroups = {
                                         'uncores':self.uncores,
                                         'boxes':self.boxes,
                                         'chipsets':self.chipsets,
                                         'threads':self.threads,  
                                         'cores':self.cores,
                                         'coregroups':self.coregroups,
                                         }

    def _add_node_capabilities(self):
        ##############
        # create nodes
        ##############
        for node in self.devs:
            node.state = State(self, node.device.did)

        # for each thread we need to add in the the thread control variable
        # that will warn folks about backwards compatibility
        for thread in self.threads:
            thread.cv = dal_compatibility.ThreadControlVar(thread)

    def _add_services(self):
        #################
        # add breaks
        #################
        self._add_breaks()

        #################
        # add stalls
        #################
        self.stalls = ipc_stalls.IpcCliStalls(self._domain)
        for node in self.devs:
            if node.device.devicetype == "Debugport":
                node.stalls = ipc_stalls.IpcCliStalls(node.device.did)

        #################
        # add state ports
        #################
        try:
            self.stateport = ipc_stateport.BaseaccessStatePorts(self) 
        except py2ipc.IPC_Error as e:
            if e.code == py2ipc.IPC_Error_Codes.Unknown_Service:
                _log.debug("StatePort was an Unknown service")
            else:
                raise

        #################
        # add device actions
        #################
        try:
            self.device_action = ipc_deviceactions.BaseaccessDeviceActions(self)
        except py2ipc.IPC_Error as e:
            if e.code == py2ipc.IPC_Error_Codes.Unknown_Service:
                _log.debug("Device Control was an Unknown service")
            else:
                raise

        ####################
        # add preconditions
        ####################
        try:
            self.precondition = ipc_precondition.BaseaccessPreconditions(self)
        except py2ipc.IPC_Error as e:
            if e.code == py2ipc.IPC_Error_Codes.Unknown_Service:
                _log.debug("Device Control was an Unknown service")
            else:
                raise

    def _add_configs(self):
        try:
            self.device_config = ipc_deviceconfig.BaseaccessDeviceConfigDeprecated(self)
            self.config = ipc_deviceconfig.BaseaccessDeviceConfig(self)
        except py2ipc.IPC_Error as e:
            if e.code == py2ipc.IPC_Error_Codes.Unknown_Service:
                _log.debug("Device Control was an Unknown service")
            else:
                raise

    def _add_breaks(self):
        self.breaks = None
        # add BREAKS for all devices
        for node in self.devs:
            if node.device.nodetype == 'domain':
                node.breaks = ipc_breaks.IpcCliBreaks(node.device, False)
                if node.device.did == self._domain_dev.did:
                    self._domain_dev.breaks = node.breaks
            elif (node.device.nodetype == 'groupcore' and node.name == 'GPC'):
                node.breaks = ipc_breaks.IpcCliBreaks(node.device, False)
                if self._version_check("OpenIPC", build=2894, error=False, revision=586334):
                    self.breaks = node.breaks
            else:
                node.breaks = ipc_breaks.IpcCliBreaks(node.device, True)
        # If there is no GPC coregroup, let's re-route the breaks to the target domain?
        if self.breaks is None:
            self.breaks = self._domain_dev.breaks

    @property
    def device_actions(self):
        import warnings
        warnings.warn("\n!!! Please use self.device_action, the plural will be removed in the future",Warning)
        return self.device_action

    def _create_arch_regs(self):
        """create arch regs on our thread nodes"""
        for thread in self.devs.group_by(nodetype="thread"):
            thread.state.regs = Regs(self,thread.did)
                    
    def _copy_global_commands(self):
        """copies commands from cmds that must be available off the base object"""
        # if you udpate this be sure to update the document the ipc_device_commands.rst
        # with the fact that a new global command was added
        cmds = ['autoscandr',
                'block_transport_from_leaving',
                'device_locker',
                'idcode',
                'irdrscan',
                'irdrscanreplace',
                'irdrscanrmw',
                'irdrscanverify',
                'irscan',
                'rawirdrscan',
                'drscan',
                'i2cscan',
                'getpin',
                'holdhook',
                'hookpins',
                'hookstatus',
                'pulsehook',
                'resettap',
                'jtag_locker',
                'jtag_goto_state',
                'jtag_shift',
                'i2c_locker',
                'power_status',
                'pulsepwrgood',
                'resettarget',
                'perfreport',
                'setpin',
                'startremotedebugagent',
                'stopremotedebugagent',
                'obspins',
                'pulsepin',
                ]
        for cmd in cmds:
            func = getattr(self.cmds,cmd,None)
            if func is not None:
                setattr(self, cmd, func)
                
    ###############################################
    # Global / customer visible functions begin here
    ################################################

    def reconnect(self, delay=None, dynamic_initialization=False):
        """
        Disconnects from the IPC, waits for the specified number of seconds, 
        then re-connects and re-initializes the IPC-CLI.

        Args:
            delay (int) : the number of seconds to wait before reconnecting and reinitializing the IPC-CLI.
            dynamic_initialization (bool) : whether to opt-in to dynamic initialization.
        """
        # must be here to prevent circular imports
        from .ipc_commands import DeviceLocker
        if DeviceLocker.lock_present():
            raise RuntimeError("reconnect cannot be called if a lock is present")
        # pick a default depending on OpenIPC version
        if delay is None:
            if self._version_check("OpenIPC", build=1452, error=False, revision=512361):
                delay = 0
            else:
                delay = 10

        if py2ipc.IPC_GetConnectionStatus() != "NotConnected":
            if self._version_check("OpenIPC", build=1784, error=False, revision=512361):
                if py2ipc.IPC_DisconnectIfLastClient():
                    _log.result("Disconnect successful. The IPC API server has been shut down.")
                else:
                    row   = "{:>4}  |  {:>4}\n"
                    delimiter = "-----------------------\n"
                    formatted = row.format("Id", "Name")
                    formatted += delimiter

                    for client in self.clients():
                        formatted += row.format(str(client['Id']), client['name'])
                    raise RuntimeError("Aborting the reconnection process because other clients are still "
                                       "connected to the IPC API server.\nHere's a list of all the clients "
                                       "currently connected to this IPC API server (see the 'ipc.clients()' command):\n\n{0}".format(formatted))
            else:
                py2ipc.IPC_Disconnect()
                _log.result("Disconnect successful. The IPC API server may or may not have shut down.")

        # re do init
        time.sleep(delay)
        if dynamic_initialization:
            self._initializing = True
            self.init = self.init = ipc_init.IpcInit(self)
        else:
            self._init()

    def isconnected(self):
        """
        Returns whether there is an active connection to an IPC API server.

        P.S.: If you require further granularity regarding the connection
              status, you can call the IPC_GetConnectionStatus() function in
              the 'py2ipc' module
        """
        connection_status = py2ipc.IPC_GetConnectionStatus()
        return (connection_status != "NotConnected") and (connection_status != "ConnectionLost")

    def polling(self,debugPort=None, enable=None):
        """
        This function has no effect. It is only here for backwards 
        compatibility with previous generations of software
        """
        if debugPort==None:
            debugPort = 0
        if enable==None: # current value requested
            p = self._pollingvalues.get(debugPort,True)
            return p
        # we must be setting it then
        self._pollingvalues[debugPort]=enable
        return

    def halt(self):
        """
        Halts all the General Purpose Cores (GPC) in the devicelist.

        Notes:
        
            * If no GPC cores are present, then an error is thrown
            * If run control is not enabled, it will enable it

        """
        runsrvc = py2ipc.IPC_GetService("RunControl")

        # Remember if run control was disabled before the halt attempt
        gpc = getattr(self.devs,"gpc", None)
        if gpc is None:
            raise ValueError("There are no GPC cores available to halt")
        run_control_was_disabled = not gpc.isruncontrolenabled()

        operation = runsrvc.HaltAll()
        operation.Flush()

        # If run control was disabled and the halt attempt succeeded, then 
        # update the supported breaks on devices
        if run_control_was_disabled:
            self._add_breaks()


    def go(self):
        """
        Attempts to make sure that no general purpose cores are 'halted'
        """
        runsrvc = py2ipc.IPC_GetService("RunControl")
        operation = runsrvc.GoAll()
        operation.Flush()
        
    def tapresetidcodes(self):
        """
        Issues a tapreset for all jtag chains in the devicelist, and reports
        status of the chain. The following status values may be returned.

          GOOD - We found at least one idcode and no JTAG corruption.
          OPEN - The pattern we shifted into TDI was not found in TDO
          SHORT - The pattern we shifted into TDI was read from TDO with no additional bits.
          INVERTED - The pattern we shifted into TDI was read from TDO but all the bits were flipped.
          CORRUPTION - The pattern we shifted into TDI was read from TDO with some incorrect bits.

        For additional jtag continuity debugging, consider using ipc.diagnostic.run().
        
        """
        jtagcfg = py2ipc.IPC_GetService("JtagDiagnostic")
        devsrvc = py2ipc.IPC_GetService("Device")
        dev_control = py2ipc.IPC_GetService("DeviceControl")
        # for each scan chain in this domain
        chains = devsrvc.GetDescendantIdsForType(self._domain,"JTAGScanChain",True)
        for chain in chains:
            try:
                idcodes,status = jtagcfg.TapResetGetIdcodes(chain,100)
            except py2ipc.IPC_Error as err:
                idcodes = None
                status = str(err)
            _log.result("Chain Device Id- 0x{0:x}".format(chain))

            try:
                tclk_rate = dev_control.GetDeviceConfig(chain, "Jtag.TclkRate") + " Hz"
            except py2ipc.IPC_Error as err:
                tclk_rate = str(err)
            _log.result("  TCLK Rate: {}".format(tclk_rate))

            if idcodes is not None:
                for idcode in idcodes:
                    _log.result("  Idcode: 0x{0:x}".format(idcode))
            _log.result("  Status: {0}".format(status))
        
    def runstatusall(self):
        """
        Returns the runstatus of the current domain
        """
        runsrvc = py2ipc.IPC_GetService("RunControl")
        return runsrvc.GetAllRunStatus(self._domain)
       
    def forcereconfig(self,phases=["Default"],device=None):
        """
        Requests a topology update from the IPC API. When the topology update
        occurs we will refresh the entire devicelist and node structure.

        Args:
            phases (list) : (optional) list of phases to be done during update
            device (int)  : (optional) the did or alias of the target domain, debug
                            port, or JTAG scan chain device to scope the topology
                            update to; if not specified then the target domain is
                            targeted

        Phases can be specified as a list of phases to update. Options are:

             **Default** - Performs the update phases that were configured in the IPC API implementation for initial device tree construction during start-up.
             
             **DetectTaps** - The JTAG chain(s) are over scanned to detect what TAP network definition to use to populate the device tree under that chain.
             
             **AdjustTaps** - The devices populated for the JTAG chain(s) are reduced or otherwise adjusted based on the device under test (e.g. TAP of a fused core is removed, physical ordering of devices are determined, etc.)
             
        If the detect phase is not being used in the update then the TAP network(s) originally detected or specified for the JTAG chain is re-loaded in its full, unadjusted form before the adjust flow is invoked. 

        Example:
          >>> ipc.forcereconfig() # Perform the configured phases of topology update for entire device tree
          >>> ipc.forcereconfig(phases=["DetectTaps","AdjustTaps"]) # Perform the "DetectTaps" and "AdjustTaps" phases of topology update for entire device tree
          >>> ipc.forcereconfig(device=ipc.devs.jtagscanchain0) # Update topology only for the specified JTAG chain and its descendants

        Raises:
             ValueError: if an invalid phase is specified

        **IPC services required:** Device
        """
        # probably needs to call some IPC function as well
        if _log.isEnabledFor(cli_logging.DEBUG):
            stack = traceback.format_stack(limit=3)[-2].strip()  # last entry is THIS file...next to last entry should be our caller
            _log.debug("CALLER: {0}".format(stack))            
            _log.debug("ENTER: forcereconfig(phases={0:s})".format(str(phases)))

        for p in phases:
            if p not in ["Default","DetectTaps","AdjustTaps"]:
                raise ValueError("Unsupported phase: %s"%p)

        did = self._domain if device is None else self.cmds._device_convert(device)
            
        # forcereconfig not enabled for everyone yet
        if self._version_check("OpenIPC", build=383, error=False, revision=454380) and \
                        self.events.ignore_reconfiguration(None) == False:
            devservice = py2ipc.IPC_GetService("Device")
            devservice.UpdateTopology(did, phases)
        else:
            self._refresh_devicelist()
        
        if _log.isEnabledFor(cli_logging.DEBUG):
            _log.debug("EXIT: forcereconfig()")
            
    def version(self,):
        """Displays version information. This output format may change in the future."""
        results = {}
        import ipccli
        results["IPC-CLI"] = "{0:s}".format(str(ipccli.__version__))
        if ipccli.__version__ == "0.0.0":
            results["IPC-CLI, SVN Rev"] = "{0:s}".format(str(ipccli.__revision__))
        import py2ipc
        results["PY2IPC Bindings"] = "{0:s}".format(str(py2ipc.__version__))
        # IPC API has multiple pieces of version info:
        file_version, header_version = py2ipc.IPC_GetAPIVersion()
        results["IPC API Interface"] = "{0:s} ({1})".format(str(file_version), header_version)
        # get implementation version info:
        gen = py2ipc.IPC_GetService("General")
        results["IPC Implementation"]  = "{0:s} ({1:s})".format(
            str(gen.GetImplementationVersion()),
            str(gen.GetImplementationIdentifier()))
        return ReportResults(results)

    def log(self,filename,accesstype='w'):
        """Begin logging stdout to a file.

        Args:
            filename (str) : name of file to log to.
            accesstype (str) : 'w' - write , or 'a' - append.

        This will log the command and its output. In order for this to grab ALL loggers
        created from the builtin python logging module, the ipccli will need to be
        imported before the logging module.
        
        Note:
            If run under IPython it will grab stdout writes, but it cannot 
            grab the "out" that is displayed to the terminal.
        """
        return stdiolog.log(filename,accesstype)

    def nolog(self):
        """Turn off logging and close log file which was initiated with the log command."""
        return stdiolog.nolog()

    def isrunning(self):
        """
        Returns whether all the general purpose threads in the system are running.
        
        This uses the domain did to prompt the IPC for this information with a single call
        vs. using "ipc.threads.isrunning" which will show you all the threads running status.

        **IPC services required:** RunControl
        """ 
        runsrvc = py2ipc.IPC_GetService("RunControl")
        running = (runsrvc.GetAllRunStatus(self._domain)=="Running")
        return running

    def ishalted(self):
        """
        Returns whether all the general purpose threads in the system are halted.
        
        This uses the domain did to prompt the IPC for this information with a single call
        vs. using "ipc.threads.halted" which will show you all the threads halted status.
        
        **IPC services required:** RunControl
        """         
        runsrvc = py2ipc.IPC_GetService("RunControl")
        halted = (runsrvc.GetAllRunStatus(self._domain)=="Halted")
        return halted
    
    def help(self):
        """
        Opens a browser and points it to the latest help for the IPC-CLI. Be sure
        to check versions and release notes since this may be run from an older
        IPC-CLI and may display the newer help.
        """
        _log.result("Opening web browser to display documentation")
        import webbrowser
        import ipccli
        localpath = os.path.dirname(ipccli.__file__) + os.sep + "html/index.html"
        doc = "file://"+ localpath.replace("\\","/")
        _log.debug("Local documentation should be here: {0}".format(doc))
        if not os.path.exists(localpath):
            _log.result("Local documentation does not exist, bailing out")
            raise RuntimeError("Unable to find documentation at {0}".format(localpath))
        webbrowser.open_new_tab(doc)

    def search(self, find, find_what="all"):
        """
        Args:
            find: String to Search For, this is expected to be a python regular expression
            find_what: "all", "command" (includes stateport), "config", "action", "precondition"

            Note:
                "all" - currently searches "command", "config", "action", at some point new items may be added
                        where "all" may not search (like descriptions), and the find_what will need to be specified

        Returns:
            path to the "thing" that was found relative to the top level ipc object

        """

        if find_what not in ["all", "command", "config", "action", "precondition"]:
            raise ValueError("invalue for find_what: %s, must be one of: %s"%(find_what, _OPTIONS))

        results = []
        find_re = re.compile(find)
        for node in self.devs:
            # find simple commands
            if find_what in ["all", "command"]:
                for cmd in node._get_added_functions():
                    path_to_search = "devs.{name}.{command}".format(
                            name=node.name.lower(),
                            command=cmd,
                        )
                    if find_re.search(path_to_search):
                        results.append(path_to_search)
            # check stateport
            if find_what in ["all", "command"]:
                for cmd in node.stateport.stateport_names:
                    path_to_search = "devs.{name}.stateport.{command}".format(
                        name=node.name.lower(),
                        command=cmd,
                    )
                    if find_re.search(path_to_search):
                        results.append(path_to_search)
            # check config
            if find_what in ["all", "config"]:
                for cmd in node.config.names:
                    path_to_search = "devs.{name}.config.{command}".format(
                        name=node.name.lower(),
                        command=cmd,
                    )
                    if find_re.search(path_to_search):
                        results.append(path_to_search)
            # action names
            if find_what in ["all", "action"]:
                for cmd in node.device_action.action_names:
                    path_to_search = "devs.{name}.device_action.{command}".format(
                        name=node.name.lower(),
                        command=cmd,
                    )
                    if find_re.search(path_to_search):
                        results.append(path_to_search)
            # precondition names
            if find_what in ["all", "precondition"]:
                for cmd in node.precondition.precondition_names:
                    path_to_search = "devs.{name}.precondition.{command}".format(
                        name=node.name.lower(),
                        command=cmd,
                    )
                    if find_re.search(path_to_search):
                        results.append(path_to_search)
        results.sort()
        return results

    def clients(self):
        """
        Returns a list of all the clients that are connected to the IPC API server.
        """
        if self._version_check("OpenIPC", build=1784, error=False, revision=512361):
            import py2ipc
            gen = py2ipc.IPC_GetService("General")
            clients = gen.GetConnectedClients()
            return [ {"name" : c.clientName, "Id" : c.clientId } for c in clients ]

    ###########################
    # These are simple ALL cases of point to all the threads
    ###########################
    
    def msr(self,*args,**kwargs):
        """runs the msr command on all threads
        
        See Also:
            - :py:meth:`~ipccli.ipc_env.ipc_commands.Commands.msr` 
        """
        threads = getattr(self, "threads", None)
        if threads is None or len(threads) == 0:
            raise RuntimeError("No threads present")
        return threads.msr(*args,**kwargs)

    def status(self,*args,**kwargs):
        """runs the thread_status command on all threads.

        See Also:
            - :py:meth:`~ipccli.ipc_env.ipc_commands.Commands.thread_status`
        """
        threads = getattr(self, "threads", None)
        if threads is None or len(threads) == 0:
            raise RuntimeError("No threads present")
        with self.device_locker():
            return threads.thread_status(*args,**kwargs)
    
    ###########################
    # Breakpoint functions...I would prefer these be under
    # their own "br" object, but not sure if that will fly 
    ###########################
        
    def brnew(self,*args,**kwargs):
        #assert getattr(self,"threads",None) != None, "No threads present"
        #return self.threads.brnew
        return self.cmds.brnew(None,*args,**kwargs)
    brnew.__doc__ = Commands.brnew.__doc__  # @UndefinedVariable    

    def brget(self,breakpointId=None):
        return self.cmds.brget(None,breakpointId)
    brget.__doc__ = Commands.brget.__doc__  # @UndefinedVariable        
    
    def brenable(self,*breakpointIds):
        return self.cmds.brenable(None,*breakpointIds)
    brenable.__doc__ = Commands.brenable.__doc__  # @UndefinedVariable
        
    def brdisable(self,*breakpointIds):
        return self.cmds.brdisable(None,*breakpointIds)
    brdisable.__doc__ = Commands.brdisable.__doc__  # @UndefinedVariable
        
    def brremove(self,*breakpointIds):
        return self.cmds.brremove(None,*breakpointIds)     
    brremove.__doc__ = Commands.brremove.__doc__   # @UndefinedVariable
        
    def brchange(self,breakpointId,**kwargs):
        return self.cmds.brchange(None,breakpointId,**kwargs)
    brchange.__doc__ = Commands.brchange.__doc__  # @UndefinedVariable
             
    def setplatformtype(self, platformtype): 
        """
        
        Sets the platform type for all debug ports.

        Args:
            
            platformtype (str): The platform type.

        Examples:

            >>> ipc.setplatformtype("Generic")
        """
        for debugport in self.debugports:
           debugport.setplatformtype(platformtype) 

    def getplatformtype(self):
        """
        
        Gets the platform type of all debug ports.

        Examples:

            >>> ipc.getplatformtype()
            DebugPort0.getplatformtype - "Generic"
        """
        return self.debugports.getplatformtype()

    def getsupportedplatformtypes(self):
        """
        
        Gets the supported platform types of all debug ports.

        Examples:

            >>> ipc.getsupportedplatformtypes()
            DebugPort0.getsupportedplatformtypes - ["Generic"]
        """
        return self.debugports.getsupportedplatformtypes()
