###############################################################################
# Copyright 2014 2017 Intel Corporation.
#
# The source code, information and material ("Material") contained herein is
# owned by Intel Corporation or its suppliers or licensors, and title to such
# Material remains with Intel Corporation or its suppliers or licensors. The
# Material contains proprietary information of Intel or its suppliers and
# licensors. The Material is protected by worldwide copyright laws and treaty
# provisions. No part of the Material may be used, copied, reproduced, modified,
# published, uploaded, posted, transmitted, distributed or disclosed in any way
# without Intel's prior express written permission.
#
# No license under any patent, copyright or other intellectual property rights
# in the Material is granted to or conferred upon you, either expressly, by
# implication, inducement, estoppel or otherwise. Any license under such
#
# intellectual property rights must be express and approved by Intel in writing.
# Unless otherwise agreed by Intel in writing, you may not remove or alter
# this notice or any other notice embedded in Materials by Intel or Intel's
# suppliers or licensors in any way.
###############################################################################

from __future__ import absolute_import

from . import cli_logging
from ._py2to3 import *
from . import settings
import re

_log = cli_logging.getLogger("device")

class Device(object):
    alias = ""
    did = None            
    debugport = None
    isenabled = False
    devicetype = ""
    devicesubtype = None  # would love to default to '' but needs to be "" for backwards compatibility
    stepping = ""
    scanchain = None
    irlength = None
    idcode = None
    dieid = None
    tapport = 0 # TODO: need to know whether to remove
    bustype = ""
    serialnum = None
    coreid = None
    threadid = None
    packageid = None
    instanceid = None
    parent = None
    # careful, dervitives that fill this in need to set this list
    # per object (or provide override via property)
    children = [] 
    # We will need a weak reference back to devicelist so that
    # a device can find children and ancestors
    _devicelist = None 
    # TODO: determine if we still need
    #device = ""
    #board = ""
    # parentid = None
    #childrenids = []

    
    def __init__(self,environ_device=None):
        """
        :param environ_device: used to set "_env_device" in case an environment needs to map
                from something in this devicelist back to some special collection of data for 
                its environment
        """
        self._env_device = environ_device


    def __repr__(self):
        # name variables how you want them displayed
        coreid    = "{0:2d}".format(self.coreid)       if self.coreid !=  None    else "" 
        threadid  = "{0:2d}".format(self.threadid)     if self.threadid != None   else ""
        packageid = "{0:2d}".format(self.packageid)    if self.packageid != None  else ""
        dieid     = "{0:2d}".format(self.dieid)        if self.dieid != None      else ""
        instanceid = "{0:2d}".format(self.instanceid)  if self.instanceid != None else ""
        debugport = "0x{0:08x}".format(self.debugport) if self.debugport != None  else "None"
        scanchain = "0x{0:08x}".format(self.scanchain) if self.scanchain != None  else "None"
        idcode    = "0x{0:08x}".format(self.idcode)    if self.idcode != None     else "-"*10
        # because it is in first column it does not need the 2 spacing, but if it gets moved, change this... 
        irlength  = "{0:d}".format(self.irlength)     if self.irlength != None   else "--" 
        
        # no if statements here, but still put formatting up here as a general rule...
        did       = "0x{0:08x}".format(self.did)
        
        
        # name variables how you want them displayed
        col1 = [ ("alias",self.alias),
                 ("devicetype",self.devicetype),
                 ("stepping",self.stepping),
                 ("did",did),
                 ("idcode",idcode),
                 ("irlength",irlength),                 
                 ("isenabled",str(self.isenabled)),
                 ("bustype",self.bustype),  
                ]
        col2 = [
                (),
                ("devicesubtype", '' if self.devicesubtype is None else self.devicesubtype),
                ("packageid",packageid),
                ("dieid",dieid),
                ("coreid",coreid),
                ("threadid",threadid),
                ("instanceid",instanceid),
                ]
        rows = []
        maxrows = 0
        maxrows = max(maxrows,len(col1))
        maxrows = max(maxrows,len(col2))
        # it somewhat of an ugly bit of code here, but hopefully you only have to touch the list above to 
        # add something
        for r in range(maxrows):
            row = []
            # if this columnn has that many rows...and this row is not empty
            if r < len(col1) and col1[r]!=():
                key,val = col1[r]
                equal = "="
            else:
                key = ""
                val = ""
                equal = " "
            row.append("{0:13s} {1} {2:<20s}".format(key,equal,val))
            if r < len(col2) and col2[r]!=():
                key,val = col2[r]
                equal = "="
            else:
                key = ""
                val = ""
                equal = " "
            row.append("{0:10s} {1} {2:<11s}".format(key,equal,val))

            rows.append(" ".join(row))
        output = "\n".join(rows)
        return output

    @property
    def parents(self):
        parentlist = []
        parent = self
        while parent.parent != None:
            parentlist.append( parent.parent )
            parent = parent.parent
        # put oldest parent first
        parentlist.reverse()
        return DeviceList(parentlist)
    
    def search_children(self,**kwargs):
        """Search for children and return a list of devices that match the specified criteria.

        Args:
            kwargs : key=value of what to search for. If value is a string, then it is treated like a regular expression.

            ignore_case : optional kwarg to specify ignoring case (defaults to True).

        Returns:
            a devicelist object.

        Raises:
            ValueError: if kwargs is empty
            ValueError: if more than 1 kwarg is passed in (not counting ignore_case)
        """
        ignore_case = kwargs.pop("ignore_case", True)
        if len(kwargs)>1:
            raise ValueError("Can only search by one kwarg currently one kwarg currently supported") 
        elif len(kwargs)==0:
            raise ValueError("Must specify what to search for")
        newdevicelist = DeviceList()
        find_key,find_value = kwargs.popitem()
        if isinstance(find_value,basestring): # if it is a string, assume it is a regular expression
            if ignore_case: regex = re.compile(find_value, re.I)
            else:           regex = re.compile(find_value)
        else:
            regex = None
        for child in self.children:
            if not hasattr(child,find_key):
                continue
            value = getattr(child,find_key)
            if regex is None:
                if value == find_value:
                    newdevicelist.addDevice(child)
            else:
                if regex.match(value): # if we found match, add to list
                    newdevicelist.addDevice(child)
            # now...search through children too, and add search results
            kwargs['ignore_case'] = ignore_case
            tempdevlist = child.search_children(**{find_key:find_value})
            for t in tempdevlist:
                newdevicelist.addDevice( t )
        return newdevicelist

    def search_parents(self,**kwargs):
        """Search parents and return a list of devices that match the specified criteria.

        Args:
            kwargs : key=value of what to search for. If value is a string, then it is treated like a regular expression.

            ignore_case : optional kwarg to specify ignoring case (defaults to True).

        Returns:
            a devicelist object.

        Raises:
            ValueError: if kwargs is empty
            ValueError: if more than 1 kwarg is passed in (not counting ignore_case)
        """
        ignore_case = kwargs.pop("ignore_case", True)
        if len(kwargs)>1:
            raise ValueError("Can only search by one kwarg currently one kwarg currently supported") 
        elif len(kwargs)==0:
            raise ValueError("Must specify what to search for")
        newdevicelist = DeviceList()
        find_key,find_value = kwargs.popitem()
        if isinstance(find_value,basestring): # if it is a string, assume it is a regular expression
            if ignore_case: regex = re.compile(find_value, re.I)
            else:           regex = re.compile(find_value)
        else:
            regex = None
        devlist = []
        parent = self.parent
        while parent != None:
            if not hasattr(parent,find_key):
                continue
            value = getattr(parent,find_key)
            if regex is None:
                if value == find_value:
                    devlist.append(parent)
            else:
                if regex.match(value): # if we found match, add to list
                    devlist.append(parent)
            # walk up to parent
            parent = parent.parent
        # put the greatest parent first...
        devlist.reverse()
        return DeviceList(devlist)

        #
        #output = ""
        #output += "alias      = %-15s  did      = %3d  debugport = 0x%08x  isenabled = %-3s\n"%(self.alias, self.did, self.debugport, self.isenabled)
        #output += "devicetype = %-15s  stepping = %-3s  scanchain = 0x%08x  irlength  = %2d\n"%(self.devicetype, self.stepping, self.scanchain, self.irlength)
        #output += "idcode     = %-15s  dieid    = %3s  tapport   = %2d  bustype   = %s\n"%(self.idcode, dieid, self.tapport, self.bustype)
        # output += "serialnum  = %-15s  coreid   = %3s  threadid  = %2s  packageid = %2s\n"%(self.serialnum, cid, tid, pid)
        # return output  
        #'''


############################
# Proudly stolen from itpii
############################
class DeviceList(object):
    #: used to control which devices show up in the display
    devicetypefilter = None
    #: used to provide a summary of the coregroups encountered
    coregroups = None

    def __init__(self, devices=None,**kwargs):
        """Constructor for the DeviceList class.

        Args::
          devices (list) : A list of Device objects or a DeviceList object to 
                     clone (copying from a DeviceList copies the references to the 
                     child nodes but does not clone the child nodes).
          noindex (bool): Optional keyword argument to supress displaying the index
        """

        self._diddict = {}
        self._devices = []
        self.coregroups = set()
        self.devicetypefilter = settings.DEVICELIST_DEVICETYPE_FILTER
        self._noindex = kwargs.pop('noindex',False)
        if len(kwargs) > 0:
            raise ValueError("Unknown kwarg specified: {0:s}".format(kwargs))
        if devices != None and isinstance(devices, DeviceList):
            import copy
            self._devices = copy.copy(devices._devices)
        elif devices != None:
            for d in devices:
                self.addDevice(d)

    def _buildDidLookup(self):
        # used to keep a lookup table so that findByDID is fast
        self._diddict = {}
        for dev in self._devices:
            self._diddict[dev.did]=dev
            
    def addDevice(self,device):
        if not isinstance(device,Device):
            raise TypeError("Device object must be an instance of 'Device'")
        self._devices.append(device)
        self._diddict[device.did] = device
        if device.nodetype in ["core", "thread"]:
            core_group = getattr(device, "coregroup", None)
            if core_group not in [None, ""]:
                self.coregroups.add(core_group)
        # maybe re-sort based on deviceid ??
        
    def removeDevice(self,device):
        # make sure we have did
        did = device.did  if isinstance(device,Device) else device
        for dev in self._devices:
            if did == dev.did:
                self._devices.remove(dev)
                del self._diddict[did]
                break
        else: # yes, else the for loop, ie. break was not called
            raise Exception("device not found")

    def __len__(self):
        return len(self._devices)

    def __getiter__(self):
        return iter(self._devices)

    def __getitem__(self, index):
        if self._devices == None:
            raise IndexError("No devices in list.")
        if isinstance(index, basestring):
            index = index.upper()
            if self._devices != None:
                for d in self._devices:
                    if d.alias.upper() == index:
                        return d
            raise IndexError("No device found with alias '%s'."%index)
        deviceIndex = int(index)
        if deviceIndex not in range(-len(self._devices)+1, len(self._devices)):
            raise IndexError("Index %s is out of range: 0 - %s"%(index, len(self._devices)))
        return self._devices[deviceIndex]

    def get(self, index, default=None):
        """Get the item from the devicelist,
        but return default  if it is not present
        """
        try:
            return self[index]
        except IndexError:
            return default

    def __getslice__(self,start,end):
        return DeviceList(self._devices[start:end])

    def findByDID(self, did):
        """
        Return the device object matching this did.
        
        Arguments:
            did : the id of the device to return.

        Returns:
          The device instance, if found.

        Raises:
            ValueError: if no matching did is found.
        """
        ret = self._diddict.get(did, None)
        if ret==None:
            raise ValueError("Unknown did: {0}".format(hex(did)))
        return self._diddict[did]

    def _getMaxWidthForAliasAndType(self):
        """Compute the maximum character width for aliases (names) and types
        in the device list and return the sizes.

        Arguments:
          None.

        Returns:
          Tuple containing the maximum Alias width and the maximum Type width:
            (maxAliasWidth, maxTypeWidth)
        """

        maxAliasWidth = 0
        maxTypeWidth = 0
        for d in self._devices:
            if len(d.alias) > maxAliasWidth:
                maxAliasWidth = len(d.alias)
            if len(d.devicetype) > maxTypeWidth:
                maxTypeWidth = len(d.devicetype)
        return (maxAliasWidth, maxTypeWidth)
    
    def __contains__(self,index):
        if isinstance(index,basestring):
            for d in self._devices:
                if d.alias == index:
                    return True
            else:
                return False
        else: # not a string
            return index in self._devices

    def __repr__(self):
        tpTag = "TP"
        aliasTag = "Alias"
        typeTag = "Type"
        maxAliasWidth, maxTypeWidth = self._getMaxWidthForAliasAndType()
        if len(aliasTag) > maxAliasWidth:
            maxAliasWidth = len(aliasTag)
        if len(typeTag) > maxTypeWidth:
            maxTypeWidth = len(typeTag)
        maxTPWidth = 2 if len(self._devices) < 100 else 3
        
        if self._noindex:
            output =  "     DID         %-*s  %-*s  Step Idcode      P/D/ C/T  Enabled\n"%(maxAliasWidth, aliasTag, maxTypeWidth, typeTag)
        else:
            output =  "Indx DID         %-*s  %-*s  Step Idcode      P/D/ C/T  Enabled\n"%(maxAliasWidth, aliasTag, maxTypeWidth, typeTag)
        output += "----------------%s--%s---------------------------------------------\n"%('-'*maxAliasWidth, '-'*maxTypeWidth)
        for d,dev in enumerate(self._devices):
            if dev.devicetype not in self.devicetypefilter:
                if self._noindex: output += "     " # 5 spaces to match below
                else:             output += "%-5d"%d
                output += "0x%08X  "%dev.did
                #output += "%-2d "%d.debugport
                #output += "%*d  "%(maxTPWidth, d.tapport)
                #output += "%-2d "%d.scanchain
                output += "%-*s  "%(maxAliasWidth, dev.alias)
                output += "%-*s  "%(maxTypeWidth, dev.devicetype)
                output += "%-4s "%dev.stepping
                if    dev.idcode==None: output+=(" "*11)
                else: output += "0x%08X "%dev.idcode
                output += " " # some space between idcode and packagid
                if dev.packageid == None: output += " -/"
                else: output += "%2d/"%dev.packageid # 
                if dev.dieid == None: output+= "-/"
                else: output += "%d/"%dev.dieid
                if dev.coreid == None: output += " -"
                else: output += "%2d"%dev.coreid
                if dev.threadid == None: output += "/-  "
                else: output += "/%d  "%dev.threadid
                if dev.isenabled: output += "%-3s"%"Yes"
                else: output += "%-3s"%"No"
                output += "\n"
        return output
                
    def empty(self):
        """
        Removes all of our existing devices
        """
        self._devices = []
        self.coregroups.clear()


    def clone(self):
        """Create a clone of this container.  Does not perform a deep copy
        of all child nodes and only creates references to the child nodes.

        Arguments:
          None.

        Returns:
          A new DeviceList object based on this DeviceList object.
        """

        return DeviceList(self)
    
    def search(self,values=None,bywhat=None,**kwargs):
        """
        
        Searches through all the known devices and looks for
        an attribute specified in the 'bywhat' variable. Returns
        a DeviceList with a list of all the devices that
        matched the search criteria.
        
        Args:
            values : this can be either a list of values 
                    or a regular expression that we will match each devices 'bywhat' against.
            bywhat : any device attribute to match against.
            kwargs : bywhat can be express via kwargs like so: devicetype=value;
                     only one kwarg is currently supported.
            ignore_case : (optional) special kwarg that will be looked for
                          (default=True).
            
        Returns:
            A devicelist object.

         
        Example:
        
            >>> # Search for aliases that match exactly with what is in the specified list
            >>> ipc.devicelist.search( ["SLM_C0_T0","SLM_C1_T0"] )
            
            >>> # Search for aliases that match the given regular expression
            >>> ipc.devicelist.search( "SLM_C._T." )
            
            >>> # search for all device types that match the given regular expression
            >>> ipc.devicelist.search( "SLM", "devicetype" )
            >>> ipc.devicelist.search( devicetype = "SLM" )

        Raises:
            ValueError: if values is None and kwargs is empty
            ValueError: if more than 1 kwarg is passed in (not including ignore_case)
            ValueError: if no devices with the given attributes were found
        """
        ignore_case = kwargs.pop("ignore_case", True)
        # doing this way in case one day I add kwargs support, and I can tell if user
        # specifically set the bywhat or not
        find_what = {}
        if len(kwargs)==0:
            if bywhat == None: 
                bywhat = "alias"
            if values==None:
                raise ValueError("Must specify values to search for")
            find_what[bywhat] = values
        # add to dictionary things we should find
        find_what.update(kwargs)
        # special error to give when the bywhat was not found
        attrfound = False
        items = list(find_what.items())
        this_loop, next_loop = items[0], items[1:]
        bywhat, values = this_loop
        newdevicelist = []
        if isinstance(values, list):
            if ignore_case:
                # convert to lower if ignore case was given
                values = [v.lower() if isinstance(v,basestring) else v
                          for v in values]
            # make sure aliases are in upper to match the device alias
            for device in self._devices:
                # make sure device has that attribute
                if not hasattr(device, bywhat):
                    continue
                attrfound = True
                if ignore_case:
                    test_value = getattr(device,bywhat)
                    if isinstance(test_value,basestring):
                        test_value = test_value.lower()
                    # now we can finally test
                    if test_value in values:
                        newdevicelist.append(device)
                else: # ignore case not set, direct match
                    if getattr(device,bywhat) in values:
                        newdevicelist.append(device)
        elif isinstance(values, string_types): 
            if ignore_case: regex = re.compile(values, re.I)
            else:           regex = re.compile(values)
            for device in self._devices:
                # make sure device has that attribute
                if not hasattr(device,bywhat):
                    continue
                attrfound = True
                # if we got a match add to our nodes
                searchval = getattr(device,bywhat)
                if regex.search(searchval):
                    newdevicelist.append( device )
        # assume some type of number or want an equal comparison
        else:
            for device in self._devices:
                # make sure device has that attribute
                if not hasattr(device,bywhat):
                    continue
                attrfound = True
                # if we got a match add to our nodes
                searchval = getattr(device,bywhat)
                if values == searchval:
                    newdevicelist.append( device )

        if not attrfound:
            raise ValueError("no devices found with attribute '{0}'".format(bywhat))
        next_list = DeviceList(newdevicelist,noindex=True)
        if len(next_loop):
            return next_list.search(**dict(next_loop))
        else:
            return next_list
