###############################################################################
# Copyright 2014 2017 Intel Corporation.
#
# The source code, information and material ("Material") contained herein is
# owned by Intel Corporation or its suppliers or licensors, and title to such
# Material remains with Intel Corporation or its suppliers or licensors. The
# Material contains proprietary information of Intel or its suppliers and
# licensors. The Material is protected by worldwide copyright laws and treaty
# provisions. No part of the Material may be used, copied, reproduced, modified,
# published, uploaded, posted, transmitted, distributed or disclosed in any way
# without Intel's prior express written permission.
#
# No license under any patent, copyright or other intellectual property rights
# in the Material is granted to or conferred upon you, either expressly, by
# implication, inducement, estoppel or otherwise. Any license under such
#
# intellectual property rights must be express and approved by Intel in writing.
# Unless otherwise agreed by Intel in writing, you may not remove or alter
# this notice or any other notice embedded in Materials by Intel or Intel's
# suppliers or licensors in any way.
###############################################################################


##############################################################################
# Theoretically the ipc_environment gets to define the "interface 
# specification" and this code would work any environment adhering to that
#
##############################################################################
import weakref
import traceback
from . import _version
from . import cli_logging
_log = cli_logging.getLogger("ipc")


class OPTIONS:
    _warn_always = False


class ThreadControlVar(object):
    # to track warnings we have already given
    # do this at the class level instead of per thread/instance
    def __init__(self, node):
        self._node = weakref.proxy(node)

    @property
    def isrunning(self):
        return self._node.isrunning()

    @property
    def ishalted(self):
        return self._node.ishalted()

    def _getbreak(self, breakcondition):
        if not hasattr(self._node, "breaks"):
            raise ValueError("this node {0} does not have breaks".format(self._node.device.alias))
        breakval = getattr(self._node.breaks, breakcondition, None)
        if breakval is None:
            raise ValueError("this node {0} does not have this break: {1}".format(
                self._node.device.alias, breakcondition))
        _log.caller()
        _log.debug("ENTER/EXIT: controlvar: 'cv.{0}break' for thread '{1}' has value: {2}".format(
            breakcondition, self._node.device.alias, breakval))
        return breakval
    
    def _setbreak(self, breakcondition, newval):
        if not hasattr(self._node, "breaks"):
            raise ValueError("this node {0} does not have breaks".format(self._node.device.alias))
        breakval = getattr(self._node.breaks, breakcondition, None)
        if breakval is None:
            raise ValueError("this node {0} does not have this break {1}".format(
                self._node.device.alias, breakcondition))
        _log.caller()
        _log.debug("ENTER/EXIT: controlvar: 'cv.{0}break' for thread '{1}' getting set to value: {2}".format(
            breakcondition, self._node.device.alias, newval))
        setattr(self._node.breaks, breakcondition, newval)

    initbreak = property(lambda self: self._getbreak("init"),
                         lambda self, value: self._setbreak("init", value))
    machinecheckbreak = property(lambda self: self._getbreak("machinecheck"),
                                 lambda self, value: self._setbreak("machinecheck", value))
    smmentrybreak = property(lambda self: self._getbreak("smmentry"),
                             lambda self, value: self._setbreak("smmentry", value))
    smmexitbreak = property(lambda self: self._getbreak("smmexit"),
                            lambda self, value: self._setbreak("smmexit", value))
    smmintobreak = property(lambda self: self._getbreak("smminto"),
                            lambda self, value: self._setbreak("smminto", value))
    vmentrybreak = property(lambda self: self._getbreak("vmentry"),
                            lambda self, value: self._setbreak("vmentry", value))
    vmexitbreak = property(lambda self: self._getbreak("vmexit"),
                           lambda self, value: self._setbreak("vmexit", value))
            
    _asm_map_to_ipc = {
               'use16': '16Bit',
               'use32': '32Bit',
               'use64': '64Bit',
                }
    # reverse lookup
    _asm_map_from_ipc = ivd = dict((v, k) for k, v in _asm_map_to_ipc.items())
    
    @property
    def asmmode(self):
        return self._asm_map_from_ipc.get(self._node.asmmode())

    @asmmode.setter
    def asmmode(self, value):
        if value not in self._asm_map_to_ipc.keys():
            raise ValueError("Invalid value for asmmode {0}".format(value))
        self._node.asmmode(self._asm_map_to_ipc[value])


###############################################
# for holding all our global control variables
################################################
################################################
#
# This admittedly VIOLATES the IPC CLI architecture.
# hopefully this is temporary until I can create a jtag.lock/unlock, etc...
#
################################################
class IpcGlobalControlVariables(object):
    def __init__(self, base):
        # needed to know for what devices we will lock or check operations against
        # domain must be an integer, an py2ipc.IPC_Device or a py2ipc.Device
        self._base = weakref.proxy(base)
        self._locker = self._base.device_locker(queue=False)
        self._domain = base._domain
        # needed to make sure we don't set Lock too many times when using the old
        # control variable mechanism
        self._manualscans_lock = False 

    @property
    def version(self):
        return "IPC-CLI Version %s" % str(_version.__version__)
    
    @property
    def manualscans(self):
        return self._manualscans_lock

    @manualscans.setter
    def manualscans(self, value):
        # if both are "True" or both are "False"
        if (self._manualscans_lock and value) or \
           (not self._manualscans_lock and not value):
            return
        # make sure we are storing True/False
        self._manualscans_lock = (value == 1)
        if self._manualscans_lock:
            self._locker.__enter__()
        else:
            self._locker.__exit__(None, None, None)

    # ###############################
    # some legacy commands
    # do we really have to keep these?
    # ##############################
    @property
    def isrunning(self):
        return self._base.isrunning()

    @property    
    def ishalted(self):
        return self._base.ishalted()
    
    @property
    def breakall(self):
        return self._base.breaks.breakall
    
    @breakall.setter
    def breakall(self, value):
        """NOTE: not all probes support this feature"""
        self._base.breaks.breakall = value
        
    @property
    def targpower(self):
        return self._base.power_status()

    @property
    def interestingthreads(self):
        return self._base.devs.gpc.interestingthreads()
    
    @interestingthreads.setter
    def interestingthreads(self, value):
        self._base.devs.gpc.interestingthreads(value)

    @property
    def keepprobemoderedirectionset(self):
        return self._base.devs.gpc.keepprobemoderedirectionset()
    
    @keepprobemoderedirectionset.setter
    def keepprobemoderedirectionset(self, value):
        self._base.devs.gpc.keepprobemoderedirectionset(value)


    @property
    def resetbreak(self):
        if len(self._base.threads) == 0:
            raise RuntimeError("no threads present, consider using ipc.breaks.reset if that is equivalent for what you need")
        else: 
            return self._getbreak("reset")
        
    @resetbreak.setter
    def resetbreak(self, value):
        if len(self._base.threads) == 0:
            raise RuntimeError("no threads present, consider using ipc.breaks.reset if that is equivalent for what you need")
        else: 
            return self._setbreak("reset", value)

    @property
    def keepprobemoderedirectioncleared(self):
        return self._base.devs.gpc.keepprobemoderedirectioncleared()
    
    @keepprobemoderedirectioncleared.setter
    def keepprobemoderedirectioncleared(self, value):
        self._base.devs.gpc.keepprobemoderedirectioncleared(value)

    @property
    def releaseallthreadswhilestepping(self):
        return self._base.devs.gpc.step_mode() == "all"

    @releaseallthreadswhilestepping.setter
    def releaseallthreadswhilestepping(self, value):
        mode = "all" if value else "targetted"
        self._base.devs.gpc.step_mode(mode)

    @property
    def stepintoexception(self):
        return self._base.devs.gpc.stepintoexception()
    
    @stepintoexception.setter
    def stepintoexception(self, value):
        self._base.devs.gpc.stepintoexception(value)
        
    def _getbreak(self, breakname):
        if len(self._base.threads) > 0:
            breakval = getattr(self._base.threads.breaks, breakname, None)
            if breakval is None:
                raise ValueError("Required break '{0}' does exist in ipc.threads.breaks".format(breakname))
            _log.caller()
            _log.debug("ENTER/EXIT: Global controlvar {0}break's value was returned as: {1}".format(
                breakname, breakval))
            return breakval
        else:
            raise RuntimeError("no threads present")
        
    def _setbreak(self, breakname, value):
        if len(self._base.threads) > 0:
            oldbreakval = getattr(self._base.threads.breaks, breakname, None)
            if oldbreakval is None:
                raise ValueError("Required break '{0}' does exist in ipc.threads.breaks".format(breakname))
            _log.caller()
            _log.debug("ENTER/EXIT: Global controlvar {0}break's value is getting set to: {1}".format(breakname, value))
            return setattr(self._base.threads.breaks, breakname, value)
        else:
            raise RuntimeError("no threads present")
        
    initbreak = property(lambda self: self._getbreak("init"),
                         lambda self, value: self._setbreak("init", value))
    machinecheckbreak = property(lambda self: self._getbreak("machinecheck"),
                                 lambda self, value: self._setbreak("machinecheck", value))
    smmentrybreak = property(lambda self: self._getbreak("smmentry"),
                             lambda self, value: self._setbreak("smmentry", value))
    smmexitbreak = property(lambda self: self._getbreak("smmexit"),
                            lambda self, value: self._setbreak("smmexit", value))
    smmintobreak = property(lambda self: self._getbreak("smminto"),
                            lambda self, value: self._setbreak("smminto", value))
    vmentrybreak = property(lambda self: self._getbreak("vmentry"),
                            lambda self, value: self._setbreak("vmentry", value))
    vmexitbreak = property(lambda self: self._getbreak("vmexit"),
                           lambda self, value: self._setbreak("vmexit", value))
    shutdownbreak = property(lambda self: self._getbreak("shutdown"),
                             lambda self, value: self._setbreak("shutdown", value))
