###############################################################################
# Copyright 2014 2017 Intel Corporation.
#
# The source code, information and material ("Material") contained herein is
# owned by Intel Corporation or its suppliers or licensors, and title to such
# Material remains with Intel Corporation or its suppliers or licensors. The
# Material contains proprietary information of Intel or its suppliers and
# licensors. The Material is protected by worldwide copyright laws and treaty
# provisions. No part of the Material may be used, copied, reproduced, modified,
# published, uploaded, posted, transmitted, distributed or disclosed in any way
# without Intel's prior express written permission.
#
# No license under any patent, copyright or other intellectual property rights
# in the Material is granted to or conferred upon you, either expressly, by
# implication, inducement, estoppel or otherwise. Any license under such
#
# intellectual property rights must be express and approved by Intel in writing.
# Unless otherwise agreed by Intel in writing, you may not remove or alter 
# this notice or any other notice embedded in Materials by Intel or Intel's 
# suppliers or licensors in any way.
###############################################################################


from __future__ import absolute_import
import logging,logging.handlers,traceback,re
import sys
import os

# stdiolog must be imported before we hook sys.stdout
from . import stdiolog
from ._py2to3 import *

###########################################################################################
# Background
#  - Python logging is complex because the logger has a level AND the handlers have levels
#
# Overall Logging Plan
# - scripts should create loggers with "ipccli.xyz"
# - scripts should use "log.result" to go to the console (or warning/error/critical)
# - logging to file <should> only be done at the root "ipccli" level, so that we go
#   to one file
# - in order for performance to be high, the logging modules we care about ("ipccli.")
#   will leave their levels to only go to console
#
# itp.logger.show()     - show list of loggers
# itp.logger.setFile()  - set file name for log path
# itp.logger.level(logger,level) - no args, is same as show? with args, sets file console level (checks for file handler first) 
# itp.logger.echo(echoOn) - enable logging to console (sets the console to be same level as file)
# itp.logger.reset() - back to defaults
#


#
# Default Logging levels for reference: 
#
# =============   ==========
# Level String    Level Num      
# =============   ==========     
# CRITICAL        50             
# ERROR           40              
# WARNING         30             
# RESULT          25       
# INFO            20              
# DEBUG           10
# TRACE           5
# =============   ==========     
# 
# New levels
#    - 5... Function call entry/exit
#
_new_log_levels = {
                   'RESULT':25
                   }
###############################################################
############ LOGGING ##########################################
################################################################
DEBUGALL = logging.DEBUGALL = 1
DEBUG    = logging.DEBUG
ACCESS   = logging.ACCESS  = 19   
INFO     = logging.INFO
RESULT   = logging.RESULT = 25
WARNING  = logging.WARNING
ERROR    = logging.ERROR
CRITICAL = logging.CRITICAL
_newlevels = {
    'DEBUGALL':DEBUGALL,
    'ACCESS'  :ACCESS,
    'RESULT'  :RESULT,
}

# Known holes in the "caller" routine
# that need to be worked around to get useful information
_stacks_to_skip = [
                   re.compile(r"node\.py.* in newf"),
                   re.compile(r"node\.py.* in __call__"),
                   #re.compile("dal_compatibility.*lambda"),
                   re.compile("dal_compatibility"), # skip logging the compatibility piece of a flow
                   ]

def _log_caller(self,func="",stacklevel=-1,*args,**kwargs):
    """
    This function attached to the logger class, and when it is invoked
    it will log the function that called the function with the log call

    Args:
        func (str) : function name to display as part of the logging.

    The output sent is sent to logger and is of the form:
        FUNC: CALLER: 
            <callback stack>
    """
    # shortcut for when this is called be logging is not enabled for debugall
    if not self.isEnabledFor(DEBUGALL):
        return
    stack = traceback.format_stack()
    # remove at least 2 (leaving -1 as the one that is whoever
    # called the ipc_commands.cmds
    stack.pop();stack.pop();
    # now circle until we find first stack entry that is known to be legit
    finished = False
    while not finished and len(stack)>1:
        finished=True # set back to false if we find match
        for skipre in _stacks_to_skip:
            if skipre.search(stack[-1]):
                stack.pop()
                finished = False
                break
    if func != "":
        self.debugall("{func}: CALLER: {stack}".\
                format(func=func,stack=stack[stacklevel].strip()))  
    else:
        self.debugall("CALLER: {stack}".\
                format(stack=stack[stacklevel].strip()))

_ansi_colors = re.compile(r'''
    \x1B  # ESC
    (?:   # 7-bit C1 Fe (except CSI)
        [@-Z\\-_]
    |     # or [ for CSI, followed by a control sequence
        \[
        [0-?]*  # Parameter bytes
        [ -/]*  # Intermediate bytes
        [@-~]   # Final byte
    )
''', re.VERBOSE)
     
     
class RemoveColorFormatter(logging.Formatter):
    """Has the same behavior as logging.Formatter, but will also remove ansi 
    coloring from the log message"""

    def format(self, record):
        msg = super().format(record)
        msg = _ansi_colors.sub('', msg)
        return msg

               
# we will create one of these after we have create our class 
class LoggerManager():
    """
    Contains all the functions for managing log files within the cli.
    """
    _logmanager = None
    _logobjects = {}
    _echoOn = False
    _root_logger = None
    _root_console = None
    _root_filehandler = None
    _base_logger = 'ipccli'

    def __init__(self):
        logClass = logging.getLoggerClass()
        for level,val in _newlevels.items():
            if PY2:
                levelNames = logging._levelNames
            else:
                levelNames = list(logging._nameToLevel.keys())
            if level not in levelNames:
                logging.addLevelName(val,level)
                newlevel = self._make_log_func(val)
                setattr(logClass,level.lower(),newlevel)
        # additionally we want to make sure logging class has some other nicities:
        if not hasattr(logClass,"caller"):
            setattr(logClass,"caller",_log_caller)
        # we should always have a console logger it's just whether the logger is on or not
        self._root_logger = self.getLogger(self._base_logger)
        self._root_logger.setLevel(1) # all logging should get to handlers
        self._root_console = logging.StreamHandler( sys.stdout )
        # simple format
        # default is to only send result to the screen
        formatter = logging.Formatter("%(message)s","%m-%d %H:%M")                
        self._root_console.setFormatter(formatter)
        self._root_console.setLevel(RESULT)
        self._root_logger.addHandler(self._root_console)
        
    def _make_log_func(self,level):
        """for making a new log function with the level name on the logger object"""
        def newlevel(self,msg,*args,**kwargs):
            self.log(level,msg,*args,**kwargs)
        return newlevel
             
    def show(self):
        """
        Display all the loggers that are available and the levels they are at.

        Example::

            >>> ipc.logger.show()
            Logname                        = Loglevel
            -------                        = --------
            ipc                            = RESULT
            state_uarch                    = RESULT
            node                           = RESULT
            bitdata                        = RESULT
            device                         = RESULT
            itpii_import_hook              = RESULT
            breakpoint                     = RESULT
            Debug logging to file is currently: OFF
            Debug logging to screen is currently: OFF                    

        """
        print("{logname:30s} = {loglevel:8s}".format(logname="Logname",loglevel="Loglevel"))    
        print("{logname:30s} = {loglevel:8s}".format(logname="-------",loglevel="--------"))
        loggernames = self.getLoggerNames()
        # only show the child loggernames
        loggernames = [ lname.replace(self._base_logger+".","") for lname in loggernames]
        if PY2:
            levelNames = logging._levelNames
        else:
            levelNames = logging._levelToName
        for lname in loggernames:
            logobj = self.getLogger(lname)
            # sometimes we can get a "placeholder" logger"
            if hasattr(logobj,"getEffectiveLevel"):
                levelnum = logobj.getEffectiveLevel()
            else:
                levelnum = logging.NOTSET
            loglevel = levelNames.get( levelnum, levelnum )
            #if isinstance(loglevel,str) or isinstance, class_or_type_or_tuple)
            print("{logname:30s} = {loglevel:8}".format(logname=lname,loglevel=loglevel))
        # if we have file handler report file handling as on or off
        filelogging   = "ON" if self._root_filehandler else "OFF"
        screenlogging = "ON" if self._echoOn           else "OFF"
        print("Debug logging to file is currently: {0}".format(filelogging))
        print("Debug logging to screen is currently: {0}".format(screenlogging))
        
    def setFile(self,filename,filemode='w',maxBytes=0,backupCount=10,
                logformat='%(asctime)s - %(name)s - %(levelname)s - %(message)s'):
        """
        
        Used to log CLI related information to a file.
        
        Args:
            filename (str) : Path to where to write file, default output
                             is the current working directory.
            filemode (str) : 'w' or 'a' - to write (default) or append to current file.
            maxBytes (int) : maximum number of bytes to write before rolling in
                             to a new file.
            backupCount (int) : if backupCount is zero, rollover never occurs; if backupCount is 
                             non-zero, the system will save old log files by appending the extensions
                             '.1', '.2' etc., to the filename. For example, with a backupCount of 5
                             and a base file name of app.log, you would get app.log, app.log.1, up to
                             app.log.5.
            logformat (str) : format of information written to log file.
            
        Returns:
            None.
            
        This function makes use of the logger module's RotateFileHandler class
        which is documented here in the standard Python documentation:
        
        https://docs.python.org/2/library/logging.html
           
        The logformat choices are documented under the LogRecord options
        in the standard Python documentation:

        https://docs.python.org/2/library/logging.html#logrecord-attributes

        Some examples of what happens with filemode/maxBytes:
            - filemode -'a', maxBytes=0 - append to existing file
            - filemode -'w', maxBytes=0 - roll over any existing file, start new one
            - filemode -'a', maxBytes=>0 - append to existing one, then rollover at maxBytes
            - filemode -'w', maxBytes=>0 - roll over any existing file, start new one, rollover again at maxBytes
        """
        if self._root_filehandler != None:
            self.closeFile()
        # if file exists and write was specified, we will make a backup
        do_rollover = os.path.exists(filename) and filemode=='w'
        # always use 'a' for actual call to rollover handler
        filemode = 'a'
        self._root_filehandler = \
                    logging.handlers.RotatingFileHandler(filename,filemode,maxBytes=maxBytes, backupCount=backupCount)
        #formatter = logging.Formatter(logformat,"%m-%d %H:%M")
        formatter = RemoveColorFormatter(logformat,"%m-%d %H:%M")         
        self._root_filehandler.setFormatter( formatter )
        if do_rollover:
            self._root_filehandler.doRollover()
        # if message makes it to root, then log it to file
        self._root_filehandler.setLevel(1)
        self._root_logger.addHandler( self._root_filehandler )
    
    def closeFile(self):
        """
        Stop logging from going to file.
        """
        if self._root_filehandler is None:
            raise RuntimeError("Cannot close file because no log file has been open")
        self._root_filehandler.close()         
        self._root_logger.removeHandler( self._root_filehandler )
        self._root_filehandler = None

    
    def echo(self,echoOn=None):
        """Enable logging debug information to the screen
        (you must use level() to set which loggers are enabled for debug).
         
        Args:
            echoOn (bool) : a boolean indicating whether logging is to be echoed to the screen (default is None).

        Returns:
            The current value of echoOn, if None is passed to the method.
         """
        if echoOn==None:
            return self._echoOn
        self._echoOn = echoOn
        if echoOn not in [0,1,True,False]:
            raise ValueError("Must specify boolean, (True or False, but not as a string) for echoOn")
        if echoOn:        
            self._root_logger.setLevel(0) # enable logging of whatever makes it to us...
            self._root_console.setLevel(0)
        else:
            self._root_console.setLevel(RESULT)
    
    def getLoggerNames(self,relative=False):
        """Return a list of the known loggers for the cli.
        
        Args:
            relative (bool) : used to return whether to only return logger names relative to the base logger name

        Returns:
            a list of logger names
        """
        loggernames = []
        # could perhaps use locall logger object dictionary instead...
        for logname,logobj in logging.Logger.manager.loggerDict.items():
            # never return the base
            if logname==self._base_logger:
                continue
            if logname.startswith(self._base_logger):
                if not relative:
                    loggernames.append( logname )
                else:
                    # remove the base level logger name
                    loggernames.append( logname.replace(self._base_logger+".","") )
        return loggernames        
        
    def level(self,loggername=None,loglevel=None):
        """
        Used to choose whether additional logging information will be
        enabled. You still have to use "echo()" or setFile()" to determine
        whether the logging goes to the screen or to a file.
        
        Args:
            loggername (str) : name of logger to enable.
            loglevel (str) : level of information to output.
        
        - if no parameters are specified, then the current loggers are displayed.
        - if only loglevel is specified, then ALL loggers will be set to that level.
        - if only loggername is specified, then the current log level is returned.
        
        Valid loglevels are:
            - CRITICAL - logs only CRITICAL messages
            - ERROR - logs ERROR and CRITICAL messages
            - NOTSET - logs WARNING, ERROR, and CRITICAL messages
            - WARN, WARNING - logs WARNING, ERROR, and CRITICAL messages
            - INFO - logs RESULT, INFO, WARNING, ERROR, and CRITICAL messages
            - DEBUG - logs RESULT, DEBUG, INFO, WARNING, ERROR, and CRITICAL messages
            - DEBUGALL - logs all messages
            - RESULT <- Default, recommended value. Logs RESULT, WARNING, ERROR, AND CRITICAL messages
            - OFF - reverts the loglevel to the default RESULT level
        
        Warning:
            If you specify WARNING or CRITICAL it will prevent logging
            and display of most output.
        
        Raises:
            ValueError: if an invalid logger name is passed in
        """
        if loggername == None and loglevel == None:
            return self.show()
        
        # if we are changing loglevel to something other than default
        # but there's nothing other than the screen to send it to..then 
        # warn the user
        if not (self._echoOn) and len(self._root_logger.handlers) <= 1:
            if isinstance(loglevel,basestring) and loglevel!=None and loglevel.lower() != "result":
                print("Warning!: don't forget to either set file or turn echo on to see log messages")
            if isinstance(loglevel,(int,long)) and loglevel != RESULT:
                print("Warning!: don't forget to either set file or turn echo on to see log messages")
            
        # use upper case when setting log level since that is what logger objects need
        if isinstance(loglevel,str):
            loglevel = loglevel.upper()

        if PY2:
            level2names = logging._levelNames
            names2level = logging._levelNames
        else:
            level2names = logging._levelToName
            names2level = logging._nameToLevel

        # in case we get a common error
        if loggername != None and loggername.upper() in names2level:
            raise ValueError("must specify the logger you would like to set the log level to")

        if loggername in [None,'all',"ALL"] and loglevel != None:
            # set all loggers to be the specified log level
            for lname in self.getLoggerNames():
                lobj = logging.getLogger(lname)
                if isinstance(loglevel,basestring):
                    # if we get "off" then put back to result
                    loglevel = "RESULT" if loglevel in ["OFF","off"] else loglevel
                    if loglevel not in names2level:
                        raise ValueError("Invalid log level name: {0}".format(loglevel))
                    loglevel = names2level[loglevel]                
                lobj.setLevel(loglevel)
            return
        elif loggername != None and loglevel == None:
            if not loggername.startswith(self._base_logger):
                # if the only specified the child, then add our base logger prefix to it
                loggername = "{0}.{1}".format(self._base_logger,loggername)
            if loggername not in self.getLoggerNames():
                raise ValueError("that is not a known {0} logger,\n\t{indent}run this function without parameters to see the valid loggers".\
                                 format(self._base_logger,indent=" "*len("ValueError: ")))
            lobj = logging.getLogger(loggername)
            level = lobj.getEffectiveLevel()
            level = level2names.get(level,level)
            return level
        else:
            if not loggername.startswith(self._base_logger):
                # if the only specified the child, then add our base logger prefix to it
                loggername = "{0}.{1}".format(self._base_logger,loggername)            
            if loggername not in self.getLoggerNames():
                raise ValueError("that is not a known {0} logger,\n{indent}run this function without parameters to see the valid loggers".\
                                 format(self._base_logger,indent=" "*len("ValueError: ")))                
            # we must be supposed to set the level
            lobj = logging.getLogger(loggername)
            # to better support python26 we need to make sure level is an int
            if isinstance(loglevel,basestring):
                # if we get "off" then put back to result
                loglevel = "RESULT" if loglevel in ["OFF","off"] else loglevel
                if loglevel not in names2level:
                    raise ValueError("Invalid log level name: {0}".format(loglevel))
                loglevel = names2level[loglevel]
            lobj.setLevel( loglevel )

    def reset(self):
        """Reset all logging to defaults."""
        self._root_logger.handlers = [self._root_console]
        self.level(loglevel=RESULT)
        if self._root_filehandler != None:
            self.closeFile()
        self.echo(False)
    
    def getLogger(self,loggername):
        """Used by modules to get a new/existing logger instance that 
        will be tied to the IPC.
        
        Loggers that go through this call will be enabled/disabled
        via the other output functions in this IPC CLI.

        Args:
          loggername (str) : the name of the logger to get

        Returns:
          the specified logger instance
        """
        # only return loggers that are children of this base logger
        if not loggername.startswith(self._base_logger) and loggername != self._base_logger:
            loggername = "{0}.{1}".format(self._base_logger,loggername)
                    
        if loggername in self._logobjects:
            # should only get here if someone is trying to get baselogger other than this manager
            if loggername == self._base_logger: 
                raise ValueError("base logger is reserved, please be sure to specify {0}.something".format(self._base_logger))
            return self._logobjects[loggername]            
        
        # otherwise, we need to initialize this one
        # go through normal 'getLogger' process 
        lobj = logging.getLogger(loggername)
        self._logobjects[loggername] = lobj
        # set default logging level to be RESULT only
        lobj.setLevel(RESULT)
        # make sure it does not have handlers due to other code that also
        # is using the logging module
        lobj.handlers = []
        return lobj

_logmanager = LoggerManager()

##################################################################
# It is important that all getLogger calls flow through this file
# instead of straight to the logging module
##################################################################
def getLogger(loggername):
    return _logmanager.getLogger(loggername)

def getManager():
    return _logmanager
