###############################################################################
# Copyright 2014 2017 Intel Corporation.
#
# The source code, information and material ("Material") contained herein is
# owned by Intel Corporation or its suppliers or licensors, and title to such
# Material remains with Intel Corporation or its suppliers or licensors. The
# Material contains proprietary information of Intel or its suppliers and
# licensors. The Material is protected by worldwide copyright laws and treaty
# provisions. No part of the Material may be used, copied, reproduced, modified,
# published, uploaded, posted, transmitted, distributed or disclosed in any way
# without Intel's prior express written permission.
#
# No license under any patent, copyright or other intellectual property rights
# in the Material is granted to or conferred upon you, either expressly, by
# implication, inducement, estoppel or otherwise. Any license under such
#
# intellectual property rights must be express and approved by Intel in writing.
# Unless otherwise agreed by Intel in writing, you may not remove or alter 
# this notice or any other notice embedded in Materials by Intel or Intel's 
# suppliers or licensors in any way.
###############################################################################




# TODO: needs to handle the VALUE not being'known' yet
#
#
import operator
import array
import binascii
import traceback
import math
from copy import copy
from ._py2to3 import *

from . import cli_logging
_log = cli_logging.getLogger("bitdata")
_debug_q = False

# special class that will be tested against for allowing
# initialization without a length parameter
Operation = None

def _setOperationClass(cls):
    """internal function used to help treat operations special when creating bit data
    an Operation object must provide a GetValue() function that returns an object, the
    returned object, must have bitSize attribute and a Value() function
    """
    global Operation
    Operation = cls

# only a global to save performance....
_intstr = set(map(str,range(10)))


# En Enum to document the possible Queued operations
class QOps:
    op        = 1
    set_slice = 2
    check     = 3
    invert    = 4
    reverse   = 5



class BitData(object):
    """
    This BitData provides numerous functions to help with displaying numbers as hex, setting/retrieving only
    certain bits, and lots of other bit granularity functions.

    Args:

        length (int, optional): the bit length of the data being passed in
        value (int, string): the value to be stored in the BitData object

    If length is not specified, then the value must be passed in as a string. Then the
    length will be guessed based on the number of bits.

    There are many ways to access the data after a new bit data object has been created.
    However, the fastest/most-efficient is to use array style syntax.

    Some Examples of BKMS for usage:

        >>> bits = ipccli.BitData(32,0xAA55)
        >>> bits[0:7]
        [8b] 0x55
        >>> bits[0]
        0x1L
        >>> bits[0:7] = 0xAA
        >>> bits
        [32b] 0x0000AAAA
        >>>

    And of course, the bitdata can also be treated as a number most of the time:
    
        >>> bits = ipccli.BitData(32,1)
        >>> bits = bits + 1
        >>> bits
        [32b] 0x00000002
        >>> bits = bits << 2
        >>> bits
        [32b] 0x00000008
        >>> bits / 2
        [32b] 0x00000004

    As you can tell from the examples above, the default display for a bitdata is in hex.

    **Bit Data Operation Queuing**

    There are additional arguments that provide the ability to queue up operations
    instead of applying them immediately. These queueing operations
    should ONLY be used during state representations. When used with simple
    numbers, there is no benefit to delaying the operations. The
    primary use case for queuing is to use with Operation Receipts. However, for
    simplicity's sake, the examples use standard numbers. In the example you can see that the
    bitdata in queue mode can pretty much be treated like a standard bitdata:

        >>> bdata = ipc.BitData(32,0x5A,queue=True)
        >>> bdata = ipc.BitData(32,0x5A,queue=True)
        >>> bdata2 = bdata[7:4]
        >>> bdata2
        [4b] 0x5
        >>> bdata3 = bdata2 + 1
        >>> bdata3
        [4b] 0x6
        >>> bdata4 = ~bdata2
        >>> bdata4
        [4b] 0xA

    If you already have a bit data, and want to enable the queuing, that can be done via:

        >>> bdata.Queue()

    "Slicing" (treating the bitdata object like an array) is the recommended way to
    do operations on portions of bitdata objects (keeping in mind that the bitsize of the result
    is kept consistent with the slice requested).

    Builtin math operations are supported, as well as the following functions that are specific
    to the queuing mode:

    - *AddErrorCheck* - see below for more examples of this function.

    Most all of the other BitData functions for extracting or displaying data
    shoud be avoided if a flush is not desired. Only the following can be executed without
    causing a flush:

    - *Reverse*

    All of the following functions will implictily force a flush. Again, most of these
    can directly be replaced by using slicing:

    - *Append, Concatenate, Extract functions, Copy, To___ functions, IsBitOn, IsBitOff*

    There are certain BitData functions that are only valid while Queuing is enabled.
    Currently those are:

        - *AddErrorCheck*

    """
    __slots__ = [
                '_length',
                '_value',
                '_q_enabled',
                '_q_length',
                '_q_value',
                '_q_operations',
                '_q_verifiers',
                'truncate',
                ]
    # whether to truncate data
    # when we overflow the BitSize
    def __init__(self, *args,**kwargs):
        """
        Expect arguments are one of:
            BitData(length,value)
            BitData(value)
        """
        # some default
        self._length = None
        self._value = None
        self.truncate = True
        # quueing related
        self._q_enabled = False
        self._q_length = None
        self._q_value = None
        self._q_operations = []
        self._q_verifiers = []
        # the fact that we take our "optional" parameter first kindof sux...
        # but we have no choice, it is legacy behavior that we have to maintain

        if _log.isEnabledFor(cli_logging.DEBUG):
            stack = traceback.format_stack(limit=3)[-2].strip()  # last entry is THIS file...next to last entry should be our caller
            _log.debug("CALLER: {0}".format(stack))
            _log.debug("ENTER: bitdata.__init__(*args={0},kwargs={1})".\
                       format( str(args), str(kwargs) ))


        # default in case we cant figure out length
        length = None

        # if no args, then that is easy
        if len(args)==0 and len(kwargs) == 0:
            self._value = 0
            self._length = 0
            return

        #### check for special queue kwargs:
        self._q_enabled = kwargs.pop("queue",False)
        self._q_operations = kwargs.pop("operations",[])
        self._q_verifiers = kwargs.pop("verifiers",[])
        # truncate args...
        self.truncate = kwargs.pop("truncate", True)

        if self._q_enabled:
            self._q_value = args[1]
            self._q_length = args[0]
            self._value = None
            self._length = None
            return

        # figure out our arguments
        if len(args)==1 and 'value' not in kwargs:
            # Example:
            # BitData("0x100") or BitData(0x100)
            if isinstance(args[0],(int,long)):
                raise Exception("Must specify value and length if initializing with numbers")
            elif isinstance(args[0],basestring):
                # we'll conver this later
                value = args[0]
            else:
                # put seperate else...because we may should check for operation here?
                # could error check that this is operation here?
                value = args[0]

        elif len(args)==1 and 'value' in kwargs:
            # Example: BitData(0,value=2)
            length = args[0]
            value = kwargs.pop('value')
        elif len(args)==2:
            # Example:
            # BitData(32,2) or BitData(32,"0x2")
            length = args[0]
            value = args[1]
        elif len(args) > 3:
            # INVALID
            raise Exception("Too many arguments passed in: {0}".format(args))
        elif len(args)==0:
            # Examples:
            #  BitData(length=32,value=5)
            #  BitData(value=5)
            if 'value' not in kwargs:
                raise ValueError("Must provide a value for BitData")
            value = kwargs.pop('value')
            if length in kwargs:
                length.kwargs.pop('length')

        #### a safety net:
        if 'value' in kwargs:
            raise RuntimeError("Sorry, you've hit a code bug here, kwargs: {0}".format(kwargs))
        if  'length' in kwargs:
            raise RuntimeError("Sorry, you've hit a code bug here, kwargs: {0}".format(kwargs))

        #####################################################
        # If queuing is enabled we do a lot less checking
        #####################################################
        if self._q_enabled:
            self._q_value = value
            self._q_length = length
            self._value = None
            self._length = None
            return

        # We should not have any unknown kwargs at this point
        if len(kwargs) != 0:
            raise ValueError("Unknown keyword arguments passed in: {0}".format(kwargs))

        #########################################
        # first off...very special case:
        #########################################
        if Operation != None:
            if isinstance(value, Operation):
                self._value = value
                self._length = None
                return
            # or if we are given a bitdata that points to an operation
            # I think maybe we should force value to be a number here in case
            # we are given an operation...the way it is coded as of 6/2/2014, is that
            # both bitdata's might point to the same operation....
            elif isinstance(value,BitData) and isinstance(value._value,Operation):
                self._value  = value._value
                self._length = value._length
                return
            # no else here on purpose, if if the value was not an
            # operation, go on, and below we fiture out what to do with it

        ##############################################
        # Make sure we convert type for length
        ##############################################
        if isinstance(length,basestring):
            length = self._parse_number( length )
        elif isinstance(length,(int,long,BitData)):
            length = long(length)

        ##############################################
        # Make sure we convert type for value
        ##############################################
        if isinstance(value,basestring):
            valuestr = value
            value = self._parse_number(value)
            if length == None:
                if len(valuestr)<2:
                    length = 8
                elif valuestr[1] in ['x', 'X']:
                    length = (len(valuestr) - 2) * 4
                elif valuestr[1] in ['b', 'B']:
                    length = (len(valuestr) - 2)
                elif valuestr[1] in ['y', 'Y']:
                    length = (len(valuestr) - 2)
                elif valuestr[1] not in _intstr: #this means we got a letter
                    raise Exception("Unknown number type for value")
                else: # TODO: decimal, this really needs to verify we have an INT at strval[1]
                    length = int( math.ceil(
                                 len(valuestr)*3.3219280948873623478703194294894
                                ))
        elif isinstance(value,array.array):
            # KRP: not sure I understand this one, but leaving in for legacy to make sure
            # i do not break anything
            value = value.tostring() if PY2 else value.tobytes()
            value = long(binascii.hexlify(value[::-1]), 16)
        elif isinstance(value,list):
            # not really sure how to handle lists...this would require some odd assumptions
            # but using code from previous implementations
            newvalue = 0
            for v,val in enumerate(value):
                newvalue |= val << (v*32)
            value = newvalue
        # if value is some odd array, convert it...
        elif isinstance(value,BitData):
            # ah, its just like us....just grab its value
            if length == None:
                length = value._length
            value = value._value

        #####################################
        # Now some more error checking
        ####################################

        # Verify we have length and value at this point
        if length==None:
            raise Exception("Could not determine BitData length")
        if value==None:  # value was specifically given as None to get here, but we don't support so turn to a number
            value = 0

        self._length = length
        self._value = value & self._compute_mask(self._length)
        if self.truncate:
            return
        if self._value != value:
            self._length = len("%x"%value)*4
            self._value = value
        return

    def __getstate__(self):
        # make sure it has been converted to true number
        # before we pickle..
        if not isinstance(self._value, long):
            long(self)
        mystate = {
            '_value': self._value,
            '_length': self._length,
            'truncate': self.truncate
        }
        return mystate

    def __setstate__(self, state):
        self._value = state['_value']
        self._length = state['_length']
        self.truncate = state.get('truncate', True)
        self._q_enabled = False
        self._q_length = None
        self._q_value = None
        self._q_operations = []
        self._q_verifiers = []
        return

    @classmethod
    def CreateFromLong(cls, length, int_value, truncate=False, verify=True):
        """
        Create using exactly two arguments.

        Args:
            length (int) : Bit length of this bitdata.
            int_value (int) : number or object that behaves as an integer.
            truncate (bool) : whether to truncate data when it grows beyond bitsize (default: False)
            verify (bool) : whether to verify length is correct for the given int_value

        Note:
            This function is aimed at improving performance and no bounds or type
            checking is done. If checking is desired, use the standard constructor.
        """
        # hopefully faster creation
        bd = BitData.__new__(BitData)
        # queueing always off for this call
        bd._q_value = bd._q_value = bd._q_length = None
        bd._q_enabled = False
        bd.truncate = truncate
        bd._q_verifiers = None
        if verify:
            bd._length = length
            bd._value = int_value & ((1 << long(length)) - 1)
            if not truncate and bd._value != int_value:
                bd._length = len("%x" % int_value) * 4
                bd._value = int_value
        else:
            bd._length = length
            bd._value = int_value
        return bd

    # python3 prep for where there is no such thing as long...
    CreateFromInt = CreateFromLong

    def _compute_mask(self, length):
        mask = (1 << long(length)) - 1
        return mask

    def _parse_number(self, strval):
        strval = strval.replace('L', '')
        if len(strval)<2:
            return long(strval)
        elif strval[1] in ['x', 'X']:
            return long(strval,16)
        elif strval[1] in ['b', 'B']:
            return long(strval,2)
        elif strval[1] in ['y', 'Y']:
            return long(strval[2:],2)
        return long(strval, 10)

    def Queue(self,operations=None):
        """
        Turns on operations Queuing for this BitData. Future operations
        will be Queued instead of being applied immediately

        Args:
            operations (list) : (optional) list of queued operations.

        The operations list should not be built by hand. It is used by
        the BitData class to build the queued list of operations that
        should be executed when we finally evaluate the object.
        
        Raises:
            Exception: if queuing is already enabled.

        """
        if self._q_enabled:
            raise Exception("queuing already enabled")
        # save current info to be part of "queue"
        self._q_enabled = True
        self._q_value = self._value
        self._q_length = self._length
        operations = [] if operations == None else operations
        self._q_operations = operations
        # now wipe the "normal" values...because once those are set again
        # we will use them all upthe time
        self._value = None
        self._length = None # ? maybe leave this one?

    def _q_flush(self):
        """Something triggered us to evaluate our current queue of operations"""
        # call this once, and use as a local variable to reduce any logging call overhead
        logenabled = True if _log.isEnabledFor(cli_logging.DEBUG) else False
        if logenabled:
            _log.caller()
            _log.debugall("ENTER : _q_flush")
        # evaulate current operations and set the value and length to be valid
        self._q_enabled = False
        # for this function
        # currval = the current value with the operations that have been applied so far
        # bitsize = the assumed bitsize of the final value

        # start by foricing our q value to give us a number, this will force a
        # flush on an operation receipt
        currval = None
        if self._q_length == None:
            # assume that it is either an operation receipt, or that it behaves like one
            currval = long(self._q_value) # this will perform an error check
            # now grab the bit data object so that we can get number of bits
            ipc_bitdata = self._q_value.GetValue()
            bitsize = ipc_bitdata.bitSize

        # we have a lenght, nothing else to other than make sure we have a number
        else:
            # not sure what it is, but assume it can be cast, and check it for a bitsize
            currval = long(self._q_value)
            bitsize = self._q_length

        # at this point we should be done with _q_value
        if logenabled:
            _log.debugall("_q_flush : starting value and bitsize: 0x{0}, {0:s} bits".format(currval,bitsize))

        for ops in self._q_operations:
            if _debug_q:
                _log.debugall("_q_flush : applying operation : {0:s}".format(ops))
            if ops[0] is QOps.op:
                operation, other, right = ops[1:]
                # sometimes we don't have an 'other'
                if other==None:
                    currval = operation(currval)
                    bitsize = bitsize
                else:
                    if isinstance(other,(int,long)):
                        other_bitsize = 0
                    else:
                        other_bitsize = getattr(other,"BitSize", 0) # not 0, because None < 0
                    bitsize = max(bitsize, other_bitsize)
                    # always long(other) in case it is a queued bitdata
                    if right: currval = operation(long(other),currval)
                    else:     currval = operation(currval,long(other))

            elif ops[0] is QOps.check: # standard operation
                # based on the current value, do a check on the specified bits
                lowerbit, numbits, exp_value, message = ops[1:]
                mask = ((1<<numbits)-1)
                checkval = (currval>>lowerbit) & mask
                if checkval != exp_value:
                    raise ValueError(message + " Expected: {0}, Actual {1}".format(
                            exp_value,checkval))
            elif ops[0] is QOps.set_slice : # ..
                operation, start, end, other  = ops
                # we cannot bounds check because we dont really know what the current end is...
                start, end = (long(start), long(end)) if end>start else (long(end), long(start))
                nbits = end - start + 1
                mask = (1<<nbits)-1
                mask = ~(mask << start)
                # make sure we cast other to long in case it is a queued bit data
                other = long(other) << start
                currval = (currval & mask) | other
            elif ops[0] is QOps.invert:
                currval = ~currval
            elif ops[0] is QOps.reverse:
                # reverse pretty much requires us to apply bitsize at this point....
                bitOffset,bitSize = ops[1:]
                if bitSize == None:
                    bitSize = bitsize
                to_reverse = (currval>>bitOffset)
                reversed_value = 0
                for b in xrange(bitSize): # note we are using the capital bitSize...the one
                    reversed_value |= ((to_reverse>>(bitSize-b-1)) & 1) << b
                # super long line generats a mask to "remove" bits that we will replace with reversed value
                mask = ~(((1 << long(bitSize)) - 1) << bitOffset)
                currval = (currval&mask) | (reversed_value<<bitOffset)
            else:
                raise RuntimeError("Unexpected operations list: {0:s}".format(ops))
        # try not to apply bitsize until the end
        if bitsize!=None:
            currval = currval & ((1<<bitsize)-1)
        # set actual value and length to go back to acting like a bit data
        if _debug_q:
            _log.debugall("_q_flush : EXIT")
        self._value = currval
        self._length = bitsize

    @property
    def BitSize(self):
        """BitSize()
        Returns the number of bits (or length) that represent the number stored here.
        """
        if self._q_enabled: self._q_flush() # make sure we are not queuing first
        if self._length != None:
            return self._length
        else: # length is None, so we might/must have an Operation object
            # this should only happen when we get an Operation...we are not going to
            # type check, because sometimes the isinstance does not work due to how
            # the Operation class gets set in this file.
            ipc_bitdata = self._value.GetValue()
            self._length = ipc_bitdata.bitSize
            self._value = long(self._value)
            return self._length

    @BitSize.setter
    def BitSize(self, value):
        if self._q_enabled: self._q_flush() # make sure we are not queuing first
        self._length = value
        if self._length != 0:
            self._value = self._value & self._compute_mask(self._length)
        else:
            self._value = 0

    def __hash__(self):
        return hash(long(self))

    def IsBitOn(self, bit_num):
        """
        Returns a boolean value that reports whether bit_num was set.

        Args:
            bitnum(int) : bit to check if == 1.

        Example:
            >>> bit2set = BitData(32,0x4)
            >>> bit2set.IsBitOn(2)
            True
            >>> bit2set.IsBitOn(0)
            False
        """
        if self._q_enabled: self._q_flush() # make sure we are not queuing first
        return self[bit_num] == 1

    def IsBitOff(self, bit_num):
        """
        Returns a boolean value that reports whether bit_num was clear.

        Args:
            bitnum(int) : bit to check if == 0.

        Example:
            >>> bit2set = BitData(32,0x4)
            >>> bit2set.IsBitOff(2)
            False
            >>> bit2set.IsBitOff(0)
            True
        """
        if self._q_enabled: self._q_flush() # make sure we are not queuing first
        return self[bit_num] == 0

    def SetBit(self, bit_num, value):
        """
        Change the specified bit to a new value.

        Args:
            bitnum(int) : bit to change.
            value (int) : new value.

        Example:
            >>> bitset = ipccli.BitData(32,4)
            >>> bitset.SetBit(3,1)
            >>> bitset
            [32b] 0x0000000C
        """
        self[bit_num] = value

    @property
    def value(self):
        """
        The value property exists for backwards compatibility, but it is not
        recommended to use directly. Use: long(bitdata_object) to get the value.
        """
        if self._q_enabled: self._q_flush() # make sure we are not queuing first
        return long(self)

    def Copy(self,startbit=None,length=None):
        """
        Copies the requested bits in to a new bitdata object and returns it.

        Args:
            startbit (int) : where to begin copying to new bitdata.
            length (int) : number of bits to copy to new bitdata.

        Returns:
            New BitData object

        This is similar to the Extract function but it returns a bitdata instead
        of a just the number.  If no args are specified, then a copy of this
        BitData is returned.
        """
        if self._q_enabled and startbit is None and length is None:
            bd = BitData.__new__(BitData)
            bd._length = bd._value = None
            bd._q_value = self._q_value
            bd._q_length = self._q_length
            bd._q_enabled = True
            bd._q_operations = copy(self._q_operations)
            bd._q_verifiers = copy(self._q_verifiers)
            bd.truncate = self.truncate
            return bd

        if startbit==None:
            startbit = 0
        if startbit<0:
            raise ValueError("Startbit must be a positive number, use slicing if you need negative numbers")
        if length==None:
            length = self.BitSize-startbit
        value = self.Extract(startbit,length)
        return BitData(length, value, truncate=self.truncate)

    def ToHex(self):
        """Display value as a hex string."""
        if self._q_enabled: self._q_flush() # make sure we are not queuing first
        char_len = ((self.BitSize + 3) // 4)
        return '0x' + ('{0:X}').format(long(self)).zfill(char_len)

    def ToBinary(self):
        """Display value as a binary string."""
        if self._q_enabled: self._q_flush() # make sure we are not queuing first
        char_len = self.BitSize
        return '0b' + ('{0:b}').format(long(self)).zfill(char_len)

    def ToDecimal(self):
        """Display value as a decimal string."""
        if self._q_enabled: self._q_flush() # make sure we are not queuing first
        return str(self._value)

    def ToUInt32(self):
        """Return raw integer value...same as int(bitdata)."""
        if self._q_enabled: self._q_flush() # make sure we are not queuing first
        return int(self)

    def ToUInt64(self):
        """Return raw integer value...same as long(bitdata)."""
        if self._q_enabled: self._q_flush() # make sure we are not queuing first
        return long(self)

    def ToRawBytes(self,bitOffset=0,byteCount=None):
        """Return as a list where each entry is one byte worth of data.

        Args:
            bitOffset : starting *bit* to begin creating array (default=0).
            byteCount : number of bytes to convert (default= use BitSize to calculate).

        This function was created to return an array of bytes. It has additional
        parameters in order to provide backwards compatibility for the ReadByteArray
        function (which was discovered after this function was created).
        """
        if self._q_enabled: self._q_flush() # make sure we are not queuing first
        num_bytes = byteCount or ((self.BitSize + 7 ) // 8)
        return [(self._value >> (i*8 + bitOffset)) & 0xFF for i in range(num_bytes)]


    def ReadByteArray(self,bitOffset=0,byteCount=None):
        """
        Return a list where each entry is one byte worth of data.

        Args:
            bitOffset : starting *bit* to begin creating array (default=0).
            byteCount : number of bytes to convert (default= use BitSize to calculate).
        """

        return self.ToRawBytes(bitOffset,byteCount)

    def __reload__(self, length, value = 0):
        # ??
        if self._q_enabled: self._q_flush() # make sure we are not queuing first
        self._length = int(length)
        self._value = value


    def __hex__(self):
        if self._q_enabled: self._q_flush() # make sure we are not queuing first
        return hex(self._value).replace("L","")

    def __format__(self,format_spec):
        if self._q_enabled: self._q_flush() # make sure we are not queuing first
        # if string or default
        if format_spec.count("s") or format_spec=="":
            return format( repr(self), format_spec )
        else:
            # assume it wants our number format
            try:
                num = long(self)
            except Exception as e:
                _log.error("ERROR formating to hex due to operation error: %s"%str(e))
                raise
            return format( num, format_spec )

    def __repr__(self):
        if self._q_enabled: self._q_flush() # make sure we are not queuing first
        try:
            length = self.BitSize
            if length > 0:
                return '[' + str(length) + 'b] ' + self.ToHex()
            else:
                return '[' + str(length) + 'b] empty'
        except Exception as e:
            _log.error("ERROR formating to hex due to operation error: %s"%str(e))
            raise

    def ToString(self):
        """Return string version that is the same as repr(bitdata)."""
        if self._q_enabled: self._q_flush() # make sure we are not queuing first
        return unicode(self.__repr__())

    # Comparison Operators
    def __long__(self):   
        # if we don't have a length, we probably have an operation receipt
        if self._q_enabled: self._q_flush() # make sure we are not queuing first    
        if self._length==None:
            ipc_bitdata = self._value.GetValue()
            self._length = ipc_bitdata.bitSize
            self._value = long(self._value)
            self._flush_verifiers()
        elif not isinstance(self._value, long):
            self._value = long(self._value)
            self._flush_verifiers()
        return self._value
    # convert to long first to make sure type flows
    def __int__(self):
        return int(self.__long__())

    def __float__(self):
        return float(self.__long__())

    def __abs__(self):
        return abs(self.__long__())

    # for python3
    def __index__(self):
        return self.__long__()
    def __nonzero__(self):   return long(self).__nonzero__()
    def __bool__(self):   return long(self).__bool__()
    def __lt__(self, other): return long(self) < other
    def __eq__(self, other):
        # special case None check to prevent long from being called
        if other == None:
            return False
        return long(self) == other
    def __le__(self,other):  
        return (long(self)<=other)

    def __ne__(self, other):
        return not self == other
    def __ge__(self, other):
        return self > other or self == other
    def __gt__(self, other):
        return not self <= other

    def __len__(self):
        return self.BitSize

    # Sequence Operators
    def __getitem__(self, index):
        if isinstance(index, slice):
            return self.__getslice__(index.start, index.stop)
        elif isinstance(index, tuple):
            newb = type(self)()
            for each_i in index:
                newb.append(self.__getitem__(each_i))
            return newb
        else:
            return self.__getslice__( index, index )

    def __setitem__(self, index, value):
        if isinstance(index, slice):
            self.__setslice__(index.start, index.stop, value)
            return
        return self.__setslice__(index, index, value)

    def __getslice__(self, start, end):
        '''BitData supports different indexing options than python list.
        Indices are inclusive, and can be given as [start:end] or [end:start]'''

        # We have to force these in to numbers due to the fact that if we get a bitdata, things
        # can go badly.
        # For Q'd bit data, this might force a cache flush on whatever is passed in to start and end
        # but we have to have that in order to handle the start/end being backwards
        start = long(start)
        if start<0:  # in python3 this can be negative, in python2 it figures it out using len
            orig_start = start
            start = len(self) + start
            if start<0: # if still less than 0 we have an error
                raise IndexError("start bit '{0}' is greater than this bit data's size '{1}'".format(orig_start, self.BitSize))

        # hardcoded bit data size is compatible with python2 and python3
        end = long(end) if end is not None else 0x7fffffffffffffff
        if end<0:  # in python3 this can be negative, in python2 it figures it out using len
            orig_end = end
            end = len(self) + end
            if end<0: # if still less than 0 we have an error
                raise IndexError("end bit '{0}' is greater than this bit data's size '{1}'".format(orig_end, self.BitSize))
        if start > end:
            start, end = end, start
        # slight tweak in code path if we are using queueing
        # do this BEFORE the BitSize below since that will cause queues to flush
        if self._q_enabled:
            nextops = []
            nextops.append( (QOps.op, operator.rshift,start, False) )
            nbits = end - start + 1
            mask = (1<<nbits)-1
            nextops.append(  (QOps.op, operator.and_, mask, False ) )
            # maximum creation efficiency
            bd = BitData.__new__(BitData)
            bd._length = bd._value = None
            bd._q_value = self._q_value
            bd._q_length = nbits
            bd._q_enabled = True
            bd._q_operations = self._q_operations + nextops
            bd._q_verifiers = copy(self._q_verifiers) ## 
            bd.truncate = self.truncate
            return bd

        # if end is too big, truncate it
        if end>self.BitSize and self.truncate:
            end = self.BitSize-1
        # if start is too bit, throw an exception
        if start >= self.BitSize and self.truncate:
            raise IndexError("start bit '{0}' is greater than this bit data's size '{1}'".format(start,self.BitSize))
        bitlen = end - start + 1
        mask = (1<<bitlen)-1
        if self.truncate:
            value = (self._value >> long(start)) & long(mask)
        else:
            value = (self._value >> long(start))
        return BitData.CreateFromInt(bitlen, value, truncate=self.truncate)

    def __setslice__(self, start, end, value):
        # slight tweak in code path if we are using queueing
        # do this BEFORE the BitSize below since that will cause queues to flush
        if self._q_enabled:
            # we have to just store off reference to value (hopefully it doesn't change ?)
            self._q_operations.append( (QOps.set_slice, start, end, value) )
            return

        # at this point it should be safe to cast start and end to make sure they are ints otherwise,bitdata
        # behaves VERY badly and does some truncating
        start = long(start)
        end = long(end)
        value = long(value)

        if start > end:
            start, end = end, start

        if end>self.BitSize and self.truncate:
            end = self.BitSize-1
        # if start is too bit, throw an exception
        if start >= self.BitSize and self.truncate:
            raise IndexError("start bit '{0}' is greater than this bit data's size '{1}'".format(start,self.BitSize))

        mask = (1<<(end + 1))-1   # mask of bit 0 to end
        mask &= ~( (1<<start)-1 ) # clear bits from 0 to start

        if self._value is not None:
            self._value &= ~mask # clear slice
        else:
            self._value = 0
        self._value |= (( value << start ) & mask )
        if not self.truncate:
            # confirm number of bits
            if end > self.BitSize:
                self._length = end + 1

    # Numeric operators
    def __apply_numeric_operation__(self, num_op, other, rightside=False ):
        '''
        Used for generic numeric operations where our bitdata is on the left
        '''
        # first check for queueing:
        if self._q_enabled:
            # maximum creation efficiency
            bd = BitData.__new__(BitData)
            bd._length = bd._value = None
            bd._q_value = self._q_value
            bd._q_length = self._q_length
            bd._q_enabled = True
            bd._q_operations = self._q_operations + [ (QOps.op, num_op, other, rightside) ]
            bd._q_verifiers = copy(self._q_verifiers)
            bd.truncate = self.truncate
            return bd
        elif isinstance(other, BitData) and other._q_enabled:
            # if the other thing is a bitdata, but the other bitdata is queued
            # then try to turn this bitdata in to a queued bitdata
            # maximum creation efficiency
            self.Queue([(QOps.op, num_op, other, rightside)])
            return self

        if self.BitSize == 0 and self.truncate:
            raise Exception("Can't perform numeric operations on an empty BitData")

        if isinstance(other, BitData) and self.truncate:
            if other.BitSize == 0:
                raise Exception("Can't perform numeric operations on an empty BitData")
            bitlen = max(self.BitSize, other.BitSize)
            val_b = long(other)
        else:
            bitlen = self.BitSize
            val_b = long(other)

        if rightside:
            return BitData.CreateFromInt(bitlen, num_op(val_b, self._value), truncate=self.truncate)
        else:
            return BitData.CreateFromInt(bitlen, num_op(self._value, val_b), truncate=self.truncate)

    def __add__(self, other):
        return self.__apply_numeric_operation__(operator.add, other)

    def __radd__(self, other):
        return self.__apply_numeric_operation__(operator.add, other,True)

    def __sub__(self, other):
        return self.__apply_numeric_operation__(operator.sub, other)

    def __rsub__(self, other):
        return self.__apply_numeric_operation__(operator.sub, other, True)

    def __mul__(self, other):
        return self.__apply_numeric_operation__(operator.mul, other)

    def __rmul__(self, other):
        return self.__apply_numeric_operation__(operator.mul, other, True)

    def __div__(self, other):
        return self.__apply_numeric_operation__(operator.div, other)

    def __rdiv__(self, other):
        return self.__apply_numeric_operation__(operator.div, other, True)

    def __truediv__(self, other):
        raise TypeError("Not supported, use // for floor or cast to float if true division needed")

    def __rtruediv__(self, other):
        raise TypeError("Not supported, use // for floor or cast to float if true division needed")

    def __floordiv__(self, other):
        if PY2:
            return self.__apply_numeric_operation__(operator.div, other)
        else:
            return self.__apply_numeric_operation__(operator.floordiv, other)

    def __rfloordiv__(self, other):
        if PY2:
            return self.__apply_numeric_operation__(operator.div, other, True)
        else:
            return self.__apply_numeric_operation__(operator.floordiv, other, True)

    def __mod__(self, other):
        return self.__apply_numeric_operation__(operator.mod, other)

    def __rmod__(self, other):
        return self.__apply_numeric_operation__(operator.mod, other, True)

    def __rshift__(self, other):
        return self.__apply_numeric_operation__(operator.rshift, other)

    def __rrshift__(self, other):
        return self.__apply_numeric_operation__(operator.rshift, other, True)

    def rshift(self, other):        
        if self._q_enabled:
            self._q_operations.append((QOps.op, operator.irshift, other, False))
        else:
            self._value >>= other
        return self

    def __lshift__(self, other):
        return self.__apply_numeric_operation__(operator.lshift, other)

    def __rlshift__(self, other):
        return self.__apply_numeric_operation__(operator.lshift, other, True)

    def lshift(self, other):       
        if self._q_enabled:
            self._q_operations.append((QOps.op, operator.ilshift, other, False))
        else:
            self._value <<= other
        return self

    def __and__(self, other):
        return self.__apply_numeric_operation__(operator.and_, other)

    def __rand__(self, other):
        return self.__apply_numeric_operation__(operator.and_, other, True)

    def __xor__(self, other):
        return self.__apply_numeric_operation__(operator.xor, other)

    def __rxor__(self, other):
        return self.__apply_numeric_operation__(operator.xor, other, True)

    def __or__(self, other):
        return self.__apply_numeric_operation__(operator.or_, other)

    def __ror__(self, other):
        return self.__apply_numeric_operation__(operator.or_, other, True)
        #if isinstance(other, BitData):
        #    return self.__apply_numeric_operation__(operator.or_, other, self)
        #else:
        #    return self.__apply_numeric_operation__(operator.or_, self, other)

    def __invert__(self):
        if self._q_enabled:
            # maximum creation efficiency
            bd = BitData.__new__(BitData)
            bd._length = bd._value = None
            bd._q_value = self._q_value
            bd._q_length = self._q_length
            bd._q_enabled = True
            bd._q_operations = self._q_operations + [(QOps.invert)]
            bd._q_verifiers = copy(self._q_verifiers)
            bd.truncate = self.truncate
            return bd
        mask = self._compute_mask(self.BitSize)
        return BitData(self.BitSize, self._value ^ mask, truncate=self.truncate)
        
    def Invert(self, *args):
        """In-place invert of the bits using the given bit mask. Optionally, 
        lowerbit and numbits can be specified. Returns itself.
        
        Example:
            >>> bdata.Invert(mask)
            >>> bdata.Invert(lowerbit, numbits, mask)
        
        Raises:
            Exception: if an invalid number of arguments are passed
        """
        if len(args) == 1:
            mask = args[0]
        elif len(args) ==  3:
            lowerbit, numbits, mask = args
            mask &= (1 << numbits) - 1
            mask <<= lowerbit
        else:
            raise Exception("invalid number of arguments, either (mask) or (lowerbit, numbits, mask)")
        if self._q_enabled:
            self._q_operations.append((QOps.op, operator.ixor, mask, False))
        else:
            self._value ^= mask
        return self        
        
    invert = Invert

    def Concatenate(self, bitdata):
        """
        This is the equivalent of Append, but returns a new bitdata object.

        This is the same as: bitdata_new = (bitdata <<  len(bitdata_orig)) | bitdata_orig

        Args:
            bitdata : must be a bitdata object.

        Example:
            >>> bitdata_orig.Concatenate( bitdata )
        """
        if self._q_enabled: self._q_flush() # make sure we are not queuing first
        if self.BitSize == 0:
            result = BitData(bitdata.BitSize, long(bitdata), truncate=self.truncate)
        elif bitdata.BitSize == 0:
            result = BitData(self.BitSize, self._value, truncate=self.truncate)
        else:
            value = ( long(bitdata) << self.BitSize) |  self._value
            # OLD code was the othe way
            #value = (self._value << other_bitdata.BitSize) + bitdata._value
            result = BitData(self.BitSize + bitdata.BitSize, value, truncate=self.truncate|bitdata.truncate)
        return result

    def Append (self, bitdata):
        """
        Takes a bitdata and adds the value to the upper bits of this bitdata.

        This is the same as: bitdata_orig = (bitdata <<  len(bitdata_orig)) | bitdata_orig

        Args:
            bitdata : must be a bitdata object

        Returns:
            returns this same object (not a new one) so that operatoins can be 
            chained if needed

        Example:
            >>> b1 = ipc.BitData(4,0x5)
            >>> b2 = ipc.BitData(4,0xA)
            >>> b1.Append(b2)
            >>> b1
            [8b] 0xA5
            >>>
        """
        if not isinstance(bitdata,BitData):
            raise ValueError("bitdata must be a BitData object")
        if self._q_enabled: self._q_flush() # make sure we are not queuing first
        # dont use _value directly on the other bitdata in case it is not simply a number (due to queuing)
        self._value = (long(bitdata) << self.BitSize) |  self._value
        self.BitSize += bitdata.BitSize
        # return self, this is needed for chaining multiple appends in downstream tools
        return self

    # for pep8 compliance and compatibility with other BitData like objects
    append = Append

    def PopCount(self,startbit=0,length=None):
        """Count the number of set bits in this bitdata.

        Args:
            startbit (int) : bit to begin counting (default=0).
            length (int) : number of bits in this bitdata to count (default=all bits).

        Example:
            >>> bdata = ipc.BitData(8,0xF0)
            >>> bdata.PopCount()
            0x4
            >>> bdata.PopCount(5)
            0x3
            >>> bdata.PopCount(0,6)
            0x2
        """
        if startbit<0:
            raise ValueError("Startbit must be a positive number, use slicing if you need negative numbers")
        if self._q_enabled: self._q_flush() # make sure we are not queuing first
        # default to  bitSize-bitOffset if bitSize is None
        if length == None:
            length = self.BitSize - startbit
        subset = self[startbit:startbit+length-1]
        return bin(subset._value).count('1')

    def Deposit(self, startbit, length, value):
        """

        Changes the specified bits within this bitdata to have the new value.

        Args:
            startbit (int) : bit to begin depositing the new value (default=0).
            length (int) : number of bits in this bitdata to change (default=all bits).
            value (int) : value to insert.

        Note:
            This is very similar to using the slicing capabilities.

        Returns:
            This bitdata object is returned (not a new one).

        Raises:
            IndexError: if the starting bit plus the length is greater than the size of the BitData object.
        """
        if startbit<0:
            raise ValueError("Startbit must be a positive number, use slicing if you need negative numbers")
        if self._q_enabled: self._q_flush() # make sure we are not queuing first
        # this indirectly tests the start bit also, but we have to leave a
        # more general error message
        if startbit+length > self.BitSize:
            raise IndexError("Mismatch in start ({0}) or length ({1}) for the bits in this bitdata ({2})"\
                             .format(startbit,length,self.BitSize))
        self.__setslice__(startbit, startbit+length-1, value)
        return self

    def Extract (self, startbit=None, length=None):
        """

        Args:
            startbit (int) : bit to begin extracting (default=0).
            length (int) : number of bits in this bitdata to extract (default=all bits).

        Returns:
            A Python long.

        Note:
            This is very similar to using the slicing capabilities, but it returns
            a long instead of a BitData.

        Example:
            >>> bdata = BitData(32,0)
            >>> long(bdata[startbit:startbit+length-1]) == bdata.Extract(startbit,length)

        Raises:
            IndexError: if the starting bit plus the length is greater than the size of the BitData object.
        """
        if self._q_enabled: self._q_flush() # make sure we are not queuing first
        if startbit==None:
            startbit = 0
        if startbit<0:
            raise ValueError("Startbit must be a positive number, use slicing if you need negative numbers")
        if length==None:
            length = self.BitSize-startbit
        # this indirectly tests the start bit also, but we have to leave a
        # more general error message
        if startbit+length > self.BitSize:
            raise IndexError("Mismatch in start ({0}) or length ({1}) for the bits in this bitdata ({2})"\
                             .format(startbit,length,self.BitSize))
        # we duplicate bitdata code here to have a performance boost vs. creating
        # a bitdata object
        #
        # cast startbit and length to long in case they were bitdata objects
        # if they were bitdata objects then their bitsize can screw things up
        mask = (1<<long(length))-1
        value = (self._value >> long(startbit)) & long(mask)
        return value

    def Extract32(self, startbit):
        """

        This is the same as calling Extract(startbit, 32).
        See Extract help for detailed info on extract.
        """
        return self.Extract(startbit, 32)

    def Extract64(self, startbit):
        """

        This is the same as calling Extract(startbit, 64).
        See Extract help for detailed info on extract.
        """
        return self.Extract(startbit, 64)

    def _reverse_int(self, a, num_bits):
        rvrsd = 0
        for i in range( num_bits ):
            if (a & (1 << i)):
                rvrsd |= (1 << (num_bits - 1 - i))
        return rvrsd

    def Reverse(self, bitOffset=0, bitSize=None):
        '''
        Returns a new bitdata object with the specified bits reversed.

        Arguments:
          bitOffset (int): Offset to first bit in bitfield to reverse.
          bitSize (int)  : Size of bitfield to reverse.

        Returns:
          A new BitData containing the reversed value.

        Example:
          >>> myBitData = BitData(16, 0xAAAA)
          >>> rbitdata = myBitData.Reverse()
          >>> myBitData
          [16b] 0xAAAA
          >>> rbitdata
          [16b] 0x5555
        '''
        if bitOffset<0:
            raise ValueError("Startbit must be a positive number, use slicing if you need negative numbers")
        if self._q_enabled:
            if bitSize == None:
                # assume we have some sort of length..if not, then later on we are going to have to hope
                # that the _q_value is an operation receipt that gives us value and bitsize
                bitSize = self._q_length
            return BitData(self._q_length,
                           self._q_value,
                           queue=True,
                           operations=self._q_operations + [(QOps.reverse,bitOffset,bitSize)],
                           truncate=self.truncate,
                           )
        # default to  bitSize-bitOffset if bitSize is None
        if bitSize == None:
            bitSize = self.BitSize - bitOffset

        newbitdata = BitData(self.BitSize, self._value, truncate=self.truncate)
        # convert to number or bitsize will kill us
        oldval = long(newbitdata[bitOffset:bitOffset+bitSize-1])
        newval = 0
        for b in xrange(bitSize):
            # (oldval >> (bitSize-b)) = bitvalue
            # << b is to put in to correct location
            # I'd rather not have all on one line, but its probably faster
            oldbit = ((oldval >> (bitSize-b-1))&1)
            newval |=  ( oldbit << b )
        newbitdata[bitOffset:bitOffset+bitSize-1] = newval
        return newbitdata

    def verify_other(self,the_other):
        """Joins the list of delayed verification operations from the_other
        and adds it to current (self)"""
        if self._q_verifiers is None:
            self._q_verifiers= []        
        self._q_verifiers = self._q_verifiers + the_other._q_verifiers
  

    def verify_delayed(self, value, mask=None, compare_type="==", msg=""):
        """Stores the verify-instructions so they get executed later
        Supports the same format as traditional verify:
        value is the numeric value to compare to. mask is the bit mask to apply to the 
        BundleMap before the compare, defaults is no mask. compare_type is the compare 
        operation to perform, defaults to equality. msg is the message to be raised 
        with the exception if compare is False.
        
        Valid compare_type values are: "==", "!=", "<", "<=", ">", ">=".
        """       
        if self._q_verifiers is None:
            self._q_verifiers = []
        self._q_verifiers.append((self, value, mask, compare_type, msg))
   
    def _flush_verifiers(self):
        """Executes the list of delayed verifications(verify_delayed must be called previously)"""     
        operations = self._q_verifiers
        failures = []
        if operations:
            for nself, nvalue, nmask, ncompare, nmsg in operations:
                try:
                    self._run_verify(q_self = nself, value=nvalue, mask=nmask, compare_type=ncompare, msg=nmsg)
                except Exception as e:
                    failures.append(str(e))
                    
        if failures:
            string_of_failures = ""
            for failure in failures:
                string_of_failures = string_of_failures + failure + "\n"
            raise ValueError("Values didn't match, see {failures}".format(failures=string_of_failures))

    def verify(self, value, mask=None, compare_type="==", msg=""):
        """Compares the value of this BitData with the value given when the bundle is 
        executed. If the compare is False then a BundleVerifyFailed is raised.
        
        value is the numeric value to compare to. mask is the bit mask to apply to the 
        BundleMap before the compare, defaults is no mask. compare_type is the compare 
        operation to perform, defaults to equality. msg is the message to be raised 
        with the exception if compare is False.
        
        Valid compare_type values are: "==", "!=", "<", "<=", ">", ">=".

        Example evaluation would be:
            long(self) & mask < value == raise ValueError if this is false
        """        
        lookup = {
                    "==": operator.eq,
                    "!=": operator.ne,
                    "<=": operator.le,
                    ">=": operator.ge,
                    "<": operator.lt,
                    ">": operator.gt,
                }
        op = lookup[compare_type]
        # & with mask if it was given
        our_value = int(self)
        cmp_value = our_value & mask if mask is not None else our_value
        if not op(cmp_value, value):
            if mask is not None:
                raise ValueError("0x{our_value:x} & 0x{mask:x}) {op_type} 0x{expected_value} - is not True\nERROR: {msg}".format(
                    our_value = our_value,
                    mask=mask,
                    op_type=compare_type,
                    expected_value=value,
                    msg=msg
                    ))
            else:
                raise ValueError("0x{our_value:x} {op_type} 0x{expected_value} - is not True.\nERROR: {msg}".format(
                    our_value = our_value,
                    op_type=compare_type,
                    expected_value=value,
                    msg=msg
                    ))      

    def _run_verify(self, q_self, value, mask=None, compare_type="==", msg=""):
        """Compares the value of this BitData with the value given when the bundle is 
        executed. If the compare is False then a BundleVerifyFailed is raised.
        
        value is the numeric value to compare to. mask is the bit mask to apply to the 
        BundleMap before the compare, defaults is no mask. compare_type is the compare 
        operation to perform, defaults to equality. msg is the message to be raised 
        with the exception if compare is False.
        
        Valid compare_type values are: "==", "!=", "<", "<=", ">", ">=".

        Example evaluation would be:
            long(self) & mask < value == raise ValueError if this is false
        """        
        lookup = {
                    "==": operator.eq,
                    "!=": operator.ne,
                    "<=": operator.le,
                    ">=": operator.ge,
                    "<": operator.lt,
                    ">": operator.gt,
                }
        op = lookup[compare_type]
        # & with mask if it was given
        our_value = int(q_self)
        cmp_value = our_value & mask if mask is not None else our_value
        if not op(cmp_value, value):
            if mask is not None:
                raise ValueError("0x{our_value:x} & 0x{mask:x}) {op_type} 0x{expected_value:x} - is not True\nERROR: {msg}".format(
                    our_value = our_value,
                    mask=mask,
                    op_type=compare_type,
                    expected_value=value),
                    msg=msg,
                    )
            else:
                raise ValueError("0x{our_value:x} {op_type} 0x{expected_value:x} - is not True.\nERROR: {msg}".format(
                    our_value = our_value,
                    op_type=compare_type,
                    expected_value=value,
                    msg=msg)
                    )        


    def AddErrorCheck(self,lowerbit,numbits,check_value,message=""):
        """Queue up a check that will be done. If the specified bits do
        not match check_value, then an exception will be thrown with the message
        that is given.

        Args:
            lowerbit (int) : bottom bit to begin compare to.
            numbits (int) : number of bits to compare against.
            check_value (int) : value to check against.
            message (str): optional string to give for the exception that should display if
                        the compare fails.

        Example:

            >>> bdata = ipccli.BitData(32,0x1,queue=True)
            >>> bdata.AddErrorCheck(0,4,0,"compare to 0 will fail, causing an exception")
            >>> print bdata
            Traceback (most recent call last):
              File "<stdin>", line 1, in <module>
              File "C:\\Python27\\lib\\site-packages\\ipccli\\bitdata.py", line 570, in __repr__
                if self._q_enabled: self._q_flush() # make sure we are not queuing first
              File "C:\\Python27\\lib\\site-packages\\ipccli\\bitdata.py", line 362, in _q_flush
                exp_value,checkval))
            ValueError: compare to 0 will fail, causing an exception Expected: 0, Actual 1

        Raises:
            RuntimeError: if queuing is not enabled for the BitData object.
        """
        if not self._q_enabled:
            raise RuntimeError("Should only use AddErrorCheck while bitdata is in queuing mode")
        newcheck = (QOps.check, lowerbit,numbits,check_value,message)
        self._q_operations.append(newcheck)



