###############################################################################
# Copyright 2014 2017 Intel Corporation.
#
# The source code, information and material ("Material") contained herein is
# owned by Intel Corporation or its suppliers or licensors, and title to such
# Material remains with Intel Corporation or its suppliers or licensors. The
# Material contains proprietary information of Intel or its suppliers and
# licensors. The Material is protected by worldwide copyright laws and treaty
# provisions. No part of the Material may be used, copied, reproduced, modified,
# published, uploaded, posted, transmitted, distributed or disclosed in any way
# without Intel's prior express written permission.
#
# No license under any patent, copyright or other intellectual property rights
# in the Material is granted to or conferred upon you, either expressly, by
# implication, inducement, estoppel or otherwise. Any license under such
#
# intellectual property rights must be express and approved by Intel in writing.
# Unless otherwise agreed by Intel in writing, you may not remove or alter 
# this notice or any other notice embedded in Materials by Intel or Intel's 
# suppliers or licensors in any way.
###############################################################################

import weakref
#################################################################
# This requires the baseaccess to implement:
#  - arch_register command
#  - ishalted command
# code leveraged from the original itp-simulator
#################################################################

class Regs(object):
    """
    Requires the baseaccess.cmds to implement an "arch_register" command  for reading/writing 
    architecture registers
    
    Tab completion may show only a subset of what is supported, if an attribute is specified
    and it does not below to this class, then we will at least attempt to write the register
    """
    _attributes = ['baseaccess','did','_attributes','_visible_regs','_hidden_regs']
    _visible_regs = []
    
    # hidden supported regs
    # in case someone uses the LTB name, we will use that fine:
    _hidden_regs = []
    # We do this odd trick to avoid tab complete from causing a function call
    # we need to switch this to a generic "lazy execute" that will only do the function cal
    # after we have returned it... 
    # TODO: GET RID OF THIS ODDITY (and the getregisterobject call)
    registers=[]
    
    def __init__(self,baseaccess,did):
        self.baseaccess = weakref.proxy(baseaccess)
        self.did = did
        if self.baseaccess._version_check("OpenIPC", build=4125, error=False, revision=643859):
            import py2ipc
            try:
                self._populate_available_registers()
            except py2ipc.IPC_Error as err:
                if (err.code == py2ipc.IPC_Error_Codes.Not_Supported) or (err.code == py2ipc.IPC_Error_Codes.Not_Implemented):
                    # If the IPC implementation does not support getting available
                    # register names then assume IA registers
                    self._populate_ia_registers()
                elif (err.code == py2ipc.IPC_Error_Codes.Invalid_Device):
                    # If the thread was disabled then there is nothing to do
                    pass
                else:
                    raise
        else:
            self._populate_ia_registers()
        
    def getregisterobject(self,regname):
        # prevents pythonsv tab completion code from doing a read...
        # definitely cheap and fast way to do this...but something better
        # would be much more drawn out
        return None

    def _get_visible_regs(self):
        if not self._visible_regs:
            # If runcontrol has become enabled, and our list is empty, we should populate it
            self._populate_available_registers()
        return self._visible_regs

    def __getattr__(self,attr):
        if (attr in self._visible_regs) or (attr in self._hidden_regs):
            return self.baseaccess.cmds.arch_register(self.did,attr)
        if not self._visible_regs:
            # If runcontrol has become enabled, and our list is empty, we should populate it
            self._populate_available_registers()
            if (attr in self._visible_regs) or (attr in self._hidden_regs):
                return self.baseaccess.cmds.arch_register(self.did,attr)
        raise AttributeError("Missing attribute {0}".format(attr))

    def __setattr__(self,attr,value):
        if attr in self._attributes:
            object.__setattr__(self,attr,value)
        elif (attr in self._visible_regs) or (attr in self._hidden_regs):
            return self.baseaccess.cmds.arch_register(self.did,attr,value)
        else:
            if not self._visible_regs:
                # If runcontrol has become enabled, and our list is empty, we should populate it
                self._populate_available_registers()
                if (attr in self._visible_regs) or (attr in self._hidden_regs):
                    return self.baseaccess.cmds.arch_register(self.did,attr,value)
            raise AttributeError("Missing attribute {0}".format(attr))
        
    def __dir__(self):
        myattrs = list(self.__dict__.keys()) + list(self.__class__.__dict__.keys())
        if not self._visible_regs:
            # If runcontrol has become enabled, and our list is empty, we should populate it
            self._populate_available_registers()
        myattrs.extend(self._visible_regs)
        return myattrs

    def __contains__(self, reg):
        return reg in dir(self)

    def _populate_available_registers(self):
        import py2ipc
        devsrvc = py2ipc.IPC_GetService("Device")
        coregroup = devsrvc.GetCoreGroupIdForDevice(self.did)
        runcontrolconfig_service = py2ipc.IPC_GetService("RunControlConfiguration")
        if not runcontrolconfig_service.IsRunControlEnabled(coregroup):
            self._visible_regs = []
            return
        register_service = py2ipc.IPC_GetService("Register")
        self._visible_regs = register_service.GetRegisters(self.did)

    def _populate_ia_registers(self):
        self._visible_regs = []
        self._visible_regs += ['rax', 'eax', 'ax', 'ah', 'al']
        self._visible_regs += ['rbx', 'ebx', 'bx', 'bh', 'bl']
        self._visible_regs += ['rcx', 'ecx', 'cx', 'ch', 'cl']
        self._visible_regs += ['rdx', 'edx', 'dx', 'dh', 'dl']
        self._visible_regs += ['rsi', 'esi', 'si']
        self._visible_regs += ['rdi', 'edi', 'di']
        for segment in ['cs','ds','ss','fs','gs', 'es']:
            self._visible_regs += [ segment, segment+'bas',segment+'lim']
        self._visible_regs += ['rbp', 'rsp', 'rf', 'rip', 'eip', 'ip']
        self._visible_regs += ['csa']
        self._visible_regs += ['esp', 'sp']
        self._visible_regs += ['ebp', 'bp']
        self._visible_regs += ['ldtr', 'bp']
        self._visible_regs += ['eflags', 'rflags']
        self._visible_regs += ['efer']
        self._visible_regs += ['r%d' % i for i in range(8,16)]
        self._visible_regs += ['dr%d' % i for i in range(4)]
        self._visible_regs += ['dr6', 'dr7']
        self._visible_regs += ['cr0', 'cr2', 'cr3', 'cr4', 'cr8', 'xcr0']
        self._visible_regs += ['mm%d' % i for i in range(0,8)]
        self._visible_regs += ['emm%d' % i for i in range(0,8)]
        #self._visible_regs += ['fp%d' % i for i in range(0,8)]
        self._visible_regs += ['ldtas','ldtb','ldtl']
        self._visible_regs += ['gdtbas','gdtlim','ldtbas','ldtlim','idtbas','idtlim']
        self._visible_regs += ['xmm%d' % i for i in range(0,16)]
        self._visible_regs += ['zmm%d' % i for i in range(0,32)]
        self._visible_regs += ['ymm%d' % i for i in range(0,32)]
        self._visible_regs += ['k%d' % i for i in range(0,8)]

