###############################################################################
# Copyright 2014 2017 Intel Corporation.
#
# The source code, information and material ("Material") contained herein is
# owned by Intel Corporation or its suppliers or licensors, and title to such
# Material remains with Intel Corporation or its suppliers or licensors. The
# Material contains proprietary information of Intel or its suppliers and
# licensors. The Material is protected by worldwide copyright laws and treaty
# provisions. No part of the Material may be used, copied, reproduced, modified,
# published, uploaded, posted, transmitted, distributed or disclosed in any way
# without Intel's prior express written permission.
#
# No license under any patent, copyright or other intellectual property rights
# in the Material is granted to or conferred upon you, either expressly, by
# implication, inducement, estoppel or otherwise. Any license under such
#
# intellectual property rights must be express and approved by Intel in writing.
# Unless otherwise agreed by Intel in writing, you may not remove or alter 
# this notice or any other notice embedded in Materials by Intel or Intel's 
# suppliers or licensors in any way.
###############################################################################


# TODO: this is missing support for the LDT:XS:OFFSET case in some spots...
import re
import operator
from ._py2to3 import *
from .datatypes import BitData, Ord1 ,Ord2, Ord4, Ord8

class AddressType:
    """This is an enumlike class that provides some backwards compatibility with legacy tools"""
    implied = 0
    linear = 1
    physical = 2
    guestphysical = 3
    twofieldvirtual = 4
    threefieldvirtual = 5
    onefieldcodevirtual = 6
    onefielddatavirtual = 7
    onefieldstackvirtual = 8
    io = 9
    msr = 10
    
    Strings = { 
        implied : 'implied',
        linear : 'linear',
        physical : 'physical',
        guestphysical : 'guestphysical',
        twofieldvirtual : 'twofieldvirtual',
        threefieldvirtual : 'threefieldvirtual',
        onefieldcodevirtual : 'onefieldcodevirtual',
        onefielddatavirtual : 'onefielddatavirtual',
        onefieldstackvirtual : 'onefieldstackvirtual',
        io: 'io',
        msr: 'msr'
    }
    # reverse lookup
    StringsToType = {v:k for k,v in Strings.items()}
    
    #TODO: Fix this up
    ShortStrings = {
        implied : '',
        linear : 'L',
        physical : 'P',
        guestphysical : '',
        twofieldvirtual : '',
        threefieldvirtual : '',
        onefieldcodevirtual : '',
        onefielddatavirtual : '',
        onefieldstackvirtual : '',
        io: '',
        msr:''
    }
    
class Address(object):
    """
    Used to create an address object that can be passed to various cli functions. The address can be initialized
    using a string that supports multiple forms.

    Linear addresses:
        - "36L" == interpreted as 36 decimal or 0x20 hexadecimal **linear** address
        - "0x20L" == interpreted as 36 decimal or 0x20 hexadecimal **linear** address
    Physical addresses:
        - "36P" == interpreted as 36 decimal or 0x20 hexadecimal **physical** address
        - "0x20P" == interpreted as 36 decimal or 0x20 hexadecimal **physical** address
    Two field virtual (both are assumed hex):
        - "10:000" == segment: 0x10, offset: 0
    Thread field virtual (both are assumed hex):
        - "B0:10:000" == table: 0xB0, segment: 0x10, offset: 0
 
    """
    AddressType = AddressType
    # ####################
    # Constructors
    # ####################
    def __init__(self, *args, **kwargs):
        self._type = None
        self._raw_addr = None
        self._offset = None
        self._selector = None # not everyone gets to be a virtual address
        self._table = None # for 3 field
        # case where all our info is pass in via kwargs
        if len(kwargs) > 0:
            # quick check for per reasons...
            table = kwargs.pop("table", None)
            selector = kwargs.pop("segment", None)
            offset = kwargs.pop("offset", None)
            address_type = kwargs.pop("address_type", None)
            # make sure nothing is left..
            if len(kwargs) > 0:
                raise TypeError("Unexpected kwargs: %s"%list(kwargs.keys()))
            # --- now do initializations...
            # double check for valid values
            if None in [table,selector, offset, address_type]:
                raise ValueError("specified values cannot be None for kwarg initialization")
            # type checks
            if not isinstance(table, (long, int)):
                raise TypeError("Unexpected type for table")
            self._table = table
            if not isinstance(selector, (long, int)):
                raise TypeError("Unexpected type for selector")
            self._selector = selector
            if not isinstance(offset, (long, int)):
                raise TypeError("Unexpected type for offset")
            self._offset = offset
            if isinstance(address_type, string_types):
                self._type = AddressType.StringsToType[address_type]
            elif isinstance(address_type, (long, int)):
                if address_type not in AddressType.Strings:
                    raise ValueError("Unexpected value for address type")
                self._type = address_type
            # initialization done...return
            return

        if len(args) == 1:
            if isinstance(args[0],Address):
                self._init_copy(args[0])
            elif isinstance(args[0], BitData):
                self._init_from_bitdata(args[0])
            elif isinstance(args[0], (int,long)):
                self._init_from_int(args[0])
            elif isinstance(args[0],basestring):
                self._init_from_string(args[0])
        elif len(args) == 2 and \
            isinstance(args[0] , (int, long)) and \
            isinstance(args[1] , (int,long)):
            self._init_from_int_and_type(args[0], args[1])
        else:
            raise TypeError("no constructor matches given arguments")
 
    
    def _init_copy(self, other):
        self._type = other._type
        self._raw_addr = other._raw_addr
        self._offset = other._offset
        self._selector = other._selector
        self._table = other._table
        

    def _init_from_bitdata(self, bd_address):
        self._type = AddressType.implied
        self._offset = bd_address._value
        self._raw_addr = "0x%016x" % (self._offset)
        
 
    def _init_from_int(self, addr):
        self._type = AddressType.implied
        self._offset = addr
        self._raw_addr = "0x%016x" % (addr)
               
     
    def _init_from_int_and_type(self, addr, type):
        self._type = type
        self._offset = addr
        self._raw_addr = "0x%016x%s" % ( addr, AddressType.ShortStrings[type])
        
        
    def _init_from_string(self, addr_string):
        # go ahead and set this
        self._raw_addr = addr_string
        # check for $ addresses ($, $+1, $-2)
        m = re.match('\$((\+|-)[0-9]+)?', addr_string)
        if (m != None):
            offset = m.groups()[0]
            if offset == None:
                self._offset = 0
            else:
                self._offset = int(offset)
            self._type = AddressType.implied
            return
           
        # virtual address
        if (':') in addr_string:
            self._init_virtual_address_from_string(addr_string)
            return

        # linear, implied or physical
        m = re.match("(0x)?([0-9a-fA-F]+)([PLpl]?)(.*)", addr_string)
        if (m == None or len(m.groups()) != 4):
            raise ValueError("Address: Unable to parse address and type from: " + \
                             "'{0}'".format(addr_string))
        groups = m.groups()
        base = 10 if groups[0] == None else 16
        self._offset = int(groups[1], base)
        type_s = groups[2]
        # if the 3rd group is set, we got some sort of invalid number
        if groups[3] != '':
            raise ValueError("Address: Unable to parse address and type from: " + \
                             "'{0}'".format(addr_string))
        if type_s.upper() == 'P':
            self._type = AddressType.physical
        elif type_s.upper() == 'L':
            self._type = AddressType.linear
        else:
            self._type = AddressType.implied
        self._raw_addr = "0x%016x%s" % ( self._offset, AddressType.ShortStrings[self._type])
    
    def _init_virtual_address_from_string(self, addr_string):
        error_message = "Address: Unable to parse virtual address from: " + \
                        "'{0}'".format(addr_string)
        self._raw_addr = addr_string

        components = addr_string.split(":")
        if len(components) > 3:
            raise ValueError(error_message)
        elif len(components) == 3:
            self._table, self._selector, self._offset = components
            self._type = AddressType.threefieldvirtual
        elif len(components) == 2:
            self._selector, self._offset = components
            self._type = AddressType.twofieldvirtual
        # make sure we convert to number
        try:
            for convert in ["_table","_selector","_offset"]:
                if getattr(self,convert) != None:
                    to_int = int(  getattr(self,convert) , 16 )
                    setattr(self,convert,to_int)
        except:
            raise ValueError(error_message)
                
    # Constructors from ITP that we ignore
    #|  Void .ctor(System.Nullable`1[System.UInt32], UInt32, UInt64)
    #|  Void .ctor(System.Object)
    
    def __str__(self):
        # if raw addr is still valid, then
        # return the one we created with the object
        if self._raw_addr != None:
            return self._raw_addr
        # if its None, then the offset or type has been changed
        # at some point since we started.
        
        # depends on type:
        if self.Type in [AddressType.physical,
                         AddressType.linear,
                         AddressType.implied]:
            return "0x%016x%s" % ( self._offset, AddressType.ShortStrings[self._type])
        elif self.Type == AddressType.twofieldvirtual:
            return "0x%04x:0x%016x%s" % ( self._selector, self._offset, AddressType.ShortStrings[self._type])
        elif self.Type == AddressType.threefieldvirtual:
            return "0x%04x:0x%04x:0x%016x%s" % ( self._table, self._selector, self._offset, AddressType.ShortStrings[self._type])
        elif self.Type == AddressType.onefieldcodevirtual:
            return "cs:0x%016x%s"%self._offset
        elif self.Type == AddressType.onefielddatavirtual:
            return "ds:0x%016x%s"%self._offset
        elif self.Type == AddressType.onefieldstackvirtual:
            return "ss:0x%016x%s"%self._offset

    def __repr__(self):
        return self.__str__()

    # Properties
    @property
    def Type(self):
        return self._type

    @Type.setter
    def Type(self,value):
        if isinstance(value,basestring) and  (value not in self.AddressType.Strings.values()):
            raise ValueError("invalid Type value: %s"%value)
        elif isinstance(value,(int,long)) and (value not in self.AddressType.Strings.keys()):
            raise ValueError("Invalid Type, must be one of AddressType's")
        else:
            raise TypeError("invalid type for Type. type was: '%s', value was: '%s'"%(type(value),value))
        # if any attribute chanes we should clear out the raw_address and rebuild it
        self._raw_addr = None
        self._type = value

    @property
    def Offset(self):
        return self._offset

    @Offset.setter
    def Offset(self,value):
        if not isinstance(value,(int,long)):
            raise TypeError("Offset must be a number")
        # if any attribute chanes we should clear out the raw_address and rebuild it
        self._raw_addr = None
        self._offset = value

    @property
    def Table(self):
        return self._table

    @property
    def Segment(self):
        return self._selector

    @property
    def RawAddress(self):
        return self.__repr__()

    @RawAddress.setter
    def RawAddress(self,value):
        raise TypeError("this is read-only attribute")
    
    @property
    def TypeAsString(self):
        return AddressType.Strings[self._type]

    @TypeAsString.setter
    def TypeAsString(self,value):
        raise TypeError("this is read-only attribute")
  
    def __eq__(self,other):
        # if not an address class, return false
        if not isinstance(other,type(self)):
            return False
        return  self._type == other._type and \
                self._raw_addr == other._raw_addr and \
                self._offset == other._offset and \
                self._selector == other._selector and \
                self._table == other._table

    def __ne__(self,other):
        return  not self.__eq__(other)

    def __add__(self, other):
        newaddr = Address(self)
        newaddr.Offset = operator.add(newaddr.Offset,other)
        return newaddr

    def __sub__(self, other):
        newaddr = Address(self)
        newaddr.Offset = operator.sub(newaddr.Offset,other)
        return newaddr

    def __mul__(self, other):
        newaddr = Address(self)
        newaddr.Offset = operator.mul(newaddr.Offset,other)
        return newaddr

    def __lshift__(self, other):
        newaddr = Address(self)
        newaddr.Offset = operator.lshift(newaddr.Offset,other)
        return newaddr

    def __rshift__(self, other):
        newaddr = Address(self)
        newaddr.Offset = operator.rshift(newaddr.Offset,other)
        return newaddr

    def __and__(self, other):
        newaddr = Address(self)
        newaddr.Offset = operator.and_(newaddr.Offset,other)
        return newaddr

    def __or__(self, other):
        newaddr = Address(self)
        newaddr.Offset = operator.or_(newaddr.Offset,other)
        return newaddr

    def __xor__(self, other):
        newaddr = Address(self)
        newaddr.Offset = operator.xor(newaddr.Offset,other)
        return newaddr

    def __mod__(self, other):
        newaddr = Address(self)
        newaddr.Offset = operator.mod(newaddr.Offset,other)
        return newaddr


