#!/bin/bash

# Adjust any of the variables below
numOfHoursToPrepDrive=4
# Variables fioRuntime and fioRamptime are in seconds
fioRuntime=600
fioRamptime=600
numOfIterations=3
# If unsure of what CPU the NVMe SSDs are on, set "affinitized=false"
affinitized=true
# If NVMe SSDs are installed on 1st CPU then set "numaCpu=0"
# If NVMe SSDs are installed on 2nd CPU then set "numaCpu=1"
numaCpu=1
# Adjust variable "fioDevice" to applicable device like /dev/nvme0n1 for passthrough 
#     or /dev/md126 for VROC RAID volume
fioDevice=/dev/md126
if [[ "${fioDevice:0:7}" == "/dev/md" ]]; then
    # Variable "raidVolume" can be changed to whatever the RAID level is
    # (i.e. "RAID0" or "4DR0" or "RAID5" or "4DR5" or etc)
    raidVolume="RAID"
    # Set group_thread_cnt to 4 for RAID 5 volume if it exists
    if [ -f /sys/block/${fioDevice:5}/md/group_thread_cnt ]; then
        echo 4 > /sys/block/${fioDevice:5}/md/group_thread_cnt
    fi
else
    raidVolume="Passthrough"
fi

CreateFioConfigFile ()
{
    echo "[global]" > FioConfig.ini
    echo "time_based" >> FioConfig.ini
    echo "ioengine=libaio" >> FioConfig.ini
    echo "direct=1" >> FioConfig.ini
    echo "buffered=0" >> FioConfig.ini
    echo "norandommap" >> FioConfig.ini
    echo "refill_buffers" >> FioConfig.ini
    echo "stonewall" >> FioConfig.ini
    echo "disable_slat=1" >> FioConfig.ini
    echo "disable_lat=1" >> FioConfig.ini
    echo "disable_bw=1" >> FioConfig.ini
    echo "randrepeat=0" >> FioConfig.ini
    echo "thread" >> FioConfig.ini
    echo "unified_rw_reporting=0" >> FioConfig.ini
    echo "group_reporting" >> FioConfig.ini
    echo "do_verify=0" >> FioConfig.ini
    if [[ $affinitized == true ]]; then
        echo "cpus_allowed_policy=split" >> FioConfig.ini
        echo "cpus_allowed=${affinity}" >> FioConfig.ini
    fi
    if [[ "${SeqOrRandom}" == "Random" ]]; then
        echo "percentage_random=100" >> FioConfig.ini
        echo "random_generator=tausworthe64" >> FioConfig.ini
    fi
    echo "runtime=${tempRuntime}" >> FioConfig.ini
    echo "ramp_time=${tempRamptime}" >> FioConfig.ini
    echo "" >> FioConfig.ini
    echo "[${tempFioOutputStr}]" >> FioConfig.ini
    echo "size=100%" >> FioConfig.ini
    echo "filename=${fioDevice}" >> FioConfig.ini
    if [ $writePercent -eq 100 ] && [[ "${SeqOrRandom}" == "Seq" ]]; then
        echo "rw=write" >> FioConfig.ini
    elif [ $writePercent -eq 0 ] && [[ "${SeqOrRandom}" == "Seq" ]]; then
        echo "rw=read" >> FioConfig.ini
    elif [ $writePercent -eq 100 ] && [[ "${SeqOrRandom}" == "Random" ]]; then
        echo "rw=randwrite" >> FioConfig.ini
        echo "rwmixwrite=100" >> FioConfig.ini
    elif [ $writePercent -eq 0 ] && [[ "${SeqOrRandom}" == "Random" ]]; then
        echo "rw=randread" >> FioConfig.ini
        echo "rwmixread=100" >> FioConfig.ini
    else
        echo "rw=randrw" >> FioConfig.ini
        echo "rwmixwrite=${writePercent}" >> FioConfig.ini
    fi
    echo "bs=${blockSize}" >> FioConfig.ini
    echo "offset=0" >> FioConfig.ini
    echo "iodepth=${iodepth}" >> FioConfig.ini
    echo "iodepth_batch=${iodepth}" >> FioConfig.ini
    echo "numjobs=${workers}" >> FioConfig.ini
    sleep 1
}

RunFioTest ()
{
    iostatRuntime=$(( tempRuntime + tempRamptime - 2 ))
    tempFioOutputStr="${configName}_${blockSize}_${SeqOrRandom}_${writePercent}%Writes"
    tempFioOutputStr="${tempFioOutputStr}_IODepth${iodepth}_Workers${workers}_${iteration}"
    iostatFile="./${resultsFolder}/${tempFioOutputStr}_iostat.txt"
    CreateFioConfigFile
    cp -a ./FioConfig.ini ./${resultsFolder}/${tempFioOutputStr}.ini
    fioParams="--minimal --eta=never"
    fioParams="${fioParams} --output=./${resultsFolder}/${tempFioOutputStr}.csv"
    echo "$(date +%x_%T)__FioTest_${tempFioOutputStr}"
    iostat -tmyx -p ${fioDevice} 1 ${iostatRuntime} > ${iostatFile} &
    $numactlCmd fio ${fioParams} ./FioConfig.ini &
    wait $(pidof fio)
    # The commands below change the results from semicolon delimited to comma delimited
    sed -i -e 's/;/,/g' ./${resultsFolder}/${tempFioOutputStr}.csv
    sed -i -e 's/_/,/g' ./${resultsFolder}/${tempFioOutputStr}.csv
}


#######################################################################################

# Check if device is present.  If not, then exit script.
if [ ! -b $fioDevice ]; then
    echo "The device specified in the fioDevice variable is not present."
    echo "Please adjust this variable in this script and try again."
    exit 1
fi

# Stop irqbalance service (this may help reduce high standard deviations between iterations)
systemctl stop irqbalance

# Setting CPU affinity command and applicable CPUs if variable "affinitized=true"
numactlCmd=""
if [[ $affinitized == true ]]; then
    numactlCmd="numactl --cpunodebind=$numaCpu"
    numOfNumaNodes=$(( $( lscpu | grep "NUMA node(s):" | rev | awk '{print $1}' | rev ) ))
    numaNodesStrings=( $( lscpu | grep "NUMA node" | grep "CPU" | awk '{print $2}' ) )
    if [ $numaCpu -ge $numOfNumaNodes ] && [ $numOfNumaNodes -ge 1 ]; then
        numaCpu=$(( numOfNumaNodes - 1 ))
    fi
    if [ $numaCpu -lt $numOfNumaNodes ]; then
        affinity=$( lscpu | grep "NUMA ${numaNodesStrings[$numaCpu]}" | awk '{print $4}' )
    fi
fi

configName="VROC-${raidVolume}"
timestamp=$(date +%Y_%m%d_%H%M)
resultsFolder=${timestamp}_${configName}
mkdir -p ./${resultsFolder}

# Start of sequential workload testing section
SeqOrRandom=Seq
for blockSize in 128k; do
    workers=1
    # Start of pre-conditioning section
    iteration=Prep; writePercent=100; iodepth=128
    tempRuntime=$(( numOfHoursToPrepDrive * 60 * 60 )); tempRamptime=30
    RunFioTest
    # End of pre-conditioning section
    for (( iteration=1; iteration<=$numOfIterations; iteration++ )); do
        for writePercent in 100 0; do
            # Modify the iodepths below based on your needs
            for iodepth in 128; do
                tempRuntime=${fioRuntime}; tempRamptime=${fioRamptime}
                RunFioTest
            done
        done
    done
done
# End of sequential workload testing section

# Start of random workload testing section
SeqOrRandom=Random
for blockSize in 4k; do
    # Start of pre-conditioning section
    iteration=Prep; writePercent=100; workers=8; iodepth=32
    tempRuntime=$(( numOfHoursToPrepDrive * 60 * 60 )); tempRamptime=30
    RunFioTest
    # End of pre-conditioning section
    for (( iteration=1; iteration<=$numOfIterations; iteration++ )); do
        for writePercent in 100 30 0; do
            for workers in 16 8; do
                # Modify the iodepths below based on your needs
                if [ $workers -eq 16 ]; then
                    iodepths=(256 128 64 32 16 8)
                elif [ $workers -eq 8 ]; then
                    iodepths=(256 128 64 32 16 8)
                elif [ $workers -eq 4 ]; then
                    iodepths=(256 128 64)
                else
                    iodepths=(256 128 64 32 16 8)
                fi
                for iodepth in "${iodepths[@]}"; do
                    tempRuntime=${fioRuntime}; tempRamptime=${fioRamptime}
                    RunFioTest
                done
            done
        done
    done
done
# End of random workload testing section

# Create the header for the CSV file
sourceString="fio_version,jobname,groupid,"
replacementStr="fio_version,jobname,blocksize,SeqOrRandom,write_percent"
replacementStr="${replacementStr},iodepth,threads,iteration,groupid,"
fioHeaderArray=( $( man fio | grep "terse_version" | sed "s/;/,/g" | sed "s%${sourceString}%${replacementStr}%g" ) )
fioHeader="${fioHeaderArray[@]}"
# Combining all results into one file (excluding any files from pre-conditioning)
echo "${fioHeader}" > ./${resultsFolder}/AllFioResults.csv
cat ./${resultsFolder}/*IODepth*[0-9].csv >> ./${resultsFolder}/AllFioResults.csv
# Convert some of the values to integers for the CSV file
sourceString="%Writes"
sed -i -e "s/${sourceString}//g" ./${resultsFolder}/AllFioResults.csv
sourceString="IODepth"
sed -i -e "s/${sourceString}//g" ./${resultsFolder}/AllFioResults.csv
sourceString="Workers"
sed -i -e "s/${sourceString}//g" ./${resultsFolder}/AllFioResults.csv

# Start irqbalance service back up
systemctl start irqbalance
