/* Copyright (c) <2012>, Intel Corporation
 *
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 *
 *  - Redistributions of source code must retain the above copyright notice, 
 *    this list of conditions and the following disclaimer.
 *  - Redistributions in binary form must reproduce the above copyright notice, 
 *    this list of conditions and the following disclaimer in the documentation 
 *    and/or other materials provided with the distribution.
 *  - Neither the name of Intel Corporation nor the names of its contributors 
 *    may be used to endorse or promote products derived from this software 
 *    without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <assert.h>
#include <memory.h>
#include <stdio.h>
#include <stdlib.h>

#include "opengl.h"
#include "system.h"
#include "shader.h"

///////////////////////////////////////////////////////////////////////////////////////////////////
// Compilers the give shader type (vertex or fragment) and returns a handle
GLuint compile_shader( GLenum type, const char* shader_data, int shader_size )
{ 
    // Create the shader 
    GLuint shader_handle = glCreateShader( type );

    if( shader_handle == 0 )
    {
        return 0;
    }

    // Set up string for #define based on Type 
    // Must keep the same length for the hard coded 17 for string size (see ShaderStringLengths)
    char* define_vertex_shader   = "#define VERTEX  \n";
    char* define_fragment_shader = "#define FRAGMENT\n";
    char* shader_define = ( type == GL_VERTEX_SHADER ) ? define_vertex_shader : define_fragment_shader;

    // Set up string table (first string is the #define for this Type and then the shader program)
    const char* shader_strings[2] = { shader_define, shader_data };
    GLint shader_strings_length[2] = { 17, shader_size };

    // Load the shader source
    glShaderSource( shader_handle, 2, shader_strings, shader_strings_length );
   
    // Compile the shader
    glCompileShader( shader_handle );

    // Check the compiler status
    GLint compile_status;
    glGetShaderiv( shader_handle, GL_COMPILE_STATUS, &compile_status );

    if( !compile_status )
    {
        GLint info_length = 0;

        glGetShaderiv( shader_handle, GL_INFO_LOG_LENGTH, &info_length );
      
        if( info_length > 1 )
        {
            char* info_log = (char*)malloc( sizeof(char) * info_length );

            glGetShaderInfoLog( shader_handle, info_length, NULL, info_log );
            
            char error_string[1024];
            sprintf( error_string, "Error lcompiling shader:\n%s\n", info_log ); 
            Log( error_string );
         
            free( info_log );
        }

        glDeleteShader( shader_handle );

        // Failed to compile the shadedrs
        assert( 0 );
        return 0;
    }

    return shader_handle;
}


///////////////////////////////////////////////////////////////////////////////////////////////////
// Links the shaders and create the final program
GLuint link_shader( GLuint vertex_shader_handle, GLuint fragment_shader_handle )
{ 
    GLuint shader_program = glCreateProgram();
    if( shader_program == 0 )
    {
        // Failed to create a program handle
        assert( 0 );
        return 0;
    }

    // Associate the shaders with the program
    glAttachShader( shader_program, vertex_shader_handle );
    glAttachShader( shader_program, fragment_shader_handle );

    // Link the program
    glLinkProgram( shader_program );

    // Check the link status
    GLint  linker_status;
    glGetProgramiv( shader_program, GL_LINK_STATUS, &linker_status );

    if( !linker_status )
    {
        // Failed to link, get the error string
        GLint info_length = 0;

        glGetProgramiv( shader_program, GL_INFO_LOG_LENGTH, &info_length );
      
        if( info_length > 1 )
        {
            char* info_log = (char*)malloc( sizeof(char) * info_length );

            glGetProgramInfoLog ( shader_program, info_length, NULL, info_log );
            
            char error_string[1024];
            sprintf( error_string, "Error linking program:\n%s\n", info_log ); 
            Log( error_string );
         
            free( info_log );
        }

        glDeleteProgram( shader_program );

        // Failed to link the shadedrs
        assert( 0 );
        return 0;
    }

    return shader_program;
}


///////////////////////////////////////////////////////////////////////////////////////////////////
// Loads, compiles, and links the shader for the given file and returns the programs id
GLuint load_shader( const char* shader_file_name )
{   
    GLuint shader_program;

    // Open files
    char* file_data = NULL;
    unsigned int file_size = 0;
    
    ReadFile( shader_file_name, &file_data, &file_size );

    // Load and compile the vertex shader
    GLuint vertex_shader = compile_shader( GL_VERTEX_SHADER, file_data, file_size );
    if( vertex_shader == 0 )
    {
        // Couldn't load the shader for the give file
        assert( 0 ); 
        return 0;
    }

    // Load and compiler the fragment shader
    GLuint fragment_shader = compile_shader( GL_FRAGMENT_SHADER, file_data, file_size );
    if( fragment_shader == 0 )
    {
        glDeleteShader( vertex_shader );

        // Couldn't load the shader for the give file
        assert( 0 );
        return 0;
    }

    // Create the program handle
    shader_program = link_shader( vertex_shader, fragment_shader );
    if( shader_program == 0 )
    {
        // Failed to link the program
        assert( 0 );
        return 0;
    }

    // Free the shader resources
    glDeleteShader( vertex_shader );
    glDeleteShader( fragment_shader );

    return shader_program;
}

