/* Copyright (c) <2012>, Intel Corporation
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *  - Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *  - Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *  - Neither the name of Intel Corporation nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <assert.h>
#include <memory.h>
#include <stdio.h>
#include <stdlib.h>

#include "opengl.h"
#include "system.h"
#include "CPUTMath.h"
#include "shader.h"
#include "model.h"

model_t* create_triangle_model( void )
{
    model_t* _triangle_model = (model_t*)malloc(sizeof(model_t));

    // triangle vertices
    //
    //      0
    //     / \
    //    /   \
    //   /     \
    //  1-------2
    //
    const vertex_t _triangle_vertices[] =
    {
        // position     //      // normal     //      // tex coords //
        { {  0.0f,  1.0f,  0.0f }, { 0.0f, 0.0f, 1.0f }, { 0.5f, 1.0f } },  // 0
        { { -1.0f, -1.0f,  0.0f }, { 0.0f, 0.0f, 1.0f }, { 0.0f, 0.0f } },  // 1
        { {  1.0f, -1.0f,  0.0f }, { 0.0f, 0.0f, 1.0f }, { 1.0f, 0.0f } },  // 2
    };

    const unsigned int _triangle_indices[] =
    {
        0, 1, 2
    };

    _triangle_model->num_vertices = 3;
    _triangle_model->num_indices = 3;

    unsigned int _size_of_vertices = sizeof(vertex_t) * _triangle_model->num_vertices;
    _triangle_model->vertices = (vertex_t*)malloc(_size_of_vertices);
    memcpy(_triangle_model->vertices, _triangle_vertices, _size_of_vertices);

    unsigned int _size_of_indices = sizeof(GLuint) * _triangle_model->num_indices;
    _triangle_model->indices = (GLuint*)malloc(_size_of_indices);
    memcpy(_triangle_model->indices, _triangle_indices, _size_of_indices);

    // generate VBOs
    glGenBuffers(1, &_triangle_model->vertex_buffer);
    glBindBuffer(GL_ARRAY_BUFFER, _triangle_model->vertex_buffer);
    glBufferData(GL_ARRAY_BUFFER, _size_of_vertices, _triangle_model->vertices, GL_STATIC_DRAW);

    glGenBuffers(1, &_triangle_model->index_buffer);
    glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, _triangle_model->index_buffer);
    glBufferData(GL_ELEMENT_ARRAY_BUFFER, _size_of_indices, _triangle_model->indices, GL_STATIC_DRAW);

    // reset default bindings
    glBindBuffer(GL_ARRAY_BUFFER, 0);
    glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, 0);


    return _triangle_model;
}

model_t* create_quad_model( unsigned int is_interleaved )
{
    model_t* _quad_model = (model_t*)malloc(sizeof(model_t));
    _quad_model->is_interleaved = is_interleaved;

    // quad vertices
    //
    //  1-------0
    //  |       |
    //  |       |
    //  |       |
    //  2-------3
    //
    // Interleaved vertex data
    const vertex_t _quad_vertices[] =
    {
        { {  1.0f,  1.0f,  0.0f }, { 0.0f, 0.0f, 1.0f }, { 1.0f, 1.0f } },
        { { -1.0f,  1.0f,  0.0f }, { 0.0f, 0.0f, 1.0f }, { 0.0f, 1.0f } },
        { { -1.0f, -1.0f,  0.0f }, { 0.0f, 0.0f, 1.0f }, { 0.0f, 0.0f } },
        { {  1.0f, -1.0f,  0.0f }, { 0.0f, 0.0f, 1.0f }, { 1.0f, 0.0f } },
    };

    // Non-interleaved vertex data
    const float3 _quad_positions[] =
    {
        {  1.0f,  1.0f, 0.0f },
        { -1.0f,  1.0f, 0.0f },
        { -1.0f, -1.0f, 0.0f },
        {  1.0f, -1.0f, 0.0f },
    };

    const float3 _quad_normals[] =
    {
        { 0.0f, 0.0f, 1.0f },
        { 0.0f, 0.0f, 1.0f },
        { 0.0f, 0.0f, 1.0f },
        { 0.0f, 0.0f, 1.0f },
    };

    const float2 _quad_texcoords[] =
    {
        { 1.0f, 1.0f },
        { 0.0f, 1.0f },
        { 0.0f, 0.0f },
        { 1.0f, 0.0f },
    };

    // indices
    const unsigned int _quad_indices[] =
    {
        0, 1, 2,
        0, 2, 3,
    };

    _quad_model->num_vertices = 4;
    _quad_model->num_indices = 6;

    unsigned int _size_of_indices = sizeof(GLuint) * _quad_model->num_indices;
    _quad_model->indices = (GLuint*)malloc(_size_of_indices);
    memcpy(_quad_model->indices, _quad_indices, _size_of_indices);

    if( _quad_model->is_interleaved )
    {
        unsigned int _size_of_vertices = sizeof(vertex_t) * _quad_model->num_vertices;
        _quad_model->vertices = (vertex_t*)malloc(_size_of_vertices);
        memcpy(_quad_model->vertices, _quad_vertices, _size_of_vertices);

        // generate VBOs
        glGenBuffers(1, &_quad_model->vertex_buffer);
        glBindBuffer(GL_ARRAY_BUFFER, _quad_model->vertex_buffer);
        glBufferData(GL_ARRAY_BUFFER, _size_of_vertices, _quad_model->vertices, GL_STATIC_DRAW);
    }
    else
    {
        unsigned int _size_of_vertices = sizeof(float3) * _quad_model->num_vertices;

        glGenBuffers(1, &_quad_model->positions_buffer);
        glBindBuffer(GL_ARRAY_BUFFER, _quad_model->positions_buffer);
        glBufferData(GL_ARRAY_BUFFER, _size_of_vertices, _quad_positions, GL_STATIC_DRAW);

        _size_of_vertices = sizeof(float3) * _quad_model->num_vertices;

        glGenBuffers(1, &_quad_model->normals_buffer);
        glBindBuffer(GL_ARRAY_BUFFER, _quad_model->normals_buffer);
        glBufferData(GL_ARRAY_BUFFER, sizeof(float3) * _quad_model->num_vertices, _quad_normals, GL_STATIC_DRAW);

        _size_of_vertices = sizeof(float2) * _quad_model->num_vertices;

        glGenBuffers(1, &_quad_model->texcoords_buffer);
        glBindBuffer(GL_ARRAY_BUFFER, _quad_model->texcoords_buffer);
        glBufferData(GL_ARRAY_BUFFER, sizeof(float2) * _quad_model->num_vertices, _quad_texcoords, GL_STATIC_DRAW);
    }

    glGenBuffers(1, &_quad_model->index_buffer);
    glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, _quad_model->index_buffer);
    glBufferData(GL_ELEMENT_ARRAY_BUFFER, _size_of_indices, _quad_model->indices, GL_STATIC_DRAW);

    // reset default bindings
    glBindBuffer(GL_ARRAY_BUFFER, 0);
    glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, 0);

    return _quad_model;
}

model_t* create_cube_model( void )
{
    model_t* _cube_model = NULL;
    _cube_model = (model_t*)malloc(sizeof(model_t));

    // cube vertices
    //
    //               9---------8
    //              /         /
    //             /         /
    //            10-------11
    //
    //    13         20-------21         4       front: {  0,  1,  2 }, {  0,  2,  3 }
    //    /|         |         |        /|       right: {  4,  5,  6 }, {  4,  6,  7 }
    //   / |         |         |       / |         top: {  8,  9, 10 }, {  8, 10, 11 }
    //  12 |      1---------0  |      5  |        left: { 12, 13, 14 }, { 12, 14, 15 }
    //  |  |      |  |      |  |      |  |      bottom: { 16, 17, 18 }, { 16, 18, 19 }
    //  | 14      |  23-----|-22      |  7        back: { 20, 21, 22 }, { 20, 22, 23 }
    //  | /       |         |         | /
    //  |/        |         |         |/
    //  15        2---------3         6
    //
    //               18-------19
    //              /         /
    //             /         /
    //            17-------16
    //
    const vertex_t _cube_vertices[] =
    {
        // FRONT
        { {  0.5f,  0.5f,  0.5f }, { 0.0f, 0.0f, 1.0f }, { 1.0f, 1.0f } },  // 0
        { { -0.5f,  0.5f,  0.5f }, { 0.0f, 0.0f, 1.0f }, { 0.0f, 1.0f } },  // 1
        { { -0.5f, -0.5f,  0.5f }, { 0.0f, 0.0f, 1.0f }, { 0.0f, 0.0f } },  // 2
        { {  0.5f, -0.5f,  0.5f }, { 0.0f, 0.0f, 1.0f }, { 1.0f, 0.0f } },  // 3

        // RIGHT
        { {  0.5f,  0.5f, -0.5f }, { 1.0f, 0.0f, 0.0f }, { 1.0f, 1.0f } },  // 4
        { {  0.5f,  0.5f,  0.5f }, { 1.0f, 0.0f, 0.0f }, { 0.0f, 1.0f } },  // 5
        { {  0.5f, -0.5f,  0.5f }, { 1.0f, 0.0f, 0.0f }, { 0.0f, 0.0f } },  // 6
        { {  0.5f, -0.5f, -0.5f }, { 1.0f, 0.0f, 0.0f }, { 1.0f, 0.0f } },  // 7

        // TOP
        { {  0.5f,  0.5f, -0.5f }, { 0.0f, 1.0f, 0.0f }, { 1.0f, 1.0f } },  // 8
        { { -0.5f,  0.5f, -0.5f }, { 0.0f, 1.0f, 0.0f }, { 0.0f, 1.0f } },  // 9
        { { -0.5f,  0.5f,  0.5f }, { 0.0f, 1.0f, 0.0f }, { 0.0f, 0.0f } },  // 10
        { {  0.5f,  0.5f,  0.5f }, { 0.0f, 1.0f, 0.0f }, { 1.0f, 0.0f } },  // 11

        // LEFT
        { { -0.5f,  0.5f,  0.5f }, { -1.0f, 0.0f, 0.0f }, { 1.0f, 1.0f } },  // 12
        { { -0.5f,  0.5f, -0.5f }, { -1.0f, 0.0f, 0.0f }, { 0.0f, 1.0f } },  // 13
        { { -0.5f, -0.5f, -0.5f }, { -1.0f, 0.0f, 0.0f }, { 0.0f, 0.0f } },  // 14
        { { -0.5f, -0.5f,  0.5f }, { -1.0f, 0.0f, 0.0f }, { 1.0f, 0.0f } },  // 15

        // BOTTOM
        { {  0.5f, -0.5f,  0.5f }, { 0.0f, -1.0f, 0.0f }, { 1.0f, 1.0f } },  // 16
        { { -0.5f, -0.5f,  0.5f }, { 0.0f, -1.0f, 0.0f }, { 0.0f, 1.0f } },  // 17
        { { -0.5f, -0.5f, -0.5f }, { 0.0f, -1.0f, 0.0f }, { 0.0f, 0.0f } },  // 18
        { {  0.5f, -0.5f, -0.5f }, { 0.0f, -1.0f, 0.0f }, { 1.0f, 0.0f } },  // 19

        // BACK
        { { -0.5f,  0.5f, -0.5f }, { 0.0f, 0.0f, -1.0f }, { 1.0f, 1.0f } },  // 20
        { {  0.5f,  0.5f, -0.5f }, { 0.0f, 0.0f, -1.0f }, { 0.0f, 1.0f } },  // 21
        { {  0.5f, -0.5f, -0.5f }, { 0.0f, 0.0f, -1.0f }, { 0.0f, 0.0f } },  // 22
        { { -0.5f, -0.5f, -0.5f }, { 0.0f, 0.0f, -1.0f }, { 1.0f, 0.0f } },  // 23
    };

    const unsigned int _cube_indices[] =
    {
         0,  1,  2,   0,  2,  3,  // front
         4,  5,  6,   4,  6,  7,  // right
         8,  9, 10,   8, 10, 11,  // top
        12, 13, 14,  12, 14, 15,  // left
        16, 17, 18,  16, 18, 19,  // bottom
        20, 21, 22,  20, 22, 23,  // back
    };

    _cube_model->num_vertices = sizeof(_cube_vertices) / sizeof(vertex_t);
    _cube_model->num_indices = sizeof(_cube_indices) / sizeof(GLuint);

    unsigned int _size_of_vertices = sizeof(vertex_t) * _cube_model->num_vertices;
    _cube_model->vertices = (vertex_t*)malloc(_size_of_vertices);
    memcpy(_cube_model->vertices, _cube_vertices, _size_of_vertices);

    unsigned int _size_of_indices = sizeof(GLuint) * _cube_model->num_indices;
    _cube_model->indices = (GLuint*)malloc(_size_of_indices);
    memcpy(_cube_model->indices, _cube_indices, _size_of_indices);

    // generate VBOs
    glGenBuffers(1, &_cube_model->vertex_buffer);
    glBindBuffer(GL_ARRAY_BUFFER, _cube_model->vertex_buffer);
    glBufferData(GL_ARRAY_BUFFER, _size_of_vertices, _cube_model->vertices, GL_STATIC_DRAW);

    glGenBuffers(1, &_cube_model->index_buffer);
    glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, _cube_model->index_buffer);
    glBufferData(GL_ELEMENT_ARRAY_BUFFER, _size_of_indices, _cube_model->indices, GL_STATIC_DRAW);
 // reset default bindings glBindBuffer(GL_ARRAY_BUFFER, 0);
    glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, 0);

    return _cube_model;
}

model_t* load_mesh( const char* mesh_file_path )
{
    char* file_data = NULL;
    unsigned int file_size = 0;

    ReadFile( mesh_file_path, &file_data, &file_size );

    mesh_data_t* mesh_data = ( mesh_data_t* )file_data;

    model_t* model = (model_t*)malloc(sizeof(model_t));

    model->num_vertices = mesh_data->vertex_count;
    model->num_indices = mesh_data->index_count;

    unsigned int _size_of_vertices = mesh_data->vertex_size * mesh_data->vertex_count;
    unsigned int _size_of_indices = mesh_data->index_size * mesh_data->index_count;

    vertex_t* vertices = ( vertex_t * )( file_data + sizeof(mesh_data_t) );
    int* indices = ( int* )( file_data + sizeof(mesh_data_t) + _size_of_vertices );

    // generate VBOs
    glGenBuffers(1, &model->vertex_buffer);
    glBindBuffer(GL_ARRAY_BUFFER, model->vertex_buffer);
    glBufferData(GL_ARRAY_BUFFER, _size_of_vertices, vertices, GL_STATIC_DRAW);

    glGenBuffers(1, &model->index_buffer);
    glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, model->index_buffer);
    glBufferData(GL_ELEMENT_ARRAY_BUFFER, _size_of_indices, indices, GL_STATIC_DRAW);

    // reset default bindings
    glBindBuffer(GL_ARRAY_BUFFER, 0);
    glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, 0);

    return model;
}

static GLuint shader_program = 0;
static GLint  position_attrib = 0;
static GLint  normal_attrib = 0;
static GLint  texcoord_attrib = 0;

void draw_model( model_t* model, float4x4* view_projection_matrix, float4x4* world_matrix )
{
    // Load the program if it's not ready
    if( !shader_program )
    {
        shader_program = load_shader( "assets/shaders/model.shader" );

        position_attrib = glGetAttribLocation( shader_program, "a_position" );
        normal_attrib   = glGetAttribLocation( shader_program, "a_normal" );
        texcoord_attrib = glGetAttribLocation( shader_program, "a_texCoord" );
    }

	// Set up shader model program
	glUseProgram( shader_program );

	// Bind VBO buffers and attributes
	glBindBuffer( GL_ARRAY_BUFFER, model->vertex_buffer );
	glBindBuffer( GL_ELEMENT_ARRAY_BUFFER, model->index_buffer );

	GLuint stride = sizeof(vertex_t);

	GLuint offset = 0;
	glEnableVertexAttribArray( position_attrib );
	glVertexAttribPointer( position_attrib, 3, GL_FLOAT, GL_FALSE, stride, (const void*)offset );

	offset += 3 * sizeof( GLfloat );
	glEnableVertexAttribArray( normal_attrib );
	glVertexAttribPointer( normal_attrib, 3, GL_FLOAT, GL_FALSE, stride, (const void*)offset );

    offset += 3 * sizeof( GLfloat );
    glEnableVertexAttribArray( texcoord_attrib );
    glVertexAttribPointer( texcoord_attrib, 2, GL_FLOAT, GL_FALSE, stride, (const void*)offset );

	// Set the MVP matrix
    GLint view_proj_matrix_loc = glGetUniformLocation( shader_program, "u_view_proj_matrix" );
    glUniformMatrix4fv( view_proj_matrix_loc, 1, GL_FALSE, (const GLfloat*)view_projection_matrix );

    // Set the model view transpose
    GLint world_matrix_loc = glGetUniformLocation( shader_program, "u_world_matrix" );
    glUniformMatrix4fv( world_matrix_loc, 1, GL_FALSE, (const GLfloat*)world_matrix );

    // Hardcode a light direction
    float3 light_direction;
    light_direction.x = 0.0f;
    light_direction.y = 0.0f;
    light_direction.z = 1.0f;
    GLint lightLoc = glGetUniformLocation( shader_program, "u_light_dir" );
    glUniform3fv( lightLoc, 1, (const GLfloat*)&light_direction );

	// Draw model
	glDrawElements( GL_TRIANGLES, model->num_indices, GL_UNSIGNED_INT, 0 );

    // Unbind buffer before we return
    glBindBuffer( GL_ARRAY_BUFFER, 0 );
    glBindBuffer( GL_ELEMENT_ARRAY_BUFFER, 0 );
}

